/* PSPP - computes sample statistics.
   Copyright (C) 1997, 1998 Free Software Foundation, Inc.
   Written by Ben Pfaff <blp@gnu.org>.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA. */

#include <config.h>
#include <ctype.h>
#include <assert.h>
#include <limits.h>
#include "common.h"
#include "str.h"

a_string *
aa_strcpy (a_string * d, const a_string * s)
{
  d->l = s->l;
  memcpy (d->s, s->s, s->l);
  return d;
}

a_string *
ab_strcpy (a_string * d, const b_string * s)
{
  d->l = s->l;
  memcpy (d->s, s->s, s->l);
  return d;
}

a_string *
ac_strcpy (a_string * d, const c_string * s)
{
  d->l = LOAD_2 (&s->l1);
  memcpy (d->s, s->s, d->l);
  return d;
}

b_string *
ba_strcpy (b_string * d, const a_string * s)
{
  d->l = s->l;
  memcpy (d->s, s->s, s->l);
  return d;
}

b_string *
bb_strcpy (b_string * d, const b_string * s)
{
  d->l = s->l;
  memcpy (d->s, s->s, s->l);
  return d;
}

c_string *
ca_strcpy (c_string * d, const a_string * s)
{
  assert (s->l <= 65535);
  STORE_2 (&d->l1, s->l);
  memcpy (d->s, s->s, s->l);
  return d;
}

a_string *
aa_strdup (const a_string * s)
{
  a_string *d = xmalloc (sizeof (a_string));
  d->l = s->l;
  d->s = xmalloc (s->l);
  memcpy (d->s, s->s, s->l);
  return d;
}

a_string *
aa_strdupcpy (a_string * d, const a_string * s)
{
  d->l = s->l;
  d->s = xmalloc (s->l);
  memcpy (d->s, s->s, s->l);
  return d;
}

b_string *
ba_strdup (const a_string * s)
{
  b_string *d = xmalloc (sizeof (b_string) + s->l - 1);
  d->l = s->l;
  memcpy (d->s, s->s, s->l);
  return d;
}

char *
sa_strdup (const a_string * s)
{
  char *d = xmalloc (s->l + 1);
  memcpy (d, s->s, s->l);
  d[s->l] = 0;
  return d;
}

/* Copies at most N-1 characters from SRC to DEST, always appending a
   null character. */
void
strmaxcpy (char *dest, const char *src, size_t n)
{
  size_t len;

  len = strlen (src);
  if (len <= n - 1)
    strcpy (dest, src);
  else
    /* len>n-1 */
    {
      memcpy (dest, src, n - 1);
      dest[n - 1] = 0;
    }
}

/* Copies SRC to DEST, truncating to N characters or right-padding
   with character PAD to N characters if necessary; does not append a
   null character.  SRC, however, must be null-terminated. */
void
strbarepadcpy (char *dest, const char *src, size_t n, int pad)
{
  size_t len;

  len = strlen (src);
  if (len >= n)
    memcpy (dest, src, n);
  else
    {
      memcpy (dest, src, len);
      memset (&dest[len], pad, n - len);
    }
}

/* Copies SRC to DEST, truncating SRC to N characters or right-padding
   with character PAD to N characters if necessary; does not append a
   null character.  SRC must be LEN characters long but does not need
   to be null-terminated. */
void
strbarepadlencpy (char *dest, const char *src, size_t n, int pad, size_t len)
{
  if (len >= n)
    memcpy (dest, src, n);
  else
    {
      memcpy (dest, src, len);
      memset (&dest[len], pad, n - len);
    }
}

/* Copies SRC to DEST, truncating SRC to N-1 characters or
   right-padding with character PAD to N-1 characters if necessary;
   always appends a null character. */
void
strpadcpy (char *dest, const char *src, size_t n, int pad)
{
  size_t len;

  len = strlen (src);
  if (len == n - 1)
    strcpy (dest, src);
  else if (len < n - 1)
    {
      memcpy (dest, src, len);
      memset (&dest[len], pad, n - 1 - len);
      dest[n - 1] = 0;
    }
  else
    /* len>n-1 */
    {
      memcpy (dest, src, n - 1);
      dest[n - 1] = 0;
    }
}

/* Sets the first N-1 characters of S to PAD, and appends a null. */
void
blpstrset (char *s, int n, int pad)
{
  memset (s, pad, n - 1);
  s[n - 1] = 0;
}

/* Reverses the order of NBYTES bytes at address P, thus converting
   between little- and big-endian byte orders.  */
void
memrev (void *p, size_t nbytes)
{
  unsigned char *h = p, *t = &h[nbytes - 1];
  unsigned char temp;

  nbytes /= 2;
  while (nbytes--)
    {
      temp = *h;
      *h++ = *t;
      *t-- = temp;
    }
}

/* Finds the last NEEDLE of length NEEDLE_LEN in a HAYSTACK of length
   HAYSTACK_LEN.  Returns a pointer to the needle found. */
char *
memrmem (const char *haystack, size_t haystack_len,
	 const char *needle, size_t needle_len)
{
  int i;
  for (i = haystack_len - needle_len; i >= 0; i--)
    if (!memcmp (needle, &haystack[i], needle_len))
      return (char *) &haystack[i];
  return 0;
}

/* Compares S0 of length S0L to S1 of length S1L.  The shorter string
   is considered to be padded with spaces to the length of the
   longer. */
int
cmp_str (const char *s0, int s0l, const char *s1, int s1l)
{
  /* 254 spaces. */
  static char blanks[254] =
  "                                                               "
  "                                                               "
  "                                                               "
  "                                                               "
  "  ";

  int diff = s0l - s1l;
  int r;

  if (diff == 0)
    {
      if (s0l == 0)
	return 0;
      return memcmp (s0, s1, s0l);
    }
  else if (diff > 0)
    {
      /* s0l > s1l */
      if (s1l)
	{
	  r = memcmp (s0, s1, s1l);
	  if (r)
	    return r;
	}
      return memcmp (&s0[s1l], blanks, diff);
    }
  else
    /* diff<0 */
    {
      /* s0l < s1l */
      if (s0l)
	{
	  r = memcmp (s0, s1, s0l);
	  if (r)
	    return r;
	}
      return memcmp (blanks, &s1[s0l], -diff);
    }
}

