/**
 *
 * $Header: /cvsroot/hungry/lesstif/lib/Xm/Label.c,v 1.80 1998/09/28 01:35:16 rwscott Exp $
 *
 * Copyright (C) 1995 Free Software Foundation, Inc.
 *
 * This file is part of the GNU LessTif Library.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 **/

static const char rcsid[] = "$Header: /cvsroot/hungry/lesstif/lib/Xm/Label.c,v 1.80 1998/09/28 01:35:16 rwscott Exp $";

#include <LTconfig.h>
#include <XmI/XmI.h>

#include <Xm/XmP.h>
#include <Xm/XmosP.h>
#include <Xm/AtomMgr.h>
#include <Xm/DragC.h>
#include <Xm/DragIconP.h>
#include <Xm/LabelP.h>
#include <Xm/LabelGP.h>
#include <Xm/MenuShellP.h>
#include <Xm/MenuUtilP.h>
#include <Xm/RowColumnP.h>
#include <Xm/TransltnsP.h>
#include <XmI/AtomMgrI.h>
#include <XmI/DragDropI.h>
#ifdef __STDC__
#include <stdarg.h>
#else
#include <varargs.h>
#endif

#include <assert.h>

#include <XmI/DebugUtil.h>

/* Forward Declarations */

static void class_initialize();

static void class_part_initialize(WidgetClass w_class);

static void initialize(Widget request, Widget new_w,
		       ArgList args, Cardinal *num_args);

static void initialize_prehook(Widget request, Widget new_w,
			       ArgList args, Cardinal *num_args);

static void initialize_posthook(Widget request, Widget new_w,
				ArgList args, Cardinal *num_args);

static void destroy(Widget w);

static void resize(Widget w);

static void expose(Widget w, XEvent *event, Region region);

static XtGeometryResult query_geometry(Widget w,
				       XtWidgetGeometry *proposed,
				       XtWidgetGeometry *answer);

static Boolean set_values(Widget current, Widget request, Widget new_w,
			  ArgList args, Cardinal *num_args);

Boolean _XmLabelShowsAccelerators(Widget w);

Boolean _XmLabelShowsMnemonic(Widget w);

Boolean _XmLabelGetBaselines(Widget w,
			     Dimension **baselines,
			     int *nbaselines);

Boolean _XmLabelGetDisplayRect(Widget w, XRectangle *rect);

/* prototypes for drag-drop */

static Boolean drag_convert_proc(Widget w, Atom *selection,
				 Atom *target, Atom *type_return,
				 XtPointer *value_return,
				 unsigned long *length_return,
				 int *format_return);

static void drag_drop_finish(Widget w,
			     XtPointer client_data,
			     XtPointer call_data);


/*
 * Resources for the label class
 */
#define Offset(field) XtOffsetOf(XmLabelRec, label.field)
static XtResource resources[] =
{
    {
	XmNshadowThickness, XmCShadowThickness, XmRHorizontalDimension,
	sizeof(Dimension), XtOffsetOf(XmLabelRec, primitive.shadow_thickness),
	XmRImmediate, (XtPointer)0
    },
    {
	XmNalignment, XmCAlignment, XmRAlignment,
	sizeof(unsigned char), Offset(alignment),
	XmRImmediate, (XtPointer)XmALIGNMENT_CENTER
    },
    {
	XmNlabelType, XmCLabelType, XmRLabelType,
	sizeof(unsigned char), Offset(label_type),
	XmRImmediate, (XtPointer)XmSTRING
    },
    {
	XmNmarginWidth, XmCMarginWidth, XmRHorizontalDimension,
	sizeof(Dimension), Offset(margin_width),
	XmRImmediate, (XtPointer)2
    },
    {
	XmNmarginHeight, XmCMarginHeight, XmRVerticalDimension,
	sizeof(Dimension), Offset(margin_height),
	XmRImmediate, (XtPointer)2
    },
    {
	XmNmarginLeft, XmCMarginLeft, XmRHorizontalDimension,
	sizeof(Dimension), Offset(margin_left),
	XmRImmediate, (XtPointer)0
    },
    {
	XmNmarginRight, XmCMarginRight, XmRHorizontalDimension,
	sizeof(Dimension), Offset(margin_right),
	XmRImmediate, (XtPointer)0
    },
    {
	XmNmarginTop, XmCMarginTop, XmRVerticalDimension,
	sizeof(Dimension), Offset(margin_top),
	XmRImmediate, (XtPointer)0
    },
    {
	XmNmarginBottom, XmCMarginBottom, XmRVerticalDimension,
	sizeof(Dimension), Offset(margin_bottom),
	XmRImmediate, (XtPointer)0
    },
    {
	XmNfontList, XmCFontList, XmRFontList,
	sizeof(XmFontList), Offset(font),
	XmRImmediate, (XtPointer)NULL
    },
    {
	XmNlabelPixmap, XmCLabelPixmap, XmRPrimForegroundPixmap,
	sizeof(Pixmap), Offset(pixmap),
	XmRImmediate, (XtPointer)XmUNSPECIFIED_PIXMAP
    },
    {
 XmNlabelInsensitivePixmap, XmCLabelInsensitivePixmap, XmRPrimForegroundPixmap,
	sizeof(Pixmap), Offset(pixmap_insen),
	XmRImmediate, (XtPointer)XmUNSPECIFIED_PIXMAP
    },
    {
	XmNlabelString, XmCXmString, XmRXmString,
	sizeof(XmString), Offset(_label),
	XmRImmediate, (XtPointer)NULL
    },
    {
	XmNmnemonic, XmCMnemonic, XmRKeySym,
	sizeof(KeySym), Offset(mnemonic),
	XmRImmediate, (XtPointer)NULL
    },
    {
	XmNmnemonicCharSet, XmCMnemonicCharSet, XmRString,
	sizeof(String), Offset(mnemonicCharset),
	XmRImmediate, (XtPointer)XmFONTLIST_DEFAULT_TAG
    },
    {
	XmNaccelerator, XmCAccelerator, XmRString,
	sizeof(String), Offset(accelerator),
	XmRImmediate, (XtPointer)NULL
    },
    {
	XmNacceleratorText, XmCAcceleratorText, XmRXmString,
	sizeof(XmString), Offset(_acc_text),
	XmRImmediate, (XtPointer)NULL
    },
    {
	XmNrecomputeSize, XmCRecomputeSize, XmRBoolean,
	sizeof(Boolean), Offset(recompute_size),
	XmRImmediate, (XtPointer)True
    },
    {
	XmNstringDirection, XmCStringDirection, XmRStringDirection,
	sizeof(XmStringDirection), Offset(string_direction),
	XmRImmediate, (XtPointer)((XmStringDirection)XmUNSPECIFIED)
    },
    /* resources we override from XmPrimitive */
    {
	XmNtraversalOn, XmCTraversalOn, XmRBoolean,
	sizeof(Boolean), XtOffsetOf(XmLabelRec, primitive.traversal_on),
	XmRImmediate, (XtPointer)False
    },
    {
	XmNhighlightThickness, XmCHighlightThickness, XmRHorizontalDimension,
      sizeof(Dimension), XtOffsetOf(XmLabelRec, primitive.highlight_thickness),
	XmRImmediate, (XtPointer)0
    }
};

/* Synthetic Resources for the Label Widget */

static XmSyntheticResource syn_resources[] =
{
    {
	XmNmarginWidth,
	sizeof(Dimension), Offset(margin_width),
	_XmFromHorizontalPixels, _XmToHorizontalPixels
    },
    {
	XmNmarginHeight,
	sizeof(Dimension), Offset(margin_height),
	_XmFromVerticalPixels, _XmToVerticalPixels
    },
    {
	XmNmarginLeft,
	sizeof(Dimension), Offset(margin_left),
	_XmFromHorizontalPixels, _XmToHorizontalPixels
    },
    {
	XmNmarginRight,
	sizeof(Dimension), Offset(margin_right),
	_XmFromHorizontalPixels, _XmToHorizontalPixels
    },
    {
	XmNmarginTop,
	sizeof(Dimension), Offset(margin_top),
	_XmFromVerticalPixels, _XmToVerticalPixels
    },
    {
	XmNmarginBottom,
	sizeof(Dimension), Offset(margin_bottom),
	_XmFromVerticalPixels, _XmToVerticalPixels
    },
    {
	XmNlabelString,
	sizeof(XmString), Offset(_label),
	_XmExportLabelString, NULL
    },
    {
	XmNmnemonicCharSet,
	sizeof(String), Offset(mnemonicCharset),
	_XmExportString, NULL
    },
    {
	XmNaccelerator,
	sizeof(String), Offset(accelerator),
	_XmExportString, NULL
    },
    {
	XmNacceleratorText,
	sizeof(XmString), Offset(_acc_text),
	_XmExportLabelString, NULL
    },
};

static void ProcessDrag(Widget w, XEvent *event,
			String *params, Cardinal *num_params);

static void Help(Widget w, XEvent *event,
		 String *params, Cardinal *num_params);

static XtActionsRec actions[] =
{
    {"Enter", _XmPrimitiveEnter},
    {"Leave", _XmPrimitiveLeave},
    {"Help", Help},
    {"ProcessDrag", ProcessDrag},
};

/* *INDENT-OFF* */
static XmBaseClassExtRec _XmLabelCoreClassExtRec = {
    /* next_extension            */ NULL,
    /* record_type               */ NULLQUARK,                             
    /* version                   */ XmBaseClassExtVersion,
    /* size                      */ sizeof(XmBaseClassExtRec),
    /* initialize_prehook        */ initialize_prehook,
    /* set_values_prehook        */ XmInheritSetValuesPrehook,
    /* initialize_posthook       */ initialize_posthook,
    /* set_values_posthook       */ XmInheritSetValuesPosthook,
    /* secondary_object_class    */ XmInheritClass,
    /* secondary_object_create   */ XmInheritSecObjectCreate,
    /* get_secondary_resources   */ XmInheritGetSecResData,
    /* fast_subclass             */ { 0 },
    /* get_values_prehook        */ XmInheritGetValuesPrehook,
    /* get_values_posthook       */ XmInheritGetValuesPrehook,
    /* class_part_init_prehook   */ NULL,
    /* class_part_init_posthook  */ NULL,
    /* ext_resources             */ NULL,
    /* compiled_ext_resources    */ NULL,
    /* num_ext_resources         */ 0,
    /* use_sub_resources         */ False,
    /* widget_navigable          */ XmInheritWidgetNavigable,
    /* focus_change              */ XmInheritFocusChange,
    /* wrapper_data              */ NULL
};

XmPrimitiveClassExtRec _XmLabelPrimClassExtRec = {
    /* next_extension      */ NULL,
    /* record_type         */ NULLQUARK,
    /* version             */ XmPrimitiveClassExtVersion,
    /* record_size         */ sizeof(XmPrimitiveClassExtRec),
    /* widget_baseline     */ _XmLabelGetBaselines,
    /* widget_display_rect */ _XmLabelGetDisplayRect,
    /* widget_margins      */ NULL
};

XmLabelClassRec xmLabelClassRec = {
    /* Core class part */
    {
	/* superclass            */ (WidgetClass) &xmPrimitiveClassRec,
        /* class_name            */ "XmLabel",
	/* widget_size           */ sizeof(XmLabelRec),
	/* class_initialize      */ class_initialize,
	/* class_part_initialize */ class_part_initialize,
	/* class_inited          */ False,
	/* initialize            */ initialize,
	/* initialize_hook       */ NULL,
	/* realize               */ XtInheritRealize,
	/* actions               */ actions,
	/* num_actions           */ XtNumber(actions),
	/* resources             */ resources,
	/* num_resources         */ XtNumber(resources),
	/* xrm_class             */ NULLQUARK,
	/* compress_motion       */ True,
	/* compress_exposure     */ XtExposeCompressMaximal,
	/* compress_enterleave   */ True,
	/* visible_interest      */ False,
	/* destroy               */ destroy,
	/* resize                */ resize, 
	/* expose                */ expose,
	/* set_values            */ set_values,
	/* set_values_hook       */ NULL,
	/* set_values_almost     */ XtInheritSetValuesAlmost,
	/* get_values_hook       */ NULL,
	/* accept_focus          */ NULL,
	/* version               */ XtVersion,
	/* callback offsets      */ NULL,
	/* tm_table              */ NULL,
	/* query_geometry        */ query_geometry,
	/* display_accelerator   */ NULL,
	/* extension             */ (XtPointer)&_XmLabelCoreClassExtRec
    },
    /* Primitive Class part */
    {
        /* border_highlight      */ XmInheritBorderHighlight,
        /* border_unhighlight    */ XmInheritBorderUnhighlight,
        /* translations          */ XtInheritTranslations,
        /* arm_and_activate_proc */ NULL,
        /* Synthetic Resources   */ syn_resources,
        /* num syn res           */ XtNumber(syn_resources),
	/* extension             */ (XtPointer)&_XmLabelPrimClassExtRec
    },
    /* Label Class part */
    {
        /* setOverrideCallback */ NULL,
        /* menuProcs           */ NULL,
        /* translations        */ NULL,
	/* extension           */ NULL
    }
};
/* *INDENT-ON* */

WidgetClass xmLabelWidgetClass = (WidgetClass)&xmLabelClassRec;

extern XmFontList _XmFontListCreateDefault(Display *);

static XtTranslations menu_trans = NULL;
static XtTranslations default_trans = NULL;

static void
class_initialize()
{
    menu_trans = XtParseTranslationTable(_XmLabel_menuTranslations);
    default_trans = XtParseTranslationTable(_XmLabel_defaultTranslations);

    _XmLabelCoreClassExtRec.record_type = XmQmotif;
}

static void
class_part_initialize(WidgetClass widget_class)
{
    XmLabelWidgetClass lwc = (XmLabelWidgetClass)widget_class;
    XmLabelWidgetClass swc =
    (XmLabelWidgetClass)(widget_class->core_class.superclass);
    XmPrimitiveClassExt ext, *extptr, *sextptr;

    DEBUGOUT(XdbDebug(__FILE__, NULL, "class_part_initialize - %s -> %s\n",
    		swc->core_class.class_name,
    		widget_class->core_class.class_name));
    /* Handle label class part inheritance */

    if (lwc->label_class.menuProcs == XmInheritMenuProc)
    {
	DEBUGOUT(XdbDebug(__FILE__, NULL, "    XmInheritMenuProc\n"));
	lwc->label_class.menuProcs =
	    swc->label_class.menuProcs;
    }
    if (lwc->label_class.setOverrideCallback == XmInheritSetOverrideCallback)
    {
	DEBUGOUT(XdbDebug(__FILE__, NULL, "    XmInheritSetOverrideCallback\n"));
	lwc->label_class.setOverrideCallback = False;	/* FIX ME */
    }
    if (lwc->label_class.translations == XtInheritTranslations)
    {
	DEBUGOUT(XdbDebug(__FILE__, NULL, "    XtInheritTranslations\n"));
	lwc->label_class.translations =
	    swc->label_class.translations;
    }
    else if (lwc->label_class.translations != NULL)
    {
	lwc->label_class.translations =
	    (String)XtParseTranslationTable(lwc->label_class.translations);
    }
    else
    {
	DEBUGOUT(XdbDebug(__FILE__, NULL, "    Use menu traversal translations\n"));
	lwc->label_class.translations =
	    (String)XtParseTranslationTable(_XmLabel_menu_traversal_events);
    }

    extptr = (XmPrimitiveClassExt *)_XmGetClassExtensionPtr(
			(XmGenericClassExt *)&(lwc->primitive_class.extension),
							       NULLQUARK);
    sextptr = (XmPrimitiveClassExt *)_XmGetClassExtensionPtr(
			(XmGenericClassExt *)&(swc->primitive_class.extension),
								NULLQUARK);

    if (extptr == NULL || *extptr == NULL)
    {
	ext = (XmPrimitiveClassExt)XtNew(XmPrimitiveClassExtRec);
	if (ext != NULL)
	{
	    ext->next_extension = lwc->primitive_class.extension;
	    ext->record_type = NULLQUARK;
	    ext->version = XmPrimitiveClassExtVersion;
	    ext->record_size = sizeof(XmPrimitiveClassExtRec);
	    lwc->primitive_class.extension = (XtPointer)ext;
	}
    }
    else
    {
	ext = *extptr;
    }

    if (sextptr && *sextptr)
    {
	if (ext->widget_baseline == XmInheritBaselineProc)
	{
	    ext->widget_baseline = (*sextptr)->widget_baseline;
	}
	if (ext->widget_display_rect == XmInheritDisplayRectProc)
	{
	    ext->widget_display_rect = (*sextptr)->widget_display_rect;
	}
	if (ext->widget_margins == XmInheritMarginsProc)
	{
	    ext->widget_margins = (*sextptr)->widget_margins;
	}
    }

    _XmFastSubclassInit(widget_class, XmLABEL_BIT);
}

static void
CreateNormalGC(Widget w)
{
    XGCValues values;
    XtGCMask mask;

    mask = GCForeground | GCBackground | GCFillStyle | GCFunction |
	GCSubwindowMode | GCGraphicsExposures | GCPlaneMask;
    values.function = GXcopy;
    values.plane_mask = -1;
    values.subwindow_mode = ClipByChildren;
    values.graphics_exposures = False;
    values.foreground = Prim_Foreground(w);
    values.background = XtBackground(w);
    values.fill_style = FillSolid;

    Lab_NormalGC(w) = XtGetGC(w, mask, &values);
}

static void
CreateInsensitiveGC(Widget w)
{
    XGCValues values;
    XtGCMask mask;
    char *stipple_name;

    mask = GCForeground | GCBackground | GCFillStyle | GCFunction | GCStipple |
	GCPlaneMask | GCSubwindowMode | GCGraphicsExposures |
	GCTileStipXOrigin | GCTileStipYOrigin;
    values.function = GXcopy;
    values.plane_mask = -1;
    values.subwindow_mode = ClipByChildren;
    values.graphics_exposures = False;
    values.foreground = Prim_Foreground(w);
    values.background = XtBackground(w);
    values.fill_style = FillStippled;
    values.ts_x_origin = values.ts_y_origin = 0;

    if ((Lab_TextRect_x(w) & 1) ^ (Lab_TextRect_y(w) & 1))
	stipple_name = XmODD_STIPPLE_IMAGE;
    else
	stipple_name = XmEVEN_STIPPLE_IMAGE;
    values.stipple = XmGetPixmapByDepth(XtScreen(w),
					stipple_name,
					_XmWhitePixelOfObject(w),
					_XmBlackPixelOfObject(w),
					1);

    Lab_InsensitiveGC(w) = XtGetGC(w, mask, &values);
}

static void
initialize_prehook(Widget request,
		   Widget new_w,
		   ArgList args,
		   Cardinal *num_args)
{
    DEBUGOUT(XdbDebug(__FILE__, new_w, "Initialize Prehook\n"));

    _XmSaveCoreClassTranslations(new_w);

    if (XmIsRowColumn(XtParent(new_w)) &&
	(RC_Type(XtParent(new_w)) == XmMENU_PULLDOWN ||
	 RC_Type(XtParent(new_w)) == XmMENU_POPUP ||
	 RC_Type(XtParent(new_w)) == XmMENU_BAR))
    {
	CoreClassTranslations(new_w) = (String)menu_trans;
    }
    else if (!XmIsDrawnButton(new_w))
    {
	CoreClassTranslations(new_w) = (String)default_trans;
    }
}

static void
initialize_posthook(Widget request, Widget new_w,
		    ArgList args, Cardinal *num_args)
{
    DEBUGOUT(XdbDebug(__FILE__, new_w, "Initialize Posthook\n"));

    _XmRestoreCoreClassTranslations(new_w);
}

static void
initialize(Widget request, Widget new_w,
	   ArgList args, Cardinal *num_args)
{
    DEBUGOUT(XdbDebug(__FILE__, new_w, "initialize()\n"));

    /* get the default fontlist if the label was created without one. */
    if (Lab_Font(new_w) == (XmFontList)XmUNSPECIFIED ||
	Lab_Font(new_w) == NULL)
    {
	Lab_Font(new_w) = _XmGetDefaultFontList(new_w, XmLABEL_FONTLIST);
    }
    /* if the user specified one, COPY it */
    else
    {
	Lab_Font(new_w) = XmFontListCopy(Lab_Font(new_w));
    }

    /* If the label was not initialized with the resource labelString set,
       use its name -- the following _XmString code comes from MegaButton */
    if (Lab_Label(new_w) == (_XmString)XmUNSPECIFIED ||
	Lab_Label(new_w) == (_XmString)0)
    {				/* Shouldn't be necessary but is */
	XmString xmstring;

	xmstring = _XmOSGetLocalizedString((char *)NULL,
					   (Widget)new_w,
					   XmNlabelString,
					   XtName(new_w));

	Lab_Label(new_w) = _XmStringCreate(xmstring);
	XmStringFree(xmstring);
    }

    if (_XmStringIsXmString((XmString)Lab_Label(new_w)))
    {
	Lab_Label(new_w) = _XmStringCreate((XmString)Lab_Label(new_w));
    }

    if (Lab_AcceleratorText(new_w) != NULL)
    {
	Lab_AcceleratorText(new_w) =
	    _XmStringCreate((XmString)Lab_AcceleratorText(new_w));
    }
    else
    {
    XmString tmp;

	tmp = XmStringCreateLocalized("");
	Lab_AcceleratorText(new_w) = _XmStringCreate(tmp);
	XmStringFree(tmp);
    }

    /* if the parent is a row column, set the menu_type to
       it's type.  Otherwise, XmNONE  (Is this right?) FIX ME */

    if (XmIsRowColumn(XtParent(new_w)))
    {
	Lab_MenuType(new_w) = RC_Type(XtParent(new_w));
    }
    else
    {
	Lab_MenuType(new_w) = XmNONE;
    }

    /*
     * have to check request since new may have been polluted by a
     * superclass 
     * MLM: Who?  *Primitive*?  The only ones it could use to change
     * width/height we override (shadowThickness/highlightThickness)
     */
    if (XtWidth(request) == (Dimension)0)
    {
	XtWidth(new_w) = 0;
    }
    if (XtHeight(request) == (Dimension)0)
    {
	XtHeight(new_w) = 0;
    }

    _XmCalcLabelDimensions(new_w);
    resize(new_w);

    /*
     * the user might have wanted something.  Change it back, if so
     */
    if (XtWidth(request) != 0)
    {
	XtWidth(new_w) = XtWidth(request);
    }
    if (XtHeight(request) != 0)
    {
	XtHeight(new_w) = XtHeight(request);
    }

    /* surge protection */
    if (XtWidth(new_w) == 0)
    {
	XtWidth(new_w) = 1;
    }
    if (XtHeight(new_w) == 0)
    {
	XtHeight(new_w) = 1;
    }

    /* allocate the normal and insensitive GC's */
    CreateNormalGC(new_w);
    CreateInsensitiveGC(new_w);

    /* Force the traversal and highlight on enter resources if
       in an popup, pulldown, and option menus. */

    if (Lab_MenuType(new_w) == XmMENU_POPUP ||
	Lab_MenuType(new_w) == XmMENU_PULLDOWN ||
	Lab_MenuType(new_w) == XmMENU_OPTION ||
	Lab_MenuType(new_w) == XmMENU_BAR)
    {
	DEBUGOUT(XdbDebug(__FILE__, new_w, "    merge in Label Class Translations\n"));
#if 0
MLM: WHY TRAVERSAL?
	Prim_TraversalOn(new_w) = False;
#endif
	Prim_HighlightOnEnter(new_w) = False;

	XtOverrideTranslations(new_w,
			       (XtTranslations)LabClass_Translations(XtClass(new_w)));
    }
    Lab_SkipCallback(new_w) = False;

    if (Lab_MnemonicCharset(new_w) != NULL)
    {
	Lab_MnemonicCharset(new_w) = XtNewString(Lab_MnemonicCharset(new_w));
    }
    else
    {
	Lab_MnemonicCharset(new_w) = XtNewString("");
    }

    if (_XmStringIsXmString((XmString)Lab_Accelerator(new_w)))
    {
	XmStringGetLtoR((XmString)Lab_Accelerator(new_w),
			XmFONTLIST_DEFAULT_TAG,
			&Lab_Accelerator(new_w));
    }

    if (Lab_Accelerator(new_w))
    {
	Lab_Accelerator(new_w) = XtNewString(Lab_Accelerator(new_w));
	_XmManagerInstallAccelerator(XtParent(new_w), new_w,
				     Lab_Accelerator(new_w));
    }

    if (Lab_Mnemonic(new_w))
    {
	_XmManagerInstallMnemonic(XtParent(new_w), new_w, Lab_Mnemonic(new_w));
    }
}

static void
destroy(Widget w)
{
    _XmManagerUninstallMnemonic(XtParent(w), w);
    _XmManagerUninstallAccelerator(XtParent(w), w);
    XtFree(Lab_Accelerator(w));
    XtFree(Lab_MnemonicCharset(w));
    XtReleaseGC(w, Lab_InsensitiveGC(w));
    XtReleaseGC(w, Lab_NormalGC(w));
    _XmStringFree(Lab_AcceleratorText(w));
    _XmStringFree(Lab_Label(w));
    XmFontListFree(Lab_Font(w));
}


/* 
 * it is bad to call _XmCalcLabelDimensions inside SetValues
 * since it is possible the parent will reject the size
 * changes and _XmLabelRecomputeSize will have changed widget 
 * internals based on the size it thinks it will get.
 * If the changes are rejected, we are left with the changed
 * internals and the unchanged Width and Height.
 *
 * We could change the label widget to use set_values_almost, but
 * I don't see the need to do so.  In fact, I can't think of any
 * simple widgets that use set_values_almost anyway.  The internal 
 * positions are always calcuated in resize.
 *
 * Lab_RecomputeSize seems to be checked in OSF/Motif when either
 *      o the pixmap changes
 *      o the labelString changes
 * it is possible to change into the other...
 *
 * What happens if we are changing the label (possibly
 * to a pixmap) and the size change is rejected or the new size evaluates 
 * to the same value as the old size?  Resize won't be called.  We need to 
 * do something so expose doesn't use the old internal values.  
 *
 * Just recompute the label internals in expose, by calling resize()
 * directly. :)
 *
 * Notes:
 *
 *    if Lab_RecomputeSize is set, the widget wants to resize to fit
 *    changes to the labelString or pixmap
 * 051496 -- There are surely missing cases in here. MLM
 */
static Boolean
set_values(Widget old, Widget request, Widget new_w,
	   ArgList args, Cardinal *num_args)
{
    Boolean refresh_needed = False, relayout_needed = False;

    DEBUGOUT(XdbDebug(__FILE__, new_w,
		      "%s(%d):set_values: %i args\n"
		      "\t    old X %5i Y %5i W %5i H %5i\n"
		      "\trequest X %5i Y %5i W %5i H %5i\n"
		      "\t  new_w X %5i Y %5i W %5i H %5i\n",
		      __FILE__, __LINE__, *num_args,
		      XtX(old), XtY(old),
		      XtWidth(old), XtHeight(old),
		      XtX(request), XtY(request),
		      XtWidth(request), XtHeight(request),
		      XtX(new_w), XtY(new_w),
		      XtWidth(new_w), XtHeight(new_w)));
    DEBUGOUT(XdbPrintArgList(__FILE__, new_w, args, *num_args, False));

    if (XtSensitive(old) != XtSensitive(new_w))
    {
    	refresh_needed = True;
	if (Lab_RecomputeSize(new_w))
	{
	    relayout_needed = True;
	}
	else
	{
	    refresh_needed = True;
	}
    }
    /* This is a Primitive resource but we have the GC's for it... */
    if (Prim_Foreground(new_w) != Prim_Foreground(old) ||
	XtBackground(new_w) != XtBackground(old))
    {
	XtReleaseGC(new_w, Lab_NormalGC(new_w));
	XtReleaseGC(new_w, Lab_InsensitiveGC(new_w));

	CreateNormalGC(new_w);
	CreateInsensitiveGC(new_w);

	refresh_needed = True;
    }

    if (Lab_AcceleratorText(new_w) != Lab_AcceleratorText(old))
    {
	Lab_AcceleratorText(new_w) =
	    _XmStringCreate((XmString)Lab_AcceleratorText(new_w));

	_XmStringFree(Lab_AcceleratorText(old));

	if (Lab_RecomputeSize(new_w))
	{
	    _XmLabelAccTextSize(new_w);
	    relayout_needed = True;
	}
	else
	{
	    refresh_needed = True;
	}
    }

    if (Lab_MnemonicCharset(new_w) != Lab_MnemonicCharset(old))
    {
	Lab_MnemonicCharset(new_w) = XtNewString(Lab_MnemonicCharset(new_w));

	XtFree(Lab_MnemonicCharset(old));

	refresh_needed = True;
    }

    if (Lab_Label(new_w) == NULL)
    {
    XmString tmp;

	tmp = XmStringCreateSimple(XtName(new_w));
	Lab_Label(new_w) = _XmStringCreate(tmp);
	XmStringFree(tmp);

	if (Lab_Label(old))
	{
	    _XmStringFree(Lab_Label(old));
	}

	if (Lab_RecomputeSize(new_w))
	{
	    relayout_needed = True;
	}
	else
	{
	    refresh_needed = True;
	}
    }
    else if (Lab_Label(new_w) != Lab_Label(old))
    {
	if (_XmStringIsXmString((XmString)Lab_Label(new_w)))
	{
	    Lab_Label(new_w) = _XmStringCreate((XmString)Lab_Label(new_w));
	}

	if (Lab_Label(old))
	{
	    _XmStringFree(Lab_Label(old));
	}

	if (Lab_RecomputeSize(new_w))
	{
	    relayout_needed = True;
	}
	else
	{
	    refresh_needed = True;
	}
    }

    if (Lab_Alignment(new_w) != Lab_Alignment(old))
    {
	refresh_needed = True;
    }

    if (Lab_Font(new_w) && Lab_Font(new_w) != (XmFontList)XmUNSPECIFIED &&
	Lab_Font(new_w) != Lab_Font(old))
    {
	XmFontListFree(Lab_Font(old));
	Lab_Font(new_w) = XmFontListCopy(Lab_Font(new_w));

	if (Lab_RecomputeSize(new_w))
	{
	    relayout_needed = True;
	}
	else
	{
	    refresh_needed = True;
	}
    }

    if (Lab_MarginTop(new_w) != Lab_MarginTop(old)
	|| Lab_MarginBottom(new_w) != Lab_MarginBottom(old)
	|| Lab_MarginLeft(new_w) != Lab_MarginLeft(old)
	|| Lab_MarginRight(new_w) != Lab_MarginRight(old)
	|| Lab_MarginWidth(new_w) != Lab_MarginWidth(old)
	|| Lab_MarginHeight(new_w) != Lab_MarginHeight(old)
	|| Lab_Mnemonic(new_w) != Lab_Mnemonic(old)
	|| Lab_StringDirection(new_w) != Lab_StringDirection(old))
    {
	if (Lab_RecomputeSize(new_w))
	{
	    relayout_needed = True;
	}
	else
	{
	    refresh_needed = True;
	}
    }

    /* check for change in insensitive pixmap */
    if ((Lab_PixmapInsensitive(new_w) != Lab_PixmapInsensitive(old))
	&& !XtSensitive(new_w) && Lab_IsPixmap(new_w))
    {
	if (Lab_RecomputeSize(new_w))
	{
	    relayout_needed = True;
	}
	else
	{
	    refresh_needed = True;
	}
    }

    /* check for change in pixmap */
    if (Lab_Pixmap(new_w) != Lab_Pixmap(old))
    {
	/* if changed pixmap to UNSPECIFIED, automatically configure to a
	 * string
	 */
	if (Lab_IsPixmap(new_w)
	    && Lab_Pixmap(new_w) == (Pixmap)XmUNSPECIFIED_PIXMAP)
	{
	    Lab_LabelType(new_w) = XmSTRING;
	}

	if (Lab_RecomputeSize(new_w))
	{
	    relayout_needed = True;
	}
	else
	{
	    refresh_needed = True;
	}
    }

    /* did the label change types? */
    if (Lab_LabelType(new_w) != Lab_LabelType(old))
    {
	if (Lab_RecomputeSize(new_w))
	{
	    relayout_needed = True;
	}
	else
	{
	    refresh_needed = True;
	}
    }

    if (Lab_Accelerator(new_w) != Lab_Accelerator(old))
    {

	if (_XmStringIsXmString((XmString)Lab_Accelerator(new_w)))
	{
	    XmStringGetLtoR((XmString)Lab_Accelerator(new_w),
			    XmFONTLIST_DEFAULT_TAG,
			    &Lab_Accelerator(new_w));
	}
	else if (Lab_Accelerator(new_w))
	{
	    Lab_Accelerator(new_w) = XtNewString(Lab_Accelerator(new_w));
	}

	_XmManagerUninstallAccelerator(XtParent(new_w), new_w);
	_XmManagerInstallAccelerator(XtParent(new_w), new_w,
				     Lab_Accelerator(new_w));

	if (Lab_RecomputeSize(new_w))
	{
	    relayout_needed = True;
	}
	else
	{
	    refresh_needed = True;
	}
    }
    if (Lab_Mnemonic(new_w) != Lab_Mnemonic(old))
    {
	_XmManagerUninstallMnemonic(XtParent(new_w), new_w);
	_XmManagerInstallMnemonic(XtParent(new_w), new_w, Lab_Mnemonic(new_w));

	if (Lab_RecomputeSize(new_w))
	{
	    relayout_needed = True;
	}
	else
	{
	    refresh_needed = True;
	}
    }

    if (XtWidth(new_w) == 0 || XtHeight(new_w) == 0)
    {
	relayout_needed = True;
    }

    if (relayout_needed)
    {
	if (Lab_RecomputeSize(new_w) && XtWidth(new_w) == XtWidth(old))
	{
	    XtWidth(new_w) = 0;
	}
	if (Lab_RecomputeSize(new_w) && XtHeight(new_w) == XtHeight(old))
	{
	    XtHeight(new_w) = 0;
	}

	_XmCalcLabelDimensions(new_w);

	resize(new_w);

	refresh_needed = True;
    }

    return refresh_needed;
}

static void
expose(Widget w, XEvent *event, Region region)
{
    XRectangle cliprect;
    GC myGC;

    DEBUGOUT(XdbDebug(__FILE__, w, "Expose\n"));

    /*
     * I'm becoming paranoid - Danny - see testXm/filesb/test3
     */
    if (!XtIsRealized(w))
    {
	return;
    }
    /* 
     * recompute the labels internals... they could be off if resize
     * hadn't been called after a change (see set_values)
     */
#if 0
    /* rws 8 May 1998
       This messes up Mozilla's Bookmark menu buttons which are sub-classed
       from PushB and CascadeB.
     */
    resize(w);
#else
    (XtClass(w)->core_class.resize)(w);
#endif

    /* use the right GC */
    if (XtSensitive(w))
    {
	myGC = Lab_NormalGC(w);
    }
    else
    {
	myGC = Lab_InsensitiveGC(w);
    }

    /* Set a clip rectangle for the GC - ensure we don't overwrite shadows */
    /* the rectangle used to include MarginWidth and MarginHeight.  Leaving
     * those two out makes Xinvest and Xmcd look better */
    /*
     * However, not accounting for the margin screws up the default button
     * rendering; PushButton will fiddle with the margins to keep the label
     * from drawing over the extra shadow.  Basically we want to make the
     * label draw itself as though it were smaller than it is but we want
     * to keep geometry management doing the "right thing" so we can't
     * just fiddle with XmNwidth and XmNheight.  The problem will only
     * show up if a PushButton is forced to be smaller than it wants to
     * be _and_ there is a defaultButton.  Inspection has revealed that
     * the (Top|Bottom|Left|Right)Margin's are the ones we want (yes,
     * I did count the pixels).
     *
     * See similar things in LabelG.c.
     */
    cliprect.x = Prim_HighlightThickness(w)
               + Prim_ShadowThickness(w)
               + Lab_MarginLeft(w);
    cliprect.y = Prim_HighlightThickness(w)
               + Prim_ShadowThickness(w)
               + Lab_MarginTop(w);
    cliprect.width  = XtWidth(w)
                    - 2 * (Prim_ShadowThickness(w) + Prim_HighlightThickness(w))
                    - Lab_MarginLeft(w) - Lab_MarginRight(w)
                    + Lab_AccTextRect(w).width + Lab_AccTextRect(w).x;
    cliprect.height = XtHeight(w)
                    - 2 * (Prim_ShadowThickness(w) + Prim_HighlightThickness(w))
                    - Lab_MarginTop(w) - Lab_MarginBottom(w)
                    + Lab_AccTextRect(w).height + Lab_AccTextRect(w).y;

    DEBUGOUT(XdbDebug(__FILE__, w,
		      "Expose: cliprect %d %d %dx%d\n",
		      cliprect.x, cliprect.y,
		      cliprect.width, cliprect.height));
    DEBUGOUT(XdbDebug(__FILE__, w,
                     "Expose: acctextrect %d %d %dx%d\n",
                     Lab_AccTextRect(w).x, Lab_AccTextRect(w).y,
                     Lab_AccTextRect(w).width, Lab_AccTextRect(w).height));

    XSetClipRectangles(XtDisplay(w), myGC, 0, 0, &cliprect, 1, Unsorted);

    if (Lab_IsText(w))
    {				/* LabelString */
	if (_XmLabelShowsMnemonic(w) && Lab_Mnemonic(w))
	{
	    char m[2];

	    m[0] = Lab_Mnemonic(w);
	    m[1] = '\0';

	    _XmStringDrawMnemonic(XtDisplay(w), XtWindow(w),
				  Lab_Font(w), Lab_Label(w), myGC,
				  Lab_TextRect_x(w),
				  Lab_TextRect_y(w),
				  Lab_TextRect_width(w),
				  Lab_Alignment(w),
				  0,
				  NULL,
				  m, Lab_MnemonicCharset(w));
	}
	else
	{
	    _XmStringDraw(XtDisplay(w),
			  XtWindow(w),
			  Lab_Font(w),
			  Lab_Label(w),
			  myGC,
			  Lab_TextRect_x(w),
			  Lab_TextRect_y(w),
			  Lab_TextRect_width(w),
			  Lab_Alignment(w),
			  0,
			  NULL);
	}
	/* AcceleratorText */
	if (_XmLabelShowsAccelerators(w))
	{
	    _XmStringDraw(XtDisplay(w),
			  XtWindow(w),
			  Lab_Font(w),
			  Lab_AcceleratorText(w),
			  myGC,
			  Lab_AccTextRect(w).x,
			  Lab_AccTextRect(w).y,
			  Lab_AccTextRect(w).width,
			  XmALIGNMENT_BEGINNING,
			  0,
			  NULL);
	}
    }
    else if (XtSensitive(w) &&
	     Lab_Pixmap(w) != XmUNSPECIFIED_PIXMAP)
    {
	XCopyArea(XtDisplay(w),
		  Lab_Pixmap(w),
		  XtWindow(w),
		  myGC,
		  0,
		  0,
		  Lab_TextRect_width(w),
		  Lab_TextRect_height(w),
		  Lab_TextRect_x(w), Lab_TextRect_y(w));
    }
    else if (!XtSensitive(w) &&
	     Lab_PixmapInsensitive(w) != XmUNSPECIFIED_PIXMAP)
    {
	XCopyArea(XtDisplay(w),
		  Lab_PixmapInsensitive(w),
		  XtWindow(w),
		  myGC,
		  0,
		  0,
		  Lab_TextRect_width(w),
		  Lab_TextRect_height(w),
		  Lab_TextRect_x(w), Lab_TextRect_y(w));
    }

    XSetClipMask(XtDisplay(w), myGC, None);
}

static void
resize(Widget w)
{
    Dimension width, height;
    Dimension pix_width, pix_height;
    Boolean showAcc;
    unsigned char beforex, beforey, afterx, aftery;

    if (!XmIsLabel(w))
    {
	return;
    }

    beforex = Lab_TextRect_x(w) & 1;
    beforey = Lab_TextRect_y(w) & 1;

    showAcc = _XmLabelShowsAccelerators(w);

    /* set the label's size so the pixmap/string fits */
    if (Lab_IsText(w))
    {
	_XmStringExtent(Lab_Font(w), Lab_Label(w), &width, &height);

	DEBUGOUT(XdbDebug(__FILE__, w,
			  "Resize: _XmStringExtent => w %d, h %d\n",
			  width, height));
    }
    else
	/* Lab_IsPixmap(w) */
    {
	_XmLabelGetPixmapSize(w, Lab_Pixmap(w), &pix_width, &pix_height);

	width = pix_width;
	height = pix_height;

	DEBUGOUT(XdbDebug(__FILE__, w,
			  "Resize: _XmLabelGetPixmapSize => w %d, h %d\n",
			  width, height));
    }

    Lab_TextRect_width(w) = width;
    Lab_TextRect_height(w) = height;


    /* The alignments only modify x values if a size was set */
    _XmLabelAccTextSize(w);

    switch (Lab_Alignment(w))
    {
    case XmALIGNMENT_END:
	Lab_TextRect_x(w) = (XtWidth(w)
			     - Lab_Highlight(w)
			     - Lab_Shadow(w)
			     - Lab_MarginWidth(w)
			     - Lab_MarginRight(w)
			     - Lab_TextRect_width(w));
	break;

    case XmALIGNMENT_BEGINNING:
	/* Default X Position of TextRect */
	Lab_TextRect_x(w) = (Lab_Highlight(w)
			     + Lab_Shadow(w)
			     + Lab_MarginWidth(w)
			     + Lab_MarginLeft(w));
	break;

    case XmALIGNMENT_CENTER:
    default:
	Lab_TextRect_x(w) = (XtWidth(w) -
			     (Lab_MarginLeft(w) +
			      Lab_MarginRight(w)) -
			     width) / 2 + Lab_MarginLeft(w);
	break;
    }

    Lab_TextRect_y(w) = ((XtHeight(w) -
			  (Lab_MarginTop(w) +
			   Lab_MarginBottom(w)) -
			  height)) / 2 + Lab_MarginTop(w);

    if (showAcc)
    {
	Lab_AccTextRect(w).x = XtWidth(w) - Lab_MarginRight(w)
	    + LABEL_ACC_PAD / 2;
	Lab_AccTextRect(w).y = Lab_TextRect_y(w);
#if 0
	DEBUGOUT(XdbDebug(__FILE__, w,
			  "Accelerator @ %d %d (x = %d - %d - %d - %d)\n",
			  Lab_AccTextRect(w).x, Lab_AccTextRect(w).y,
			  XtWidth(w), Lab_Shadow(w), Lab_Highlight(w),
			  Lab_AccTextRect(w).width));
#endif
    }

    afterx = Lab_TextRect_x(w) & 1;
    aftery = Lab_TextRect_y(w) & 1;
    if (beforex ^ afterx || beforey ^ aftery)
    {
	XtReleaseGC(w, Lab_InsensitiveGC(w));

	CreateInsensitiveGC(w);
    }
}

/*
 * ask how we want to look
 */
#define	Wants(x)	(proposed->request_mode & x)

static XtGeometryResult
query_geometry(Widget w, XtWidgetGeometry *proposed, XtWidgetGeometry *answer)
{
#if 0
    XtWidgetGeometry a;
    Dimension wd, ht;

    DEBUGOUT(XdbDebug(__FILE__, w, "query_geometry\n"));

    wd = XtWidth(w);
    ht = XtHeight(w);

    /* rws 28 Feb 1997
     * Setting height and width to 0 will calculate the minimum
     * size of the label. If recompute is false we do not want to
     * do that. This helps form/test4 (and some of my own apps)
     * This, however requires a change to the initialize code
     *
     * rws 22 Mar 1997
     * As Mitch suspects recompute size is only dealt with in set_values
     */

    if (proposed->request_mode & CWWidth)
    {
	XtWidth(w) = proposed->width;
    }
    if (proposed->request_mode & CWHeight)
    {
	XtHeight(w) = proposed->height;
    }

    if (!Lab_RecomputeSize(w))
    {
	a.width = XtWidth(w);
	a.height = XtHeight(w);
	a.request_mode = CWWidth | CWHeight;

	XtWidth(w) = wd;
	XtHeight(w) = ht;

	if ((proposed->request_mode & (CWWidth | CWHeight)) ==
	    (CWWidth | CWHeight) &&
	    proposed->width == answer->width &&
	    proposed->height == answer->height)
	{
	    return XtGeometryYes;
	}
	else if (answer->width == XtWidth(w) && answer->height == XtHeight(w))
	{
	    return XtGeometryNo;
	}
	else
	{
	    return XtGeometryAlmost;
	}
    }

    if (XmIsCascadeButton(w))
    {
	extern void _XmCBCalcDimensions(Widget w);

	_XmCBCalcDimensions(w);
    }
    else
    {
	_XmCalcLabelDimensions(w);
    }

    a.width = XtWidth(w);
    a.height = XtHeight(w);
    a.request_mode = CWWidth | CWHeight;

    XtWidth(w) = wd;
    XtHeight(w) = ht;

    if (answer)
    {
	*answer = a;
    }

    DEBUGOUT(XdbDebug(__FILE__, w, "query_geometry => %s\n",
		      XdbWidgetGeometry2String(answer)));

    if ((proposed->request_mode & (CWWidth | CWHeight)) ==
	(CWWidth | CWHeight) &&
	proposed->width == answer->width && proposed->height == answer->height)
    {
	return XtGeometryYes;
    }
    else if (answer->width == XtWidth(w) && answer->height == XtHeight(w))
    {
	return XtGeometryNo;
    }
    else
    {
	return XtGeometryAlmost;
    }
#else
    Dimension wd = XtWidth(w);
    Dimension ht = XtHeight(w);
    XtWidgetGeometry pc;

	pc = *proposed;
	/* rws 27 Sep 1998
	   label/test8 should prove whether this IsRealized should be here
	   or not, and it should not.
	   label/test9 should prove whether Lab_RecomputeSize should be here
	   or not,
	 */
	if (Lab_RecomputeSize(w))
	{
	    _XmCalcLabelDimensions(w);
	    answer->width = XtWidth(w);
	    answer->height = XtHeight(w);
	    XtWidth(w) = wd;
	    XtHeight(w) = ht;
	}
	else
	{
	    answer->width = XtWidth(w);
	    answer->height = XtHeight(w);
	}

#if 0
	if (((pc.request_mode & (CWWidth | CWHeight)) == (CWWidth | CWHeight)) && 
		pc.width == answer->width && 
		pc.height == answer->height)
	{
	    return XtGeometryYes;
	}
	else if (answer->width == XtWidth(w) && answer->height == XtHeight(w))
	{
	    return XtGeometryNo;
	}
	else
	{
	    return XtGeometryAlmost;
	}
#else
	return _XmGMReplyToQueryGeometry(w, &pc, answer);
#endif
#endif
}

static Boolean
drag_convert_proc(Widget w, Atom *selection,
		  Atom *target, Atom *type_return,
		  XtPointer *value_return,
		  unsigned long *length_return,
		  int *format_return)
{
    Atom COMPOUND_TEXT;
    Atom MOTIF_DROP;

    COMPOUND_TEXT = XmInternAtom(XtDisplay(w), _XA_COMPOUND_TEXT, False);
    MOTIF_DROP = XmInternAtom(XtDisplay(w), _XA_MOTIF_DROP, False);

    if (*selection != MOTIF_DROP)
    {
	return False;
    }

    DEBUGOUT(XdbDebug(__FILE__, w, "We're dealing with a motif drop\n"));

    return False;
}

static void
drag_drop_finish(Widget w, XtPointer client_data, XtPointer call_data)
{
}

static void
ProcessDrag(Widget w, XEvent *event, String *params, Cardinal *num_params)
{
    Atom export_target;
    Arg args[10];
    int n = 0;
    Widget dc;

    DEBUGOUT(XdbDebug(__FILE__, w, "Processing a drag-drop request\n"));

    if (Lab_IsPixmap(w))
    {
	export_target = XmInternAtom(XtDisplay(w),
				     _XA_PIXMAP,
				     False);
    }
    else
    {
	export_target = XmInternAtom(XtDisplay(w),
				     _XA_COMPOUND_TEXT,
				     False);
    }

    XtSetArg(args[n], XmNexportTargets, &export_target); n++;
    XtSetArg(args[n], XmNnumExportTargets, 1); n++;
    XtSetArg(args[n], XmNdragOperations, XmDROP_COPY); n++;
    XtSetArg(args[n], XmNconvertProc, drag_convert_proc); n++;
    XtSetArg(args[n], XmNclientData, w); n++;
    XtSetArg(args[n], XmNsourceCursorIcon, _XmGetTextualDragIcon(w)); n++;

    dc = XmDragStart(w, event, args, n);

    if (dc)
    {
	XtAddCallback(dc, XmNdragDropFinishCallback, drag_drop_finish, NULL);
    }
}

static void
Help(Widget w, XEvent *event, String *params, Cardinal *num_params)
{
    XtCallActionProc(w, "PrimitiveHelp", event, params, *num_params);
}

void
_XmExportLabelString(Widget w, int offset, XtArgVal *value)
{
    _XmString str;
    XmString ret;

    str = *(_XmString *)(((char *)w) + offset);
    if (str)
    {
	if (XmIsLabelGadget(w))
	{
	    ret = _XmStringCreateExternal(LabG_Font(w), str);
	}
	else if (XmIsLabel(w))
	{
	    ret = _XmStringCreateExternal(Lab_Font(w), str);
	}
	else
	{
	    ret = NULL;
	}
    }
    else
    {
	ret = NULL;
    }

    *value = (XtArgVal)ret;
}

void
_XmLabelGetPixmapSize(Widget w, Pixmap Pix,
		      Dimension *width, Dimension *height)
{
    unsigned int tmp, tmpw, tmph;
    int tmpx, tmpy;
    unsigned Depth;
    Window tmpwin;

    if (Pix == XmUNSPECIFIED_PIXMAP || Pix == None)
    {
	*width = *height = 0;
	return;
    }

    XGetGeometry(XtDisplayOfObject(w),
		 Pix,
		 &tmpwin,
		 &tmpx, &tmpy,
		 &tmpw, &tmph,
		 &tmp, &Depth);

    *width = (Dimension)tmpw;
    *height = (Dimension)tmph;
}

Boolean
_XmLabelGetBaselines(Widget w, Dimension **baseelines, int *nbaselines)
{
    return False;
}

Boolean
_XmLabelGetDisplayRect(Widget w, XRectangle *rect)
{
    rect->x = Lab_TextRect_x(w);
    rect->y = Lab_TextRect_y(w);
    rect->width = Lab_TextRect_width(w);
    rect->height = Lab_TextRect_height(w);

    return True;
}

Boolean
_XmLabelShowsMnemonic(Widget w)
{
    /*
     * It looks like mnemonics have to be displayed even when not in a
     * menu system. Careful reading of the spec reveals that the mnemonics
     * only *work* when in a menu system. Guess that means they should be
     * displayed even if they can't work.
     *
     * Xmcd-2.2 bug #1. Danny 29/8/1997
     */
#if 0
    if (!XtIsSubclass(XtParent(w), xmRowColumnWidgetClass))
    {
	return False;
    }
    if (RC_Type(XtParent(w)) != XmMENU_PULLDOWN &&
	RC_Type(XtParent(w)) != XmMENU_BAR)
    {
	return False;
    }
#endif
    return True;
}

Boolean
_XmLabelShowsAccelerators(Widget w)
{
    Dimension aw = 0, ah = 0;

    if (!XtIsSubclass(XtParent(w), xmRowColumnWidgetClass))
    {
	return False;
    }
    if (RC_Type(XtParent(w)) != XmMENU_PULLDOWN &&
        RC_Type(XtParent(w)) != XmMENU_POPUP)
    {
	return False;
    }
    if (XmIsLabel(w))
    {
	if (Lab_IsPixmap(w) || Lab_AcceleratorText(w) == NULL)
	{
	    return False;
	}

	_XmStringExtent(Lab_Font(w), Lab_AcceleratorText(w),
			&aw, &ah);
	if (aw == 0)
	{
	    return False;
	}
    }
    else if (XmIsLabelGadget(w))
    {
	if (LabG_IsPixmap(w) || LabG_AcceleratorText(w) == NULL)
	{
	    return False;
	}

	_XmStringExtent(LabG_Font(w), LabG_AcceleratorText(w),
			&aw, &ah);
	if (aw == 0)
	{
	    return False;
	}
    }
    else
    {
	return False;
    }

    return True;
}

void
_XmLabelAccTextSize(Widget w)
{
    Dimension width, height;

    if (XmIsLabel(w))
    {
	Lab_AccTextRect(w).width = Lab_AccTextRect(w).height = 0;

	if (!_XmLabelShowsAccelerators(w))
	{
	    return;
	}

	_XmStringExtent(Lab_Font(w), Lab_AcceleratorText(w), &width, &height);

	Lab_AccTextRect(w).height = height;
	Lab_AccTextRect(w).width = width;

	if (Lab_MarginRight(w) < Lab_AccTextRect(w).width + LABEL_ACC_PAD)
	{
	    Lab_MarginRight(w) = Lab_AccTextRect(w).width + LABEL_ACC_PAD;

	    DEBUGOUT(XdbDebug(__FILE__, w,
			      "_XmLabelAccTextSize: set right margin to %d\n",
			      Lab_MarginRight(w)));
	}
    }
    else if (XmIsLabelGadget(w))
    {
	LabG_AccTextRect(w).width = LabG_AccTextRect(w).height = 0;

	if (!_XmLabelShowsAccelerators(w))
	{
	    return;
	}

	_XmStringExtent(LabG_Font(w), LabG_AcceleratorText(w), &width, &height);

	LabG_AccTextRect(w).height = height;
	LabG_AccTextRect(w).width = width;

	if (LabG_MarginRight(w) < LabG_AccTextRect(w).width + LABELG_ACC_PAD)
	{
	    _XmAssignLabG_MarginRight((XmLabelGadget)w,
				      LabG_AccTextRect(w).width +
				      LABELG_ACC_PAD);

	    DEBUGOUT(XdbDebug(__FILE__, w,
			      "_XmLabelAccTextSize: set right margin to %d\n",
			      LabG_MarginRight(w)));

	    _XmReCacheLabG(w);
	}
    }
}

/*
 * private label functions 
 * converted to work for both widgets and gadgets -- CT
 *
 * --- JRA
 * this should NEVER be called with XtWidth or XtHeight == 0
 * 
 * this touches the internal dimensions of the widget, it should
 * NOT be called from inside a SetValues call as a result of a 
 * size change since the size may be REJECTED by the PARENT
 *
 * MLM: Hmmm.  The Motif version of this function appears to be
 * always called with XtWidth/XtHeight = 0.  Also, I don't think
 * the statement about the size rejection is entirely true (I've
 * run into a situation with XmPushB where it is essential that
 * the size be changed in the set_values method).  I believe if
 * the parent doesn't like the change, it will reject it -- AND
 * call the child's resize procedure -- FIX ME.
 *
 * FIX ME
 *
 * This thing is to be changed for PushButtons (including gadgets)
 * when they're in a menu, in the case that XmNacceleratorText
 * is non-NULL.
 */
void
_XmCalcLabelDimensions(Widget w)
{
    Dimension cw, ch, width, height;

    if (Lab_IsText(w))
    {
	_XmStringExtent(Lab_Font(w), Lab_Label(w), &cw, &ch);
    }
    else
	/* pixmap */
    {
	if (XtIsSensitive(w))
	{
	    _XmLabelGetPixmapSize(w, Lab_Pixmap(w), &cw, &ch);
	}
	else
	{
	    _XmLabelGetPixmapSize(w, Lab_PixmapInsensitive(w), &cw, &ch);
	}
    }

    width = (Lab_Highlight(w)
	     + Lab_Shadow(w)
	     + Lab_MarginLeft(w)
	     + Lab_MarginWidth(w)
	     + cw
	     + Lab_MarginWidth(w)
	     + Lab_MarginRight(w)
	     + Lab_Shadow(w)
	     + Lab_Highlight(w));

    height = (Lab_Highlight(w)
	      + Lab_Shadow(w)
	      + Lab_MarginTop(w)
	      + Lab_MarginHeight(w)
	      + ch
	      + Lab_MarginHeight(w)
	      + Lab_MarginBottom(w)
	      + Lab_Shadow(w)
	      + Lab_Highlight(w));

    /* Do something about menus... */
    /* MLM.  Nah, it's figured into the right margin */

    /* rws 17 Apr 1997
       test5 shows that Label was returning wrong values from a query_geometry.
       It seems that it should always be returning its preferred size. 
       Besides with this here the preferred size is _always_ the current size.
       Taking it out breaks the day menu of plan though (BulletinBoard) so I
       will leave it in for now.
       This prevents MessageBox's from resizing when the messageString is
       changed.
     */
    XtWidth(w) = width;
    XtHeight(w) = height;
}

Widget
XmCreateLabel(Widget parent, char *name,
	      Arg *arglist, Cardinal argcount)
{
    return XtCreateWidget(name, xmLabelWidgetClass, parent,
			  arglist, argcount);
}
