/* $Id: gii.c,v 1.8 1998/10/26 11:40:22 ajapted Exp $
***************************************************************************

   Graphics library for GGI. General Input Interface.

   Copyright (C) 1998 Andreas Beck	[becka@ggi-project.org]
   Copyright (C) 1997 Jason McMullan        [jmcc@ggi-project.org]
   Copyright (C) 1998 Jim Ursetto    [jim.ursetto@ggi-project.org]
   Copyright (C) 1998 Andrew Apted  [andrew.apted@ggi-project.org]

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
  
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
  
   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

***************************************************************************
*/

/* FIXME!!! Autoconf this */
#ifdef __linux__
#define HAVE_LINUXCOMPATIBLE_SELECT	1
#endif


#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>

#define GII_POLLINTERVAL 10000	/* Poll in 10ms intervals */

#include <sys/types.h>
#include <sys/time.h>
#include <unistd.h>

#define GII_DLINIT_SYM		GII_SYMPREFIX##"GIIdlinit"

#define GII_VERSIONCODE		0x00000001
#define GII_COMPATIBLE_MASK	0xffffff00

#include <ggi/internal/gii.h>

/* 
 * ** Internal standard handlers **
 */

static int _GIIstdseteventmask(struct gii_input *inp, gii_event_mask evm)
{
	inp->curreventmask=( evm & inp->targetcan );
	return GGI_OK;
}

static gii_event_mask _GIIstdgeteventmask(struct gii_input *vis)
{
	return vis->curreventmask;
}

static int _GIIstdgetselectfd(struct gii_input *vis, fd_set *readfds)
{
	int x;
	for (x=0;x<vis->maxfd;x++)
		if (FD_ISSET(x,&(vis->fdset)))
			FD_SET(x,readfds);
	return vis->maxfd;
}


/*
 * ** EvQueue mechanisms **
 * 
 * These are intended for internal use only
 */

/* Check for presence of a queue for that event type. Set it up if not.
 * Returns the queue.
 */
static gii_ev_queue *_giiEvQueueSetup(gii_input *inp, int type)
{
	if (inp->queue->queues[type] == NULL) {
		gii_ev_queue *qp = (gii_ev_queue *) malloc(sizeof(gii_ev_queue));
		DPRINT_EVENTS("_giiEvQueueSetup(%p, %d) malloc failed\n", inp, type);
		memset(qp, 0, sizeof(gii_ev_queue));
		inp->queue->queues[type] = qp;
	}
	
	return inp->queue->queues[type];
}

/* Destroy the queue for that input. Set pointer to NULL to catch late calls.
 */
static void _giiEvQueueDestroy(gii_input *inp)
{
	int queue;

	DPRINT_EVENTS("_giiEvQueueDestroy(%p) called\n", inp);

	for (queue=0; queue < evLast; queue++) {
		if (inp->queue->queues[queue])
			free(inp->queue->queues[queue]);
	}
	free(inp->queue);
	inp->queue=NULL;	/* Assure segfault ... */
}

/* Allocate the queue for that input. Individual queues are created on demand.
 */
static int _giiEvQueueAllocate(gii_input *inp)
{
	DPRINT_EVENTS("_giiEvQueueAllocate(%p) called\n", inp);

	if (NULL==(inp->queue = (gii_ev_queue_set *) malloc(sizeof(gii_ev_queue_set))))
		return 1;

	/* O.K. - initialize it */
	memset(inp->queue, 0, sizeof(gii_ev_queue_set));

	return 0;
}

#if 0
/* Check which queues (from mask) have at least 1 event.
 */
static gii_event_mask _giiEvQueueSeen(gii_input *inp, gii_event_mask mask)
{
	DPRINT_EVENTS("_giiEvQueueSeen(%p, 0x%x) called\n", inp, mask);
	return (inp->queue->seen & mask);
}
#endif

/* Release an event from the queue. Select the earliest one.
 * returns event-size. 0=fail.
 */
static int _giiEvQueueRelease(gii_input *inp, gii_event *ev,
			      gii_event_mask mask)
{
	gii_ev_queue *qp = NULL;
	gii_event_mask evm;
	struct timeval t_min;
	int queue;

	DPRINT_EVENTS("_giiEvQueueRelease(%p, %p, 0x%x) called\n",
		      inp, ev, mask);

	evm = mask & inp->queue->seen;

	/* Got nothing.. */
	if (evm == 0) {
		return 0;
	}

	/* Max timestamp.. */
	t_min.tv_sec= 0x7FFFFFFF;
	t_min.tv_usec=0x7FFFFFFF;

	/* Get the specified event out of the queue.  If the user asks
	 * for more than one event type, return the one that has been
	 * waiting the longest.
	 */
	for (queue=0; queue < evLast; queue++) {
		struct gii_ev_queue *qp_tmp = inp->queue->queues[queue];

		if (qp_tmp && qp_tmp->count && (evm & (1 << queue))) {
			gii_event *e_tmp = (gii_event *)
				(qp_tmp->buf + qp_tmp->tail);
			struct timeval t_tmp = e_tmp->any.time;

			if (t_tmp.tv_sec < t_min.tv_sec || 
			    (t_tmp.tv_sec == t_min.tv_sec &&
                             t_tmp.tv_usec < t_min.tv_usec)) {

				DPRINT_EVENTS("_giiEvQueueRelease: Plausible found.\n");
				qp = qp_tmp;
				t_min = t_tmp;
			}
		}
	}

	/* Shouldn't happen.. */
	LIBGII_ASSERT(qp != NULL, "_giiEvQueueRelease: Arrgghh!! Nothing plausible");
	if (qp == NULL) {
		return 0;
	}
	
	/* Pull event out of queue.. */
	memcpy(ev, qp->buf + qp->tail, qp->buf[qp->tail]);

	qp->count--;
	qp->tail += ev->size;

	if (qp->tail > GII_Q_THRESHOLD) {
		qp->tail = 0;
	}
	if (qp->count == 0) {
		inp->queue->seen &= ~(1 << ev->any.type);
	}
	
	DPRINT_EVENTS("_giiEvQueueRelease: Retrieved event type %d (0x%08x), size %d.\n", 
	       ev->any.type, 1 << ev->any.type, ev->size);

	return ev->size;
}

/* Set all queue entries to the queue. Used to join inputs.
 */
static void	_giiSetQueue(struct gii_input *inp,gii_ev_queue_set *set)
{
	struct gii_input *curr;

	DPRINT_EVENTS("_giiSetQueue called for %p.\n",inp);

	if (!(curr=inp)) return;
	do {
		curr->queue=set;
		curr=curr->next;

	} while(curr!=inp);	/* looped through once. */
}

/*
 * ** These are externally callable by GII modules **
 */

void _giiEventBlank(gii_event *ev)
{
	memset(ev, 0, sizeof(gii_event));
	EV_TIMESTAMP(ev);
}

/* Add an event. Return 0 on either success or unrecognized event type,
 * and return -1 is the buffer is full
 */
int _giiEvQueueAdd(gii_input *inp, gii_event *ev)
{
	gii_ev_queue *qp;
	gii_event_mask evm = 1 << ev->any.type;
	int freespace;

	DPRINT_EVENTS("_giiEvQueueAdd(%p, %p) called\n", inp, ev);

	/* Check if type is in range */

	if (ev->any.type >= evLast) {
		DPRINT_EVENTS("_giiEvQueueAdd: bad type!\n");
		return 0;
	}
	
	qp = _giiEvQueueSetup(inp, ev->any.type);

	/* Enough free space ? */

	freespace=1;

	if (qp->head < qp->tail) {
		if ((qp->tail - qp->head - 1) < ev->size) {
			freespace=0;
		}	
	} else if (qp->head > qp->tail) {
		if ((qp->head + ev->size) > GII_Q_THRESHOLD) {
		
			/* Event crosses threshold, thus we need space
			 * at start of buffer to put head (tail may be
			 * at 0, but head == tail means an empty buffer).
			 */
			
			if (qp->tail == 0) {
				freespace=0;
			}
		}
	}

	if (! freespace) {
		DPRINT_EVENTS("_giiEvQueueAdd: Queue overflow\n");
		return -1;
	}

	DPRINT_EVENTS("_giiEvQueueAdd: Adding event type %d (0x%08x), size %d at pos %d.\n", 
		     ev->any.type, evm, ev->size, qp->count);

	/* Add the event, and mark that we've seen it.. */
	memcpy(qp->buf + qp->head, ev, ev->size);
	
	qp->count++;
	qp->head += ev->size;

	if (qp->head > GII_Q_THRESHOLD) {
		qp->head = 0;
	}
	
	inp->queue->seen |= evm;

	return 0;
}

static int giiorigincount=0;

/* allocate memory for an input descriptor and initialize it.
 * It also allocates an event queue.
 */
struct gii_input	*_giiInputAlloc(void)
{
	struct gii_input *ret;

	/* allocate memory for input descriptor.
	 */
	if (NULL==(ret=malloc(sizeof(gii_input)))) return ret;
	if (_giiEvQueueAllocate(ret))
	{ free(ret); return NULL; }

	ret->version=GII_VERSIONCODE;
	ret->origin=giiorigincount++;

	/* ret->mutex; ??? */

	ret->next=ret->prev=ret;        /* Ring structure ... self<->self */

	ret->curreventmask=0;	/* The target should set those. */
	ret->targetcan    =0;	/* If it doesn't, it is broken. */
	ret->maxfd=0;
	FD_ZERO(&(ret->fdset));

	ret->GIIeventpoll=NULL;
	ret->GIIsendevent=NULL;
	ret->GIIseteventmask=_GIIstdseteventmask;
	ret->GIIgeteventmask=_GIIstdgeteventmask;
	ret->GIIgetselectfdset=_GIIstdgetselectfd;
	ret->GIIclose=NULL;

	return ret;
}

/*
 * ** Opening/Closing/Joining of sources **
 */

extern gii_dlhandle *_giiLoadDL(const char *filename);

struct gii_input *    giiOpen(const char *input,...)
{
	struct gii_input *ret;
	va_list	drivers;
	char target[1024];
	char *cp;
	const char *fname;
	int err;

	/* Leave space for up to 256 subsystems ... */
	/* I don't think we will wrap around - will we ? */
	giiorigincount+=0x100;	/* Start at next boundary */
	giiorigincount&=~0x0ff;
	giiorigincount&=~EV_ORIGIN_SENDEVENT;

	if (NULL==(ret=_giiInputAlloc())) return ret;

	va_start(drivers,input);

	for (;input!=NULL;input=va_arg(drivers,const char *)) {

		DPRINT_CORE("Loading input %s\n",input);

		if (ggParseTarget((char *)input,target,1024) == NULL) {
			continue;
		}
		cp=strchr(target, ':');

		if (cp != NULL) {
			*cp++ = 0;
		}

		DPRINT_LIBS("giiOpen adding (%p, \"%s\", \"%s\")\n",
		    ret, input, cp);
	
		fname = ggMatchConfig(_giiconfhandle, input, cp);

		if (fname==NULL) {
			fprintf(stderr, "LibGII: libgii.conf doesn't have an entry for: %s\n", input);
			free(ret->queue);
			free(ret);
			return NULL;
		}

		ret->dlhand=_giiLoadDL(fname);
		DPRINT_LIBS("_giiLoadDL returned %p\n", ret->dlhand);
		if (ret->dlhand==NULL)
		{
			free(ret->queue);
			free(ret);
			return NULL;
		}
	
		err=ret->dlhand->init((void *)ret,cp);	/* Yeah - I know ... FIXME ! */
		DPRINT_LIBS("%d=dlh->init(%p,\"%s\") - %s %s\n",err,ret,cp,input,fname);
		if (err) {
			free(ret->queue);
			free(ret->dlhand);
			free(ret);
			return NULL;
		}
	}
	if (ret->targetcan==0)
		DPRINT_LIBS("Bogus target %p loaded ... targetcan==0.\n",ret);
	return ret;
}

int	giiClose(struct gii_input *inp)
{
	struct gii_input *curr;
	int rc=-1;

	DPRINT_LIBS("giiClose called for %p.\n",inp);

	_giiEvQueueDestroy(inp);	/* This destroys _all_ queues ! */

	if (!(curr=inp)) return -1;
	do {
		curr->queue=NULL;	/* For better error catching. */

		if (curr->GIIclose)
			rc=curr->GIIclose(curr);

		curr=curr->next;
		/* FIXME ! dlclose ! */
		free(curr->dlhand);
		free(curr->prev);
	} while(curr!=inp);	/* looped through once. */
	return rc;
}

/* Take two inputs and merge them together. For the program, this is as if
 * inp2 has been giiClosed() and inp has taken over all of its properties.
 */
struct gii_input *giiJoinInputs(struct gii_input *inp, struct gii_input *inp2)
{
	DPRINT_LIBS("giiJoinInputs called for %p and %p.\n",inp,inp2);

	if (!inp2) return inp;
	if (!inp)  return inp2;	/* This shouldn't happen, but ... */

	_giiEvQueueDestroy(inp2);	/* Destroy inp2 queue */
	_giiSetQueue(inp2,inp->queue);	/* Set inp queue instead */

	inp2->prev->next=inp ->next;
	inp ->next->prev=inp2->prev;
	inp ->next=inp2;
	inp2->prev=inp;

	return inp;
}

/*
 * ** Selection/polling/reading **
 */

int	giiGetSelectFdset(gii_input *inp, fd_set *readfds,int *haspolled)
{
	struct gii_input *curr;
	fd_set	hlpfd;
	int x,xx,max;
	
	DPRINT_LIBS("giiGetSelectFdset called for %p.\n",inp);
	FD_ZERO(readfds);max=0;
	if (haspolled) *haspolled=0;

	if (!(curr=inp)) return 0;
	do {

		if (curr->GIIgetselectfdset)
		{
			xx=curr->GIIgetselectfdset(curr,&hlpfd);
			if (haspolled && curr->maxfd==0) *haspolled=1;
			for (x=0;x<xx;x++)
				if (FD_ISSET(x,&hlpfd))
					FD_SET(x,readfds);
			if (xx>max) max=xx;
		}
		curr=curr->next;

	} while(curr!=inp);	/* looped through once. */
	return max;
}

/* Tests all possible sources for events that match mask. If there are any
 * polled sources for that mask, haspolled is set to true, if it is not NULL.
 * It returns a maks of all newly acquired events.
 */
static gii_event_mask giiPollall(struct gii_input *inp,gii_event_mask mask)
{
	struct gii_input *curr;
	gii_event_mask	max,xx;
	
	DPRINT_LIBS("giiPollAll called for %p mask %08x.\n",inp,mask);
	max=0;

	if (!(curr=inp)) return 0;
	do {
		if ( (curr->curreventmask & mask) && curr->GIIeventpoll)
		{
			/* This is expected to do the following:
			 * 1. queue all pending events.
			 * 2. return the or-ed mask of all queued events
			 */
			xx=curr->GIIeventpoll(curr);
			max|=xx;
		}
		curr=curr->next;

	} while(curr!=inp);	/* looped through once. */
	return max;
}

gii_event_mask giiEventPoll(struct gii_input *inp,gii_event_mask mask,struct timeval *t)
{
	fd_set	readset;
	int maxfd,haspolled;
	gii_event_mask rc;

	/* Do we have something already queued ? */
	rc = mask & inp->queue->seen;
	if (rc) return rc;
	        
	/* Give the sources a try. rc is not used ... */
	giiPollall(inp,mask);

	/* Something queued now ? */
	rc = mask & inp->queue->seen;
	if (rc) return rc;

	/* Catch common case, avoid underflow and select() */
	if (t && t->tv_sec==0 && t->tv_usec==0) return 0;

	maxfd=giiGetSelectFdset(inp,&readset,&haspolled);

	if(!haspolled) {		/* No polled drivers - good ! So we just select();. */
		if (maxfd<=0)	/* Nothing here at all ... */
			return 0;
		while(1) {
#ifdef HAVE_LINUXCOMPATIBLE_SELECT
			/* FIXME !!! doesn't handle -EINTR */
			if (select(maxfd,&readset,NULL,NULL,t)<=0) 
				return 0; /* Time is up, or error ... */
#else
			struct timeval old, new, temp, *ptemp=NULL;
			if (t) {
				/* We need to have a pointer to a _copy_ ... */
				temp = *t;
				ptemp = &temp;
			}
			gettimeofday(&old, NULL);
			/* FIXME !!! doesn't handle -EINTR */
			rc = select(maxfd,&readset,NULL,NULL,ptemp);
			gettimeofday(&new, NULL);
			if (t) {
				t->tv_sec -= (new.tv_sec - old.tv_sec);
				t->tv_usec -= (new.tv_usec - old.tv_usec);
				if (t->tv_usec < 0) {
					t->tv_usec += 1000000;
					t->tv_sec--;
					if (t->tv_sec < 0) {
						/* Time is up, set to zero */
						t->tv_sec = 0;
						t->tv_usec = 0;
					}
				}
			}
			/* FIXME !!! doesn't handle -EINTR */
			if (rc<=0) return 0; /* Time is up, or error ... */
#endif

			giiPollall(inp,mask);
			rc = mask & inp->queue->seen;
			if (rc) return rc;
			if (t && t->tv_sec == 0 && t->tv_usec == 0) {
				/* Time is up, no need to select() again */
				return 0;
			}
		}
	}

	/* We have polled drivers, so we have to split the select and
	   poll at regular intervals. */

	while(1)
	{
		struct timeval tv;
		
		tv.tv_sec=0;tv.tv_usec=GII_POLLINTERVAL;

		if (t && t->tv_sec==0 && t->tv_usec < tv.tv_usec) tv=*t;

		/* FIXME: We disregard the actual polling time ...
		 * FIXME !!! doesn't handle -EINTR
		 */
		select(maxfd,&readset,NULL,NULL,&tv);	

		/* Time is up or we have something ... */

		rc=giiPollall(inp,mask);
		rc = mask & inp->queue->seen;
		if (rc) return rc;

		if (t)
		{
			if (t->tv_usec>=GII_POLLINTERVAL) 
			{
				t->tv_usec-=GII_POLLINTERVAL;
				if (t->tv_usec==0&&t->tv_sec==0) return rc;
			}
			else 
			{
				t->tv_usec+=1000000-GII_POLLINTERVAL;
				if (t->tv_sec) t->tv_sec--;
				else return rc;
			}
		}
	}
	return 0;
}

int giiEventRead(struct gii_input *inp,gii_event *ev,gii_event_mask mask)
{
	giiEventPoll(inp,mask,NULL);
	return _giiEvQueueRelease(inp,ev,mask);
}

/*
 * ** Event Masks **
 */

int giiSetEventMask(struct gii_input *inp, gii_event_mask evm)
{
	struct gii_input *curr;
	int rc=-1;
	int i;

	/* Tell all the sources
	 */
	if (!(curr=inp)) return -1;
	do {
		if (curr->GIIseteventmask)
			rc=curr->GIIseteventmask(curr,evm);
		curr=curr->next;

	} while(curr!=inp);	/* looped through once. */

	DPRINT_EVENTS("GIIseteventmask(%p, 0x%x) called\n", inp, evm);

	/* Flush any events whose bit is 0 in the new mask.
	 * Is that really desireable ? Andy
	 */
	for (i=0; i < evLast; i++) {
		if (((evm & (1 << i)) == 0) && inp->queue->queues[i]) {
			inp->queue->queues[i]->head  = 0;
			inp->queue->queues[i]->tail  = 0;
			inp->queue->queues[i]->count = 0;
		}
	}

	return rc;
}

gii_event_mask giiGetEventMask(struct gii_input *inp)
{
	struct gii_input *curr;
	gii_event_mask	mask,hlp;

	mask=0;

	if (!(curr=inp)) return 0;
	do {

		if (curr->GIIgeteventmask)
		{
			hlp=inp->GIIgeteventmask(curr);
			mask|=hlp;
		}
		curr=curr->next;

	} while(curr!=inp);	/* looped through once. */
	return mask;
}

int giiEventSend(struct gii_input *inp, gii_event *event)
{
	struct gii_input *curr = inp;

	LIBGII_APPASSERT(inp != NULL, "giiEventSend: inp is NULL");

	event->any.origin |= EV_ORIGIN_SENDEVENT;
	do {
		/* Notify all clients that care */
		if (curr->GIIsendevent)
		{
			inp->GIIsendevent(curr, event);
		}
		curr = curr->next;

	} while(curr != inp);	/* looped through once. */
	return _giiEvQueueAdd(inp, event);
}
