/*****************************************************************************
 *                                                                           *
 * Programm:  paul                                                           *
 *            (P)rogramm zur (A)uswertung und (U)mformung von                *
 *            (L)aserbildern                                                 *
 * Modul:     paulspec.c                                                     *
 *            Einige Sachen zur Spezifikation der Bilder                     *
 * Autor:     Andreas Tille                                                  *
 * Datum:     25.05.1998                                                     *
 *                                                                           *
 *****************************************************************************/

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <unistd.h>

#include "paul.h"

#ifdef __DMALLOC__
#include <dmalloc.h>
#endif

char TypAusschnitt[]        = "Cut section",
     TypDelRand[]           = "Border deleted",
     TypDelScanRand[]       = "Scanner border deleted",
     TypDifferenzBild[]     = "Difference",
     TypFilter[]            = "Filter",
     TypGray[]              = "Monochrom",
     TypKontrastBild[]      = "Scaled",
     TypNegativ[]           = "Negativ",
     TypMove[]              = "Moved",
     TypMirror[]            = "Mirrored",
     TypRotate[]            = "Rotated",
     TypScan[]              = "Scan",
     TypShrink[]            = "Shrinked";

char ChunkNameSource[]      = "Source",
     ChunkNameSoftware[]    = "Software",
     ChunkNameTyp[]         = "Typ",
     ChunkNameDescription[] = "Description",
     ChunkNameAuthor[]      = "Author",
     ChunkFilmType[]        = "Film type",
     ChunkBrightness[]      = "Brightness",
     ChunkContrast[]        = "Contrast",
     ChunkReferenceFile[]   = "Referenzdatei",
     *ChunkNames[]          = { ChunkNameSource,   
	                        ChunkNameSoftware, 
                                ChunkNameTyp,
                                ChunkNameDescription,
                                ChunkNameAuthor,
                                ChunkFilmType,
                                ChunkBrightness,
                                ChunkContrast,
                                ChunkReferenceFile,
                                NULL
                              };

char *GetSpec(CHUNK *chunks, const char *spec)
/* Liest Spezifikationsparameter
 * --- Parameter: ---
 * CHUNK      *chunks: alle Chunks mit spezifizierenden Daten
 * const char *spec  : Name des gesuchten Chunks
 * --- R"uckgabe: ---
 * char  *GetSpec    : Wert des gesuchten Chunks bzw. NULL, falls dieser nicht existiert
 */
{
   CHUNK *ap;

   for ( ap = chunks; ap->key != NULL; ap++ )
      if ( !strcmp(ap->key, spec) ) return ap->value;
   return NULL;
}


int SetSpec(CHUNK *chunks, const char *spec, char *value)
/* Setzt Spezifikationsparameter
 * --- Parameter: ---
 * CHUNK      *chunks: alle Chunks mit spezifizierenden Daten
 * const char *spec  : Name des zu setzenden Chunks
 * const char *value : Wert des zu setzenden Chunks
 * --- R"uckgabe: ---
 * CHUNK      *chunks: neu gesetzter Chunk mit NEU ALLOCIERTEM Speicher
 * char  *GetSpec    : 0 fr OK, -1 falls Chunk nicht existiert
 */
{
   CHUNK *ap;

   for ( ap = chunks; ap->key != NULL; ap++ )
      if ( !strcmp(ap->key, spec) ) {
	 if ( ap->value ) free(ap->value);
	 ap->value = value;
	 return 0;
      }
   return -1;
}


int CopySpec(CHUNK *chunks, const char *spec, const char *value)
/* Kopiert Spezifikationsparameter
 * --- Parameter: ---
 * CHUNK      *chunks: alle Chunks mit spezifizierenden Daten
 * const char *spec  : Name des zu setzenden Chunks
 * const char *value : Wert des zu setzenden Chunks
 * --- R"uckgabe: ---
 * CHUNK      *chunks: neu gesetzter Chunk mit NEU ALLOCIERTEM Speicher
 * char  *GetSpec    : 0 fr OK, -1 falls Chunk nicht existiert
 */
{
   CHUNK *ap;

   for ( ap = chunks; ap->key != NULL; ap++ )
      if ( !strcmp(ap->key, spec) ) {
         if ( spec == ChunkNameDescription && ap->value ) {
            assert ( (ap->value = realloc(ap->value, strlen(ap->value) + strlen(value) + 2)) );
	    strcat(strcat(ap->value, "\n"), value);
         } else {
            if ( ap->value ) free(ap->value);
            assert ( (ap->value = strdup(value)) );
	 }
	 return 0;
      }
   return -1;
}


PICTURE *InitSpec(void)
/* Initialisiert Spezifikationsparameter
 * --- R"uckgabe: ---
 * PICTURE *InitSpec() : bild initialisiert
 */
{
   CHUNK   *ap;
   char   **cn;
   int      nchunks;
   PICTURE *bild;
   
   if ( !(bild = malloc(sizeof(bild[0]))) ) return NULL;
   
   bild->id       = PAUL_ID;
   bild->im       = NULL;
   bild->W        = 0;
   bild->H        = 0;
   bild->DATA     = NULL;
   bild->file     = NULL;
   bild->ext      = NULL;
   bild->dir      = NULL;
   bild->size     = 0;   
   bild->spp      = 3; /* at first assume RGB image at any case                                  */
   bild->storepix = 3; /* paul uses 3 byte per pixel; paulscan has to set storepix explicitely!! */
   bild->trans    = 0;
   bild->res      = 0;
   bild->x_offset = bild->y_offset = (unsigned long)-1;
   bild->n_gamma  = 0;
   bild->gamma    = NULL;
   bild->his      = NULL;
   bild->zeit     = 0;
   bild->roff_x   = bild->roff_y   = 0;
   bild->phys     = NULL;
   bild->physname = NULL;
   bild->flag     = 0;
   bild->label    = NULL;
   for ( cn = ChunkNames, nchunks = 1; *cn != NULL; cn++ ) nchunks++;
   assert ( (bild->spec = malloc(nchunks*sizeof(CHUNK))) );
   for ( ap = bild->spec, cn = ChunkNames; *cn != NULL; ap++, cn++ ) {
      ap->key   = *cn;
      ap->value = NULL;
   }
   ap->key = NULL;

   return bild;
}

time_t convert_png_time_to_time_t(png_time *ptime)
/* Konvertiert png_time Struktur in time_t
 * --- Parameter: ---
 * png_time *zeit                       : zu konvertierende Zeit
 * --- R"uckgabe: ---
 * time_t   convert_png_time_to_time_t(): konvertierte Zeit
 */
{
   struct tm ttime;

   ttime.tm_year = ptime->year  - 1900;
   ttime.tm_mon  = ptime->month - 1;
   ttime.tm_mday = ptime->day;
   ttime.tm_hour = ptime->hour;
   ttime.tm_min  = ptime->minute;
   ttime.tm_sec  = ptime->second;

   return mktime(&ttime);
}

static void GetPictureSource(PICTURE *bild)
/* obtain source of image (scanner, filename, source tag stored in file)
 * --- Parameter: ---
 * PICTURE    *bild     : Bild
 * char       *file     : Dateiname (hier kann NICHT bild->file verwendet werden, da
 *                        letzteres nur noch `basename` enthlt!)
 * png_struct *png_ptr  : PNG-Struktur, falls PNG-Datei, sonst NULL
 * png_info   *info_ptr : PNG-Info, falls PNG-Datei, sonst NULL
 */
{
   if ( GetSpec(bild->spec, ChunkNameSource) ) return;
   CopySpec(bild->spec, ChunkNameSource, bild->file);
}


static void GetPictureTime(PICTURE *bild, char *file)
/* obtain time of picture creation (creation of file, tag inside file)
 * --- Parameter: ---
 * PICTURE    *bild     : Bild
 * char       *file     : Dateiname (hier kann NICHT bild->file verwendet werden, da
 *                        letzteres nur noch `basename` enthlt!)
 * png_struct *png_ptr  : PNG-Struktur, falls PNG-Datei, sonst NULL
 * png_info   *info_ptr : PNG-Info, falls PNG-Datei, sonst NULL
 */
{
   struct    stat stat_buf;

   if ( bild->zeit > 0 ) return;
   if ( file && !stat(file, &stat_buf) ) bild->zeit = stat_buf.st_mtime;
   else                                  bild->zeit = time(NULL);
}


void GetPictureAuthor(PICTURE *bild)
/* Kriegt heraus, wer zur Zeit an der Kiste sitzt
 * --- Parameter: ---
 * PICTURE    *bild     : Bild
 */
{
   char  buf[256], *s;
   FILE  *p;
   int   uid;
   
   if ( GetSpec(bild->spec, ChunkNameAuthor) ) return;
   uid = getuid();
   sprintf(buf,"grep :%i: /etc/passwd | sed \"s/[^:]*:[^:]*:%i:[0-9]*:\\([^:^,]*\\).*/\\1/\"", 
                uid, uid);
   if ( (p=popen(buf,"r")) != NULL ) {
      fgets(buf,256,p);
      pclose(p);
      if ( (s = strchr(buf,'\n')) ) *s = 0;
   } else
      sprintf(buf, "User-ID = %i", uid);
   CopySpec(bild->spec, ChunkNameAuthor, buf);
}


void GetPictureSpecs(PICTURE *bild, char *file)
/* obtain all specification chunks of picture
 * --- Parameter: ---
 * PICTURE    *bild     : Bild
 * char       *file     : Dateiname (hier kann NICHT bild->file verwendet werden, da
 *                        letzteres nur noch `basename` enthlt!)
 * png_struct *png_ptr  : PNG-Struktur, falls PNG-Datei, sonst NULL
 * png_info   *info_ptr : PNG-Info, falls PNG-Datei, sonst NULL
 */
{
   GetPictureAuthor(bild);
   if ( file ) GetPictureSource(bild);
   GetPictureTime(bild, file);
   if ( !GetSpec(bild->spec, ChunkNameSoftware) )
      CopySpec(bild->spec, ChunkNameSoftware, exename);
}


void CreateDescription(PICTURE *bild, char *was)
/* Erzeugt eine Bildbeschreibung
 * was+wann
 * --- Parameter: ---
 * PICTURE *bild : Bildstruktur
 * char    *was  : was ist damit
 * --- R"uckgabe: ---
 * PICTURE *bild : bild->spec ChunkNameDescription erzeugt
 */
{
   char     *desc, zeit[50];
   time_t   t;

   if ( was == NULL ) was = "";
   t = time(NULL);
   sprintf(zeit, " (%s", asctime(localtime(&t)));
   *(strchr(zeit, '\n')) = ')';
   assert ( (desc = malloc(strlen(was) + strlen(zeit) + 1) ) 
            != NULL ); 
   strcat(strcpy(desc, was), zeit);
   SetSpec(bild->spec, ChunkNameDescription, desc);
}
		       
