/***************************************************************************
 * sted 0.01 - Simple (and/or Stupid) Text Editor, by Linus Akerlund.
 *                e-mail: uxm165t@tninet.se
 *                homepage: http://user.tninet.se/~uxm165t/index.htm
 *
 * (C) 1998 Linus Akerlund
 *
 * sted is a little program that lets you edit text files. It's supposed
 * to be small and simple. The code is also simple, but as I'm not very
 * much of a programmer, it may not be as small as it should be. You'll
 * find lots of strange things if you take a look at the code, because I'm
 * not very good at this (yet?). If you find something that could have been
 * done in a better (more economical, more efficient) way, please send me
 * an e-mail and tell me about it, or just fix it and send me a patch. I've
 * seen quite a few things in the code that I can improve a lot, and I'll 
 * probably get around to doing something about that, before I start
 * implementing functions that makes this resemble a real text editor.
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *
 **************************************************************************/

 
/* added same public domain i18n by Ričardas Čepas <rch@WriteMe.Com> */

#define _BSD_SOURCE
#define _POSIX_C_SOURCE 2
#define _XOPEN_SOURCE_EXTENDED

#include <ctype.h>
#include <curses.h>
#include <langinfo.h>
#include <limits.h>
#include <locale.h>
#include <regex.h>
#include <signal.h>
#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
#include <term.h>
#include <wchar.h>

#define LINE_LENGTH 600		/* Maximum length of a line. */
#define TAB_SPACING 8		/* Number of characters per tab. */

#define FILE_NAME_LEN_MAX 257
 /* fix it - use erasechar too */
#define is_backspace(key_code, key_type) (((key_code) == KEY_BACKSPACE && (key_type) == KEY_CODE_YES) || (((key_code) == 8 || (key_code) == erasechar()) && (key_type) != KEY_CODE_YES))
/* ISO/IEC 10646-1:1993
  <backspace>     /x00/x08        BACKSPACE (BS)
*/
#define MAX(n, m) ((n) > (m) ? (n) : (m))
#define MIN(n, m) ((n) < (m) ? (n) : (m))
#define elements(array) (sizeof(array)/sizeof(array[0]))

struct line
  {				/* The storage for the text. */
    char *content;
    wchar_t *wcontent;
    int number;
    struct line *next;
  };

typedef struct line rad;	/* And a typedef, because I'm to lazy to type
				 * 'struct' all the time. */

void add_line (wchar_t *wbuffer, int line_number);
int Addnwstr(wchar_t *, int);
int addwcolumn(wchar_t *wcontent, int left_col, int realCOLS);
void clean_up (void);
void del_line (int line_number);
static void finish (int sig);
wchar_t *get_line (int line_number);
char *get_regerror (int errcode, regex_t *compiled_regex);
int Get_wch(wint_t *);
int load_file (char *file_name, int *Lines);
void modify_line (wchar_t *wbuffer, int line_number);
void process_load_err(int load_err, char *file_name);
void push_numbers (int line_number);
int save_file (char *file_name);
void show_list (int top_line, int left_col);
int Wcswidth(wchar_t *wbuffer, size_t n);
void *xmalloc (size_t size);
void *xrealloc (void *pointer, size_t size);
char yesno_dialog(int num_of_args, ...);

size_t myMB_LEN_MAX = MB_LEN_MAX > 6 ? MB_LEN_MAX : 6;
SCREEN *screen;
char *yesstr, *nostr, *yesexpr, *noexpr;
regex_t compiled_yesexpr, compiled_noexpr;
rad *head = NULL;		/* The head pointer for the linked list. */
int realCOLS;
  
/* skip byte order mark */
#define Wcwidth(wchar) ((wchar == 0xFEFF || wchar == '\r') ? 0 : \
    ((wchar < 0x3000 \
    || (wchar >= 0xF000 && wchar < 0xF900) \
    || (wchar >= 0xFF60 && wchar < 0xFFE0) \
    || wchar >= 0xFFE8) ? 1 : 2))

  int
Wcswidth(wchar_t *wbuffer, size_t n)
{
  int Columns, i;

  for (i=0, Columns=0; wbuffer[i] != 0 && i < n; i++)
    if (wbuffer[i] == '\t')
      Columns = (Columns+TAB_SPACING) / TAB_SPACING * TAB_SPACING;
    else
      Columns += Wcwidth(wbuffer[i]);
  return Columns;
}

int
Addnwstr(wchar_t *wstring, int max)
{
  char mbstr[myMB_LEN_MAX];
  int bytes, count=0, i, width;
  
  while (count < max && wstring[count] != 0)
    if (wstring[count] == '\t')
    {
      width = Wcswidth(wstring, count);
      for (i = width; i < (width+TAB_SPACING) / TAB_SPACING * TAB_SPACING; i++)
        addch(' ');
      count++;
    }
    else if ((bytes=wctomb(mbstr, wstring[count++])) > 0)
      addnstr(mbstr, bytes);
  return(TRUE);
}

int
addwcolumn(wchar_t *wcontent, int left_col, int realCOLS)
{
  int count2, width;

  mblen(NULL, 0); /* initialize the shift state */
  for (count2 = 0, width = 0; \
      width < (left_col + realCOLS - 1) && (wcontent[count2] != 0); \
      count2++)
    if (wcontent[count2] == '\t')
      width = (width+TAB_SPACING) / TAB_SPACING * TAB_SPACING;
    else
      width += Wcwidth(wcontent[count2]);
  if (width > (left_col + realCOLS - 1) && count2 > 0)
    count2--;
  return(Addnwstr(&(wcontent[left_col]), count2-left_col));
}

int Get_wch(wint_t *wintp)
{
  int c, mbcl=0, ret=0;
  char mbcs[myMB_LEN_MAX+3];
  wchar_t wc;
  wint_t key_code;
  int key_type;

  while (mbcl <= myMB_LEN_MAX)
    if ((wc = c = getch()) >= KEY_CODE_YES)
    {
      ret = KEY_CODE_YES;
      break;
    }
    else
    {
      mbcs[mbcl++] = c;
      if (mblen(mbcs, mbcl)<=0)
      {
	if (mbcl > myMB_LEN_MAX)
	  mbcl = 0;
	continue;
      }
      else
      {
	ret=mbtowc(&wc, mbcs, mbcl);
	break;
      }
    }
  if (ret != KEY_CODE_YES && wc == 'O' - '@')
  {
    key_type = Get_wch (&key_code);	/* accept single key as input */
    if (key_type != KEY_CODE_YES)
    {
      key_type = KEY_CODE_YES;
      switch (key_code)
      {
	case 'x':
	case 's':
	  wc = KEY_DC;
	  break;
	case 'S':
	case 'D':
	  wc = KEY_SDC;
	  break;
	case 'O':
	  wc = KEY_SIC;
	  break;
	case 'h':
	case 'H'-'@':
	  wc = KEY_LEFT;
	  break;
	case 'l':
	case ' ':
	  wc = KEY_RIGHT;
	  break;
	case '0':
	case '^':
	  wc = KEY_HOME;
	  break;
	case '$':
	  wc = KEY_END;
	  break;
	case 'k':
	case '+':
	case 'P'-'@':
	  wc = KEY_UP;
	  break;
	case 'j':
	case '-':
	case '_':
	case 'J'-'@':
	case 'N'-'@':
	  wc = KEY_DOWN;
	  break;
	case 'G':
	  wc = KEY_SEND;
	  break;
	case 'g':
	  wc = KEY_SHOME;
	  break;
	case 'w':
	case 'W':
	case 'e':
	case 'E':
	  wc = KEY_SRIGHT;
	  break;
	case 'b':
	case 'B':
	  wc = KEY_SLEFT;
	  break;
	case 'E'-'@':
	case 'D'-'@':
	case 'F'-'@':
	  wc = KEY_NPAGE;
	  break;
	case 'Y'-'@':
	case 'U'-'@':
	case 'B'-'@':
	  wc = KEY_PPAGE;
	  break;
	case 'z':
	  wc = KEY_CLEAR;
	  break;
	case 'L'-'@':
	  wc = KEY_CLEAR;
	  break;
	default:
	  key_type = ret;
      }
    }
   ret = key_type;
  }
  *wintp = wc;
  return (ret);
}

int
Move(int y, int x)
{
  move(y, x);
  refresh();
/*  printf("\x1B[%i;%iH", y+1, x+1); *//* ECMA-48 CUP */
  tputs(tparm(cursor_address, y, x), 1, &putchar);
  return(OK);
}

void
read_file_name(char *file_name)
{
  int count;
  wchar_t wfile_name[FILE_NAME_LEN_MAX+1];
  wint_t key_code;
  int key_type;

  move (12, 0);
  clrtoeol ();
  addstr ("Filename: ");
  refresh ();
  count = 0;
  do
  {
    key_type = Get_wch (&key_code);
    if (is_backspace(key_code, key_type))
    {
      if (count > 0)
      {
	wfile_name[--count] = '\0';
	move(12, 0);
	clrtoeol();
	refresh(); /* to prevent ncurses optimizations */
	addstr("Filename: ");
	Addnwstr (wfile_name, wcslen(wfile_name));
	refresh();
      }
    }
    else if (key_type != KEY_CODE_YES)
    {
      wfile_name[count] = key_code;
      Addnwstr (&wfile_name[count], 1);
      count++;
    }		    
    refresh ();
  }
  while (!((key_code == KEY_ENTER && key_type == KEY_CODE_YES) || key_code == '\n' || key_code == '\0'));

  wfile_name[count - 1] = '\0';
  wcstombs(file_name, wfile_name, myMB_LEN_MAX*FILE_NAME_LEN_MAX);
  file_name[myMB_LEN_MAX*FILE_NAME_LEN_MAX]='\0';
}

char *
get_regerror (int errcode, regex_t *compiled_regex)
{
  char *buffer;
  size_t length;

  buffer = xmalloc (length = regerror(errcode, compiled_regex, NULL, 0));
  regerror (errcode, compiled_regex, buffer, length);
  fprintf (stderr, "sted: Something is wrong in current locale's YESEXPR or NOEXPR \n %s", buffer);
  exit (1);
}

void *
xmalloc (size_t size)
{
  register void *pointer = malloc (size);

  if (pointer == NULL)
  {
    fprintf (stderr, "sted: Unable to allocate memory!");
    exit (1);
  }
  return pointer;
}

void *
xrealloc (void *pointer, size_t size)
{
  register void *new_pointer = realloc (pointer, size);

  if (new_pointer == NULL)
  {
    fprintf (stderr, "sted: Failed to reallocate memory!");
    exit (1);
  }
  return new_pointer;
}

  char
yesno_dialog(int num_of_args, ...)
{
  wint_t key_code;
  wchar_t wc;
  int i, key_type, mbcsl, reg_ret=0, ret=0;
  char mbcs[myMB_LEN_MAX+3];
  va_list arg_p;

  va_start (arg_p, num_of_args);
  move (11, 0);
  clrtoeol ();
  move (13, 0);
  clrtoeol ();
  move (12, 0);
  clrtoeol ();
  for (i=1; i <= num_of_args; i++)
    addstr(va_arg(arg_p, char *));
  va_end(arg_p);
  addstr("("); addstr(yesstr); addstr("/"); addstr(nostr); addstr(")");
  refresh ();
  key_type = Get_wch (&key_code);
  wc = key_code;
  if (key_type != KEY_CODE_YES && (mbcsl = wctomb(mbcs, wc)) > 0)
  {
    mbcs[mbcsl] = '\0';
    reg_ret = regexec(&compiled_yesexpr, mbcs, 0, NULL, 0);
    if (reg_ret == REG_NOMATCH)
    {
      reg_ret = regexec(&compiled_noexpr, mbcs, 0, NULL, 0);
      if (reg_ret == REG_NOMATCH)
	ret = 0;
      else if (reg_ret == 0)
	ret = 'n';
    }
    else if (reg_ret == 0)
      ret = 'y';

    if (reg_ret == REG_ESPACE)
    {
      fprintf (stderr, "sted: Out of memory in regexec().");
      exit (1);
    }
  }
  else
    ret = 0;
  return(ret);
}


/**************************************************************************
 * The main function
 * Returns: nothing (will change).
 * Arguments: the usual ones.
 *
 * Does some initializations and then starts waiting for input from the
 * user. Processes the input and stores it.
 **************************************************************************/

void
main (int argc, char *argv[])
{
  int count = 0, count2 = 0, x = 0, y = 0;
  int line_number = 0, x_pos = 0, x_col = 0, x_col_new = 0;
  int Lines = 0, *Lines_p = &Lines; 
  int top_line = 0, left_col = 0;
  int save_err = 0, load_err = 0, colour = 2, colour_on = 0;
  int errcode;
  wchar_t clipboard[LINE_LENGTH + 1], wbuffer[LINE_LENGTH + 1], temp_wbuffer[LINE_LENGTH + 1];
  char buf[20];
  char file_name[(FILE_NAME_LEN_MAX+1)*myMB_LEN_MAX];
  wchar_t wfile_name[FILE_NAME_LEN_MAX+1];
  wint_t key_code;
  int key_type;
  enum dir {LR, RL} dir = LR;
  char *Help[] = {
    "                     Motions:                        ",
    "Ctrl-O h = left                          Ctrl-O l, ^O space = left  ",
    "Shift-Left, ^O b, ^O B = beginning of word       ",
    "Shift-Right, ^O e, ^O E ^O w, ^O W = next word   ",
    "^O 0, ^O ^  = beginning of line          ^O $  = end of line ",
    "^O j, ^O -, ^O _, ^O ^J, ^O ^N  = down   ^O k, ^O +, ^O ^P  = up   ",
    "^O ^E, ^O ^D, ^O ^F  = next page   ^O ^Y, ^O ^U, ^O ^B  = previous page",
    "Shift-Home, ^O g  = beginning of file    Shift-End, ^O G  = end of file",
    "                     Changing text:                  ",
    "Backspace, ^H = delete backwards    Delete, ^O x, ^O s = delete forward",
    "Shift-Delete, ^O D, ^O S = delete line    Shift-Ins, ^O O = insert line",
    "                     Other:                          ",
    "F1 = help    F9 = quit   F11 = load file   F12 = save   ",
    "Clear key, ^L, ^O ^L = redraw screan",
    "" };

  for (count = 0; count <= LINE_LENGTH; count++)
    clipboard[count] = temp_wbuffer[count] = wbuffer[count] = '\0';
  for (count = 0; count <= FILE_NAME_LEN_MAX; count++)
    wfile_name[count] = '\0';
  for (count = 0; count <= myMB_LEN_MAX*FILE_NAME_LEN_MAX; count++)
    file_name[count] = '\0';

  (void) signal (SIGINT, finish); /* arrange interrupts to terminate */
  /* fix it - add SIGWINCH */
  
  setlocale (LC_ALL, "");
  if (*nl_langinfo(YESSTR) && *nl_langinfo(NOSTR))
  {
    yesstr = nl_langinfo(YESSTR);
    nostr = nl_langinfo(NOSTR);
  }
  else
  {
    yesstr = "Yes";
    nostr = "No";
  }
  if (*nl_langinfo(YESEXPR) && *nl_langinfo(NOEXPR))
  {
    yesexpr = nl_langinfo(YESEXPR);
    noexpr = nl_langinfo(NOEXPR);
  }
  else
  {
    yesexpr = "^[Yy]";
    noexpr = "^[Nn]";
  }
  if ((errcode=regcomp(&compiled_yesexpr, yesexpr, REG_NOSUB)))
    get_regerror(errcode, &compiled_yesexpr);
  if ((errcode=regcomp(&compiled_noexpr, noexpr, REG_NOSUB)))
    get_regerror(errcode, &compiled_noexpr);
  
  fputs("\033[?7l", stderr); /* wrap off */
  /* fool ncurses to prevent cutting of longer lines */
  screen = newterm(NULL, stdin, stdout);
  realCOLS = COLS; /* save real COLS value */
  sprintf(buf, "%i", MIN(COLS*myMB_LEN_MAX, 500)); /* why 500 ?? */
  setenv("COLUMNS", buf, TRUE);
  sprintf(buf, "%i", LINES);
  setenv("LINES", buf, TRUE);
  endwin ();
  (void) initscr ();	  /* initialize the curses library */
  keypad (stdscr, TRUE);  /* enable keyboard mapping */
  idlok (stdscr, TRUE);  /* enable hardware ins/del line */
  (void) cbreak ();	  /* take input chars one at a time, no wait for NL */
  (void) noecho ();	  /* don't echo input */
  meta (stdscr, TRUE);  /* don't mangle eight bit */
  refresh ();		  /* to clear the screen */

    if (has_colors ())
    {
      start_color ();
      /*
       * simple colour assigment, often all we need.
       * This comes straight from the ncurses tutorial...
       * No colours are used in this program.
       */
      init_pair (COLOR_BLACK, COLOR_BLACK, COLOR_BLACK);
      init_pair (COLOR_GREEN, COLOR_GREEN, COLOR_BLACK);
      init_pair (COLOR_RED, COLOR_RED, COLOR_BLACK);
      init_pair (COLOR_CYAN, COLOR_CYAN, COLOR_BLACK);
      init_pair (COLOR_WHITE, COLOR_WHITE, COLOR_BLACK);
      init_pair (COLOR_MAGENTA, COLOR_MAGENTA, COLOR_BLACK);
      init_pair (COLOR_BLUE, COLOR_BLUE, COLOR_BLACK);
      init_pair (COLOR_YELLOW, COLOR_YELLOW, COLOR_BLACK);
    } 

  if (argc > 1)
    {
      for (count2 = 1; count2 < argc; count2++)
	{
	  if (argv[count2][0] == '-')
	    {
	      if (argv[count2][1] == 'p')
		colour_on = 1;
	      count = 1;
	    }
	  else
	    {
	      strcpy (file_name, argv[count2]);
	      mbstowcs(wfile_name, file_name, FILE_NAME_LEN_MAX);
	      wfile_name[FILE_NAME_LEN_MAX] = '\0';
	      load_err = load_file (file_name, Lines_p);
	      process_load_err(load_err, file_name);
	      if (load_err < 7)
		wcscpy (wbuffer, get_line (0));
	      else
		wbuffer[0] = 0;
	    }
	  show_list (top_line, left_col);
	  refresh ();
	  move (0, 0);
	}
    }
/* This is the main loop, which lets you input characters, and processes
 * them in a more or less appropriate way. */
  getsyx (y, x);		/* Get cursor coordinates */
  while (1)
  {
    key_type = Get_wch (&key_code);	/* accept single key as input */

    if ((key_code == KEY_ENTER && key_type == KEY_CODE_YES)
	|| key_code == '\n' 	/* It's ugly, isn't it? Enter... */ \
        || key_code == '\0'
    	|| (key_type != KEY_CODE_YES \
	    && (key_code == 0x2028 || key_code == 0x2029)))
    {
      if (x_pos != wcslen(wbuffer) || left_col != 0)
	/*touchline(stdscr, y, 2);*/
	clearok(stdscr, TRUE);
      wcscpy(temp_wbuffer, &wbuffer[x_pos]);
      wbuffer[x_pos] = '\0';
      move (y, 0);
      push_numbers (line_number);
      add_line (wbuffer, line_number);
      wcscpy(wbuffer, temp_wbuffer);
      if (y == LINES - 1)
      {
	++top_line;
	move (y, 0);
      }
      else
	move (++y, 0);
      ++line_number;
      ++Lines;
      left_col = x_pos = x_col = 0;
      move(y, 0);
    }
    else if ((key_code == KEY_SIC || key_code == KEY_IL) && key_type == KEY_CODE_YES)
    {
      clearok(stdscr, TRUE);
      push_numbers (line_number);
      wcscpy(wbuffer, clipboard);
      add_line (wbuffer, line_number);
      ++Lines;
      left_col = x_pos = x_col = 0;
      move(y, 0);
    }
    else if (is_backspace(key_code, key_type))
    {
      /* Backspace is not delete */
      if (x_pos == 0 && line_number == 0)
	continue;              /* Removed all the beeps */
      else if (x_pos == 0)
      {
	wcscpy (temp_wbuffer, get_line (line_number - 1));
	if (wcslen (temp_wbuffer) + wcslen (wbuffer) <
	    LINE_LENGTH)
	{
	  if (Lines > line_number)
	  {
	    del_line (line_number);
	    --Lines;
	  }
	  --line_number;
	  x_pos = wcslen (temp_wbuffer);
	  x_col = Wcswidth(temp_wbuffer, x_pos);
	  if (x_col >= realCOLS - 1)
	  {
	    left_col = x_col - (realCOLS - 1);
	  }
	  else
	    left_col = 0;
	  clearok(stdscr, TRUE);
	  wcscat (temp_wbuffer, wbuffer);
	  wcscpy (wbuffer, temp_wbuffer);

	  if (y == 0)
	  {
	    --top_line;
	    move (y, 0);
	  }
	  else
	    move (--y, 0);
	  for (count = 0; count < LINE_LENGTH; count++)
	    temp_wbuffer[count] = '\0';
	}
	else
	  continue;            /* Removed all the beeps. */
      }
      else
      {
	for (count = x_pos - 1; count < wcslen (wbuffer); count++)
	  wbuffer[count] = wbuffer[count + 1];
	wbuffer[wcslen (wbuffer) + 1] = '\0';
	x_col = Wcswidth(wbuffer, --x_pos);
	if (left_col > 0 && x_col<=left_col)
	  left_col = x_col;
      }
    }
    else if (key_code == KEY_DC && key_type == KEY_CODE_YES)
    {
      /* Delete key, the way I want it to work */
      if (x_pos == wcslen (wbuffer))
      {
	if (line_number >= Lines - 1)
	  continue;          /* Removed all the beeps. */
	else
	{
	  wcscpy (temp_wbuffer, get_line (line_number + 1));
	  if (wcslen (temp_wbuffer) + wcslen (wbuffer) <
	      LINE_LENGTH)
	  {
	    del_line (line_number + 1);
	    --Lines;
	    wcscat (wbuffer, temp_wbuffer);
	    for (count = 0; count < LINE_LENGTH; count++)
	      temp_wbuffer[count] = '\0';
	  }
	  else
	    continue;       /* Removed all the beeps. */
	}
      }
      else
      {
	for (count = x_pos; count < wcslen (wbuffer); count++)
	  wbuffer[count] = wbuffer[count + 1];
	wbuffer[wcslen (wbuffer) + 1] = '\0';
      }
    }
    else if ((key_code == KEY_SDC || key_code == KEY_DL) \
	&& key_type == KEY_CODE_YES)
    {
      clearok(stdscr, TRUE);
      wcscpy (clipboard, wbuffer);
      wbuffer[0] = 0;
      x_pos = 0;
      x_col = 0;
      left_col = 0;
      if (Lines > 1)
      {
	  del_line (line_number);
	  --Lines;
	  if (line_number >= Lines)
	  {
	    if (y > 0)
	      y--;
	    else
	      top_line--;
	    line_number = Lines - 1;
	  }
	  wcscpy (wbuffer, get_line (line_number));
      }
      else
	modify_line (wbuffer, line_number);
    }
    else if ((key_code == KEY_LEFT || key_code == KEY_SLEFT) \
	&& key_type == KEY_CODE_YES)
    {
      if (line_number == 0 && x_pos == 0)
	continue;               /* Removed all the beeps. */
      /* Cursor left. */
      if (x_col == 0 && left_col == 0)
      {
	{
	  if (line_number == Lines)
	  {
	    add_line (wbuffer, line_number);
	    ++Lines;
	  }
	  else
	    modify_line (wbuffer, line_number);
	  wcscpy (wbuffer, get_line (--line_number));
	  if (wbuffer == NULL)
	  {
	    fprintf (stderr, "Couldn't get any line!");
	    exit (1);
	  }
	  if (y == 0)
	  {
	    --top_line;
	    move (y, 0);
	  }
	  else
	    move (--y, 0);
	  x_pos = wcslen (wbuffer);
	  x_col = Wcswidth(wbuffer, x_pos);
	  if (x_col >= realCOLS - 1)
	  {
	    left_col = x_col - (realCOLS - 1);
	    clearok(stdscr, TRUE);
	  }
	  else
	    left_col = 0;
	}
      }
      else
      {
	--x_pos;
	if (key_code == KEY_SLEFT)
	{
	  while (x_pos && isspace((char)wbuffer[x_pos]))
	    --x_pos;
	  if (!isspace((char)wbuffer[x_pos]))
	  {
	    while (x_pos && !isspace((char)wbuffer[x_pos]))
	      --x_pos;
	    if (isspace((char)wbuffer[x_pos]))
	      ++x_pos;
	  }
	}
	x_col = Wcswidth(wbuffer, x_pos);
	if (x_col<=left_col)
	  left_col = x_col;
      }
    }
    else if ((key_code == KEY_RIGHT || key_code == KEY_SRIGHT) \
       && key_type == KEY_CODE_YES)
    {
      /* Cursor right. */
      if (x_pos == wcslen(wbuffer))
      {
	if (line_number >= Lines - 1)
	  continue;             /* Removed all the beeps. */
	else
	{
	  modify_line (wbuffer, line_number);
	  wcscpy (wbuffer, get_line (++line_number));
	  if (wbuffer == NULL)
	  {
	    fprintf (stderr, "Couldn't get any line!");
	    exit (1);
	  }
	  if (y == LINES - 1)
	  {
	    ++top_line;
	    move (y, 0);
	  }
	  else
	    move (++y, 0);
	  x_pos = 0; x_col = 0;
	  if (left_col > 0)
	    clearok(stdscr, TRUE);
	  left_col = 0;
	  /*move (y, 0);*/
	}
      }
      else 
      {
	if (key_code == KEY_SRIGHT)
	{
	  while (x_pos < wcslen(wbuffer) && !isspace((char)wbuffer[x_pos]))
	    ++x_pos;
	  while (x_pos < wcslen(wbuffer) && isspace((char)wbuffer[x_pos]))
	    ++x_pos;
	}
	else
	  ++x_pos;
	x_col_new = Wcswidth(wbuffer, x_pos);
	if (x_col >= (realCOLS - 1))
	  left_col += x_col_new - x_col;
	x_col = x_col_new;
      }
    }
    else if ((key_code == KEY_HOME || key_code == KEY_BEG) \
	&& key_type == KEY_CODE_YES)
    {
      /* Home: to beginning of line. */
      x_pos = 0;
      x_col = 0;
      left_col = 0;
      clearok(stdscr, TRUE);
    }
    else if (key_code == KEY_END && key_type == KEY_CODE_YES)
    {
      /* End: to end of line. */
      x_col = Wcswidth(wbuffer, x_pos = wcslen(wbuffer));
      if ((x_col) > (left_col + (realCOLS - 1)))
      {
	left_col = x_col - (realCOLS - 1);
	clearok(stdscr, TRUE);
      }
    }
    else if (key_code == KEY_END && key_type == KEY_CODE_YES)
    {
      /* End: to end of line. */
      x_col = Wcswidth(wbuffer, x_pos = wcslen(wbuffer));
      if ((x_col) > (left_col + (realCOLS - 1)))
      {
	left_col = x_col - (realCOLS - 1);
	clearok(stdscr, TRUE);
      }
    }
    else if (key_code == KEY_UP && key_type == KEY_CODE_YES)
    {
      /* Cursor up. */
      if (line_number == 0)
	continue;              /* Removed all the beeps. */
      else
      {
	if (line_number == Lines)
	{
	  add_line (wbuffer, line_number);
	  ++Lines;
	}
	else
	  modify_line (wbuffer, line_number);
	wcscpy (wbuffer, get_line (--line_number));
	if (wbuffer == NULL)
	{
	  fprintf (stderr, "Couldn't get any line!");
	  exit (1);
	}
	if (y == 0)
	{
	  --top_line;
	  move (y, 0);
	}
	else
	  move (--y, 0);
	if (x_pos > wcslen (wbuffer))
	  x_pos = wcslen (wbuffer);
	x_col = Wcswidth (wbuffer, x_pos); /* fix it - x_col should not change */
	if (x_col < left_col)
	  left_col = x_col;
      }
    }
    else if (key_code == KEY_DOWN && key_type == KEY_CODE_YES)
    {
      /* Cursor down. */
      if (line_number >= Lines - 1)
	continue;              /* Removed all the beeps. */
      else
      {
	modify_line (wbuffer, line_number);
	wcscpy (wbuffer, get_line (++line_number));
	if (wbuffer == NULL)
	{
	  fprintf (stderr, "Couldn't get any line!");
	  exit (1);
	}
	if (y == LINES - 1)
	{
	  ++top_line;
	  move (y, 0);
	}
	else
	  move (++y, 0);
	if (x_pos > wcslen (wbuffer))
	  x_pos = wcslen (wbuffer);
	x_col = Wcswidth (wbuffer, x_pos); /* fix it - x_col should not change */
	if (x_col < left_col)
	  left_col = x_col;
      }
    }
    else if ((key_code == KEY_NPAGE || key_code == KEY_NEXT) && key_type == KEY_CODE_YES)
    {
      if (line_number >= Lines - 1)
	continue;		/* Removed all the beeps. */
      else
      {
	modify_line (wbuffer, line_number);
	if (y == LINES - 1)
	{
	  line_number += (LINES - 1);
	  if (line_number >= Lines)
	    line_number = Lines - 1;
	  top_line = line_number - (LINES - 1);
	}
	else
	{
	  y = MIN(LINES - 1, Lines-1 - top_line);
	  line_number = MIN(top_line + (LINES - 1), Lines-1);
	}
	wcscpy (wbuffer, get_line (line_number));
	if (x_pos > wcslen (wbuffer))
	  x_pos = wcslen (wbuffer);
	x_col = Wcswidth (wbuffer, x_pos);
	left_col = x_col - (realCOLS - 1);
	if (left_col < 0)
	  left_col = 0;
	clearok(stdscr, TRUE);
      }
    }
    else if (key_code == KEY_SEND && key_type == KEY_CODE_YES)
    {
      modify_line (wbuffer, line_number);
      line_number = Lines - 1;
      top_line = line_number - (LINES - 1);
      if (top_line < 0)
	top_line = 0;
      y = MIN(LINES - 1, line_number);
      wcscpy (wbuffer, get_line (line_number));
      x_pos = wcslen (wbuffer);
      x_col = Wcswidth (wbuffer, x_pos);
      left_col = x_col - (realCOLS - 1);
      if (left_col < 0)
	left_col = 0;
      clearok(stdscr, TRUE);
    }
    else if ((key_code == KEY_SHOME || key_code == KEY_SBEG) \
      && key_type == KEY_CODE_YES)
    {
	if (line_number >= Lines)
	{
	  add_line (wbuffer, line_number);
	  ++Lines;
	}
	else
	  modify_line (wbuffer, line_number);
	y = line_number = top_line = 0;
	wcscpy (wbuffer, get_line (line_number));
	x_pos = x_col = left_col = 0;
	clearok(stdscr, TRUE);
    }
    else if ((key_code == KEY_PPAGE || key_code == KEY_PREVIOUS) && key_type == KEY_CODE_YES)
    {
      if (line_number == 0)
	continue;		/* Removed all the beeps. */
      else
      {
	if (line_number >= Lines)
	{
	  add_line (wbuffer, line_number);
	  ++Lines;
	}
	else
	  modify_line (wbuffer, line_number);
	if (y == 0)
	{
	  line_number -= (LINES - 1);
	  if (line_number < 0)
	    line_number = 0;
	  top_line = line_number;
	  y = 0;
	}
	else
	{
	  y = 0;
	  line_number = top_line;
	  if (line_number < 0)
	  {
	    line_number = 0;
	    y = 0;
	  }
	}
	wcscpy (wbuffer, get_line (line_number));
	if (x_pos > wcslen (wbuffer))
	  x_pos = wcslen (wbuffer);
	x_col = Wcswidth (wbuffer, x_pos);
	left_col = x_col - (realCOLS - 1);
	if (left_col < 0)
	  left_col = 0;
	clearok(stdscr, TRUE);
      }
    }
    else if ((key_code == (KEY_F0 + 1) || key_code == KEY_HELP) \
	&& key_type == KEY_CODE_YES)
    {
      clear();
      for (count=0; count < elements(Help); count++)
	mvaddstr(count+1, 0, Help[count]);
      mvaddstr(count+1, 0, "  Press any key to continue ");
      refresh ();
      key_type = Get_wch (&key_code);
      clearok(stdscr, TRUE);
    }
    else if ((key_code == KEY_EXIT || key_code == (KEY_F0 + 9)) && key_type == KEY_CODE_YES)
    {
      /* F9 key exits */
      if (yesno_dialog(1, "Quit without saving? ") == 'y')
	finish (0);
    }
    else if (key_code == (KEY_F0 + 10) && key_type == KEY_CODE_YES)
    { /* fix it - add menu */
    }
    else if ((key_code == KEY_OPEN || key_code == (KEY_F0 + 11)) && key_type == KEY_CODE_YES)
    {
      /* load a file */
      load_err = 0;
      if (yesno_dialog(1, "Load a file and clear the current buffer without saving? ") == 'y')
      {
	read_file_name(file_name);
	clean_up ();
	load_err = load_file (file_name, Lines_p);
	process_load_err(load_err, file_name);
	y = x = x_pos = x_col = 0;
	top_line = left_col = 0;
	line_number = 0;
	if (load_err < 7)
	  wcscpy (wbuffer, get_line (0));
	else
	  wbuffer[0] = 0;
      }
    }
    else if ((key_code == KEY_SAVE || key_code == (KEY_F0 + 12)) \
      && key_type == KEY_CODE_YES)
    {
      /* F12 saves the file */
      save_err = 0;
      if (line_number >= Lines)
      {
	add_line (wbuffer, line_number);
	++Lines;
      }
      else
	modify_line (wbuffer, line_number);
      while (strlen (file_name))
      {
	if (yesno_dialog(3, "Save as ``", file_name, "'' ") == 'y')
	{
	  save_err = save_file (file_name);
	  if (save_err == 1)
	  {
	    move (12, 0);
	    clrtoeol ();
	    addstr ("Error saving file!");
	    refresh ();
	    key_type = Get_wch (&key_code);
	  }
	  break;
	}
	else
	{
	  file_name[0] = '\0';
	  continue;
	}
      }
      if (!strlen (file_name))
      {
	read_file_name(file_name);
	save_err = save_file (file_name);
	if (save_err == 1)
	{
	  move (12, 0);
	  clrtoeol ();
	  addstr ("Error saving file!");
	  refresh ();
	  key_type = Get_wch (&key_code);
	}
      }
    }
    else if ((key_code == (KEY_CLEAR) && key_type == KEY_CODE_YES) || (key_code == 'L'-'@' && key_type != KEY_CODE_YES))
    {
      clearok(stdscr, TRUE);
    }
    else if (key_code == 'B'-'@' && key_type != KEY_CODE_YES)
    {
      if (dir == LR)
	dir=RL;
      else
	dir=LR;
      /* somebody should end this */
    }
    else if (key_code == 'O'-'@' && key_type != KEY_CODE_YES)
    {
    }
    else if (key_type != KEY_CODE_YES)	/* Most characters... */
    {
      if (wcslen (wbuffer) >= LINE_LENGTH - 1)
	continue;		/* Removed all the beeps. */
      else
      {
	for (count = wcslen (wbuffer) + 1; count > x_pos; count--)
	  wbuffer[count] = wbuffer[count - 1];
	wbuffer[x_pos++] = key_code;
	x_col = Wcswidth(wbuffer, x_pos);
	while (x_col - left_col >= realCOLS - 1) 
	{
	  ++left_col;
	}
      }
    }

    show_list(top_line, left_col);
    move (y, 0);
    clrtoeol ();
    refresh();/* to trick ncurses */
    addwcolumn(wbuffer, left_col, realCOLS);
    if (colour_on == 1)
    {
      attrset(COLOR_PAIR(colour++));
      if (colour > 7)
	colour = 1;
    }
    Move(y, x_col - left_col);
    refresh ();		/* Updating the screen after each character. */
  }
  finish (0);			/* This is the end... */
}

/**************************************************************************
 * the finish function
 * Returns: nothing
 * Arguments: basically nothing.
 *
 * Just cleans up and returns you to the shell.
 **************************************************************************/

static void
finish (int sig)
{
  clean_up ();

  endwin ();

  delscreen (screen);

  fputs("\033[?7h", stderr); /* wrap on */

  exit (0);
}

/**************************************************************************
 * The clean_up() function.
 * Returns: nothing
 * Arguments: none
 *
 * Just goes through the linked lists and free()s all allocated memory.
 *************************************************************************/

void
clean_up (void)
{
  rad *temp = head, *prev = NULL;

  if (head != NULL)
    {
      while (temp != NULL)	/* Freeing all allocated memory. */
	{
	  free (temp->content);
	  free (temp->wcontent);
	  prev = temp->next;
	  free (temp);
	  temp = prev;
	}
    }
  head = NULL;
}

/**************************************************************************
 * The push_numbers function.
 * Returns: nothing (probably should, though)
 * Arguments: line_number, the line at which to begin pushing.
 *
 * To make place for a new line, it pushes all the line numbers from that
 * line up one step, so that we don't end up with several lines with the
 * same line numbers.
 **************************************************************************/

void
push_numbers (int line_number)
{
  rad *temp = head;

  while (temp != NULL)
    {
      if (temp->number == line_number)
	{
	  while (temp != NULL)
	    {
	      ++temp->number;
	      temp = temp->next;
	    }
	  return;
	}
      temp = temp->next;
    }
}

/*************************************************************************
 * The add_line function. Allocates memory and adds a line to the text.
 * Returns: nothing (probably should, though)
 * Arguments: buffer, the line to be stored.
 *            line_number, the number of the line to store.
 *
 * Linked list, not anything I'm very good at, but it seems to work quite
 * well.
 *************************************************************************/

void
add_line (wchar_t *wbuffer, int line_number)
{
  rad *new = NULL, *temp = NULL, *prev = NULL;

  new = (rad *) xmalloc (sizeof (rad));
  new->wcontent = (wchar_t *) xmalloc (sizeof(wchar_t) * (wcslen (wbuffer) + 1));
  new->content = (char *) xmalloc (myMB_LEN_MAX*(wcslen (wbuffer) + 1));
  wcstombs(new->content, wbuffer, myMB_LEN_MAX*wcslen(wbuffer));
  new->content[myMB_LEN_MAX*wcslen(wbuffer)]='\0';
  wcscpy (new->wcontent, wbuffer);
  new->number = line_number;
  new->next = NULL;


  if (head == NULL)
    {
      head = new;
      new->next = NULL;
    }
  else
    {
      if (new->number < head->number)
	{
	  new->next = head;
	  head = new;
	}
      else
	{
	  temp = head->next;
	  prev = head;

	  if (temp == NULL)
	    {
	      prev->next = new;
	      new->next = NULL;
	    }
	  else
	    {
	      while ((temp->next != NULL))
		{
		  if (new->number < temp->number)
		    {
		      new->next = temp;
		      prev->next = new;
		      break;
		    }
		  else
		    {
		      temp = temp->next;
		      prev = prev->next;
		    }
		}

	      if (temp->next == NULL)
		{
		  if (new->number < temp->number)
		    {
		      new->next = temp;
		      prev->next = new;
		    }
		  else
		    {
		      temp->next = new;
		      new->next = NULL;
		    }
		}
	    }
	}
    }
}

/*************************************************************************
 * The modify_line function. Reallocates memory and modifies a previously
 *                           stored line in memory.
 * Returns: nothing (probably should, though).
 * Arguments: buffer, the line to be stored.
 *            line_number, the number of the line to be stored.
 *
 * Self-explanatory, I suppose. Just reallocates memory and puts the
 * buffer into the struct, instead of the old line.
 *************************************************************************/

void
modify_line (wchar_t *wbuffer, int line_number)
{
  rad *temp = head;

  while (temp != NULL)
    {
      if (temp->number == line_number)
	{
	  temp->content[0] = '\0';
	  temp->content = (char *) xrealloc (temp->content, sizeof(wchar_t)*(wcslen (wbuffer) + 1)*myMB_LEN_MAX);
	  temp->wcontent[0] = '\0';
	  temp->wcontent = (wchar_t *) xrealloc (temp->wcontent, sizeof(wchar_t)*(wcslen (wbuffer) + 1));
	  wcstombs(temp->content, wbuffer, myMB_LEN_MAX*wcslen(wbuffer));
	  temp->content[myMB_LEN_MAX*wcslen(wbuffer)]='\0';
	  wcscpy (temp->wcontent, wbuffer);
	  return;
	}
      temp = temp->next;
    }
}

/**************************************************************************
 * The del_line function
 * Returns: nothing (probably should)
 * Arguments: line_number, the line to be deleted
 *
 * Goes through the linked list until it finds the line to delete, deletes
 * it and free()s the memory, after linking together the previous and next
 * structs. Then it decrements the line numbers of each of the following
 * structs, so that we don't get any holes in the linked list.
 **************************************************************************/

void
del_line (int line_number)
{
  rad *temp = head, *prev = head;

  if (head->number == line_number)
    {
      head = temp->next;
      free (temp->content);
      free (temp->wcontent);
      free (temp);
    }
  else
    {
      while (temp != NULL)
	{
	  prev = temp;
	  temp = temp->next;
	  if (temp->number == line_number)
	    {
	      prev->next = temp->next;
	      free (temp->content);
	      free (temp->wcontent);
	      free (temp);
	      break;
	    }
	}
    }
  temp = head;
  while (temp != NULL)
    {
      if (temp->number >= line_number)
	--temp->number;
      temp = temp->next;
    }
}

/**************************************************************************
 * The get_line function. Finds a specific line and returns it.
 * Returns: a pointer to the string that was found.
 * Arguments: line_number, the number of the line to find.
 *
 * Searches through the list to see if there's a line that corresponds to
 * the one we're after.
 *************************************************************************/

wchar_t *
get_line (int line_number)
{
  rad *temp = head;

  while (temp != NULL)
    {
      if (temp->number == line_number)
	return temp->wcontent;
      else
	temp = temp->next;
    }
  return NULL;
}

/*************************************************************************
 * The show_list function, prints what's in the memory.
 * Returns: nothing
 * Arguments: y, the current y coordinate, x, the current x coordinate,
 * top_line, the first line to print, left_col, the leftmost column.
 *
 * Prints the lines that we've stored in the linked list to the screen.
 * And of course it doesn't print more than the current screen-full.
 **************************************************************************/

void
show_list (int top_line, int left_col)
{
  rad *temp = head;
  int count;

  erase ();

  for (count = 0; (temp != NULL) && ((count - top_line) < LINES); count++)
    {
      if (temp->number >= top_line)
      {
	move (count - top_line, 0);
	addwcolumn(temp->wcontent, left_col, realCOLS);
      }
      temp = temp->next;
    }
}

/**************************************************************************
 * The save_file() function
 * Returns: 0 on success, 1 if there was some error.
 * Arguments: file_name, the file name (wasn't that quite obvious?)
 *
 * Opens a file, writes to it with fputc(), closes it.
 **************************************************************************/

int
save_file (char *file_name)
{
  char buffer[(LINE_LENGTH + 1)*myMB_LEN_MAX];
  int error = 0;
  rad *temp = head;
  FILE *fp;

  if ((fp = fopen (file_name, "w")) != NULL)
    {
      while (temp != NULL)
	{
	  wcstombs(buffer, temp->wcontent, LINE_LENGTH*myMB_LEN_MAX);
	  buffer[LINE_LENGTH*myMB_LEN_MAX] = 0;
	  fputs(buffer, fp);
	  fputc('\n', fp);
	  temp = temp->next;
	}
      fclose (fp);
    }
  else
    error = 1;

  return error;
}

/***************************************************************************
 * The load_file() function
 * Returns: 0 on success, 1 on failure
 * Arguments: file_name, the file name and *lines, a pointer the number of
 *            liness, so that I can modify it.
 * 
 * Quite simply, it opens a file, reads it line by line, puts in spaces
 * instead of tabs, and stores everything with the add_line()-function.
 **************************************************************************/

int
load_file (char *file_name, int *Lines_p)
{
  char *buffer;
  int BUFSZ = 4096, count, error = 0, is_eol, mbcharlen, next_line_num = 0;
  size_t buf_end_ind, buf_ind, chars_read, wbuf_ind;
  FILE *fp;
  wchar_t wbuffer[LINE_LENGTH + 1];
  wchar_t eols[]= { 0x2028, 0x2029, '\n', '\r', 0 };

  if ((fp = fopen (file_name, "r")) != NULL)
    {
      buffer = xmalloc(BUFSZ);
      wbuf_ind = buffer[0] = buf_ind = buf_end_ind = 0;
      chars_read = 1;
      mblen(NULL, 0); /* initialize the shift state */
      do
      {
	if (buf_end_ind-buf_ind < 4*myMB_LEN_MAX && chars_read)
	{
	  memmove (buffer, &buffer[buf_ind], BUFSZ - buf_ind);
	  buf_end_ind -= buf_ind;
	  buf_ind = 0;
	  chars_read=fread(&buffer[buf_end_ind], 1, BUFSZ - 1 - buf_end_ind, fp);
	  if (ferror(fp))
	  {
	    free(buffer);
	    error=8;
	    goto eERR;
	  }
	  buf_end_ind += chars_read;
	  buffer[buf_end_ind]='\0';
	}
	if ((wbuf_ind+1 >= elements(wbuffer)-1) || buffer[buf_ind] == '\0')
	  wbuffer[wbuf_ind] = 0;
	else
	{
	  if ((mbcharlen=mblen(&buffer[buf_ind], myMB_LEN_MAX)) < 0)
	  {
	    if (buffer[buf_ind] < 0xA0)
	      wbuffer[wbuf_ind] = 0xFFFD; /* replacement character */
	    else
	      wbuffer[wbuf_ind] = buffer[buf_ind];

	    mblen(NULL, 0); /* initialize the shift state */
	  }
	  else
	    mbcharlen=mbtowc(&wbuffer[wbuf_ind], &buffer[buf_ind], myMB_LEN_MAX);
	  buf_ind += MAX(1, mbcharlen);
	}
	is_eol = FALSE;
	for (count=0; eols[count] != 0; count++)
	    if (eols[count] == wbuffer[wbuf_ind])
		is_eol = TRUE;
	if (is_eol || wbuffer[wbuf_ind] == 0 || buf_ind == buf_end_ind)
	{
	  if (wbuffer[wbuf_ind]!='\n' && is_eol)
	    error=1; /* non-Unix EOL */
	  if (wbuffer[wbuf_ind]=='\r' && buffer[buf_ind]=='\n')
	    buf_ind++;
	  if (is_eol)
	    wbuffer[wbuf_ind]=0;
	  else if (buf_ind == buf_end_ind)
	    wbuffer[wbuf_ind+1]=0;
	  add_line (wbuffer, next_line_num);
	  next_line_num++;
	  wbuf_ind=0;
	  mblen(NULL, 0); /* initialize the shift state */
	}
	else 
	  wbuf_ind++;
      }
      while (chars_read || buf_ind < buf_end_ind);
      free (buffer);
    }
  else
eERR:
    error=7;

  *Lines_p = next_line_num;

  return error;
}

void
process_load_err(int load_err, char *file_name)
{
  wint_t key_code;
  int count;

  if (load_err != 0)
  {
    move (12, 0);
    clrtoeol ();
    if (load_err == 1)
      addstr ("Opening file ``");
    else
      addstr ("Could not open file ``");
    for (count = 0; count < (strlen (file_name)); count++)
      addch (file_name[count]);
    if (load_err == 1)
    {
      addstr ("'' with non-Unix end-of-line format.");
      move (13, 0);
      clrtoeol ();
      addstr (" It will be saved in Unix format.");
    }
    else
      addstr ("''. Starting with empty buffer.");
    refresh ();
    Get_wch (&key_code);
    move (13, 0);
    clrtoeol ();
    move (13, 0);
    clrtoeol ();
    refresh ();
  }
}

