/*
 * SXPDBR.C - PDB read functionality in SX
 *
 * Source Version: 3.0
 * Software Release #92-0043
 *
 */

#include "cpyright.h"
 
#include "sx.h"

/*--------------------------------------------------------------------------*/

#define READ_IO(pp, ptype)                                                   \
    READ_IO1(pp, ptype)                                                      \
    READ_IO2(pp, ptype)                                                      \
    READ_IO3(pp, ptype)

/*--------------------------------------------------------------------------*/

#define READ_IO1(pp, ptype)                                                  \
    switch (SC_arrtype(obj, -1))                                             \
       {case SC_INTEGER_I:                                                   \
            *pp = (ptype) SS_INTEGER_VALUE(obj);                             \
            break;                                                           \
        case SC_FLOAT_I:                                                     \
            *pp = (ptype) SS_FLOAT_VALUE(obj);                               \
            break;

/*--------------------------------------------------------------------------*/

#define READ_IO2(pp, ptype)                                                  \
        case CONS:                                                           \
            if (SS_consp(SS_car(obj)))                                       \
	       obj = SS_car(obj);                                            \
            for (i = 0; i < nitems; i++)                                     \
                {obj1 = SS_car(obj);                                         \
                 switch (SC_arrtype(obj1, -1))                               \
                    {case SC_INTEGER_I:                                      \
                         pp[i] = (ptype) SS_INTEGER_VALUE(obj1);             \
                         break;                                              \
                     case SC_FLOAT_I:                                        \
                         pp[i] = (ptype) SS_FLOAT_VALUE(obj1);               \
                         break;                                              \
                     default:                                                \
                         SS_error("EXPECTED A NUMBER", obj1);};              \
	         obj1 = SS_cdr(obj);                                         \
	         if (obj1 != SS_null)                                        \
		    obj = obj1;}                                             \
             break;

/*--------------------------------------------------------------------------*/

#define READ_IO3(pp, ptype)                                                  \
        case VECTOR:                                                         \
            va = SS_VECTOR_ARRAY(obj);                                       \
            for (i = 0; i < nitems; i++)                                     \
                {switch (SC_arrtype(va[i], -1))                              \
                    {case SC_INTEGER_I:                                      \
                        pp[i] = (ptype) SS_INTEGER_VALUE(va[i]);             \
                        break;                                               \
                     case SC_FLOAT_I:                                        \
                        pp[i] = (ptype) SS_FLOAT_VALUE(va[i]);               \
                        break;                                               \
                     default:                                                \
                        SS_error("EXPECTED A NUMBER", va[i]);};};            \
            break;                                                           \
        default:                                                             \
            SS_error("EXPECTED A NUMBER", obj);}

/*--------------------------------------------------------------------------*/

void
 SC_DECLARE(_SX_rd_indirection_list,
         (object *obj, PDBfile *file, char **vr, char *type)), 
 SC_DECLARE(_SX_rd_leaf_list,
         (object *obj, PDBfile *file, char *vr,
          long nitems, char *type, dimdes *dims)), 
 SC_DECLARE(_SX_rd_io_list, 
            (object *obj, char *vr, long nitems, char *type));

syment
 SC_DECLARE(*_SX_rd_data,
         (PDBfile *file, char *name, syment *ep, SC_address *addr,
          object *name_obj));

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_RD_TREE_LIST - read NITEMS of TYPE from the PDBfile FILE
 *                  - into the location pointed to by VR
 *                  - return the number of items successfully read
 */

void _SX_rd_tree_list(obj, file, vr, nitems, type, dims)
   object *obj;
   PDBfile *file;
   char *vr;
   long nitems;
   char *type;
   dimdes *dims;
   {long i;
    char **lvr;
    char *dtype;

    if (!_PD_indirection(type))
       _SX_rd_leaf_list(obj, file, vr, nitems, type, dims);
    else
       {lvr = (char **) vr;
        dtype = PD_dereference(SC_strsavef(type,
                               "char*:_SX_RD_TREE_LIST:dtype"));
        for (i = 0L; i < nitems; i++, obj = SS_cdr(obj))
            _SX_rd_indirection_list(SS_car(obj), file, &lvr[i], dtype);
        SFREE(dtype);};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_RD_INDIRECTION_LIST - read the information about an indirection,
 *                         - allocate the space, connect the pointer, and
 *                         - read in the data
 */

void _SX_rd_indirection_list(obj, file, vr, type)
   object *obj;
   PDBfile *file;
   char **vr;
   char *type;
   {long bytepitem, nitems;
    char *pv;

    nitems = (long) _SX_get_object_length(obj);

    if (nitems == 0L)
       *vr = NULL;
    else
       {bytepitem = _PD_lookup_size(type, file->host_chart);
        if (bytepitem == -1)
           SS_error("CAN'T FIND NUMBER OF BYTES - _SX_RD_INDIRECTION_LIST",
                    SS_null);

        DEREF(vr) = pv = SC_alloc(nitems, bytepitem,
                                  "_SX_RD_INDIRECTION_LIST:pv");
        _SX_rd_tree_list(obj, file, pv, nitems, type, (dimdes *) NULL);};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_RD_LEAF_LIST - read NITEMS of TYPE from the PDBfile FILE
 *                  - into the location pointed to by VR
 *                  - at this level it guaranteed that the type will
 *                  - not be a pointer
 *                  - return the number of items successfully read
 */

void _SX_rd_leaf_list(obj, file, vr, nitems, type, dims)
   object *obj;
   PDBfile *file;
   char *vr;
   long nitems;
   char *type;
   dimdes *dims;
   {long i, sz;
    defstr *dp;
    memdes *desc, *mem_lst;
    object *obj1;
    char *svr;

/* if this was a derived type and some of its members are pointers
 * fetch in the pointered data
 */
    dp = PD_inquire_host_type(file, type);
    if (dp == NULL)
       SS_error("BAD TYPE - _SX_RD_LEAF_LIST", SS_null);

    mem_lst = dp->members;
    if (mem_lst == NULL)
/* use dp->type to get past typedef's */
        _SX_rd_io_list(obj, vr, nitems, dp->type);

/* for an array of structs write the indirects for each array element */
    else
       {sz  = dp->size;
	svr = vr;
	for (i = 0L; i < nitems; i++, svr += sz, obj  = SS_cdr(obj))
            {obj1 = SS_car(obj);

             for (desc = mem_lst; desc != NULL; desc = desc->next)
                 {_SX_rd_tree_list(SS_car(obj1), file,
				   svr + desc->member_offs,
                                   desc->number, desc->type,
				   desc->dimensions);
                  obj1 = SS_cdr(obj1);};};};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_RD_IO_LIST - if 'type' is a primitive type, display the data,
 *                - otherwise, lookup the type, and display each member.
 */

void _SX_rd_io_list(obj, vr, nitems, type)
   object *obj;
   char *vr;
   long nitems;
   char *type;
   {int i;
    short *sp;
    int *ip;
    long *lp;
    float *fp;
    double *dp;
    object *obj1,  **va;
    char msg[80];

/* print out the type */
    if (strcmp(type, "char") == 0)
       {switch (SC_arrtype(obj, -1))
           {case SC_STRING_I:
                strncpy(vr, SS_STRING_TEXT(obj), nitems);
                break;
            case CONS:
                strncpy(vr, SS_STRING_TEXT(SS_car(obj)), nitems);
                break;
            default:
                SS_error("EXPECTED A STRING - _SX_RD_IO_LIST", obj);};

        return;};

/* only need to check non-char types 
 * NOTE: the '\0' terminator on char strings cause the following
 *       test to fail unnecessarily (DRS.SCM for example)
 */
    if (nitems < (long) _SX_get_object_length(obj))
       {sprintf(msg, "DATA TOO LONG FOR TYPE %s - _SX_RD_IO_LIST",
                type);
        SS_error(msg, obj);};

    if (strcmp(type, "short") == 0)
       {sp = (short *) vr;
        READ_IO(sp, short)}

    else if ((strcmp(type, "integer") == 0) || (strcmp(type, "int") == 0))
       {ip = (int *) vr;
        READ_IO(ip, int)}

    else if (strcmp(type, "long") == 0)
       {lp = (long *) vr;
        READ_IO(lp, long)}

    else if (strcmp(type, "float") == 0)
       {fp = (float *) vr;
        READ_IO(fp, float)}

    else if (strcmp(type, "double") == 0)
       {dp = (double *) vr;
        READ_IO(dp, double)}

    else
       SS_error("ILLEGAL TYPE - _SX_RD_IO_LIST", SS_mk_string(type));

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_READ_NUMERIC_DATA - read data from a PDB file into a numeric array
 *                      - if file is missing, use the internal virtual file
 *                      -
 *                      - (pdb-read-numeric-data [file] name)
 */

object *SX_read_numeric_data(argl)
   object *argl;
   {char *name, *s;
    object *ret, *name_obj;
    PDBfile *file;
    syment *ep;
    SC_address addr;
    C_array *arr;

    if (!SS_consp(argl))
       SS_error("BAD ARGUMENT LIST - SX_READ_NUMERIC_DATA", argl);

/* see if the first object is a pdbfile, if so use it */
    argl = SX_get_pdbfile(argl, &file);

    name = NULL;
    SS_args(argl,
            SC_STRING_I, &name,
            0);

    s = name;
    name = _PD_expand_hyper_name(file, s);
    SFREE(s);
    if (name == NULL)
       return(SS_null);

/* look up the variable name and return FALSE if it is not there */
    ep = _PD_effective_ep(file, name, TRUE, NULL);
    if (ep == NULL)
       return(SS_null);

/* GOTCHA: this will die screaming if name is a member with a cast
 *         an error stack is needed to do this right
 */
    name_obj = SS_null;
    switch (setjmp(_PD_read_err))
       {case ABORT    : SS_error(PD_err, name_obj);
        default       : memset(PD_err, 0, MAXLINE);
                        ep = _SX_rd_data(file, name, ep, &addr, name_obj);
        case ERR_FREE : break;};

    arr = FMAKE(C_array, "SX_READ_NUMERIC_DATA:arr");
    if (!_PD_indirection(PD_entry_type(ep)))
       {arr->type   = PD_entry_type(ep);
	arr->length = PD_entry_number(ep);}
    else
       {PD_itag itag;
	char *dtype;

	dtype = PD_dereference(PD_entry_type(ep));
	arr->type = dtype;

	if (file == SX_vif)
	   arr->length = SC_arrlen(addr.memaddr)/
	                 _PD_lookup_size(dtype, file->chart);
	else
	   {io_seek(file->stream, PD_entry_address(ep), SEEK_SET);
	    _PD_rd_itag(file, &itag);
	    arr->length = itag.nitems;};};
    arr->data = addr.memaddr;

    ret = SX_mk_C_array(arr);

    _PD_rl_syment_d(ep);

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_RD_DATA - grab some data from the specified file */

syment *_SX_rd_data(file, name, ep, addr, name_obj)
   PDBfile *file;
   char *name;
   syment *ep;
   SC_address *addr;
   object *name_obj;
   {long n;
    char *type, *dtype;

    n    = PD_entry_number(ep);
    type = PD_entry_type(ep);

    if (!_PD_indirection(type))

       {if (!_PD_prim_typep(type, file->host_chart, PD_READ))
	   SS_error("MUST BE PRIMITIVE TYPE - _SX_RD_DATA", name_obj);

	if (file == SX_vif)
	   addr->diskaddr = PD_entry_address(ep);

	else
	   {addr->memaddr = _PD_alloc_entry(file, type, n);
	    PD_read(file, name, addr->memaddr);};}

    else

       {dtype = PD_dereference(SC_strsavef(type,
                               "char*:_SX_RD_DATA:dtype"));

	if (!_PD_prim_typep(dtype, file->host_chart, PD_READ))
	   SS_error("MUST BE PRIMITIVE TYPE - _SX_RD_DATA", name_obj);

	if (file == SX_vif)
	   {addr->diskaddr = PD_entry_address(ep);
	    addr->memaddr  = DEREF(addr->memaddr);}

	else
	    PD_read(file, name, &addr->memaddr);

	SFREE(dtype);};

    return(ep);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
