/*
 *	Ohio Trollius
 *	Copyright 1996 The Ohio State University
 *	GDB/RBD/NJN
 *
 *	$Id: rpcreate.c,v 6.1 96/11/24 00:26:05 nevin Rel $
 *
 * 	Function:	- creates a process on a given node
 * 			- executable already loaded with flatd
 * 	Accepts:	- target node ID
 *			- program file name
 * 			- process runtime flags
 * 			- arguments argv structure
 *			- process ID (out)
 *			- process index (out)
 * 	Returns:	- 0 or ERROR
 */

#include <lam_config.h>
#include <sfh.h>

#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <args.h>
#include <events.h>
#include <kio.h>
#include <ksignal.h>
#include <net.h>
#include <portable.h>
#include <preq.h>
#include <rreq.h>
#include <terror.h>
#include <typical.h>
#include <t_types.h>

/*
 * private functions
 */
static int		send_stdio_fds();

/*
 * external functions
 */
extern char		*getworkdir();

/*
 * external variables
 */
extern struct kio_t	_kio;

int
rpcreate(nodeid, filename, rtf, argv, pid, idx)

int4			nodeid;
CONST char		*filename;
int4			rtf;
char			**argv;
int4			*pid;
int4			*idx;

{
	struct nmsg	nhead;			/* network header */
	struct preq	request;		/* kenya request */
	struct preply	reply;			/* kenya reply */
	char		*cwd;			/* current working directory */
	char		*args;			/* glued argv structure */
	int		packsize;		/* size of packed argv */
	int4		mask;
	int4		tag;			/* program/argv flat tag */
	int		err;			/* return code */
	int		sd;			/* fd server file descriptor */
	char		server[LAM_PATH_MAX];	/* fd server socket name */
/*
 * Check trivial cases.
 */
	if (filename == 0) {
		errno = EINVAL;
		return(LAMERROR);
	}

	if (strlen(filename) >= PSMAXNAME) {
		errno = ENAMETOOLONG;
		return(LAMERROR);
	}
/*
 * Pack and tag the arguments.
 */
	if (argv) {
		tag = getpid();
		args = argvglue(argv, 0xa, MAXNMSGLEN);
		if (args == 0) return(LAMERROR);

		packsize = strlen(args) + 1;

		if (rflat(nodeid, args, (int4) packsize,
				(int4) packsize, tag)) {
			rflclean(nodeid, tag);
			return(LAMERROR);
		}

		free(args);
	} else {
		tag = 0;
	}
/*
 * Send the `create' request.
 */
	request.pq_req = ltot(PQCREATE);
	request.pq_rtf = ltot(rtf);
	request.pq_argv = ltot(tag);
	request.pq_jobid = ltot(_kio.ki_jobid);
	request.pq_stdin = ltot(_kio.ki_stdin);
	request.pq_stdout = ltot(_kio.ki_stdout);
	request.pq_stderr = ltot(_kio.ki_stderr);
	request.pq_world = ltot(_kio.ki_world);
	request.pq_parent = ltot(_kio.ki_parent);
	strcpy(request.pq_name, filename);

	request.pq_src_node = ((nodeid == LOCAL) ||
			(tiscast(getrtype(nodeid)))) ?
			ltot(nodeid) : ltot(getnodeid());
	request.pq_src_event = ltot(-((int4) getpid()));
/*
 * Set working directory.
 */
	if (rtf & RTF_CWD) {
		if ((cwd = getworkdir()) == 0) return(LAMERROR);
		strncpy(request.pq_wrkdir, cwd, LAM_PATH_MAX);
		free(cwd);
	}
/*
 * If passing file descriptors open the server.
 */
	if (rtf & RTF_PFDIO) {
		if (lam_mktmpid( (int) getpid(), server, sizeof(server))) {
			return(LAMERROR);
		}

		if ((sd = sfh_sock_open_srv_unix_stm(server)) < 0) {
			return(LAMERROR);
		}
	}

	nhead.nh_node = nodeid;
	nhead.nh_event = EVKENYAD;
	nhead.nh_flags = 0;
	nhead.nh_type = 0;
	nhead.nh_length = sizeof(request);
	nhead.nh_msg = (char *) &request;
	mask = ksigblock(sigmask(SIGUDIE) | sigmask(SIGARREST));

	if (nsend(&nhead)) {
		if (rtf & RTF_PFDIO) {
			close(sd);
			unlink(server);
		}
		ksigsetmask(mask);
		return(LAMERROR);
	}
/*
 * Pass the stdio file descriptors to kenyad.
 */
	if (rtf & RTF_PFDIO) {
		err = send_stdio_fds(sd);
		close(sd);
		unlink(server);
		if (err) {
			ksigsetmask(mask);
			return(LAMERROR);
		}
	}

	nhead.nh_event = -getpid();
	nhead.nh_length = sizeof(reply);
	nhead.nh_msg = (char *) &reply;

	if (nrecv(&nhead)) {
		ksigsetmask(mask);
		return(LAMERROR);
	}

	ksigsetmask(mask);

	if (reply.pr_reply) {
		errno = ttol(reply.pr_reply);
		return(LAMERROR);
	} else {
		*pid = ttol(reply.pr_pid);
		*idx = ttol(reply.pr_index);
		return(0);
	}
}


/*
 *	send_stdio_fds
 *
 *	Function:	- accept connection and send stdio file
 *			  descriptors to client
 *	Accepts:	- server file descriptor
 *	Returns:	- 0 or LAMERROR
 */
static int
send_stdio_fds(sd)

int			sd;

{
	int		pass_fd;		/* stream to pass fds over */
	
	if ((pass_fd = sfh_sock_accept_tmout(sd, -1)) < 0) {
		return(LAMERROR);
	}

	if (sfh_send_fd(pass_fd, 0)
		|| sfh_send_fd(pass_fd, 1) || sfh_send_fd(pass_fd, 2)) {

		close(pass_fd);
		return(LAMERROR);
	}

	close(pass_fd);
	return(0);
}
