/* 
   BrowserMatrix.m

   NSBrowser Matrix for Workspace's browser

   Copyright (C) 1996 Free Software Foundation, Inc.

   Author:  Felipe A. Rodriguez <far@ix.netcom.com>
   Date: August 1998
   
   This file is part of the GNUstep GUI X/RAW Backend.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; see the file COPYING.LIB.
   If not, write to the Free Software Foundation,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/ 

#include "BrowserMatrix.h"
#include <gnustep/xraw/XR.h>

#define ASSIGN(a, b)	[b retain]; \
						[a release]; \
						a = b;

typedef struct {							// point in a list mode selection
  int x;									
  int y;
} MPoint;

typedef struct {
  int x;
  int y;
  int width;
  int height;
} MRect;

typedef struct _tMatrix {
  int numRows, numCols;
  int allocatedRows, allocatedCols;
  BOOL** matrix;
} *tMatrix;

static inline MPoint MakePoint (int x, int y)
{
  MPoint point = { x, y };
  return point;
}

@interface BrowserMatrix (BrowserMatrixPrivateMethods)

											// Returns by reference the row and 
											// column of cell that is above or 
- (BOOL)_getRow:(int*)row					// below or to the left or to the
	 	 column:(int*)column				// right of point.  `point' is in
       	 forPoint:(NSPoint)point			// the matrix coordinates.  Returns
	  	 above:(BOOL)aboveRequired			// NO if the point is outside the
	  	 right:(BOOL)rightRequired			// bounds of matrix, YES otherwise.
 		 isBetweenCells:(BOOL*)isBetweenCells;
@end


@implementation BrowserMatrix

//
// Class methods
//
+ (void)initialize
{
  if (self == [BrowserMatrix class])
    {
      NSDebugLog(@"Initialize BrowserMatrix class\n");

      // Initial version
      [self setVersion:1];
    }
}

- (void)mouseDown:(NSEvent*)theEvent
{
BOOL isBetweenCells, insideBounds;
int row, column;
unsigned eventMask = NSLeftMouseUpMask | NSLeftMouseDownMask | NSMouseMovedMask 
						| NSLeftMouseDraggedMask | NSPeriodicMask;
//NSPoint lastLocation = [theEvent locationInWindow];
NSPoint lastLocation;
NSEvent* lastEvent = nil;
BOOL done = NO;
NSRect rect;
id aCell, previousCell = nil, selectedCellTarget;
NSRect previousCellRect;
NSApplication *app = [NSApplication sharedApplication];
static MPoint anchor = {0, 0};
static NSPoint selectedLocation;

	if ((mode != NSTrackModeMatrix) && (mode != NSHighlightModeMatrix)) 
		[NSEvent startPeriodicEventsAfterDelay:0.05 withPeriod:0.05];

	ASSIGN(lastEvent, theEvent);
	[window _captureMouse: self];							// grab the mouse
	[self lockFocus];					// selection involves two steps, first
										// a loop that continues until the left
	while (!done) 						// mouse goes up; then a series of 
		{								// steps which send actions and display
		BOOL shouldProceedEvent = NO;	// the cell as it should appear after
										// the selection process is complete
		lastLocation = [lastEvent locationInWindow];
		lastLocation = [self convertPoint:lastLocation fromView:nil];
    	insideBounds = [self _getRow:&row 
							 column:&column
							 forPoint:lastLocation
							 above:NO right:NO
							 isBetweenCells:&isBetweenCells];
		if (insideBounds && !isBetweenCells) 			
			{											 
      		aCell = [self cellAtRow:row column:column];
      		rect = [self cellFrameAtRow:row column:column];

      		switch (mode) 
				{
				case NSTrackModeMatrix:				// in Track mode the cell
					selectedCell = aCell;			// should track the mouse
					selectedRow = row;				// until the cursor either
					selectedColumn = column;		// leaves the cellframe or
													// NSLeftMouseUp occurs
					if([aCell trackMouse:lastEvent			
							  inRect:rect					
							  ofView:self				
							  untilMouseUp:YES])			// YES if mouse 
						done = YES;							// went up in cell
					break;									

				case NSHighlightModeMatrix:			// Highlight mode is like
					[aCell setState:1];				// Track mode except that
					selectedCell = aCell;			// the cell is lit before
					selectedRow = row;				// it begins tracking and
					selectedColumn = column;		// unlit afterwards		
					[aCell highlight: YES withFrame: rect inView: self];
					[window flushWindow];
															
					if([aCell trackMouse:lastEvent			 
							  inRect:rect					
							  ofView:self				
							  untilMouseUp:YES])			// YES if mouse 
						done = YES;							// went up in cell

					[aCell setState:0];					
					[aCell highlight: NO withFrame: rect inView: self];
					[window flushWindow];
					break;									

				case NSRadioModeMatrix:					// Radio mode allows no
					if (previousCell == aCell)			// more than one cell
						break;				 			// to be selected
					
					if(selectedCell)					
						{
						[selectedCell setState:0];		// deselect previously
						if (!previousCell)				// selected cell
							previousCellRect = [self cellFrameAtRow:selectedRow 
									  			 	 column:selectedColumn];
						[selectedCell highlight:NO 			
									  withFrame:previousCellRect 
									  inView:self];
	  					((tMatrix)selectedCells)->matrix[selectedRow]
														[selectedColumn] = NO;	
						}			
					selectedCell = aCell;				// select current cell
					selectedRow = row;					
					selectedColumn = column;
					[aCell setState:1];	
					[aCell highlight:YES withFrame:rect inView:self];
	  				((tMatrix)selectedCells)->matrix[row][column] = YES;				
					[window flushWindow];
	  				break;
														// List mode allows
				case NSListModeMatrix: 			 		// multiple cells to be
					{									// selected
	  				unsigned modifiers = [lastEvent modifierFlags];
													// drag op if mouse moves
	  				if (previousCell == aCell)		// more than x pixels while
						{							// left mouse is down
						if((lastLocation.x > selectedLocation.x + 4.0) || 
								(lastLocation.x < selectedLocation.x - 4.0) ||
								(lastLocation.y > selectedLocation.y + 4.0) || 
								(lastLocation.y < selectedLocation.y - 4.0))
							{		// offset compensates for the distance the
									// cursor has traveled since the cell was
									// selected.  it also compensates for the
									// position of the image cell within the
									// larger browser matrix cell
							NSSize offset = NSMakeSize((lastLocation.x - 
												selectedLocation.x) + 
												((rect.size.width - 48.0)/2.0), 
												(lastLocation.y - 
												selectedLocation.y) + 17);

							if ((mode != NSTrackModeMatrix) && 
									(mode != NSHighlightModeMatrix))
								[NSEvent stopPeriodicEvents];
													// perform drag mechanics
							[self dragImage:[aCell image]
								  at:previousCellRect.origin
								  offset:offset
								  event:lastEvent
								  pasteboard:nil
								  source:self
								  slideBack:YES];
							done = YES;			
							}

	    				break;		
						}	

					if(selectedCell == aCell)		// Edit the browsercell's
						{							// text 			
						if([aCell trackMouse:lastEvent 
								  inRect:rect 
								  ofView:self 
								  untilMouseUp:YES])
//							return;			 
							done = YES;					// YES if the mouse 

						if(done)
							break;
						}
//					else
										// When the user first clicks on a cell 
										// we clear the existing selection 
	  				if (!previousCell) 	// unless the Alternate or Shift keys
						{				// have been pressed.
						if (!(modifiers & NSShiftKeyMask) && 
								!(modifiers & NSAlternateKeyMask))
							{
	      					[self deselectAllCells];
							anchor = MakePoint (column, row);
							}			// Consider the selected cell as the 
										// anchor from which to extend the 
										// selection to the current cell
	    				if (!(modifiers & NSAlternateKeyMask))
							{
							selectedCell = aCell;		// select current cell
							selectedRow = row;
							selectedColumn = column;
							selectedLocation = lastLocation;	
		
							[selectedCell setState:1];
							[selectedCell highlight:YES 
										  withFrame:rect 
										  inView:self];
							((tMatrix)selectedCells)->matrix[row][column] =YES;
							[window flushWindow];
							break;
							}
	  					}

					done = YES;			

//	  				if (selectionByRect)
//	    				[self _selectRectUsingAnchor:anchor
//							  last:MakePoint (selectedColumn, selectedRow)
//							  current:MakePoint (column, row)];
//	  				else
//	    				[self _selectContinuousUsingAnchor:anchor
//							  last:MakePoint (selectedColumn, selectedRow)
//							  current:MakePoint (column, row)];

					[window flushWindow];
					selectedCell = aCell;				// select current cell
					selectedRow = row;
					selectedColumn = column;
					break;
					}
      			}
			previousCell = aCell;
			previousCellRect = rect;
			[self scrollRectToVisible:rect];
    		}

    	if (done)										// if done break out of
      		break;										// the selection loop

		while (!shouldProceedEvent) 					
			{											// Get the next event
      		theEvent = [app nextEventMatchingMask:eventMask
							untilDate:[NSDate distantFuture]
							inMode:NSEventTrackingRunLoopMode
							dequeue:YES];
      		switch ([theEvent type]) 
				{
				case NSPeriodic:
					NSDebugLog(@"NSMatrix: got NSPeriodic event\n");
					shouldProceedEvent = YES;			// it's time to cycle
					break;								// thru the loop again
				case NSLeftMouseUp:
					done = YES;			// Track and Highlight modes do not use
				case NSLeftMouseDown:	// periodic events so we must break out
				default:				// and check if the mouse is in a cell
					if ((mode == NSTrackModeMatrix) || 
							(mode == NSHighlightModeMatrix))
				  		shouldProceedEvent = YES;	 
					NSDebugLog(@"NSMatrix: got event of type: %d\n",
								[theEvent type]);
					ASSIGN(lastEvent, theEvent);
					continue;
				}
    		}
//		lastLocation = [lastEvent locationInWindow];
//		lastLocation = [self convertPoint:lastLocation fromView:nil];
  		}
						
	[window _releaseMouse: self];					// Release the mouse

  	switch (mode) 									// Finish the selection
		{								 			// process
		case NSRadioModeMatrix:	
			if(selectedCell)			
				[selectedCell highlight:NO withFrame:rect inView:self];
		case NSListModeMatrix:						
//			[self setNeedsDisplayInRect:rect];		// not needed by XRAW
			[window flushWindow];
		case NSHighlightModeMatrix:
		case NSTrackModeMatrix:
			break;
  		}					

	if(selectedCell)
		{											// send single click action
		if(!(selectedCellTarget = [selectedCell target]))
			{					// selected cell has no target so send single  
			if (target)			// click action to matrix's (self's) target			
				[target performSelector:action withObject:self];
			}					// in Track and Highlight modes the single
		else 					// click action has already been sent by the 
			{					// cell to it's target (if it has one)
			if ((mode != NSTrackModeMatrix) && (mode != NSHighlightModeMatrix))
				[selectedCellTarget performSelector:[selectedCell action] 
									withObject:self];
			}
		}
								// click count > 1 indicates a double click					
	if (target && doubleAction && ([lastEvent clickCount] > 1))				
		[target performSelector:doubleAction withObject:self];	
																
	[self unlockFocus];

	if ((mode != NSTrackModeMatrix) && (mode != NSHighlightModeMatrix))
		[NSEvent stopPeriodicEvents];

	[lastEvent release];
}

@end

//*****************************************************************************
//
// 		private methods 
//
//*****************************************************************************

@implementation BrowserMatrix (BrowserMatrixPrivateMethods)

#define SET_POINTER_VALUE(pointer, value) \
  do { if (pointer) *pointer = value; } while (0)

/* Returns by reference the row and column of cell that is above or below or
   to the left or to the right of point. `point' is in the matrix coordinates.
   Returns NO if the point is outside the bounds of matrix, YES otherwise.

   Note that the cell numbering is flipped relative to the coordinate system.
 */
- (BOOL)_getRow:(int*)row
		 column:(int*)column
		 forPoint:(NSPoint)point
		 above:(BOOL)aboveRequired
		 right:(BOOL)rightRequired
		 isBetweenCells:(BOOL*)isBetweenCells
{
BOOL rowReady = NO, colReady = NO;
BOOL betweenRows = NO, betweenCols = NO;
NSRect theBounds = [self bounds];

  SET_POINTER_VALUE(isBetweenCells, NO);

  /* First check the limit cases */
  if (point.x > theBounds.size.width) {
    SET_POINTER_VALUE(column, numCols);
    colReady = YES;
  }
  else if (point.x < 0) {
    SET_POINTER_VALUE(column, 0);
    colReady = YES;
  }

  if (point.y > theBounds.size.height) {
    SET_POINTER_VALUE(row, numRows);
    rowReady = YES;
  }
  else if (point.y < 0) {
    SET_POINTER_VALUE(row, 0);
    rowReady = YES;
  }

  if (rowReady && colReady)
    return NO;

  if (!rowReady) {
    int approxRow = point.y / (cellSize.height + intercell.height);
    float approxRowsHeight = approxRow * (cellSize.height + intercell.height);

    /* Determine if the point is inside the cell */
    betweenRows = !(point.y > approxRowsHeight
		    && point.y <= approxRowsHeight + cellSize.height);

    /* If the point is between cells then adjust the computed row taking into
       account the `aboveRequired' flag. */
    if (aboveRequired && betweenRows)
      approxRow++;

#if HAS_FLIPPED_VIEWS
    SET_POINTER_VALUE(row, approxRow);
#else
    SET_POINTER_VALUE(row, numRows - approxRow - 1);
#endif
    if (*row < 0) {
      *row = -1;
      rowReady = YES;
    }
    else if (*row >= numRows) {
      *row = numRows - 1;
      rowReady = YES;
    }
  }

  if (!colReady) {
    int approxCol = point.x / (cellSize.width + intercell.width);
    float approxColsWidth = approxCol * (cellSize.width + intercell.width);

    /* Determine if the point is inside the cell */
    betweenCols = !(point.x > approxColsWidth
		    && point.x <= approxColsWidth + cellSize.width);

    /* If the point is between cells then adjust the computed column taking
       into account the `rightRequired' flag. */
    if (rightRequired && betweenCols)
      approxCol++;

    SET_POINTER_VALUE(column, approxCol);
    if (*column < 0) {
      *column = -1;
      colReady = YES;
    }
    else if (*column >= numCols) {
      *column = numCols - 1;
      colReady = YES;
    }
  }

  /* If the point is outside the matrix bounds return NO */
  if (rowReady || colReady)
    return NO;

  if (betweenRows || betweenCols)
    SET_POINTER_VALUE(isBetweenCells, YES);
  return YES;
}
										
@end
