/*
 *	VME Linux/m68k Loader
 *
 *	(c) Copyright 1997 by Nick Holgate
 *
 *	This file is subject to the terms and conditions of the GNU General Public
 *	License.  See the file COPYING for more details.
 */

/*--------------------------------------------------------------------------*/

#include <stdarg.h>

/* linux specific include files */
#include <linux/a.out.h>
#include <linux/elf.h>
#include <linux/linkage.h>
#include <asm/page.h>

#include "bootinfo.h"
#include "loader.h"
#include "loaderlib.h"
#include "stream.h"

/* declarations of stream modules */
extern MODULE file_mod;
extern MODULE gunzip_mod;

/* startcode declarations */
extern char startcode_beg;
extern char startcode_end;

/*--------------------------------------------------------------------------*/
/*  Program Entry
 */

__asm__(".text\n"
__ALIGN_STR "\n"
".globl " SYMBOL_NAME_STR(_start) ";\n"
SYMBOL_NAME_STR(_start) ":
	jbra	(" SYMBOL_NAME_STR(Main) ");
");

/*--------------------------------------------------------------------------*/
/* Console Echo Modes
 */

#define ECHO_NONE	(0)
#define ECHO_NORMAL	(1)
#define ECHO_DOT	(2)

#define NO_TIMEOUT	(0)

/*--------------------------------------------------------------------------*/

#define VALUE(ptr)		((ptr) ? (*(ptr)) : 0)
#define BO_VALUE(member) VALUE(v->boot_options->member)

/*--------------------------------------------------------------------------*/
/*	Map Data
 */

static const struct
{
    unsigned long	ID[2];
    unsigned long	Size;
    unsigned char	Data[0];

} Map __attribute__ ((section (".tail")))

 =	{
		{ LILO_ID, LILO_MAPDATAID }
	};

/*--------------------------------------------------------------------------*/
/* bootinfo (internal format)
 */

struct internal_bootinfo
{
	unsigned long	machtype;			/* machine type						*/
	unsigned long	cputype;			/* system CPU						*/
	unsigned long	fputype;			/* system FPU						*/
	unsigned long	mmutype;			/* system MMU						*/
	int				num_memory;			/* # of memory blocks found			*/
	struct mem_info memory[NUM_MEMINFO];/* memory description				*/
	struct mem_info ramdisk;			/* ramdisk description				*/
	char command_line[CL_SIZE];			/* kernel command line parameters	*/
};

/*--------------------------------------------------------------------------*/

char							debug_mode;

static struct {
	const BOOTRECORD			*default_boot_record;
	BOOTOPTIONS					*boot_options;
	BOOTRECORD					*boot_records;
	FILEDEF						*boot_files;
	char						force_prompt;
	unsigned long				map_offset;
	unsigned long				detected_mem_size;
	char						input_buffer[CL_SIZE];
	char						cmdln_buffer[CL_SIZE];
	Elf32_Ehdr					kexec_elf;
	struct internal_bootinfo	bi;

#ifdef BOOTINFO_COMPAT_1_0
	struct compat_bootinfo		compat_bootinfo;
#endif /* BOOTINFO_COMPAT_1_0 */

#define MAX_BI_SIZE				(4096)
	unsigned long				bi_size;
	union {
		struct bi_record	record;
		unsigned char		fake[MAX_BI_SIZE];
	} bi_union;

	void						(*startcode_entry)(void);
	int							leader_adjust;
} *v;

#define IOBUF_SIZE				(4096)
#define STACK_SIZE				(8192)

/*--------------------------------------------------------------------------*/
/* Read a Line from the Console
 *
 * Returns input length.
 */

int
read_line
(	int				mode,		/* echo mode								*/
	unsigned long	timeout
)
{	int				c;
    int				length    = 0;
	long			time_left = NO_TIMEOUT;

	v->leader_adjust = 0;

	if (timeout)
	{
		/* get time at timeout */
		timeout += get_time();
	}

	while (1)
	{
		if (timeout)
		{
			/* get amount of time left */
			if ((time_left = timeout - get_time()) < 1)
			{
				/* timeout */
				break;
			}
		}

		if ((c = get_char(time_left)) == -1)
		{
			/* timeout */
			break;
		}

		switch (c)
		{
		    case '\r':
		    case '\n':
		    {
				v->input_buffer[length] = '\0';

				if ((mode == ECHO_NORMAL) || (mode == ECHO_DOT))
				{
				    put_char('\n');
				}

				return length;
			}

		    case '\b':
		    case 0x7f:
		    {
				if (length)
				{
				    length--;
				    if ((mode == ECHO_NORMAL) || (mode == ECHO_DOT))
				    {
						put_str("\b \b");
					}
				}
				break;
			}

			case 0x1b:
			{
			    if ((mode == ECHO_NORMAL) || (mode == ECHO_DOT))
			    {
					while (length)
					{
					    length--;
						put_str("\b \b");
					}
				}
				break;
			}

		    case '\t':
		    {
		    	if (mode == ECHO_NORMAL)
		    	{
				    put_str("help\n");
				    str_cpy(v->input_buffer, "help");
				    return 4;
				}
				break;
			}

		    default:
		    {
		    	if ((c >= ' ')
				&&	(c <= '~')
				&&	(length < sizeof(v->input_buffer) - 1))
				{
				    v->input_buffer[length++] = c;
				    switch (mode)
					{
						case ECHO_NORMAL:
						{
						    put_char(c);
						    break;
						}

						case ECHO_DOT:
						{
						    put_char('+');
						    break;
						}
				    }
				}
				break;
			}
		}
	}

	if ((mode == ECHO_NORMAL) || (mode == ECHO_DOT))
	{
	    put_str(" ** timeout **\n");
	}

	/* timeout */
	v->input_buffer[0] = '\0';
	return -1;
}

/*--------------------------------------------------------------------------*/
/* Find the File Map for a File Path
 */

const FILEMAP *
find_file_map
(	const char		*path
)
{	const FILEDEF	*file;

	for (file = v->boot_files; file; file = file->next)
	{
		if (case_equal_strings(path, file->path))
		{
			return file->map;
		}
	}

	return NULL;
}

/*--------------------------------------------------------------------------*/
/* Get the Name for a Tag
 */

static struct {
    unsigned long	tagid;
    const char		*name;
    const char		*param;
    char			type;
#define TT_DAT	0
#define TT_STR	1
#define TT_OTH	2
} tag_names[] = {
	{ TAG_LILO,				"TAG_LILO",				NULL,			TT_OTH },
	{ TAG_EOF,				"TAG_EOF",				NULL,			TT_OTH },
	{ TAG_HEADER,			"TAG_HEADER",			NULL,			TT_OTH },
	{ TAG_HEADER_END,		"TAG_HEADER_END",		NULL,			TT_OTH },
	{ TAG_DEFAULT,			"TAG_DEFAULT",			"default",		TT_STR },
	{ TAG_MESSAGE,			"TAG_MESSAGE",			"message",		TT_STR },
	{ TAG_DELAY,			"TAG_DELAY",			"delay",		TT_DAT },
	{ TAG_TIMEOUT,			"TAG_TIMEOUT",			"timeout",		TT_DAT },
	{ TAG_DEBUG,			"TAG_DEBUG",			"debug",		TT_DAT },
	{ TAG_PROMPT,			"TAG_PROMPT",			"prompt",		TT_DAT },
	{ TAG_BOOT_RECORD,		"TAG_BOOT_RECORD",		"label",		TT_STR },
	{ TAG_BOOT_RECORD_END,	"TAG_BOOT_RECORD_END",	NULL,			TT_OTH },
	{ TAG_ALIAS,			"TAG_ALIAS",			"alias",		TT_STR },
	{ TAG_KERNEL,			"TAG_KERNEL",			"kernel",		TT_STR },
	{ TAG_ARGUMENTS,		"TAG_ARGUMENTS",		"cmdline",		TT_STR },
	{ TAG_PASSWORD,			"TAG_PASSWORD",			"password",		TT_STR },
	{ TAG_RESTRICTED,		"TAG_RESTRICTED",		"restricted",	TT_DAT },
	{ TAG_RAMDISK,			"TAG_RAMDISK",			"ramdisk",		TT_STR },
	{ TAG_SYMTAB,			"TAG_SYMTAB",			"symbols",		TT_STR },
	{ TAG_MEMSIZE,			"TAG_MEMSIZE",			"memsize",		TT_DAT },
	{ TAG_CALLMONITOR,		"TAG_CALLMONITOR",		"callmonitor",	TT_DAT },
	{ TAG_FILE_DEF,			"TAG_FILE_DEF",			NULL,			TT_OTH },
	{ TAG_FILE_DEF_END,		"TAG_FILE_DEF_END",		NULL,			TT_OTH },
	{ TAG_FILEMAP,			"TAG_FILEMAP",			NULL,			TT_OTH }
};

void
show_tag
(	const TAGRECORD		*tr
)
{	unsigned			i;

    for (i = 0; i < arraysize(tag_names); i++)
    {
		if (tag_names[i].tagid == tr->tagid)
		{
			if (tag_names[i].type == TT_DAT)
			{
				Printf(" %-10s = %ld\n", tag_names[i].param, *tr->data);
			}
			else if (tag_names[i].type == TT_STR)
			{
				Printf(" %-10s = `%s'\n", tag_names[i].param, (char *)tr->data);
			}
			break;
		}
	}
}

/*--------------------------------------------------------------------------*/

const char *
get_tag_name
(	unsigned long	tag
)
{	unsigned		i;

    for (i = 0; i < arraysize(tag_names); i++)
    {
		if (tag_names[i].tagid == tag)
		{
		    return tag_names[i].name;
		}
	}

    return NULL;
}

/*--------------------------------------------------------------------------*/
/* Unexpected Tag
 */

void
unexpected_tag
(	const TAGRECORD		*tr
)
{	const char			*name;

    if ((name = get_tag_name(tr->tagid)) != NULL)
    {
		Printf("Unexpected tag `%s' at offset 0x%08lx\n",
			name, v->map_offset);
	}
    else
    {
		Printf("Unknown tag 0x%08lx at offset 0x%08lx\n",
			tr->tagid, v->map_offset);
	}
}

/*--------------------------------------------------------------------------*/
/* Get the Value of the Debug Tag (if it exists)
 */

int
get_debug_tag
(	void
)
{	const TAGRECORD	*tr;
    unsigned long	offset;
    int				debug = FALSE;

    for (offset = 0;
			offset + sizeof(TAGRECORD) + sizeof(unsigned long) <= Map.Size;
		    	 offset += sizeof(TAGRECORD) + tr->size)
	{
		tr = (TAGRECORD *) &Map.Data[offset];

		if (tr->tagid == TAG_DEBUG)
		{
			debug = tr->data[0];
	   		break;
		}
    }

    return debug;
}

/*--------------------------------------------------------------------------*/
/* Get the Next Tag from the Map Data
 */

const TAGRECORD *
get_next_tag
(	void
)
{	static const TAGRECORD	*tr = NULL;

    if (tr)
    {
		v->map_offset += sizeof(TAGRECORD) + tr->size;
	}

	if (v->map_offset + sizeof(TAGRECORD) > Map.Size)
	{
		panic("Unexpected End of configuration data");
	}

    tr = (TAGRECORD *) &Map.Data[v->map_offset];

	return tr;
}

/*--------------------------------------------------------------------------*/
/* Read the Map Data
 */

void
read_map_data
(	void
)
{	const TAGRECORD	*tr;
	BOOTRECORD		*record;
	BOOTRECORD		**p1		= &v->boot_records;
	FILEDEF			*file;
	FILEDEF			**p2 		= &v->boot_files;

#define CASE_TAG(tag, item)\
	case TAG_ ## tag:\
		item = (typeof(item)) tr->data;\
		if (debug_mode) show_tag(tr);\
		break

	/* File Identification */
	tr = get_next_tag();
	if (tr->tagid != TAG_LILO)
	{
		panic("Invalid configuration data");
	}

	/* Boot Options */
	if (debug_mode)
	{
		put_str("\n[Boot Options]\n");
	}

	tr = get_next_tag();
	if (tr->tagid != TAG_HEADER)
	{
		panic("Invalid configuration data");
	}

	v->boot_options = (BOOTOPTIONS *) mem_alloc(sizeof(BOOTOPTIONS));

	for (tr = get_next_tag(); tr->tagid != TAG_HEADER_END; tr = get_next_tag())
	{
		switch (tr->tagid)
		{
			CASE_TAG( DEFAULT, v->boot_options->boot_default );
			CASE_TAG( MESSAGE, v->boot_options->boot_message );
			CASE_TAG( PROMPT,  v->boot_options->boot_prompt  );
			CASE_TAG( DELAY,   v->boot_options->boot_delay   );
			CASE_TAG( TIMEOUT, v->boot_options->boot_timeout );
			CASE_TAG( DEBUG,   v->boot_options->boot_debug   );
			default:
			{
				unexpected_tag(tr);
				break;
			}
		}
	}

	v->force_prompt = BO_VALUE(boot_prompt);
	debug_mode      = BO_VALUE(boot_debug );

	/* Boot Records */
	if (debug_mode)
	{
		put_str("\n[Boot Records]");
	}

	for (tr = get_next_tag(); tr->tagid == TAG_BOOT_RECORD; tr = get_next_tag())
	{
		record        = (BOOTRECORD *) mem_alloc(sizeof(BOOTRECORD));
		record->label = (const char *) tr->data;

		if (debug_mode)
		{
			put_char('\n');
			show_tag(tr);
		}

		for (tr = get_next_tag();
				tr->tagid != TAG_BOOT_RECORD_END;
					 tr = get_next_tag())
		{
			switch (tr->tagid)
			{
				CASE_TAG( ALIAS,       record->alias      );
				CASE_TAG( KERNEL,      record->kernel     );
				CASE_TAG( ARGUMENTS,   record->args       );
				CASE_TAG( PASSWORD,    record->password   );
				CASE_TAG( RESTRICTED,  record->restricted );
				CASE_TAG( RAMDISK,     record->ramdisk    );
				CASE_TAG( MEMSIZE,     record->memsize    );
				CASE_TAG( SYMTAB,      record->symtab     );
				CASE_TAG( CALLMONITOR, record->callmonitor);
				default:
				{
					unexpected_tag(tr);
					break;
				}
			}
		}

		*p1       = record;
		p1        = &record->next;
	}

	/* File Definitions */
	if (debug_mode)
	{
		put_str("\n[File Definitions]\n");
	}

	while (tr->tagid == TAG_FILE_DEF)
	{
		file = (FILEDEF *) mem_alloc(sizeof(FILEDEF));

		file->path = (const char *) tr->data;

		for (tr = get_next_tag();
				tr->tagid != TAG_FILE_DEF_END;
					tr = get_next_tag())
		{
			switch (tr->tagid)
			{
				CASE_TAG( FILEMAP, file->map );
				default:
				{
					unexpected_tag(tr);
					break;
				}
			}
		}

		if (debug_mode)
		{
			Printf("%s (", file->path);
			if (file->map)
			{
				Printf("%ld bytes in %ld fragments)\n",
					MAP_FILESIZE(file->map), MAP_NUMFRAGS(file->map));
			}
			else
			{
				put_str("not available)\n");
			}
		}

		*p2 = file;
		p2  = &file->next;
		tr  = get_next_tag();
	}

	if (tr->tagid != TAG_EOF)
	{
		unexpected_tag(tr);
	}

	if (debug_mode)
	{
		put_char('\n');
	}
}

/*--------------------------------------------------------------------------*/
/* Find a Boot Record by Name
 */

const BOOTRECORD *
find_boot_record
(	char				*name
)
{	const BOOTRECORD	*record;
	char				*base = name;

    if (name == NULL)
    {
		name = (char *) v->boot_options->boot_default;
	}

    for (record = v->boot_records; record; record = record->next)
    {
		if (prefix_string(name, record->label))
		{
			name += str_len(record->label);
			break;
		}

		if (prefix_string(name, record->alias))
		{
			name += str_len(record->alias);
	    	break;
	    }
	}

	if (record && base)
	{
		/* skip white space */
		while (*name && *name < '!')
		{
			name++;
		}

		/* strip name from start of string */
		str_cpy(base, name);
	}

    return record;
}

/*--------------------------------------------------------------------------*/
/* Check whether a Boot Image is available
 */

int
bootrecord_available
(	const BOOTRECORD	*record
)
{
	if (find_file_map(record->kernel) == NULL)
	{
		return FALSE;
	}

	if (record->ramdisk && (find_file_map(record->ramdisk) == NULL))
	{
	    return FALSE;
    }

    return TRUE;
}

/*--------------------------------------------------------------------------*/
/* List all available Boot Records
 */

void
list_records
(	void
)
{	const BOOTRECORD	*record;

	put_str("\nBoot images:\n");

	for (record = v->boot_records; record; record = record->next)
	{
		Printf("    %c%-32s  %s\n",
				(record == v->default_boot_record) ? '*' : ' ',
				record->label,
				record->alias ? record->alias : ""
		);
	}
	put_char('\n');
}

/*--------------------------------------------------------------------------*/
/* Ask the User for a Boot Operating System
 */

const BOOTRECORD *
get_boot_record
(	unsigned long		timeout
)
{	const BOOTRECORD	*record;

	while (1)
	{
		put_str("Boot: ");
		read_line(ECHO_NORMAL, timeout);

		/* timeout or no input */
		if (v->input_buffer[0] == '\0')
		{
			return v->default_boot_record;
		}

		if (equal_strings(v->input_buffer, "help")
		|| 	equal_strings(v->input_buffer, "?"   ))
		{
			list_records();
			continue;
		}

		if ((record = find_boot_record(v->input_buffer)) == NULL)
		{
			Printf("Nonexistent boot image `%s'.\n", v->input_buffer);
			continue;
		}

		if (!bootrecord_available(record))
		{
			Printf("Boot image `%s' is not available.\n", v->input_buffer);
			continue;
		}

		/* preserve any extra command line parameters
		 * following the image name
		 */
		str_cpy(v->cmdln_buffer, v->input_buffer);

		/* all done */
		return record;
	}
}

/*--------------------------------------------------------------------------*/
/* Compare the Bootstrap and Kernel Versions
 */

int
check_bootinfo_version
(	const char					*memptr
)
{	const struct bootversion	*bv		= (struct bootversion *) memptr;
	unsigned long				version	= 0;
	int							i;
	int							kernel_major;
	int							kernel_minor;
	int							boots_major;
	int							boots_minor;
	unsigned long				machtype;

	if (bv->magic == BOOTINFOV_MAGIC)
	{
		machtype = get_machtype();

		for (i = 0; bv->machversions[i].machtype != 0; ++i)
		{
			if (bv->machversions[i].machtype == machtype)
			{
				version = bv->machversions[i].version;
				break;
			}
		}

#ifdef BOOTINFO_COMPAT_1_0
		/* if no 2.1.xx version info */
		if (!version)
		{
			machtype = get_compat_machtype();

			/* look for 2.0.xx version info (has different machine type code) */
			for (i = 0; bv->machversions[i].machtype != 0; ++i)
			{
				if (bv->machversions[i].machtype == machtype)
				{
					version = bv->machversions[i].version;
				}
			}
		}
#endif
	}

	if (!version)
	{
		Printf("Kernel has no bootinfo version info, assuming 1.0\n");
		version  = get_compat_booti_version();
	}

	kernel_major = BI_VERSION_MAJOR(version);
	kernel_minor = BI_VERSION_MINOR(version);
	boots_major	 = BI_VERSION_MAJOR(get_booti_version());
	boots_minor	 = BI_VERSION_MINOR(get_booti_version());

	Printf("Bootstrap's bootinfo version : %ld.%ld\n",
		boots_major, boots_minor);

	Printf("Kernel's bootinfo version    : %ld.%ld\n",
		kernel_major, kernel_minor);

	if (kernel_major == boots_major)
	{
	    if (kernel_minor > boots_minor)
		{
			Printf(
			"Warning: Bootinfo version of bootstrap and kernel differ!\n"
			"         Certain features may not work.\n"
			);
	    }
	}

#ifdef BOOTINFO_COMPAT_1_0
	else if (kernel_major == BI_VERSION_MAJOR(get_compat_booti_version()))
	{
	    Printf("(using backwards compatibility mode)\n");
	}
#endif /* BOOTINFO_COMPAT_1_0 */

	else
	{
	    Printf("\nThis bootstrap is too %s for this kernel!\n",
		   boots_major < kernel_major ? "old" : "new");
	   	return 0;
    }

    return kernel_major;
}

/*--------------------------------------------------------------------------*/
/* Determine CPU type:
 *
 * returns 40 if 68040
 * or      60 if 68060
 */

int
cpu_type
(	void
)
{	int		cpu;

	__asm__ __volatile__ (
	"	move.w	%%sr,%%d0
		movec	%%vbr,%%a0
		move.l	0x10(%%a0),%%d1
		lea.l	Illegal(%%pc),%%a1
		move.l	%%a1,0x10(%%a0)
		move.l	%%sp,%%a1
		moveq	#60,%%d3
		dc.l	0x4e7aa803			| movec	%%msp,%%a2
		moveq	#40,%%d3
Illegal:
		move.l	%%a1,%%sp
		move.l	%%d1,0x10(%%a0)
		move.w	%%d0,%%sr
		move.l	%%d3,%0
	"
	: "=d" (cpu)
	: /* no inputs */
	: "d0", "d1", "d3", "a0", "a1", "a2", "memory"
);

	return cpu;
}

/*--------------------------------------------------------------------------*/
/* Determine FPU type:
 *
 * returns 0 not present
 */

int
fpu_type
(	void
)
{	int		fpu;

	__asm__ __volatile__ (
	"	move.w	%%sr,%%d0
		movec	%%vbr,%%a0
		move.l	0x2c(%%a0),%%d1
		lea.l	FLine(%%pc),%%a1
		move.l	%%a1,0x2c(%%a0)
		move.l	%%sp,%%a1
		moveq	#0,%%d3
		dc.l	0xf2800000
		moveq	#1,%%d3
FLine:
		move.l	%%a1,%%sp
		move.l	%%d1,0x2c(%%a0)
		move.w	%%d0,%%sr
		move.l	%%d3,%0
	"
	: "=d" (fpu)
	: /* no inputs */
	: "d0", "d1", "d3", "a0", "a1", "a2", "memory"
);

	return fpu;
}

/*--------------------------------------------------------------------------*/
/* Probe given address for the existance of RAM
 *
 * Return non-zero if RAM found.
 */

int
ram_probe
(	unsigned long	where
)
{	int				rv;

	__asm__ __volatile__ (
	"	move.w	%%sr,%%d4				| save trace flag
		movec	%%vbr,%%a0				| get vector table ptr
		move.l	0x8(%%a0),%%d5			| save BERR vector
		lea.l	BusError(%%pc),%%a1		| new BERR handler
		move.l	%%a1,0x8(%%a0)			| set it
		move.l	%%sp,%%a1				| save stack frame
		moveq	#0,%0					| say no memory here
		move.l	#0x7e57ed17,%%d1		| get pattern
		move.l	(%1),%%d2				| save memory
		move.l	4(%1),%%d3				| save memory + 4
		move.l	%%d1,(%1)				| store pattern
		not.l	%%d1					| new pattern
		move.l	%%d1,4(%1)				| store new pattern
		not.l	%%d1					| old pattern
		cmp.l	(%1),%%d1				| did it store
		bne.s	BusError				| branch if not
		move.l	%%d2,(%1)				| restore memory
		move.l	%%d3,4(%1)				| restore memory + 4
		moveq	#1,%0					| say memory found
BusError:
		move.l	%%a1,%%sp				| restore SP
		move.l	%%d5,0x8(%%a0)			| restore BERR handler
		move.w	%%d4,%%sr				| restore trace flag
	"
	: "=d" (rv)
	: "a" (where)
	: "d1", "d2", "d3", "d4", "d5", "a0", "a1", "memory"
);

	return rv;
}

/*--------------------------------------------------------------------------*/

void
loading
(	const char	*what
)
{	int			n;

	Printf("Loading %s %n", what, &n);
	n = 40 - n - v->leader_adjust;
	v->leader_adjust = 0;

	while (n-- > 0)
	{
		put_char('.');
	}

	put_char(' ');
}

/*--------------------------------------------------------------------------*/

void
load_symbols
(	const char	*filename
)
{
	if (can_do_symbols())
	{
		char 			*iobuff;
		int				iocount;
		char			*p;
		int				len;
		unsigned long	sofar;

		if ((iobuff = mem_alloc(IOBUF_SIZE)) == NULL)
		{
			put_str("\nNot enough memory to load symbols!\n");
			return;
		}

	    stream_init();
	    stream_push( &file_mod );
	    stream_push( &gunzip_mod );
	
		loading("Symbols");
	
		if (sopen(filename) < 0)
		{
			Printf("\nUnable to open symbol file `%s'\n", filename);
			goto bye;
		}
	
		/* get rid of all existing symbols */
		clear_symbols();
	
		p        = iobuff;
		iocount  = 0;
		sofar    = 0;
	
		while (1)
		{
			len = 0;
			while (1)
			{
				/* if buffer empty */
				if (iocount == 0)
				{
					/* read a chunk */
					if ((iocount = sread(iobuff, IOBUF_SIZE)) < 1)
					{
						/* error or end of file */
						if (iocount != 0)
						{
							Printf("\nError reading symbol file `%s'\n",
									filename);
							goto fail;
						}
	
						goto done;
					}
	
					/* reset empty pointer */
					p      = iobuff;
	
					/* count bytes read */
					sofar += iocount;
				}
	
				/* count byte examined */
				iocount--;
	
				/* end of line? */
				if (*p == 0x0a || *p == 0x0d)
				{
					/* skip it */
					p++;
					break;
				}
	
				/* save character */
				v->input_buffer[len++] = *p++;
			}
	
			/* null terminate */
			v->input_buffer[len] = '\0';
	
			/* if not empty line */
			if (len)
			{
				if (add_symbol(v->input_buffer))
				{
					goto fail;
				}
			}
		}
	
	done:
		percent_term("OK\n");
	
	fail:
		sclose();

	bye:
		mem_free(iobuff);
	}
}

/*--------------------------------------------------------------------------*/

void
display_message_file
(	const char		*filename
)
{	char 			*iobuff;
	int				iocount;
	char			*p;
	unsigned long	sofar;

	if ((iobuff = mem_alloc(IOBUF_SIZE)) == NULL)
	{
		put_str("Not enough memory to display message file!\n");
		return;
	}

	if (file_open(filename) != 0)
	{
		Printf("Unable to open message file `%s'\n", filename);
		goto bye;
	}

	p        = iobuff;
	iocount  = 0;
	sofar    = 0;

	while (1)
	{
		/* if buffer empty */
		if (iocount == 0)
		{
			/* read a chunk */
			if ((iocount = file_read(iobuff, IOBUF_SIZE)) < 1)
			{
				/* error or end of file */
				if (iocount)
				{
					Printf("\nError reading message file `%s'\n", filename);
				}

				goto done;
			}

			/* reset empty pointer */
			p      = iobuff;

			/* count bytes read */
			sofar += iocount;
		}

		/* count byte examined */
		iocount--;

		/* display it */
		put_char(*p++);
	}

done:
	file_close();

bye:
	mem_free(iobuff);
}

/*--------------------------------------------------------------------------*/
/* Add a Record to the Bootinfo Structure
 */

int
add_bi_record
(	unsigned short		tag,
	unsigned short		size,
	const void			*data
)
{	struct bi_record	*record;
    unsigned short		size2;

    size2 = (sizeof(struct bi_record) + size + 3) & -4;

    if (v->bi_size + size2 + sizeof(v->bi_union.record.tag) > MAX_BI_SIZE)
	{
		Printf("Can't add bootinfo record. Ask a wizard to enlarge MAX_BI_SIZE.\n");
		return 0;
    }

    record       = (struct bi_record *)
						((unsigned long)&v->bi_union.record + v->bi_size);
    record->tag  = tag;
    record->size = size2;
    mem_move(record->data, data, size);
    v->bi_size  += size2;

    return 1;
}

/*--------------------------------------------------------------------------*/
/* Add a String Record to the Bootinfo Structure
 */

int
add_bi_string
(	unsigned short		tag,
	const unsigned char	*s
)
{
    return add_bi_record(tag, str_len(s) + 1, s);
}

/*--------------------------------------------------------------------------*/
/* Create the Bootinfo Structure
 */

int
create_bootinfo
(	void
)
{	int					i;
    struct bi_record	*record;

    /* initialization */
    v->bi_size = 0;

    /* Generic tags */
    if (!add_bi_record(BI_MACHTYPE, sizeof(v->bi.machtype), &v->bi.machtype))
		return 0;

    if (!add_bi_record(BI_CPUTYPE, sizeof(v->bi.cputype), &v->bi.cputype))
		return 0;

    if (!add_bi_record(BI_FPUTYPE, sizeof(v->bi.fputype), &v->bi.fputype))
		return 0;

    if (!add_bi_record(BI_MMUTYPE, sizeof(v->bi.mmutype), &v->bi.mmutype))
		return 0;

    for (i = 0; i < v->bi.num_memory; i++)
		if (!add_bi_record(BI_MEMCHUNK, sizeof(v->bi.memory[i]), &v->bi.memory[i]))
		    return 0;

    if (v->bi.ramdisk.size)
		if (!add_bi_record(BI_RAMDISK, sizeof(v->bi.ramdisk), &v->bi.ramdisk))
		    return 0;

    if (!add_bi_string(BI_COMMAND_LINE, v->bi.command_line))
		return 0;

    /* Trailer */
    record      = (struct bi_record *)
						((unsigned long)&v->bi_union.record + v->bi_size);
    record->tag = BI_LAST;
	v->bi_size += sizeof(v->bi_union.record.tag);

    return 1;
}

/*--------------------------------------------------------------------------*/
/*  Create the Bootinfo structure for backwards compatibility mode
 */

#ifdef BOOTINFO_COMPAT_1_0

int
create_compat_bootinfo
(	void
)
{	unsigned	i;

	v->compat_bootinfo.machtype = v->bi.machtype;

    if (v->bi.cputype & CPU_68020)
		v->compat_bootinfo.cputype = COMPAT_CPU_68020;
    else if (v->bi.cputype & CPU_68030)
		v->compat_bootinfo.cputype = COMPAT_CPU_68030;
    else if (v->bi.cputype & CPU_68040)
		v->compat_bootinfo.cputype = COMPAT_CPU_68040;
    else if (v->bi.cputype & CPU_68060)
		v->compat_bootinfo.cputype = COMPAT_CPU_68060;
    else
	{
		Printf("CPU type 0x%08lx not supported by kernel\n", v->bi.cputype);
		return(0);
    }

	if (v->bi.fputype & FPU_68881)
		v->compat_bootinfo.cputype |= COMPAT_FPU_68881;
    else if (v->bi.fputype & FPU_68882)
		v->compat_bootinfo.cputype |= COMPAT_FPU_68882;
    else if (v->bi.fputype & FPU_68040)
		v->compat_bootinfo.cputype |= COMPAT_FPU_68040;
    else if (v->bi.fputype & FPU_68060)
		v->compat_bootinfo.cputype |= COMPAT_FPU_68060;
    else if (v->bi.fputype)
	{
		Printf("FPU type 0x%08lx not supported by kernel\n", v->bi.fputype);
		return(0);
    }

    v->compat_bootinfo.num_memory = v->bi.num_memory;
    if (v->compat_bootinfo.num_memory > COMPAT_NUM_MEMINFO)
	{
		Printf("Warning: using only %ld blocks of memory\n",
			       COMPAT_NUM_MEMINFO);
		v->compat_bootinfo.num_memory = COMPAT_NUM_MEMINFO;
    }

    for (i = 0; i < v->compat_bootinfo.num_memory; i++)
	{
		v->compat_bootinfo.memory[i].addr = v->bi.memory[i].addr;
		v->compat_bootinfo.memory[i].size = v->bi.memory[i].size;
    }

    if (v->bi.ramdisk.size)
	{
		v->compat_bootinfo.ramdisk_size = (v->bi.ramdisk.size + 1023) / 1024;
		v->compat_bootinfo.ramdisk_addr = v->bi.ramdisk.addr;
    }
	else
	{
		v->compat_bootinfo.ramdisk_size = 0;
		v->compat_bootinfo.ramdisk_addr = 0;
    }

    str_ncpy(v->compat_bootinfo.command_line,
					v->bi.command_line, COMPAT_CL_SIZE);
    v->compat_bootinfo.command_line[COMPAT_CL_SIZE - 1] = '\0';

    return 1;
}
#endif /* BOOTINFO_COMPAT_1_0 */

/*--------------------------------------------------------------------------*/
/* Call the copy-and-go-code
 */

void
start_kernel
(	unsigned long	kernel_dest_addr,
	char			*kernel_load_addr,
	unsigned long	kernel_mem_size,
	unsigned long	ramdisk_dest_addr,
	char			*ramdisk_load_addr,
	unsigned long	ramdisk_mem_size,
	int				call_monitor
)
{	register void (*a0)() __asm("a0") = v->startcode_entry;
	register long   a1    __asm("a1") = kernel_dest_addr;
	register char  *a2    __asm("a2") = kernel_load_addr;
	register long   a3    __asm("a3") = ramdisk_dest_addr;
	register char  *a4    __asm("a4") = ramdisk_load_addr;
	register long   d0    __asm("d0") = kernel_mem_size;
	register long   d1    __asm("d1") = ramdisk_mem_size;
	register long   d2    __asm("d2") = call_monitor;

	__asm __volatile ("jmp (%%a0)"
					  : /* no outputs */
					  : "r" (a0), "r" (a1), "r" (a2), "r" (a3),
						"r" (a4), "r" (d0), "r" (d1), "r" (d2)
	);

	/* fake a noreturn */
	for (;;);
}

/*--------------------------------------------------------------------------*/
/* load kernel image into a chunk of memory
 */

#ifdef NOT_USED

int
load_kernel
(	const char			*file_name
)
{	void				*memptr;
	unsigned long		memsize;
	long				iosize;
	Elf32_Ehdr			*ehp;			/* elf header pointer				*/
	Elf32_Shdr			*shp;			/* elf section header pointer		*/
#define MAX_SHDR		16
	Elf32_Shdr			saved[MAX_SHDR];/* saved elf section headers		*/
	int					shnum;			/* number of saved section headers	*/
	int					i;
	int					j;
	unsigned			bi;
#define RESV_MEM		(256 * 1024)	/* memory to reserve for io buffers	*/

	loading(file_name);

    stream_init();
    stream_push( &file_mod   );
    stream_push( &gunzip_mod );

	/* open kernel executable and read exec header */
	if (sopen(file_name) < 0)
	{
		Printf("\nUnable to open kernel file `%s'\n", file_name);
		return -1;
	}

	/* get size of largest free memory block */
	if ((memsize = mem_maxalloc()) < RESV_MEM)
	{
		sclose();
		goto no_mem;
	}

	/* get size of memory to allocate */
	memsize -= RESV_MEM;

	/* allocate chunk of memory */
	if ((memptr = mem_alloc(memsize)) == NULL)
	{
		sclose();
		goto no_mem;
	}


	/* read kernel image into memory */
	iosize = sread(memptr, memsize);

	/* close file */
	sclose();


	/* error or end of file */
	if (iosize < 1)
	{
		Printf("\nError reading kernel image \"%s\"\n", file_name);
		return -1;
	}

	/* if we filled the buffer */
	if (iosize == memsize)
	{
		goto free_mem;
	}

	/* get pointer to elf header */
	ehp = (Elf32_Ehdr *) memptr;

	/* get pointer to first section header */
	shp = (Elf32_Shdr *) ((char *) ehp + ehp->e_shoff);

	/* scan through section headers and save
	 * those whose section resides in memory
	 */
	for (shnum = i = 0; i < ehp->e_shnum; i++)
	{
		/* if segment resides in memory */
		if (shp->sh_flags & SHF_ALLOC)
		{
			/* make sure there's enough room */
			if (shnum == MAX_SHDR)
			{
				mem_free(memptr);
				Printf(
					"\nKernel image \"%s\" has too many ELF section headers\n",
						file_name);
				return -1;
			}

			/* save and count it */
			mem_move(&saved[shnum++], shp, sizeof(saved[0]));
		}

		/* get pointer to next section header */
		shp = (Elf32_Shdr *) ((char *)shp + ehp->e_shentsize);
	}

	/* make sure saved section headers are in accending address order */
	for (i = 0; i < shnum - 1; i++)
	{
		for (j = i + 1; j < shnum; j++)
		{
			if (saved[i].sh_addr > saved[j].sh_addr)
			{
				Elf32_Shdr temp = saved[i];
				saved[i] = saved[j];
				saved[j] = temp;
			}
		}
	}

	/* initialise bootinfo offset */
	bi = 0;
		 
	/*
	 * Work from last segment down, copy or zero fill as required.
	 * Assumes segments are in ascending address order, and that any which
	 * need to move need to move to higher addresses.
	 */

	/* copy first */
	i = shnum;
	while (i--)
	{
		/* remember the address past the highest segment
		 * this is where the bootinfo will be placed
		 */
		if ((saved[i].sh_addr + saved[i].sh_size) > bi)
		{
			bi = saved[i].sh_addr + saved[i].sh_size;

			/* check this section doesn't exceed available memory */
			if (bi > memsize)
			{
				goto free_mem;
			}
		}

		/* if section occupies no space in loaded image */
		if (saved[i].sh_type == SHT_NOBITS)
		{
			/* no data to copy */
			continue;
		}

		/* if section needs moving */
		if (saved[i].sh_addr != saved[i].sh_offset)
		{
			/* move it */
			mem_move((char *)memptr + saved[i].sh_addr,
					 (char *)memptr + saved[i].sh_offset,
					 saved[i].sh_size);
		}
	}

	/* fill second */	
	i = shnum;
	while (i--)
	{
		/* if section occupies no space in loaded image */
		if (saved[i].sh_type == SHT_NOBITS)
		{
			/* zero fill it */
			mem_clear((char *)memptr + saved[i].sh_addr, saved[i].sh_size);
		}
	}

	/* resize kernel memory allocation */
	if (mem_resize(memptr, bi) != 0)
	{
	free_mem:
		mem_free(memptr);
	no_mem:
		put_str("\nNot enough memory to boot!\n");
		return -1;
	}

	return bi;
}

#endif /* NOT_USED */

/*--------------------------------------------------------------------------*/
/* Boot the Linux/m68k Operating System
 */

void
boot_linux
(	const BOOTRECORD	*boot_record
)
{	int					i;
	unsigned long		start_mem;
	unsigned long		mem_size;
	unsigned long		ramdisk_mem_size;
    unsigned long		kernel_mem_size;
	unsigned long		kernel_size;
	Elf32_Phdr			*kernel_phdrs      = NULL;
	char				*kernel_load_addr  = NULL;
	char				*ramdisk_load_addr = NULL;
	int					file_opened        = FALSE;
    void				*bi_ptr;
	unsigned long		min_addr;
	unsigned long		max_addr;

    mem_clear(&v->bi, sizeof(v->bi));

	/* get default (2.1.xx) machine type */
	v->bi.machtype = get_machtype();

	/* detect and configure cpu/fpu type */
	switch (i = cpu_type())
	{
		case 40:
		{
			v->bi.cputype = CPU_68040;
			break;
		}

		case 60:
		{
			v->bi.cputype = CPU_68060;
			break;
		}
	}
	v->bi.mmutype = v->bi.cputype;
	if (fpu_type())
		v->bi.fputype = v->bi.cputype;
	else
		v->bi.fputype = 0;

	if (debug_mode)
	{
		put_char('\n');
		print_model(i, v->bi.fputype);
		Printf(" CPU:680%ld with%s FPU\n",
			i, v->bi.fputype ? " internal" : "out");
	}

	/* if no ram size has been specified */
	if ((mem_size = VALUE(boot_record->memsize)) == 0)
	{
		/* use detected memory size */
		mem_size = v->detected_mem_size;
	}

	/* if too much ram has been specified */
	if (mem_size > v->detected_mem_size)
	{
		/* use detected memory size */
		mem_size = v->detected_mem_size;

		if (debug_mode)
		{
			Printf("Specified memory size greater than detected size.\n"
				   "Using specified memory size %ldK\n",
					mem_size >> 10
			);
		}
	}

	else if (debug_mode)
	{
		Printf("Using %s memory size %ldK\n",
				boot_record->memsize ? "specified" : "detected",
				mem_size >> 10
		);
	}

	v->bi.num_memory     = 1;
	v->bi.memory[0].addr = 0x00000000;
	v->bi.memory[0].size = mem_size;

	if (debug_mode)
	{
		Printf("Found %ld block%s of memory\n", v->bi.num_memory,
			v->bi.num_memory > 1 ? "s" : "");

		for (i = 0; i < v->bi.num_memory; i++)
		{
			Printf(" Block %ld: 0x%08lx to 0x%08lx (%ldK)\n", i,
					v->bi.memory[i].addr,
					v->bi.memory[i].addr + v->bi.memory[i].size - 1,
					v->bi.memory[i].size >> 10
			);
		}
	}

	/* get command line */
	str_ncpy(v->bi.command_line, "BOOT_IMAGE=",      CL_SIZE);
	str_ncat(v->bi.command_line, boot_record->label, CL_SIZE);
	if (!v->force_prompt)
	{
		str_ncat(v->bi.command_line, " auto", CL_SIZE);
	}
	if (boot_record->args)
	{
		str_ncat(v->bi.command_line, " ",               CL_SIZE);
   		str_ncat(v->bi.command_line, boot_record->args, CL_SIZE);
   	}

   	/* any parameters entered on the boot prompt command line */
   	if (v->cmdln_buffer[0])
   	{
		str_ncat(v->bi.command_line, " ",             CL_SIZE);
   		str_ncat(v->bi.command_line, v->cmdln_buffer, CL_SIZE);
   	}

   	if (debug_mode)
   	{
		Printf("Using command line `%s'\n", v->bi.command_line);
	}

	start_mem = v->bi.memory[0].addr;
	mem_size  = v->bi.memory[0].size;

	/* support for ramdisk */
	if (boot_record->ramdisk)
	{
		unsigned long	mem_end = start_mem + mem_size;
		int				size;

		if ((size = file_size(boot_record->ramdisk)) == -1)
		{
			Printf("Unable to find size of ramdisk file `%s'\n",
				boot_record->ramdisk);
			goto fail;
		}

		/* record ramdisk size */
		v->bi.ramdisk.size = size;

		/* don't allow ramdisk to be moved over the startup code */
		if (mem_end > (unsigned long) v->startcode_entry)
			mem_end = (unsigned long) v->startcode_entry;
	
		ramdisk_mem_size   = (v->bi.ramdisk.size + 1023) & ~1023;
		v->bi.ramdisk.addr = mem_end - ramdisk_mem_size;
	}
	else
	{
		v->bi.ramdisk.size = 0;
		v->bi.ramdisk.addr = 0;
		ramdisk_mem_size   = 0;
	}

	loading(boot_record->label);

    stream_init();
    stream_push( &file_mod );
    stream_push( &gunzip_mod );

	/* open kernel executable and read exec header */
	if (sopen(boot_record->kernel) < 0)
	{
		Printf("\nUnable to open kernel file `%s'\n", boot_record->kernel);
		goto fail;
	}
	file_opened = TRUE;

	if (sread(&v->kexec_elf, sizeof(v->kexec_elf)) != sizeof(v->kexec_elf))
	{
		Printf("\nCannot read ELF header of kernel image `%s'\n",
				boot_record->kernel);
		goto fail;
	}

	/* Allow images starting '0x00001000 0x00000800' as they are
	 * probably valid kernels on which 16xcfg has been run.
	 */
	if (mem_cmp(&v->kexec_elf.e_ident[EI_MAG0], ELFMAG, SELFMAG) != 0 &&
		mem_cmp(&v->kexec_elf.e_ident[EI_MAG0],
			"\000\000\020\000\000\000\010\000", 8) != 0)
	{
		Printf("\nKernel image `%s' is not an ELF executable\n",
				boot_record->kernel);
		goto fail;
	}

	/* A few plausibility checks */
	if ((v->kexec_elf.e_type    != ET_EXEC   )
	||	(v->kexec_elf.e_machine != EM_68K    )
	||	(v->kexec_elf.e_version != EV_CURRENT))
	{
		Printf("\nInvalid ELF header contents in kernel `%s'\n",
				boot_record->kernel);
		goto fail;
	}

	/* allocate memory for program headers */
	if ((kernel_phdrs = (Elf32_Phdr *)
		mem_alloc(v->kexec_elf.e_phnum * sizeof(Elf32_Phdr))) == NULL)
	{
		goto no_mem;
	}

	/* Load the program headers */
	sseek(v->kexec_elf.e_phoff, SEEK_SET);
	if (sread(kernel_phdrs,
			v->kexec_elf.e_phnum * sizeof(*kernel_phdrs)) !=
				v->kexec_elf.e_phnum * sizeof(*kernel_phdrs))
	{
		Printf("\nUnable to read program headers from kernel file `%s'\n",
				boot_record->kernel);
		goto fail;
	}

	/* Load the kernel at one page after start of mem */
	start_mem += PAGE_SIZE;
	mem_size  -= PAGE_SIZE;

	/* calculate the total required amount of memory */
	min_addr = 0xffffffff;
	max_addr = 0;

	for (i = 0; i < v->kexec_elf.e_phnum; i++)
	{
		if (min_addr > kernel_phdrs[i].p_vaddr)
		{
			min_addr = kernel_phdrs[i].p_vaddr;
		}
		if (max_addr < kernel_phdrs[i].p_vaddr + kernel_phdrs[i].p_memsz)
		{
			max_addr = kernel_phdrs[i].p_vaddr + kernel_phdrs[i].p_memsz;
		}
	}

	/* This is needed for newer linkers that include the header in
	 * the first segment.
	 */
	if (min_addr == 0)
	{
		min_addr                  = PAGE_SIZE;
		kernel_phdrs[0].p_vaddr  += PAGE_SIZE;
		kernel_phdrs[0].p_offset += PAGE_SIZE;
		kernel_phdrs[0].p_filesz -= PAGE_SIZE;
		kernel_phdrs[0].p_memsz  -= PAGE_SIZE;
	}

	/* get size of kernel */
	kernel_size = max_addr - min_addr;

#ifdef NOT_USED
	/* round up to long word boundry to match new linkers (ld 2.9.1) _end symbol
	 * which is where the kernel expects to find the bootinfo
	 */
	kernel_size = (kernel_size + 3) & ~3;
#endif /* NOT_USED */

    /* create the bootinfo structure */
    if (!create_bootinfo())
    {
		Printf("\nCouldn't create bootinfo\n");
		goto fail;
	}

	/* get size of memory required for kernel and bootinfo */
    kernel_mem_size = kernel_size + v->bi_size;

#ifdef BOOTINFO_COMPAT_1_0
	/* make sure we have enough room for old bootinfo */
    if (sizeof(v->compat_bootinfo) > v->bi_size)
    {
		kernel_mem_size = kernel_size + sizeof(v->compat_bootinfo);
	}
#endif /* BOOTINFO_COMPAT_1_0 */

	/* force ramdisk load address onto long word boundry */
	kernel_mem_size = (kernel_mem_size + 3) & ~3;

	/* allocate memory for kernel and ramdisk */
	if ((kernel_load_addr = mem_alloc(kernel_mem_size + ramdisk_mem_size))
			 == NULL)
	{
		goto no_mem;
	}

	/* get pointer to memory for ramdisk image (above kernel) */
	if (ramdisk_mem_size)
	{
		ramdisk_load_addr = kernel_load_addr + kernel_mem_size;
	}

	/* read the text and data segments from the kernel image */
	for (i = 0; i < v->kexec_elf.e_phnum; i++)
	{
		if (sseek(kernel_phdrs[i].p_offset, SEEK_SET) == -1)
		{
			Printf("\nFailed to seek to segment %ld\n", i);
			goto fail;
		}

		if (sread(kernel_load_addr + kernel_phdrs[i].p_vaddr - PAGE_SIZE,
				 kernel_phdrs[i].p_filesz) != kernel_phdrs[i].p_filesz)
		{
			Printf("\nFailed to read segment %ld\n", i);
			goto fail;
		}
	}

	sclose();
	percent_term("OK\n");
	file_opened = FALSE;

	if (boot_record->ramdisk)
	{
		loading("Ramdisk Image");

		stream_init();
		stream_push(&file_mod);

		if (sopen(boot_record->ramdisk) < 0)
		{
			Printf("\nUnable to open ramdisk file `%s'\n",
					boot_record->ramdisk);
			goto fail;
		}

		if (sread(ramdisk_load_addr, v->bi.ramdisk.size) < 0)
		{
			sclose();
			Printf("\nFailed to read ramdisk file\n");
			goto fail;
		}

		sclose();
		percent_term("OK\n");
	}

	if (boot_record->symtab)
	{
		load_symbols(boot_record->symtab);
	}

    /* Check kernel's bootinfo version */
    {	unsigned long kernel_major = check_bootinfo_version(kernel_load_addr);

    	if (kernel_major == BI_VERSION_MAJOR(get_booti_version()))
		{
			bi_ptr = &v->bi_union.record;
		}

#ifdef BOOTINFO_COMPAT_1_0
		else if (kernel_major == BI_VERSION_MAJOR(get_compat_booti_version()))
		{
			v->bi.machtype = get_compat_machtype();

			if (!create_compat_bootinfo())
			{
	    		Printf("Couldn't create compat bootinfo\n");
	    		goto fail;
	    	}

			bi_ptr     = &v->compat_bootinfo;
			v->bi_size = sizeof(v->compat_bootinfo);
		}
#endif /* BOOTINFO_COMPAT_1_0 */

		else
		{
			Printf("Kernel has unsupported bootinfo version\n");
			goto fail;
    	}
    }

	/* copy the bootinfo to the end of the kernel image */
	mem_move(kernel_load_addr + kernel_size, bi_ptr, v->bi_size);

 	if (debug_mode)
	{
		if (v->bi.ramdisk.size)
		{
			Printf("RAM disk at 0x%08lx, size is %ld bytes\n",
					ramdisk_load_addr, v->bi.ramdisk.size
			);
		}

		for (i = 0; i < v->kexec_elf.e_phnum; i++)
		{
			Printf("Kernel segment %ld at 0x%08lx, size %ld, align %ld\n",
				i,
				start_mem + kernel_phdrs[i].p_vaddr - PAGE_SIZE,
				kernel_phdrs[i].p_memsz,
				kernel_phdrs[i].p_align
			);
		}

		Printf("Boot info at 0x%08lx\n", start_mem + kernel_size);

		Printf("Kernel entry is 0x%08lx\n", v->kexec_elf.e_entry);

		put_str("\nPress a key to continue booting... ");
		get_char(NO_TIMEOUT);
		put_char('\n');
	}

	/* move the copy-and-go code to it's proper place */
	mem_move((void *)v->startcode_entry, &startcode_beg,
								&startcode_end - &startcode_beg);
	invalidate_icache();

	/* execute the copy-and-go code (**never returns**) */
	start_kernel(	start_mem,          kernel_load_addr,  kernel_mem_size,
					v->bi.ramdisk.addr, ramdisk_load_addr, ramdisk_mem_size,
					VALUE(boot_record->callmonitor));

no_mem:
	put_str("\nNot enough memory to boot!\n");

fail:
	if (file_opened)
	{
		put_char('\n');
		sclose();
	}
	mem_free(kernel_phdrs    );
	mem_free(kernel_load_addr);
}

/*--------------------------------------------------------------------------*/

void
start_loader
(	unsigned long		detected_mem_size
)
{	const BOOTRECORD	*boot_record;
	unsigned long		delay;
	unsigned long		timeout;
	char				save_force_prompt;

	/* allocate global storage */
	if ((v = mem_alloc(sizeof(*v))) == NULL)
	{
		panic("Not enough memory");
	}

	/* save detected memory size for later */
	v->detected_mem_size = detected_mem_size;

	/* the kernel start code is to be placed at the bottom of
	 * the area we have reserved for the stack
	 */
	v->startcode_entry = (void (*)(void))(detected_mem_size - STACK_SIZE);

	debug_mode = get_debug_tag();

	/* get configuration */
	read_map_data();

	/* find the default boot record */
	if ((v->default_boot_record = find_boot_record(NULL)) == NULL)
	{
		panic("No boot Operating System!");
	}

	save_force_prompt = v->force_prompt;

	/* display boot message file */
	if (v->boot_options->boot_message)
	{
		display_message_file(v->boot_options->boot_message);
	}

restart:
	v->leader_adjust = 5;
	Printf("LILO ");

	if (!v->force_prompt)
	{
		/* get delay */
		delay = BO_VALUE(boot_delay);

		/* if a delay was specified */
		if (delay)
		{
			/* wait a while for a key press */
			if (get_char(delay) != -1)
			{
				/* key pressed, force boot prompt */
				v->force_prompt = TRUE;
			}
		}
	}

	/* get timeout */
	timeout = BO_VALUE(boot_timeout);

	while (1)
	{
		/* clear old command line */
		v->cmdln_buffer[0] = '\0';

		/* if interactive */
		if (v->force_prompt)
		{
			/* ask for boot record */
			boot_record = get_boot_record(timeout);
		}
		else
		{
			/* use default boot record */
			boot_record = v->default_boot_record;
		}

		/* if there's a password */
		if (boot_record->password)
		{
			/* does password only apply to extra command line */
			int restricted = VALUE(boot_record->restricted);

			if (!restricted || (restricted && v->cmdln_buffer[0]))
			{
				put_str("Password: ");
				if (read_line(ECHO_DOT, timeout) == -1)
				{
					/* timeout */
					v->force_prompt = save_force_prompt;
					goto restart;
				}
	
				if (!case_equal_strings(v->input_buffer,
										boot_record->password))
				{
					put_str("Incorrect password.\n");
					goto again;
				}
			}
		}

		/* go for it */
		boot_linux(boot_record);

		Printf("\nCouldn't boot the %s configuration\n\n",
				v->force_prompt ? "specified" : "default");

	again:
		v->force_prompt = TRUE;
	}
}

/*--------------------------------------------------------------------------*/

void
Main
(	unsigned long		arg
)
{	unsigned long		detected_mem_size;
	unsigned long		free_mem_start;

	/* enable instruction cache */
	enable_icache();

	/* do board specific initialisations */
	loader_init(arg);

	/* detect memory size */
	detected_mem_size = 0;
	while (ram_probe(detected_mem_size))
	{
		detected_mem_size += (256 * 1024);
	}

	/* initialise heap */
	free_mem_start = ((unsigned long)&Map.Data[Map.Size] + 3) & ~3;
	mem_alloc_init((void *)free_mem_start,
			detected_mem_size -  STACK_SIZE - free_mem_start);

	/* move stack pointer to end of detected memory
	 * and execute: start_loader(detected_mem_size);
	 */
	asm volatile (	"movel	%0,%%sp\n"		/* switch stack				*/
					"movel	%0,-(%%sp)\n"	/* arg - detected_mem_size	*/
					"movel	%1,-(%%sp)\n"	/* fake return address		*/
					"jmp	(%1)\n"			/* jump to function			*/
					: /* no outputs */
					: "r" (detected_mem_size), "a" (&start_loader)
					: "sp", "memory"
	);
}

/*-----------------------------< end of file >------------------------------*/
