/*======================================================================
  
    Cardbus device configuration
    
    cardbus.c 1.33 1998/08/20 23:39:38

    The contents of this file are subject to the Mozilla Public
    License Version 1.0 (the "License"); you may not use this file
    except in compliance with the License. You may obtain a copy of
    the License at http://www.mozilla.org/MPL/

    Software distributed under the License is distributed on an "AS
    IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
    implied. See the License for the specific language governing
    rights and limitations under the License.

    The initial developer of the original code is David A. Hinds
    <dhinds@hyper.stanford.edu>.  Portions created by David A. Hinds
    are Copyright (C) 1998 David A. Hinds.  All Rights Reserved.
    
    These routines handle allocating resources for Cardbus cards, as
    well as setting up and shutting down Cardbus sockets.  They are
    called from cs.c in response to Request/ReleaseConfiguration and
    Request/ReleaseIO calls.
    
======================================================================*/

#include <pcmcia/config.h>
#define __NO_VERSION__
#include <pcmcia/k_compat.h>

#ifdef __LINUX__
#include <linux/kernel.h>
#include <linux/string.h>
#include <linux/malloc.h>
#include <linux/mm.h>
#include <linux/pci.h>
#include <linux/ioport.h>
#include <asm/irq.h>
#include <asm/io.h>
#endif

#ifndef PCMCIA_DEBUG
#define PCMCIA_DEBUG 1
#endif
static int pc_debug = PCMCIA_DEBUG;

#define IN_CARD_SERVICES
#include <pcmcia/version.h>
#include <pcmcia/cs_types.h>
#include <pcmcia/ss.h>
#include <pcmcia/cs.h>
#include <pcmcia/bulkmem.h>
#include <pcmcia/cistpl.h>
#include "cs_internal.h"
#include "rsrc_mgr.h"

/*====================================================================*/

#define FIND_FIRST_BIT(n)	((n) - ((n) & ((n)-1)))

#define pci_readb		pcibios_read_config_byte
#define pci_writeb		pcibios_write_config_byte
#define pci_readw		pcibios_read_config_word
#define pci_writew		pcibios_write_config_word
#define pci_readl		pcibios_read_config_dword
#define pci_writel		pcibios_write_config_dword

#define CB_BAR(n)		(PCI_BASE_ADDRESS_0+(4*(n)))
#define CB_ROM_BASE		0x0030

/* Offsets in the Expansion ROM Image Header */
#define ROM_SIGNATURE		0x0000	/* 2 bytes */
#define ROM_DATA_PTR		0x0018	/* 2 bytes */

/* Offsets in the CardBus PC Card Data Structure */
#define PCDATA_SIGNATURE	0x0000	/* 4 bytes */
#define PCDATA_VPD_PTR		0x0008	/* 2 bytes */
#define PCDATA_LENGTH		0x000a	/* 2 bytes */
#define PCDATA_REVISION		0x000c
#define PCDATA_IMAGE_SZ		0x0010	/* 2 bytes */
#define PCDATA_ROM_LEVEL	0x0012	/* 2 bytes */
#define PCDATA_CODE_TYPE	0x0014
#define PCDATA_INDICATOR	0x0014

typedef struct cb_config_t {
    u_int		size[7];
#if (LINUX_VERSION_CODE < VERSION(2,1,90))
    struct {
	unsigned int	irq;
	unsigned long	base_address[6];
	unsigned long	rom_address;
    } dev;
#else
    struct pci_dev	dev;
#endif
} cb_config_t;

/* There are three classes of bridge maps: IO ports,
   non-prefetchable memory, and prefetchable memory */
typedef enum { B_IO, B_M1, B_M2 } bridge_type;

/*===================================================================*/

static void dump_rom(u_char *b)
{
    u_int img = 0, ofs = 0, sz;
    u_short data;
    DEBUG(0, ("ROM image dump:\n"));
    while ((readb(b) == 0x55) && (readb(b+1) == 0xaa)) {
	data = readb(b+ROM_DATA_PTR) +
	    (readb(b+ROM_DATA_PTR+1) << 8);
	sz = 512 * (readb(b+data+PCDATA_IMAGE_SZ) +
		    (readb(b+data+PCDATA_IMAGE_SZ+1) << 8));
	DEBUG(0, ("  image %d: 0x%06x-0x%06x, signature %c%c%c%c\n",
		  img, ofs, ofs+sz-1,
		  readb(b+data+PCDATA_SIGNATURE),
		  readb(b+data+PCDATA_SIGNATURE+1),
		  readb(b+data+PCDATA_SIGNATURE+2),
		  readb(b+data+PCDATA_SIGNATURE+3)));
	b += sz; ofs += sz; img++;
	if (readb(b+data+PCDATA_INDICATOR) & 0x80) break;
    }
    if (img == 0) printk(KERN_INFO "  no valid images found!\n");
}

static u_int xlate_rom_addr(u_char *b, u_int addr)
{
    u_int img = 0, ofs = 0, sz;
    u_short data;
    while ((readb(b) == 0x55) && (readb(b+1) == 0xaa)) {
	if (img == (addr >> 28))
	    return (addr & 0x0fffffff) + ofs;
	data = readb(b+ROM_DATA_PTR) + (readb(b+ROM_DATA_PTR+1) << 8);
	sz = 512 * (readb(b+data+PCDATA_IMAGE_SZ) +
		    (readb(b+data+PCDATA_IMAGE_SZ+1) << 8));
	b += sz; ofs += sz; img++;
	if (readb(b+data+PCDATA_INDICATOR) & 0x80) break;
    }
    return 0;
}

/*===================================================================*/

void read_cb_mem(socket_info_t *s, u_char fn, int space,
		 u_int addr, u_int len, void *ptr)
{
    DEBUG(3, ("cs: read_cb_mem(%d, %#x, %u)\n", space, addr, len));
    if (space == 0) {
	for (; len; addr++, ptr++, len--)
	    pci_readb(s->cap.cardbus, fn, addr, ptr);
    } else {
	cb_setup_cis_mem(s, space);
	if (space == 7)
	    addr = xlate_rom_addr(s->cb_cis_virt, addr);
	if (s->cb_cis_virt != NULL)
	    for (; len; addr++, ptr++, len--)
		*(u_char *)ptr = readb(s->cb_cis_virt+addr);
    }
}

/*=====================================================================

    These are similar to setup_cis_mem and release_cis_mem for 16-bit
    cards.  The "result" that is used externally is the cb_cis_virt
    pointer.
    
=====================================================================*/

int cb_setup_cis_mem(socket_info_t *s, int space)
{
    cb_bridge_map *m = &s->cb_cis_map;
    u_long base = 0;
    u_int sz, br;

    if (space == s->cb_cis_space)
	return CS_SUCCESS;
    else if (s->cb_cis_space != 0)
	cb_release_cis_mem(s);
    DEBUG(1, ("cs: cb_setup_cis_mem(space %d)\n", space));
    /* If socket is configured, then use existing memory mapping */
    if (s->lock_count) {
	s->cb_cis_virt =
	    ioremap(s->cb_config[0].dev.base_address[space-1],
		    s->cb_config[0].size[space-1] & ~3);
	s->cb_cis_space = space;
	return CS_SUCCESS;
    }
    /* Not configured?  Then set up temporary map */
    br = (space == 7) ? CB_ROM_BASE : CB_BAR(space-1);
    s->cb_cis_space = space;
    pci_writel(s->cap.cardbus, 0, br, 0xffffffff);
    pci_readl(s->cap.cardbus, 0, br, &sz);
    sz &= PCI_BASE_ADDRESS_MEM_MASK;
    sz = (FIND_FIRST_BIT(sz) + 0x0fff) & ~0x0fff;
    if (find_mem_region(&base, sz, "cb_enabler", 0) != 0) {
	printk(KERN_NOTICE "cs: could not allocate %dK memory for"
	       " CardBus socket %d\n", sz/1024, s->sock);
	return CS_OUT_OF_RESOURCE;
    }
    s->cb_cis_virt = ioremap(base, sz);
    DEBUG(1, ("  phys 0x%08lx-0x%08lx, virt 0x%08lx\n",
	      base, base+sz-1, (u_long)s->cb_cis_virt));
    pci_writel(s->cap.cardbus, 0, br, base | 1);
    pci_writeb(s->cap.cardbus, 0, PCI_COMMAND, PCI_COMMAND_MEMORY);
    m->map = 0; m->flags = MAP_ACTIVE;
    m->start = base; m->stop = base+sz-1;
    s->ss_entry(s->sock, SS_SetBridge, m);
    if (space == 7) dump_rom(s->cb_cis_virt);
    return CS_SUCCESS;
}

void cb_release_cis_mem(socket_info_t *s)
{
    cb_bridge_map *m = &s->cb_cis_map;
    u_int br;
    if (s->cb_cis_virt) {
	DEBUG(1, ("cs: cb_release_cis_mem()\n"));
	iounmap(s->cb_cis_virt);
	s->cb_cis_virt = NULL;
	s->cb_cis_space = 0;
    }
    if (m->start) {
	/* This is overkill: we probably only need to release the
	   memory region, but the rest should be safe */
	br = (s->cb_cis_space == 7) ?
	    CB_ROM_BASE : CB_BAR(s->cb_cis_space-1);
	m->map = 0; m->flags = 0;
	s->ss_entry(s->sock, SS_SetBridge, m);
	pci_writeb(s->cap.cardbus, 0, PCI_COMMAND, 0);
	pci_writel(s->cap.cardbus, 0, br, 0);
	release_mem_region(m->start, m->stop - m->start + 1);
	m->start = 0;
    }
}

/*=====================================================================

    cb_enable() has the job of configuring a socket for a Cardbus
    card, and initializing the card's PCI configuration registers.

    It first sets up the Cardbus bridge windows, for IO and memory
    accesses.  Then, it initializes each card function's base address
    registers, interrupt line register, and command register.

    It is called as part of the RequestConfiguration card service.
    It should be called after a previous call to cb_config() (via the
    RequestIO service).
    
======================================================================*/

void cb_enable(socket_info_t *s)
{
    u_char i, j, bus = s->cap.cardbus;
    cb_config_t *c = s->cb_config;
    
    DEBUG(0, ("cs: cb_enable(bus %d)\n", bus));
    
    /* Configure bridge */
    if (s->cb_cis_map.start)
	cb_release_cis_mem(s);
    for (i = 0; i < 3; i++) {
	cb_bridge_map m;
	switch (i) {
	case B_IO:
	    m.map = 0; m.flags = MAP_IOSPACE | MAP_ACTIVE;
	    m.start = s->io[0].BasePort;
	    m.stop = m.start + s->io[0].NumPorts - 1;
	    break;
	case B_M1:
	    m.map = 0; m.flags = MAP_ACTIVE;
	    m.start = s->win[0].base;
	    m.stop = m.start + s->win[0].size - 1;
	    break;
	case B_M2:
	    m.map = 1; m.flags = MAP_PREFETCH | MAP_ACTIVE;
	    m.start = s->win[1].base;
	    m.stop = m.start + s->win[1].size - 1;
	    break;
	}
	if (m.start == 0) continue;
	DEBUG(0, ("  bridge %s map %d (flags 0x%x): 0x%x-0x%x\n",
		  (m.flags & MAP_IOSPACE) ? "io" : "mem",
		  m.map, m.flags, m.start, m.stop));
	s->ss_entry(s->sock, SS_SetBridge, &m);
    }

    /* Set up base address registers */
    for (i = 0; i < s->functions; i++) {
	for (j = 0; j < 6; j++) {
	    if (c[i].dev.base_address[j] != 0)
		pci_writel(bus, i, CB_BAR(j), c[i].dev.base_address[j]);
	}
	if (c[i].dev.rom_address != 0)
	    pci_writel(bus, i, CB_ROM_BASE, c[i].dev.rom_address | 1);
    }

    /* Set up PCI interrupt and command registers */
    for (i = 0; i < s->functions; i++)
	pci_writeb(bus, i, PCI_COMMAND, PCI_COMMAND_MASTER |
		   PCI_COMMAND_IO | PCI_COMMAND_MEMORY);
    if (s->irq.AssignedIRQ) {
	for (i = 0; i < s->functions; i++)
	    pci_writeb(bus, i, PCI_INTERRUPT_LINE,
		       s->irq.AssignedIRQ);
	s->socket.io_irq = s->irq.AssignedIRQ;
	s->ss_entry(s->sock, SS_SetSocket, &s->socket);
    }

#if (LINUX_VERSION_CODE >= VERSION(2,1,90))
    /* Link into PCI device chain */
    c[s->functions-1].dev.next = pci_devices;
    pci_devices = &c[0].dev;
#if (LINUX_VERSION_CODE >= VERSION(2,1,103))
    /* pci_proc_attach_device(&c[0].dev); */
#endif
#endif
}

/*======================================================================

    cb_disable() unconfigures a Cardbus card previously set up by
    cb_enable().

    It is called from the ReleaseConfiguration service.
    
======================================================================*/

void cb_disable(socket_info_t *s)
{
    u_char i;
    cb_bridge_map m = { 0, 0, 0, 0xffff };
#if (LINUX_VERSION_CODE >= VERSION(2,1,90))
    struct pci_dev **p, *q;
    cb_config_t *c = s->cb_config;
    
    /* Unlink from PCI device chain */
#if (LINUX_VERSION_CODE >= VERSION(2,1,103))
    /* pci_proc_detach_device(&c[0].dev); */
#endif
    for (p = &pci_devices; *p; p = &((*p)->next))
	if (*p == &c[0].dev) break;
    for (q = *p; q; q = q->next)
	if (q == &c[s->functions-1].dev) break;
    if (p && q) {
	*p = q->next;
    }
#endif
    
    DEBUG(0, ("cs: cb_disable(bus %d)\n", s->cap.cardbus));
    
    /* Turn off bridge windows */
    if (s->cb_cis_map.start)
	cb_release_cis_mem(s);
    for (i = 0; i < 3; i++) {
	switch (i) {
	case B_IO: m.map = 0; m.flags = MAP_IOSPACE; break;
	case B_M1: m.map = m.flags = 0; break;
	case B_M2: m.map = 1; m.flags = 0; break;
	}
	s->ss_entry(s->sock, SS_SetBridge, &m);
    }
}

/*=====================================================================

    cb_config() has the job of allocating all system resources that
    a Cardbus card requires.  Rather than using the CIS (which seems
    to not always be present), it treats the card as an ordinary PCI
    device, and probes the base address registers to determine each
    function's IO and memory space needs.

    It is called from the RequestIO card service.
    
======================================================================*/

int cb_config(socket_info_t *s)
{
    u_short vend, v, dev;
    u_char i, j, fn, bus = s->cap.cardbus, *name;
    u_int sz, m, mask[3], num[3], base[3];
    cb_config_t *c;
    int irq, try, ret;
    
    pci_readw(bus, 0, PCI_VENDOR_ID, &vend);
    pci_readw(bus, 0, PCI_DEVICE_ID, &dev);
    printk(KERN_INFO "cs: cb_config(bus %d): vendor 0x%04x, "
	   "device 0x%04x\n", bus, vend, dev);

    pci_readb(bus, 0, PCI_HEADER_TYPE, &fn);
    if (fn != 0) {
	/* Count functions */
	for (fn = 0; fn < 8; fn++) {
	    pci_readw(bus, fn, PCI_VENDOR_ID, &v);
	    if (v != vend) break;
	}
    } else fn = 1;
    s->functions = fn;
    
    c = kmalloc(fn * sizeof(struct cb_config_t), GFP_KERNEL);
    memset(c, fn * sizeof(struct cb_config_t), 0);
    s->cb_config = c;

#if (LINUX_VERSION_CODE >= VERSION(2,1,90))
    s->pci_bus = kmalloc(sizeof(struct pci_bus), GFP_KERNEL);
    memset(s->pci_bus, sizeof(struct pci_bus), 0);
    s->pci_bus->number = bus;
    for (i = 0; i < fn; i++) {
	c[i].dev.bus = s->pci_bus;
	if (i < fn-1) {
	    c[i].dev.sibling = c[i].dev.next = &c[i+1].dev;
	}
	c[i].dev.devfn = i;
	c[i].dev.vendor = vend; c[i].dev.device = dev;
	pci_readl(bus, i, PCI_CLASS_REVISION, &c[i].dev.class);
	c[i].dev.class >>= 8;
	pci_readb(bus, i, PCI_HEADER_TYPE, &j);
	c[i].dev.hdr_type = j;	
    }
#endif
    
    /* Determine IO and memory space needs */
    num[B_IO] = num[B_M1] = num[B_M2] = 0;
    mask[B_IO] = mask[B_M1] = mask[B_M2] = 0;
    for (i = 0; i < fn; i++) {
	for (j = 0; j < 6; j++) {
	    pci_writel(bus, i, CB_BAR(j), 0xffffffff);
	    pci_readl(bus, i, CB_BAR(j), &sz);
	    if (sz == 0) continue;
	    if (sz & PCI_BASE_ADDRESS_SPACE) {
		m = B_IO;
		sz &= PCI_BASE_ADDRESS_IO_MASK;
	    } else {
		m = (sz & PCI_BASE_ADDRESS_MEM_PREFETCH) ? B_M2 : B_M1;
		sz &= PCI_BASE_ADDRESS_MEM_MASK;
	    }
	    sz = FIND_FIRST_BIT(sz);
	    num[m] += sz; mask[m] |= sz;
	    c[i].size[j] = sz | m;
	}
	pci_writel(bus, i, CB_ROM_BASE, 0xffffffff);
	pci_readl(bus, i, CB_ROM_BASE, &sz);
	if (sz != 0) {
	    sz = FIND_FIRST_BIT(sz & ~0x00000001);
	    num[B_M1] += sz; mask[B_M1] |= sz;
	    c[i].size[6] = sz | B_M1;
	}
    }

    /* Allocate system resources */
    name = "cb_enabler";
    s->io[0].NumPorts = num[B_IO];
    s->io[0].BasePort = 0;
    if (num[B_IO]) {
	if (find_io_region(&s->io[0].BasePort, num[B_IO], name) != 0) {
	    printk(KERN_NOTICE "cs: could not allocate %d IO ports for"
		   " CardBus socket %d\n", num[B_IO], s->sock);
	    goto failed;
	}
	base[B_IO] = s->io[0].BasePort + num[B_IO];
    }
    s->win[0].size = (num[B_M1] + 0x0fff) & ~0x0fff;
    s->win[0].base = 0;
    if (num[B_M1]) {
	if (find_mem_region(&s->win[0].base, num[B_M1], name, 0) != 0) {
	    printk(KERN_NOTICE "cs: could not allocate %dK memory for"
		   " CardBus socket %d\n", num[B_M1]/1024, s->sock);
	    goto failed;
	}
	base[B_M1] = s->win[0].base + num[B_M1];
    }
    s->win[1].size = (num[B_M2] + 0x0fff) & ~0x0fff;
    s->win[1].base = 0;
    if (num[B_M2]) {
	if (find_mem_region(&s->win[1].base, num[B_M2], name, 0) != 0) {
	    printk(KERN_NOTICE "cs: could not allocate %dK memory for"
		   " CardBus socket %d\n", num[B_M2]/1024, s->sock);
	    goto failed;
	}
	base[B_M2] = s->win[1].base + num[B_M2];
    }
    
    /* Set up base address registers */
    while (mask[B_IO] | mask[B_M1] | mask[B_M2]) {
	num[B_IO] = FIND_FIRST_BIT(mask[B_IO]); mask[B_IO] -= num[B_IO];
	num[B_M1] = FIND_FIRST_BIT(mask[B_M1]); mask[B_M1] -= num[B_M1];
	num[B_M2] = FIND_FIRST_BIT(mask[B_M2]); mask[B_M2] -= num[B_M2];
	for (i = 0; i < fn; i++) {
	    for (j = 0; j < 7; j++) {
		sz = c[i].size[j];
		m = sz & 3; sz &= ~3;
		if (sz && (sz == num[m])) {
		    base[m] -= sz;
		    if (j < 6)
			printk(KERN_INFO "  fn %d bar %d: ", i, j+1);
		    else
			printk(KERN_INFO "  fn %d rom: ", i);
		    printk("%s 0x%x-0x%x\n", (m) ? "mem" : "io",
			   base[m], base[m]+sz-1);
		    c[i].dev.base_address[j] = base[m];
		}
	    }
	}
    }
    
    /* Allocate interrupt if needed */
    s->irq.AssignedIRQ = irq = 0; ret = -1;
    for (i = 0; i < fn; i++) {
	pci_readb(bus, i, PCI_INTERRUPT_PIN, &j);
	if (j == 0) continue;
	if (irq == 0) {
	    if (s->cap.irq_mask == (1 << s->cap.pci_irq))
		irq = s->cap.pci_irq;
#ifdef CONFIG_ISA
	    else
		for (try = 0; try < 2; try++) {
		    for (irq = 0; irq < NR_IRQS; irq++)
			if ((s->cap.irq_mask >> irq) & 1) {
			    ret = try_irq(IRQ_TYPE_EXCLUSIVE, irq, try);
			    if (ret == 0) break;
			}
		    if (ret == 0) break;
		}
	    if (ret != 0) goto failed;
#endif
	    s->irq.AssignedIRQ = irq;
	}
    }
    c[0].dev.irq = irq;
    
    return CS_SUCCESS;

failed:
    cb_release(s);
    return CS_OUT_OF_RESOURCE;
}

/*======================================================================

    cb_release() releases all the system resources (IO and memory
    space, and interrupt) committed for a Cardbus card by a prior call
    to cb_config().

    It is called from the ReleaseIO() service.
    
======================================================================*/

void cb_release(socket_info_t *s)
{
    cb_config_t *c = s->cb_config;
    
    DEBUG(0, ("cs: cb_release(bus %d)\n", s->cap.cardbus));
    
    if (s->win[0].size > 0)
	release_mem_region(s->win[0].base, s->win[0].size);
    if (s->win[1].size > 0)
	release_mem_region(s->win[1].base, s->win[1].size);
    if (s->io[0].NumPorts > 0)
	release_region(s->io[0].BasePort, s->io[0].NumPorts);
    s->io[0].NumPorts = 0;
#ifdef CONFIG_ISA
    if ((c[0].dev.irq != 0) && (c[0].dev.irq != s->cap.pci_irq))
	undo_irq(IRQ_TYPE_EXCLUSIVE, c[0].dev.irq);
#endif
    kfree(s->cb_config);
    s->cb_config = NULL;
#if (LINUX_VERSION_CODE >= VERSION(2,1,90))
    kfree(s->pci_bus);
    s->pci_bus = NULL;
#endif
}
