#define _GNU_SOURCE
#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <elf.h>
#include <getopt.h>
#include <assert.h>
#include <sys/stat.h>
#include <errno.h>

#include "elfload.h"

static int verbose = 1;
static int force = 0;
static int preserve = 0;
const char *curfile;

/* Replace any DT_RELENT .dynamic entry with DT_RELAENT.  */
static int
fixrelent(int fd, Elf32_Off dyn_offset)
{
  int i;

  if (lseek(fd, dyn_offset, SEEK_SET) == -1)
    {
      fprintf (stderr, "%s: positioning for dynamic: %m\n", curfile);
      return -1;
    }
  for (i = 0; ; i++)
    {
      Elf32_Dyn dent;
      if (read(fd, &dent, sizeof(dent)) != sizeof(dent))
	{
	  fprintf (stderr, "%s: reading .dynamic entry %d: %m\n",
		   curfile, i);
	  return -1;
	}
      switch (dent.d_tag)
	{
	case DT_NULL:
	  fprintf(stderr, "%s: No DT_REL[A]ENT??\n", curfile);
	  return -1;
	case DT_RELAENT:
	  if (verbose > 1)
	    printf("%s: Already has DT_RELAENT.\n", curfile);
	  return 0;
	case DT_RELENT:
	  if (lseek(fd, dyn_offset + i*sizeof(Elf32_Dyn), SEEK_SET) == -1)
	    {
	      fprintf (stderr, "%s: positioning for write: %m\n",
		       curfile);
	      return -1;
	    }
	  dent.d_tag = DT_RELAENT;
	  if (write(fd, &dent, sizeof(dent)) != sizeof(dent))
	    {
	      fprintf (stderr, "%s: writing DT_RELAENT: %m\n",
		       curfile);
	      return -1;
	    }
	  if (verbose)
	    printf("%s: DT_RELENT fixed.\n", curfile);
	  return 0;
	default:
	  /* continue */
	}
    }
}

static int
startcmp(const Elf32_Word *broken_start, const Elf32_Word *start_data,
	 int length)
{
  int i;

  for (i = 0; i < length; i++)
    if (((broken_start[i] & 0xffff) != 0x5555
	 || (broken_start[i] & ~0xffff) != (start_data[i] & ~0xffff))
	&& broken_start[i] != start_data[i])
      return 0;
  return 1;
}

/* Returns -1 on error, 0 otherwise.  */
static int
fixstart(int fd, const Elf32_Ehdr *ehdr)
{
  enum { start_length = 0xb8 };
  static const Elf32_Word broken_start[] = {
    0x3da05555, 0x61ad5555, 0x2c1f7654, 0x41820028,
    0x7c3d0b78, 0x3821ffc0, 0x807d0000, 0x389d0004,
    0x7c852378, 0x84050004, 0x2c000000, 0x4082fff8,
    0x38a50004, 0x3fe05555, 0x90bf5555, 0x7c771b78,
    0x7c982378, 0x7cb92b78, 0x2c1f7654, 0x41820008,
    0x7fa3eb78
  };
  static const Elf32_Word new_start[] = {
    0x3821FFF0, /*        addi %r1,%r1,-16 */
    0x38000000, /*        li   %r0,0 */
    0x90010000, /*        stw  %r0,0(1) */
    0x7C0803A6, /*        mtlr %r0 */
    0x7C771B78, /*        mr   %r23,%r3 */
    0x7C982378, /*        mr   %r24,%r4 */
    0x7CE33B79, /*        mr.  %r3,%r7 */
    0x7CB92B78, /*        mr   %r25,%r5 */
    0x41820008, /*        beq  0f */
    0x48000001, /*        bl   atexit */
    0x7EE3BB78, /* 0:     mr   %r3,%r23 */
    0x7F04C378, /*        mr   %r4,%r24 */
    0x7F25CB78, /*        mr   %r5,%r25 */
    0x48000018, /*        b    1f */
    0x60000000, /*        nop */
    0x60000000, /*        nop */
    0x60000000, /*        nop */
    0x60000000, /*        nop */
    0x60000000  /*        nop */
                /* 1:     */
  };

  Elf32_Word *startp;
  Elf32_Off start_offset;
  int i;

  if (ehdr->e_entry == 0)
    {
      if (verbose)
	printf ("%s: No entry point!\n", curfile);
      return 0;
    }

  startp = (Elf32_Word *)find_location(ehdr->e_entry, 0);
  
  if (!startp)
    {
      fprintf(stderr,"%s: No section containing entry point (0x%08x).\n",
	      curfile, ehdr->e_entry);
      return -1;
    }

  /* Check to see that it's the start routine we expect.  */
  if (find_location(ehdr->e_entry, 26*4)
      && startcmp(broken_start, startp,
		  sizeof(broken_start)/sizeof(broken_start[0])))
    startp += 2;
  else if (find_location(ehdr->e_entry, 24*4)
	   && startcmp(broken_start+2, startp,
		       sizeof(broken_start)/sizeof(broken_start[0])-2))
    {
      if (verbose > 1)
	printf("%s: Warning: Start routine does not set r13.\n",
	       curfile);
    }
  else
    {
      if (verbose > 1)
	printf("%s: Start routine not known to fail, not changed.\n",
	       curfile);
      return 0;
    }

  /* Fix it.  */
  for (i = 0; (size_t)i < sizeof(new_start)/sizeof(new_start[0]); i++)
    startp[i] = new_start[i];
  /* Get bl instruction for atexit.  */
  startp[9] |= 0x03fffffc   &   startp[23] + 56;
  
  changed_segment();

  if (verbose)
    printf("%s: Start routine fixed.\n", curfile);

  return 0;
}


static const char *const fixsyms[][2] = {
  { "sigisemptyset","s!,isemptyset" },
  { "sigemptyset",  "s!,emptyset" },
  { "sigfillset",   "s!,fillset" },
  { "sigprocmask",  "s!,procmask" },
  { "sigpending",   "s!,pending" },
  { "sigaction",    "s!,action" },
  { "__sigsetjmp",  "__s!,setjmp" },
  { "__sigjmp_save","__s!,jmp_save" },
  { "tcgetattr",    "!,getattr" },
  { "tcsetattr",    "!,setattr" },
  { "__fxstat",     "__fx,!at" },
  { "__xstat",      "__x,!at" },
  { "__lxstat",     "__lx,!at" },
  { "readdir",      "readd!," },
  { "readdir_r",    "readd!,_r" },
  { "scandir",      "scand!," },
  { "alphasort",    "alphaso,!" },
  { "getdirentries","getd!,entries" },
  { "__res_init",   "res_init" },
  { "__socket",     "socket" },
  { "__recv",       "recv" },
  { "_Xsetlocale",  "setlocale" }
};

/* Returns -1 on error, 0 otherwise.  */
static int
checksyms(int fd, int dyn_offset)
{
  enum { num_need = 32 };
  Elf32_Addr needed[num_need];
  int numneeded = 0;
  int i;
  Elf32_Addr sym_offset, string_offset, hash_offset;

  if (lseek(fd, dyn_offset, SEEK_SET) == -1)
    {
      fprintf (stderr, "%s: positioning for dynamic: %m\n", curfile);
      return -1;
    }
  sym_offset = string_offset = hash_offset = 0;
  for (;;)
    {
      Elf32_Dyn dent;
      if (read(fd, &dent, sizeof(dent)) != sizeof(dent))
	{
	  fprintf (stderr, "%s: reading .dynamic entry: %m\n",
		   curfile);
	  return -1;
	}
#ifndef DT_VERNEED
#define	DT_VERNEED	0x6ffffffe
#endif
      switch (dent.d_tag)
	{
	case DT_VERNEED:
	  if (verbose > 1)
	    printf("%s: Not looking for problem functions, has versioning.\n",
		   curfile);
	  return 0;
	case DT_SYMTAB:
	  sym_offset = dent.d_un.d_ptr;
	  break;
	case DT_STRTAB:
	  string_offset = dent.d_un.d_ptr;
	  break;
	case DT_HASH:
	  hash_offset = dent.d_un.d_ptr;
	  break;
	case DT_NEEDED:
	  if (numneeded == num_need)
	    {
	      fprintf(stderr, "%s: uses too many shared objects.\n",
		      curfile);
	      return -1;
	    }
	  needed[numneeded++] = dent.d_un.d_ptr;
	  break;
	case DT_NULL:
	  goto endloop;
	default:
	  break;
	}
    }
endloop:
  if (sym_offset == 0 || string_offset == 0 || hash_offset == 0)
    {
      fprintf(stderr, "%s: No strings or no symbols!\n", curfile);
      return -1;
    }

#define findlocn(addr) ({ \
      char *result = find_location(addr, 1); \
      if (result == 0) \
	{ \
	  fprintf (stderr, "%s: Unloaded address %08x!\n", curfile, addr); \
	  return -1; \
	} \
      result; \
    })

  {
    Elf32_Word nchain, ichain, nbucket, c;
    Elf32_Addr *bucket, *chain;
    Elf32_Sym *syms;
    int hit = 0;
    char *name, *namep;

    nchain = *(Elf32_Word *) findlocn(hash_offset + 4);
    syms = (Elf32_Sym *) findlocn(sym_offset);
    for (ichain = 0; ichain < nchain; ichain++)
      if (syms[ichain].st_name != 0)
	{
	  size_t k;
	  name = findlocn(string_offset + syms[ichain].st_name);
	  for (k = 0; k < sizeof(fixsyms)/sizeof(fixsyms[0]); k++)
	    if (strcmp(name, fixsyms[k][0]) == 0)
	      {
		if (!hit && verbose == 1)
		  printf("%s: Fixing functions.\n", curfile);
		else if (verbose > 1)
		  printf("%s: Changing function %s to %s.\n", curfile,
			 fixsyms[k][0], fixsyms[k][1]);
		assert(strlen(fixsyms[k][0]) >= strlen(fixsyms[k][1]));
		strcpy(name, fixsyms[k][1]);
		hit = 1;
	      }
	}
    if (!hit && verbose > 1)
      printf("%s: Found no functions that needed replacing.\n", curfile);
    if (!hit)
      return 0;

    changed_segment();

    /* Rebuild the hash table.  */
    nbucket = *(Elf32_Word *) findlocn(hash_offset);
    bucket = (Elf32_Addr *) findlocn(hash_offset + 8);
    chain = bucket + nbucket;
    memset(bucket, 0, sizeof(Elf32_Addr)*nbucket);
    memset(chain, 0, sizeof(Elf32_Addr)*nchain);
    
    for (c = 1; c < nchain; c++)
      if (syms[c].st_name != 0)
	{
	  unsigned long h=0, g, j;
	  
	  name = findlocn(string_offset + syms[c].st_name);
	  for (j = 0; name[j]; j++)
	    {
	      h = (h << 4) + name[j];
	      if (g = h & 0xf0000000)
		h ^= g >> 24;
	      h &= ~g;
	    }
	  h %= nbucket;
	  if (bucket[h] == 0)
	    bucket[h] = c;
	  else
	    {
	      for (h = bucket[h]; chain[h] != 0; h = chain[h])
		;
	      chain[h] = c;
	    }
	}
  }
  {
    int libcct = -1;
    int ldsoct = -1;
    static const char libname[] = "fix!.so";

    for (i = 0; i < numneeded; i++)
      {
	char *name;
	name = findlocn(string_offset + needed[i]);
	if (strcmp(name, "libc.so.6") == 0)
	  libcct = i;
	else if (strcmp (name, "ld.so.1") == 0)
	  {
	    if (verbose > 1)
	      printf("%s: Replacing %s with %s\n", curfile, name, libname);
	    strcpy(name, libname);
	    ldsoct = i;
	  }
      }
    if (libcct == -1)
      {
	fprintf (stderr, "%s: Does not use libc.so.6!\n", curfile);
	return -1;
      }
    if (ldsoct == -1)
      {
	char *libcname;
	libcname = findlocn(string_offset + needed[libcct]);
	if (verbose > 1)
	  printf("%s: Replacing %s with %s\n", curfile, libcname, libname);
	strcpy(libcname, libname);
      }
  }

  return 0;
}

/* Returns -1 on error, 0 otherwise.  */
static int
fixglobdat(int fd, int dyn_offset)
{
  Elf32_Addr rela_offset;
  Elf32_Word numrels, i, hit = 0;
  Elf32_Rela *rels;

  if (lseek(fd, dyn_offset, SEEK_SET) == -1)
    {
      fprintf (stderr, "%s: positioning for dynamic: %m\n", curfile);
      return -1;
    }
  rela_offset = numrels = 0;
  for (;;)
    {
      Elf32_Dyn dent;
      if (read(fd, &dent, sizeof(dent)) != sizeof(dent))
	{
	  fprintf (stderr, "%s: reading .dynamic entry: %m\n",
		   curfile);
	  return -1;
	}
      switch (dent.d_tag)
	{
	case DT_RELA:
	  rela_offset = dent.d_un.d_ptr;
	  break;
	case DT_RELASZ:
	  numrels = dent.d_un.d_val / sizeof(Elf32_Rela);
	  break;
	case DT_RELAENT:
	  if (dent.d_un.d_val != sizeof(Elf32_Rela))
	    {
	      fprintf (stderr, "%s: RELA entry size %d, expected %d!\n",
		       curfile, dent.d_un.d_val, sizeof(Elf32_Rela));
	      return -1;
	    }
	  break;
	case DT_NULL:
	  goto endloop;
	default:
	  break;
	}
    }
endloop:
  if (rela_offset == 0 || numrels == 0)
    {
      if (verbose > 1)
	printf("%s: No relocations.\n", curfile);
      return 0;
    }

  rels = (Elf32_Rela *) find_location(rela_offset, sizeof(Elf32_Rela)*numrels);
  if (rels == NULL)
    {
      fprintf (stderr, "%s: Relocations (0x%08x-0x%08x) not loaded.\n",
	       curfile, rela_offset, rela_offset+sizeof(Elf32_Rela)*numrels);
      return -1;
    }
  
  for (i = 0; i < numrels; i++)
    if (ELF32_R_SYM(rels[i].r_info) == SHN_UNDEF
	&& ELF32_R_TYPE(rels[i].r_info) == R_PPC_GLOB_DAT)
      {
	hit++;
	rels[i].r_info = ELF32_R_INFO(SHN_UNDEF, R_PPC_RELATIVE);
      }

  if (hit)
    {
      if (verbose)
	printf("%s: %d R_PPC_GLOB_DAT relocs changed to R_PPC_RELATIVE.\n",
	       curfile, hit);
      changed_segment();
    }
  else if (verbose > 1)
    printf("%s: No broken R_PPC_GLOB_DAT relocs found.\n", curfile);
  
  return 0;
}

static void
fixit(const char *whichfile)
{
  Elf32_Ehdr ehdr;
  Elf32_Off dyn_offset;
  int fd;
  struct stat oldperms;
  int permschanged = 0;

  curfile = whichfile;
  if (preserve || force)
    if (stat(curfile, &oldperms) != 0)
      {
	fprintf (stderr, "%s: couldn't stat: %m\n", curfile);
	return;
      }
    
  fd = open(curfile, O_RDWR);
  if (force && fd == -1 && errno == EACCES)
    if (chmod(curfile, oldperms.st_mode | S_IRUSR | S_IWUSR) == -1)
      {
	fprintf (stderr, "%s: couldn't open and couldn't change modes: %m\n",
		 curfile);
	return;
      }
      else
      {
	fd = open(curfile, O_RDWR);
	permschanged = 1;
      }
  if (fd == -1)
    {
      fprintf (stderr, "%s: open failed: %m\n", curfile);
      if (permschanged)
	chmod(curfile, oldperms.st_mode);
      return;
    }
  
  { int ehdr_result = read_check_ehdr(fd, &ehdr);
    if (ehdr_result == -2 && verbose > 1)
      printf("%s: Not a PPC executable.\n", curfile);
    if (ehdr_result != 0)
      goto donefile;
  }

  { int read_seg_result = read_segment_list(fd, &ehdr, &dyn_offset);
    if (read_seg_result == 1 && verbose > 1)
      printf("%s: No dynamic section.\n", curfile);
    if (read_seg_result != 0)
      goto donefile;
  }

  if (fixrelent(fd, dyn_offset) != 0)
    goto donefile;
      
  if (load_segments(fd) != 0)
    goto donefile;

  if (fixstart(fd, &ehdr) == -1)
    goto donefile;

  if (checksyms(fd, dyn_offset) == -1)
    goto donefile;

  if (ehdr.e_type == ET_DYN
      && fixglobdat(fd, dyn_offset) == -1)
    goto donefile;

  write_segments(fd);

donefile:
  free_segments();
  if (permschanged || preserve)
    fchmod(fd, oldperms.st_mode);
  close(fd);
}

int
main(int argc, char **argv)
{
  int c;
  size_t i, ct;

  while (1)
    {
      static const struct option long_options[] =
      {
	/* These options set a flag. */
	{"verbose", 0, &verbose, 2},
	{"quiet", 0, &verbose, 0},
	{"force", 0, &force, 1},
	{"preserve", 0, &preserve, 1},
	{"help", 0, 0, 'h'},
	{"version", 0, 0, 'e'},
	{0, 0, 0, 0}
      };
      /* `getopt_long' stores the option index here. */
      int option_index = 0;
      
      c = getopt_long (argc, argv, "svpf",
		       long_options, &option_index);
     
      /* Detect the end of the options. */
      if (c == -1)
	break;
     
      switch (c)
	{
	case 0:
	  break;
     
	case 's':
	  verbose = 0;
	  break;
	case 'v':
	  verbose++;
	  break;
	case 'p':
	  preserve = 1;
	  break;
	case 'f':
	  force = 1;
	  break;
	case 'h':
	  printf (
"Usage: %s [OPTION] FILE ...\n"
"Fix FILE to make it work with libc 2.1 on powerpc.\n\n"
"  -s, --quiet          don't print output indicating changes were made\n"
"  -v, --verbose        if a file was not changed, say why\n"
"  -p, --preserve       if the file was originally setuid, leave it that way\n"
"  -f, --force          ignore absence of read permission\n"
"      --help           display this help and exit\n"
"      --version        output version information and exit\n\n"
"The changes made are that:\n"
"1. A DT_RELENT program header entry is changed to the correct DT_RELAENT.\n"
"2. The startup procedure is modified to conform to the PPC version of the\n"
"   SVR4 ELF ABI.\n"
"3. Routines which use data structures that changed size between the 971212\n"
"   libc and the final libc 2.0, or which use constants that are different\n"
"   between the two versions, are redirected to call a stub library that\n"
"   emulates the old library behaviour.  The affected routines are:\n"
"       ", argv[0]);
	  ct = 9;
	  for (i = 0; i < sizeof(fixsyms)/sizeof(fixsyms[0]); i++)
	    ct += printf("%s%s",
			 (ct + sizeof(fixsyms[i][0]) > 70
			  ? (ct=0), ",\n       "
			  : (i==0 ? "" : ", ")),
			 fixsyms[i][0]);
	  printf(".\n"
"4. R_PPC_GLOB_DAT relocations in shared objects that point to an absolute\n"
"   address are changed to be R_PPC_RELATIVE relocations.\n\n"
"If FILE is not a PowerPC ELF executable or shared object, or the changes\n"
"have already been made, or the changes don't need to be made, does nothing.\n"
);
	  exit(0);
	case 'e':
	  printf("Version 1.1\n");
	  exit(0);
	case '?':
	  /* `getopt_long' already printed an error message. */
	  fprintf(stderr, "Try `%s --help' for more information.\n", argv[0]);
	  exit(1);
	default:
	  abort ();
	}
    }
     
  while (optind < argc)
    fixit(argv[optind++]);
  return 0;
}
