/*
 * isomarkboot.c
 *
 * This file is part of aboot, the SRM bootloader for Linux/Alpha
 * Copyright (C) 1996 David Mosberger.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*
 * Making an ISO9660 filesystem bootable is straight-forward since all
 * files are contiguous.
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <sys/fcntl.h>

#include <config.h>
#include <isolib.h>

const char * prog_name;

static int disk;


void
__memcpy (void * dest, const void * src, size_t n)
{
	memcpy (dest, src, n);
}


long
iso_dev_read (void * buf, long offset, long size)
{
	if (lseek(disk, offset, SEEK_SET) != offset) {
		perror("lseek");
		return -1;
	}
	return read(disk, buf, size);
}



int
main (int argc, char ** argv)
{
	unsigned long sector[512 / sizeof(unsigned long)], sum;
	size_t nbytes, aboot_size;
	off_t aboot_pos;
	int i, aboot_fd;

	prog_name = argv[0];

	if (argc != 3) {
	    fprintf(stderr, "usage: %s filesys path\n", prog_name);
	    exit(1);
	}
	disk = open(argv[1], O_RDWR);
	if (disk < 0) {
		perror(argv[1]);
		exit(1);
	}

	if (iso_read_super (0, 0) < 0) {
		fprintf(stderr, "%s: cannot mount\n", argv[1]);
		exit(1);
	}

	aboot_fd = iso_open(argv[2]);
	if (aboot_fd < 0) {
		fprintf(stderr, "%s: file not found\n", argv[2]);
		exit(1);
	}

	aboot_size = iso_fsize(aboot_fd);
	aboot_pos = iso_map (aboot_fd, 0);

	printf("%s: %s is at offset %ld and is %lu bytes long\n",
	       prog_name, argv[2], aboot_pos, aboot_size);

	if (lseek(disk, 0, SEEK_SET) != 0) {
		perror("lseek");
		return -1;
	}

	nbytes = read(disk, sector, sizeof(sector));
	if (nbytes != sizeof(sector)) {
		if ((long) nbytes < 0) {
			perror("read");
		} else {
			fprintf(stderr, "%s: short read\n", prog_name);
		}
		exit(1);
	}

	strcpy((char *) sector, "Linux/Alpha aboot for ISO filesystem.");
	sector[60] = aboot_size / 512;		/* sector count */
	sector[61] = aboot_pos / 512;		/* starting LBM */
	sector[62] = 0;				/* flags */

	/* update checksum: */
	sum = 0;
	for (i = 0; i < 63; ++i)
		sum += sector[i];
	sector[63] = sum;

	if (lseek(disk, 0, SEEK_SET) != 0) {
		perror("lseek");
		return -1;
	}

	nbytes = write(disk, sector, sizeof(sector));
	if (nbytes != sizeof(sector)) {
		if ((long) nbytes < 0) {
			perror("write");
		} else {
			fprintf(stderr, "%s: short write\n", prog_name);
		}
		exit(1);
	}
	return 0;
}
