package BasicLyX;
# This package includes subroutines to translate "clean" LaTeX to LyX.
# It translates only "basic" stuff, which means it doesn't do class-specific
#     things. (It uses the TeX parser Text::TeX)
use strict;

######
#Next text starts a new paragraph?
# $INP = 0 for plain text not starting a new paragraph
# $INP = 1 for text starting a new paragraph, so that we write a new
#    \layout command and renew any font changes for the new paragraph
# Starts as 1 cuz if the first text in the document is plain text
#    (not, e.g., in a \title command) it will start a new paragraph
my $IsNewParagraph = 1;
my $OldINP; #Save $IsNewParagraph during footnotes

# $MBD = 1 if we're in a list, but haven't seen an '\item' command
#    In that case, we may need to start a nested "Standard" paragraph
my $MayBeDeeper = 0;
my $OldMBD; #Save $MBD during footnotes -- this should very rarely be necessary!

# Stack to take care of environments like Enumerate, Quote
# We need a separate stack for footnotes, which have separate layouts etc.
# Therefore we use a reference to an array, not just an array
my @LayoutStack = ("Standard"); #default if everything else pops off
my $CurrentLayoutStack = \@LayoutStack;

# Status of various font commands
# Every font characteristic (family, series, etc.) needs a stack, because
#    there may be nested font commands, like \textsf{blah \texttt{blah} blah}
# CurrentFontStatus usually points to the main %FontStatus hash, but
#     when we're in a footnote, it will point to a temporary hash
my %FontStatus = (
    '\emph' => ["default"],
    '\family' => ["default"],
    '\series' => ["default"],
    '\shape' => ["default"],
    '\underline' => ["default"],
    '\size' => ["default"],
);
my $CurrentFontStatus = \%FontStatus;

#############  INFORMATION ABOUT LATEX AND LYX   #############################
# LyX translations of LaTeX command tokens like \section{}
my %TokenTransTable = (
   # Font commands
   '\emph' => "\n\\emph on\n",
   '\underline' => "\n\\bar under\n",
   '\textbf' => "\n\\series bold\n",
   '\textmd' => "\n\\series medium\n",
   '\textsf' => "\n\\family sans\n",
   '\texttt' => "\n\\family typewriter\n",
   '\textrm' => "\n\\family roman\n",
   '\textsc' => "\n\\shape smallcaps\n",
   '\textsl' => "\n\\shape slanted\n",
   '\textit' => "\n\\shape italic\n",
   '\textup' => "\n\\shape up\n",
   # Font size commands
   '\tiny' => "\n\\size tiny\n",
   '\scriptsize' => "\n\\size scriptsize\n",
   '\footnotesize' => "\n\\size footnotesize\n",
   '\small' => "\n\\size small\n",
   '\normalsize' => "\n\\size default\n",
   '\large' => "\n\\size large\n",
   '\Large' => "\n\\size Large\n",
   '\LARGE' => "\n\\size LARGE\n",
   '\huge' => "\n\\size huge\n",
   '\Huge' => "\n\\size Huge\n",

   # Footnote, Margin note
   '\footnote' => "\n\\begin_float footnote\n",
   '\marginpar' => "\n\\begin_float margin\n",

    # LaTeX escaped characters
    '\_' => '_',
    '\%' => '%',
    '\$' => '$',
    '\&' => '&',
    '\{' => '{',
    '\}' => '}',
    '\#' => '#',

    # Misc simple LaTeX tokens
    '~' => "\n\\protected_separator \n",
    '@' => "@", # TeX.pm considers this a token, but it's not really
    '\@' => "\\SpecialChar \\@",
    '\ldots' => "\n\\SpecialChar \\ldots\{\}\n",
    '\-' => "\\SpecialChar \\-\n",
    # \\\\ necessary instead of \\ because of Perl string interpretations
    '\\\\' => "\n\\newline\n",
    '\LaTeX' => "LaTeX",
    '\TeX' => "TeX",
    '\LyX' => "LyX",
    '\tableofcontents' =>
             "\n\\begin_inset LatexCommand \\tableofcontents\n\n\\end_inset\n",
);
# Simple LaTeX tokens which are turned into small pieces of LyX text
# Since these tokens can appear in the midst of text, we may have to, e.g.,
#     start a new paragraph right before one of them.
my $TextTokens = '~|@|\\\\([\-%_$&{}#@\\\\]|ldots|tableofcontents|TeX|LaTeX|LyX)';

# LyX translations of some plain LaTeX text (TeX parser won't recognize it
#     as a Token, so we need to translate the Text::TeX::Text token.)
my %TextTransTable = (
    # Double quotes
    "``" => "\n\\begin_inset Quotes eld\n\\end_inset\n\n",
    "''" => "\n\\begin_inset Quotes erd\n\\end_inset\n",
);


# Environments in LaTeX \begin{environment} statements
# These are converted to LyX Layouts
my $MathEnvironments = "(math|equation|displaymath|eqnarray(\\*)?)";
my $NonMathEnvironments="(itemize|enumerate|description|quote|quotation|verse)";
my %EnvTransTable = (    # LyX translations of LaTeX environments
   'quote' => "Quote",
   'quotation' => "Quotation",
   'verse' => "Verse",

   # List commands
   'itemize' => "Itemize",
   'enumerate' => "Enumerate",
   'description' => "Description",
);
my $NonListLayouts = "Quote|Quotation|Verse";
my $ListLayouts = "Itemize|Enumerate|Description";

# Sectioning commands which turn into \layout commands
my %SecTransTable = ( # LyX layout translations of LaTeX sectioning commands
    '\section' => "Section",
    '\section*' => "Section*",
    '\subsection' => "Subsection",
    '\subsection*' => "Subsection*",
    '\subsubsection' => "Subsubsection",
    '\subsubsection*' => "Subsubsection*",
    '\paragraph' => "Paragraph",
    '\subparagraph' => "Subparagraph",
    '\title' => "Title",
    '\author' => "Author",
    '\date' => "Date",
);

# Types of LaTeX floats we can translate
my $FloatTypes = "(footnote|margin)";

# Types of LyX font commands
my $FontDescriptors = "(emph|family|series|shape|underline)";
my $FontSizeCommands = "\\\\(tiny|scriptsize|footnotesize|small|normalsize|large|Large|LARGE|huge|Huge)";

# Things that LyX translates as "LatexCommand"s
my $LatexCommands = "\\\\(ref|pageref|label|cite)";

#####################   PARSER INVOCATION   ##################################
sub call_parser {
# This subroutine calls the TeX parser & translator

    # Hash of tokens passed to the TeX parser
    # Many values are $Text::TeX::Tokens{'\operatorname'}, which has
    #    Type=>report_args and count=>1
    # Note that we don't have to bother putting in tokens which will be simply
    #    translated (e.g., from $TextTokens).
    my %MyTokens = ( 
	'{' => $Text::TeX::Tokens{'{'},
	'}' => $Text::TeX::Tokens{'}'},
	'\begin' => $Text::TeX::Tokens{'\begin'},
	'\end' => $Text::TeX::Tokens{'\end'},

	# Sectioning commands
	'\chapter' => $Text::TeX::Tokens{'\operatorname'},
	'\section' => $Text::TeX::Tokens{'\operatorname'},
	'\subsection' => $Text::TeX::Tokens{'\operatorname'},
	'\subsubsection' => $Text::TeX::Tokens{'\operatorname'},
	'\paragraph' => $Text::TeX::Tokens{'\operatorname'},
	'\subparagraph' => $Text::TeX::Tokens{'\operatorname'},
	'\chapter*' => $Text::TeX::Tokens{'\operatorname'},
	'\section*' => $Text::TeX::Tokens{'\operatorname'},
	'\subsection*' => $Text::TeX::Tokens{'\operatorname'},
	'\subsubsection*' => $Text::TeX::Tokens{'\operatorname'},
	'\paragraph*' => $Text::TeX::Tokens{'\operatorname'},
	'\subparagraph*' => $Text::TeX::Tokens{'\operatorname'},

	# Title commands
	'\title' => $Text::TeX::Tokens{'\operatorname'},
	'\author' => $Text::TeX::Tokens{'\operatorname'},
	'\date' => $Text::TeX::Tokens{'\operatorname'},

	# Font commands (non-local)
	'\emph' => $Text::TeX::Tokens{'\operatorname'},
	'\underline' => $Text::TeX::Tokens{'\operatorname'},
	'\textbf' => $Text::TeX::Tokens{'\operatorname'},
	'\textmd' => $Text::TeX::Tokens{'\operatorname'},
	'\textrm' => $Text::TeX::Tokens{'\operatorname'},
	'\textsf' => $Text::TeX::Tokens{'\operatorname'},
	'\texttt' => $Text::TeX::Tokens{'\operatorname'},
	'\textup' => $Text::TeX::Tokens{'\operatorname'},
	'\textit' => $Text::TeX::Tokens{'\operatorname'},
	'\textsl' => $Text::TeX::Tokens{'\operatorname'},
	'\textsc' => $Text::TeX::Tokens{'\operatorname'},

	# Font sizing commands (local)
	'\tiny' => {Type => 'local'},
	'\small' => {Type => 'local'},
	'\scriptsize' => {Type => 'local'},
	'\footnotesize' => {Type => 'local'},
	'\small' => {Type => 'local'},
	'\normalsize' => {Type => 'local'},
	'\large' => {Type => 'local'},
	'\Large' => {Type => 'local'},
	'\LARGE' => {Type => 'local'},
	'\huge' => {Type => 'local'},
	'\Huge' => {Type => 'local'},

	# List commands
	'\item' => {},

	# Footnote and margin note
	'\footnote' => $Text::TeX::Tokens{'\operatorname'},
	'\marginpar' => $Text::TeX::Tokens{'\operatorname'},

	# Cross references and such
	'\ref' => $Text::TeX::Tokens{'\operatorname'},
	'\pageref' => $Text::TeX::Tokens{'\operatorname'},
	'\label' => $Text::TeX::Tokens{'\operatorname'},
	'\cite' => $Text::TeX::Tokens{'\operatorname'},

	# Tokens to ignore (which make a new paragraph)
	# Just pretend they actually ARE new paragraph markers!
	'\maketitle' => {'class' => 'Text::TeX::Paragraph'},
    );

    my ($InFileName, $OutFileName) = (shift,shift);
    
    # Output LyX file
    warn "Basic LaTeX to LyX translation... \n";
    open (OUTFILE,">$OutFileName");

    # Open the file to turn into LyX.
    my $infile = new Text::TeX::OpenFile $InFileName,
	'defaultact' => \&basic_lyx,
	'tokens' => \%MyTokens;

    # Process what's left of the file (everything from \begin{document})
    $infile->process;

    # Last line of the LyX file
    print OUTFILE "\n\\the_end\n";
    close OUTFILE;
    warn "Done with basic translation\n";
    return;
} # end subroutine call_parser


##########################   MAIN TRANSLATOR SUBROUTINE   #####################
sub basic_lyx {
# This subroutine is called by Text::TeX::process each time subroutine
#     eat returns a value.
# Argument 0 is the return value from subroutine eat
# Argument 1 is the Text::TeX::OpenFile (namely, $TeXfile)
    my $eaten = shift;
    my $fileobject = shift;

    # This handles most but maybe not all comments
    # THere shouldn't be any if we've run CleanTeX.pl
    if (defined $eaten->[1]) {print "Comment: $eaten->[1]"};

    my $type = ref($eaten);
    print "$type ";

    # This loop is basically just a switch. However, making it a for
    #    (1) makes $_ = $type (convenient)
    #    (2) allows us to use things like next and last
    TYPESW: for ($type) {

        # some pre-case work
        s/^Text::TeX:://o or die "unknown token?!";
	my ($outstr, $env, $layout, $command, $dummy, $tok, @grouparr);
        
        # Handle blank lines.
        if (m/^Paragraph$/o) {
	    # $INP <>0 means We will need a new layout command
	    $IsNewParagraph = 1;

	    # $MBD means start a begin_deeper within list environments
	    #    unless there's an \item command
	    $MayBeDeeper = 1;

            last TYPESW;
        }

	# This is usually the text that makes up (a part of) the token
        $outstr = $eaten->[0];

	# If, e.g., there's just a comment in this token, don't do anything
	# This actually shouldn't happen if CleanTeX has already removed them
	last TYPESW if !defined $outstr;
        
        # Handle LaTeX tokens
        if (/^Token$/o) {

	    print "$outstr ";

	    # Tokens which turn into a bit of LyX text
	    if ($outstr =~ /^$TextTokens$/o) {
	        &CheckForNewParagraph; #Start new paragraph if necessary

		# Print the translated text
		print OUTFILE "$TokenTransTable{$outstr}";

	    # Math -- copy verbatim until you're done
	    } elsif ($outstr eq '\(' || $outstr eq '\[') {
	        &copy_math($eaten, $fileobject);
	    } elsif ($outstr eq '\)' || $outstr eq '\]') {
	        # end math
		print OUTFILE "$outstr\n\\end_inset\n";
		print "\nDone copying math ending with '$outstr'";

	    # Items in list environments
	    } elsif ($outstr eq '\item') {
		
		# What if we had a nested "Standard" paragraph?
		# Then write \end_deeper to finish the standard layout
		#     before we write the new \layout ListEnv command
		if ($$CurrentLayoutStack[-1] eq "Standard") {
		    pop (@$CurrentLayoutStack); # take "Standard" off the stack
		    print OUTFILE "\n\\end_deeper\n";
		    print "\nCurrent Layout Stack: @$CurrentLayoutStack";
		} # end deeper if

		# Upcoming text (the item) will be a new paragraph, 
		#    requiring a new layout command based on whichever
		#    kind of list environment we're in
		$IsNewParagraph = 1;

		# But since we had an \item command, DON'T nest a
		#    deeper "Standard" paragraph in the list
		$MayBeDeeper = 0;

	    # Font sizing commands
	    } elsif ($outstr =~ /^$FontSizeCommands$/) {
		$command = $TokenTransTable{$outstr}; #e.g., '\size large'

		if (! $IsNewParagraph) {
		    print OUTFILE "$command";
		} #otherwise, wait until we've printed the new paragraph command

		# Store the current font change
		($dummy = $command) =~ s/\s*(\S*)\s+(\w+)\s*/$1/;
		die "Font command error" if !exists $$CurrentFontStatus{$dummy};
		push (@{$CurrentFontStatus->{$dummy}}, $2);
		print "\nCurrent $dummy Stack: @{$CurrentFontStatus->{$dummy}}";

	    # Otherwise it's an unknown token, which must be copied
	    #     in TeX mode, along with its arguments, if any
	    } else {
		&copy_latex_token($outstr, $fileobject);
	    }

            last TYPESW;
        }
        
	# Handle tokens that take arguments, like \section{},\section*{}
	if (/^BegArgsToken$/) {
	    $env = $eaten->[0]->[0]->[0];
	    print "$env";

	    # Handle things that LyX translates as a "LatexCommand" inset
	    # \ref{blah} -> '\begin_inset LatexCommand \ref{blah}\n\n\end_inset'
	    if ($env =~ /^$LatexCommands$/o) {
		&CheckForNewParagraph; #Start new paragraph if necessary

	        print OUTFILE "\n\\begin_inset LatexCommand $env\{";
	        last TYPESW; # skip to the end of the switch
	    }

	    # Sectioning and Title environments (using a LyX \layout command)
	    if (exists $SecTransTable{$env}) {
		$layout = $SecTransTable{$env};
		push @$CurrentLayoutStack, $layout;
		print "\nCurrent Layout Stack: @$CurrentLayoutStack";

		# Check for optional argument to, e.g., \section
		$tok = $fileobject->eatOptionalArgument;
		if (defined ($dummy = $tok->[0])) {
		    print "\nIgnoring TOC entry '$dummy'";
		}

		# Upcoming text is a new paragraph, so that we type
		#    the \layout whatever command.
		# However, section stuff CANNOT be nested
		$IsNewParagraph = 1;
		$MayBeDeeper = 0;

		last TYPESW; #done translating
	    }

	    # Otherwise, we know what the token means
	    $command = $TokenTransTable{$env};

	    # Font characteristics
	    if ($command =~ /$FontDescriptors (\w+)/o) {
		# If we're about to start a new paragraph, save writing
		#    this command until *after* we write '\layout Standard'
		if (! $IsNewParagraph) {
		    print OUTFILE "$command";
		}

		# Store the current font change
		$dummy = "\\" . $1;
		die "Font command error" if !exists $$CurrentFontStatus{$dummy};
		push (@{$CurrentFontStatus->{$dummy}}, $2);


	    # Handle footnotes and margin notes
	    # Make a new font table & layout stack which will be local to the 
	    #    footnote or marginpar
	    } elsif ($command =~ /$FloatTypes/o) {

		# Open the footnote
		print OUTFILE "$command";

		# Make $CurrentFontStatus point to a new (anonymous) font table
	        $CurrentFontStatus =  {
		    '\emph' => ["default"],
		    '\family' => ["default"],
		    '\series' => ["default"],
		    '\shape' => ["default"],
		    '\underline' => ["default"],
		};

		# And make $CurrentLayoutStack point to a new (anon.) stack
		$CurrentLayoutStack = ["Standard"];

		# Store whether we're at the end of a paragraph or not
		#    for when we get to end of footnote AND 
		# Note that the footnote text will be starting a new paragraph
		$OldINP = $IsNewParagraph; $OldMBD = $MayBeDeeper;
		$IsNewParagraph = 1;
		$MayBeDeeper = 0; #can't be deeper at beginning of footnote

	    } # end of if on $command

	    # Exit the switch
	    last TYPESW;
	}

	if (/^EndArgsToken$/) {
	    my $env = $eaten->[0]->[0]->[0];
	    print "$env";

	    # Handle things that LyX translates as a "LatexCommand" inset
	    # \ref{blah} -> '\begin_inset LatexCommand \ref{blah}\n\n\end_inset'
	    if ($env =~ /^$LatexCommands$/o) {
	        print OUTFILE "\}\n\n\\end_inset\n";
	        last TYPESW; # skip to the end of the switch
	    }

	    if (exists $SecTransTable{$env}) {
		$layout = $SecTransTable{$env};
		my $mylayout = pop (@$CurrentLayoutStack);
		if ($mylayout ne $layout) { die "Problem with Layout Stack!\n"};
		print "\nCurrent Layout Stack: @$CurrentLayoutStack";

		# Text after this would be a new paragraph
		$IsNewParagraph = 1;
		$MayBeDeeper = 0; #No nesting near section commands
		last TYPESW; # done translating

	    }

	    $command = $TokenTransTable{$env};

	    # Font characteristics
	    # Pop the current FontStatus stack for a given characteristic
	    #    and give the new command (e.g., \emph default)
	    if ($command =~ /\\$FontDescriptors/) {
		$dummy = $&;
		pop @{$CurrentFontStatus->{$dummy}};
		$command = "\n$dummy $CurrentFontStatus->{$dummy}->[-1]\n";
		print OUTFILE "$command";

	    # Footnotes and marginpars
	    } elsif ($command =~ /$FloatTypes/o) {
	        print OUTFILE "\n\\end_float\n";

		# Reset the layout stack and font status table pointers to
		#    point to the global stack/table again, instead of the
		#    footnote-specific stack/table
		$CurrentFontStatus = \%FontStatus;
		$CurrentLayoutStack = \@LayoutStack;

		# We need to reissue any font commands (but not layouts)
		foreach $dummy (keys %$CurrentFontStatus) {
		    if ($CurrentFontStatus->{$dummy}->[-1] ne "default") {
			print OUTFILE $TokenTransTable{$dummy};
		    }
		}

		# Same paragraph status as we had before the footnote started
		$IsNewParagraph = $OldINP; $MayBeDeeper = $OldMBD;

	    } # End if on $command

	    # Exit main switch
	    last TYPESW;
	} # end if EndArgsToken

	# Handle END of scope of local commands like \large
	if (/^EndLocal$/) {
	    $tok = $outstr; # in this case, outstr is the token, not the string
	    $dummy = "$tok->[0]"; #local cmd which started this, e.g., '\large'
	    print $dummy;
	    $command = $TokenTransTable{$dummy};
	    ($dummy = $command) =~ s/\s*(\S*)\s+(\w+)\s*/$1/; #e.g., '\size'
	    die "Font command error" if !exists $$CurrentFontStatus{$dummy};
	    # TT::Text->check_presynthetic returns local commands FIFO, not LIFO
	    # pop font stack, but warn if we pop the wrong thing
	    warn " font confusion?" if
	               pop @{$CurrentFontStatus->{$dummy}} ne $2;
	    print "\nCurrent $dummy Stack: @{$CurrentFontStatus->{$dummy}}";
	    my $newfont = $CurrentFontStatus->{$dummy}->[-1];
	    $command = "\n$dummy $newfont\n";
	    print OUTFILE "$command";
	    last TYPESW;
	}

	# Handle '{'
	if (/^Begin::Group$/) {
	    #print "$outstr";
	    last TYPESW;
	}

	# Handle '{'
	if (/^End::Group$/) {
	    #print "$outstr";
	    last TYPESW;
	}

        # Handle \begin{foo}
        if (/^Begin::Group::Args$/) {
	    print "$outstr ";
	    @grouparr = @{ $eaten->[3]};
	    $env = $grouparr[0]->[0];
	    print "$env";
	    
	    # Environments, including list environments + verse, quote
	    if ($env =~ /$NonMathEnvironments/o) {
		$layout = $EnvTransTable{$env};

		# Nest if the layout stack has more than just "Standard" in it
		if ($#{$CurrentLayoutStack} > 0) {
		    print " Nesting!";
		    print OUTFILE "\n\\begin_deeper\n";
		}

		# Put the new layout onto the layout stack
		push @$CurrentLayoutStack, $layout;
		print "\nCurrent Layout Stack: @$CurrentLayoutStack";

		# Upcoming text will be new paragraph, needing a new layout cmd
		$IsNewParagraph = 1;
		# Test for nested "Standard" paragraph in upcoming text
		$MayBeDeeper = 1;

	    # Math environments
	    } elsif ($env =~ /^$MathEnvironments$/o) {
	        &copy_math($eaten, $fileobject);

	    # \begin document
	    } elsif ($env eq "document") {
	        print "\nStarting to translate actual document";

	    # otherwise, it's an unknown environment
	    # Don't bother putting stuff into CurrentLayoutStack etc.
	    #     but do nest if necessary
	    } else {
	        &copy_latex($env, $fileobject);
	    }

            last TYPESW;
        }
        
        # Handle \end{foo}
        if (/^End::Group::Args$/) {
	    print "$outstr ";
	    @grouparr = @{ $eaten->[3]};
	    $env = $grouparr[0]->[0];
	    print "$env";

	    # End of list or quote/verse environment
	    if ($env =~ /$NonMathEnvironments/o){
		$layout = $EnvTransTable{$env};
		my $mylayout = pop (@$CurrentLayoutStack);

		# If a standard paragraph was the last thing in a list, then
		#     we need to end_deeper and then pop the actual list layout
		if ($mylayout eq "Standard") {
		    print OUTFILE "\n\\end_deeper\n";
		    print " End Standard Nesting!";
		    $mylayout = pop (@$CurrentLayoutStack);
		}

		# The layout we popped off the stack had better match the
		#    environment we're ending!
		if ($mylayout ne $layout) { die "Problem with Layout Stack!\n"};
		print "\nCurrent Layout Stack: @$CurrentLayoutStack";

		# If we're finishing a nested layout, we need to end_deeper
		# Note that in we're nested in a list environment and the
		#     NEXT paragraph is Standard, then we'll have an extra
		#     \end_deeper\n\begin_deeper in the LyX file. It's sloppy
		#     but it works, and LyX will get rid of it when it
		#     resaves the file.
		if ($#{$CurrentLayoutStack} > 0) {
		    print " End Nesting!";
		    print OUTFILE "\n\\end_deeper\n";
		}

		# Upcoming text will be new paragraph, needing a new layout cmd
		$IsNewParagraph = 1;
		# Test for nested "Standard" paragraph in upcoming text
		$MayBeDeeper = 1;

	    # End of math environments
	    } elsif ($env =~ /^$MathEnvironments$/o) {
		print OUTFILE "\\end{$env}\n\\end_inset\n";
		print "\nDone copying math environment '$env'\n";

	    } elsif ($env eq "document") {
	        print "\nDone with document!";

	    # End of unknown environments. We're already in TeX mode
	    } else {
		print OUTFILE "\\backslash\nend{$env}\n";
		# End nesting if necessary
		if ($#{$CurrentLayoutStack} > 0) {
		    print " End Nesting!";
		    print OUTFILE "\n\\end_deeper\n";
		}

		# Upcoming text will be a new paragraph (e.g., Standard)
		$IsNewParagraph = 1;
	    }

            last TYPESW;

        }

	if (/^Text$/) {
	    if ($outstr !~ /^\s+$/) { #don't bother printing out whitespace

		# Only write '\layout Standard' once per paragraph
		&CheckForNewParagraph;

		# Print the LaTeX text to STDOUT
		print $outstr;

		# LyX runs together text separated by just \n. Add a space
		$outstr =~ s/\n/ \n/g;

		# Translate any LaTeX text that requires special LyX handling
		foreach $dummy (keys %TextTransTable) {
		    $outstr =~ s/$dummy/$TextTransTable{$dummy}/g;
		}

		# Actually print the text
		print OUTFILE "$outstr";
	    }
	    last TYPESW;
	}

	# Don't need to handle LookAhead or BegArgsTokenLookedAhead
	#    because only _ and ^ cause that, and they're copied by
	#    copy_math, not make_lyx

	# The default action - print the string.
	print "I don't know $outstr";

    } # end for ($type)

    print "\n";

} #end sub basic_lyx


#######################  VERBATIM COPYING SUBROUTINES  ########################
# Note: one or more of these subroutines may need to be moved elsewhere, if
#    they'll also be used by class-specific translating routines

sub copy_verbatim {
# This subroutine eats and prints text verbatim until a certain text is reached
# The end text itself is not eaten or printed; this is necessary so that
#    environments are properly nested (otherwise, TeX.pm complains)
#
# Arg 0 is the Text::TeX::OpenFile file object, arg 1 is the ending text
# If Arg2 exists, we're in TeX mode and need to translate \ and \n
#
# Because the searches will generally be matching expressions with backslashes
#    and other meta-characters, we put \Q\E around the expressions
#
    my $fileobject = shift;
    my $end_text = shift;
    my $tex_mode = shift; # UNDEFINED if there's no argument

    my $textref; # reference to stuff we read in to print
    my $to_print; #text to print
    my $dummy = "";
    $dummy = "in TeX mode " if defined $tex_mode;
    print "\nsub copy_verbatim going to copy $dummy until '$end_text'\n";

    # (Eat and) Print out paragraphs until you find $end_text
    # paragraph returns "" if it's time to get a new paragraph -- if that
    #    happens, we want to continue, but we can't dereference $textref
    while (defined ($textref = $fileobject->paragraph)) {
	next unless $textref; # new paragraph; keep going
	last if $$textref =~ /\Q$end_text\E/; #stop looping; we found it!

	$to_print = $$textref;
	print $to_print;
	if (defined $tex_mode) {
	    $to_print =~ s/\\/\\backslash /g;
	    $to_print =~ s/\n/\\newline\n/g;
	}
	print OUTFILE $to_print; # print the whole paragraph
	$$textref = ""; # Eat the paragraph (erase $fileobject->{"paragraph"})
    } #end while
    die "eof without finding matching text $end_text" if (!defined $textref);

    # $textref points to the paragraph which has $end_text in it.
    # Eat and print out only the part of the paragraph up until $end_text
    # Use 's' option to match since paragraph may contain '\n's
    $$textref =~ s/^(.*?)\Q$end_text\E/$end_text/s or die "couldn't find match";
    $to_print = $1;
    print $to_print;
    if (defined $tex_mode) {
	$to_print =~ s/\\/\\backslash /g;
	$to_print =~ s/\n/\\newline\n/g;
    }
    print OUTFILE $to_print;
    print "\nExiting sub copy_verbatim";
} # end copy_verbatim

# This subroutine copies math verbatim
# It figures out what token or command will end the math, then calls 
#    copy_verbatim to copy
# It also prints the lyx commands to start a math formula, and it prints
#    the beginning and end text
# Arg 0 is the beginning token/command, arg 1 is the fileobject
#
# \( ends with \) and \begin{equation} ends with \end{equation}
#
sub copy_math {
    my %endtokentbl = (  '\(' => '\)' , '\[' => '\]'  );
    my $begin_token = shift;
    my $fileobject = shift;

    my $type = ref($begin_token);
    $type =~ s/^Text::TeX:://o or die "unknown token?!";

    my ($begin_text, $end_text);
    $begin_text = $begin_token->[0]; # '\(' or the token '\begin'

    if ($type =~ /^Token$/) {  # \( or \[
        die "unknown begin_text" unless exists $endtokentbl{$begin_text};
	$end_text = $endtokentbl{$begin_text};

    } elsif (/^Begin::Group::Args$/) {    # \begin{mathenvironment}
	my $env = ${$begin_token->[3]}[0]->[0];
        $begin_text = "$begin_text" . "{$env}";
        ($end_text = $begin_text) =~ s/begin/end/
	               or die "missing begin in $begin_text";

    } else {
        die "copy_math called with unknown token type $type!";
    }

    print "\nCopying math beginning with '$begin_text'\n";
    print OUTFILE "\n\\begin_inset Formula $begin_text ";
    &copy_verbatim($fileobject, $end_text);
} #end sub copy_math

# This subroutine copies a latex environment verbatim in a lyx 'Latex' layout
# It nests if necessary
sub copy_latex {
    my $env = shift; #the environment we're starting
    my $fileobject = shift;

    # Nest if the layout stack has more than just "Standard" in it
    if ($#{$CurrentLayoutStack} > 0) {
	print " Nesting!";
	print OUTFILE "\n\\begin_deeper";
    }
    print OUTFILE "\n\\layout LaTeX\n";
    print OUTFILE "\n\\backslash\nbegin{$env}";
    &copy_verbatim($fileobject,"\\end{$env}","tex_mode");
} #end sub copy_latex

# This subroutine copies a latex token and its arguments if any.
# It puts everything in a \latex block IF the thing we're copying is 
#    short (for now, that means if it has no \n's in it). Otherwise,
#    it puts everything in a \layout LaTeX paragraph.
sub copy_latex_token {
    my $outstr = shift;
    my $fileobject = shift;
    my ($dummy, $tok, $count, $laststr);

    # Check whether the unknown token came right at the beginning
    #    of a new paragraph
    &CheckForNewParagraph;

    # Copy the actual word. Copy while you've still got
    #     arguments. Assume all args must be in the same paragraph
    #     (There could be new paragraphs *within* args)
    # We can't use copy_verbatim (unless we make it smarter) because
    #     it would choke on nested braces
    print "\nUnknown token: '$outstr': Copying in TeX mode\n";
    while (($dummy = $fileobject->lookAheadToken) &&
	   ($dummy =~ /^[[{]$/)) {
	if ($dummy eq '[') { #copy optional argument - assume it's simple
	    $tok = $fileobject->eatOptionalArgument;
	    $outstr .= "[" . $tok->print . "]";
	} else {
	    $count = 0; $laststr = "";
	    EAT: { #copied from eatBalanced, but allow paragraphs
	        die unless defined ($tok = $fileobject->eatMultiToken);
		$outstr .= "\n",next EAT if ! $tok; #new paragraph
		$dummy = $tok->print;
		# Replace space between a command & text (eatMultiToken ate it)
		$outstr.=" " if $laststr=~/\\[a-zA-Z]+/ && $dummy=~/^[a-zA-Z]/;
		$outstr .= $dummy;
		$laststr = $dummy;
		$count++ if $dummy eq '{'; # add a layer of nesting
		$count-- if $dummy eq '}'; # end one layer of nesting
		redo EAT if $count; #don't dump out until all done nesting
	    } #end EAT block
	} # end if $dummy = [{

    } #end while

    # Now we need to do TeX mode translation
    print $outstr;
    $outstr =~ s/\\/\\backslash /g;
    $outstr =~ s/\n/\\newline\n/g;

    # Get into TeX mode or LaTeX layout (and nest if nec.)
    $count = ($outstr =~ /\n/); # NUMBER OF MATCHES OF \n
    if ($count) { #long token. Use \layout LaTeX
	if ($#{$CurrentLayoutStack} > 0) { #need to nest
	    print " Nesting!";
	    print OUTFILE "\n\\begin_deeper";
	}
	print OUTFILE "\n\\layout LaTeX\n";

    } else { #short token. Just use \latex
        print OUTFILE "\n\\latex latex\n";
    }

    # Print the actual token + arguments if any
    print OUTFILE $outstr;

    # Get OUT of LaTeX mode (and end nesting if nec.)
    if ($count) { #long token. Used \layout LaTeX

	# End nesting if necessary
	if ($#{$CurrentLayoutStack} > 0) {
	    print " Done Nesting!";
	    print OUTFILE "\n\\end_deeper";
	}

	# New stuff will be in new paragraph, need a new layout command
	$IsNewParagraph = 1;

    } else { #short token. Just used \latex
        print OUTFILE "\n\\latex default\n";
    }

    print "\nDone copying unknown token";
} # end sub copy_latex_token

#########################  TRANSLATOR SUBROUTINES  ########################
sub CheckForNewParagraph {
# This subroutine makes sure we only write \layout command once per paragraph
# If $IsNewParagraph is 0, it does nothing
# If $INP==1, It starts a new paragraph
# If $MayBeDeeper==1 and we're currently within a list environment,
#    it starts a "deeper" Standard paragraph
    my $dummy; 
    my $layout = $$CurrentLayoutStack[-1];

    # Handle standard text within a list environment specially
    if ($MayBeDeeper) {
        if ($layout =~ /^$ListLayouts$/o) {
	    push (@$CurrentLayoutStack, "Standard");
	    print "\nCurrent Layout Stack: @$CurrentLayoutStack\n";
	    print OUTFILE "\n\\begin_deeper\n";
	    $layout = "Standard";
	}
	$MayBeDeeper = 0; # Don't test again until new paragraph
    }

    # Now write the new layout & font commands if we need to
    if ($IsNewParagraph) {
	print OUTFILE "\n\\layout $layout\n\n";

	# Now that we've written the command, it's no longer a new paragraph
	$IsNewParagraph = 0;

	# When you write a new paragraph, reprint any font commands
	foreach $dummy (keys %$CurrentFontStatus) {
	    if ($CurrentFontStatus->{$dummy}->[-1] ne "default") {
		print OUTFILE $TokenTransTable{$dummy};
	    }
	}
    } # end if $INP
} # end sub CheckForNewParagraph


1; # return true to calling subroutine

