/*
 *  Apple II emulator joystick calibration routines
 *  by Aaron Culliney - chernabog@baldmountain.bbn.com - (C) 1997
 *  This code has nothing to do with my employer, BBN.  It was written
 *  completely on my own time and on my own machine.
 *
 *  joystick.c - joystick calibration routines.  you need the joystick
 *  kernel loadable module for this to work and to pick up the
 *  <linux/joystick.h> include.
 *
 *  $Id: joystick.c,v 1.1 1997/06/27 03:38:32 chernabog Exp $
 *
 **/

#include "defs.H"
#include <linux/joystick.h>
#include <stdio.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <values.h>

/* from interface.c */
extern void c_interface_redo_bottom();
extern void c_interface_print( int x, int y, unsigned char *s );
extern half_joy_range;

int js_fd = -1;			/* joystick file descriptor */
struct JS_DATA_TYPE js;		/* joystick data struct */
long js_timelimit = 10;		/* 10 milliseconds between joystick samples */

int js_max_x = -1,
    js_max_y = -1,
    js_min_x = MAXINT,
    js_min_y = MAXINT;

short
    js_center_x,
    js_center_y,
    js_lowerrange_x,
    js_upperrange_x,
    js_lowerrange_y,
    js_upperrange_y,
    js_offset_x,
    js_offset_y;

float
    js_adjustlow_x,
    js_adjustlow_y,
    js_adjusthigh_x,
    js_adjusthigh_y;


/* -------------------------------------------------------------------------
    c_open_joystick() - opens joystick device and sets timelimit value
   ------------------------------------------------------------------------- */
int c_open_joystick() {
    if (js_fd < 0) {
	if ((js_fd = open("/dev/js0", O_RDONLY)) < 0) {

	    /* try again with another name */
	    if ((js_fd = open("/dev/joystick", O_RDONLY)) < 0)
		return 1;/* problem */
	}
	/* set timelimit value */
	if (ioctl(js_fd, JS_SET_TIMELIMIT, &js_timelimit) == -1)
	    return 1;/* problem */
    }
    return 0;/* no problem */
}

/* -------------------------------------------------------------------------
    c_close_joystick() - closes joystick device
   ------------------------------------------------------------------------- */
void c_close_joystick() {
    if (js_fd < 0)
	return;
    close(js_fd);
    js_fd = -1;
}


/* -------------------------------------------------------------------------
    c_calculate_joystick_parms() - calculates parameters for joystick
    device.  assumes that device extremes have already been determined.
   ------------------------------------------------------------------------- */
void c_calculate_joystick_parms() {

    js_lowerrange_x = js_center_x - js_min_x;
    js_upperrange_x = js_max_x - js_center_x;
    js_lowerrange_y = js_center_y - js_min_y;
    js_upperrange_y = js_max_y - js_center_y;

    js_offset_x = js_min_x - 0;
    js_offset_y = js_min_y - 0;

    js_adjustlow_x = (float)half_joy_range / (float)js_lowerrange_x;
    js_adjustlow_y = (float)half_joy_range / (float)js_lowerrange_y;
    js_adjusthigh_x = (float)half_joy_range / (float)js_upperrange_x;
    js_adjusthigh_y = (float)half_joy_range / (float)js_upperrange_y;
}

/* -------------------------------------------------------------------------
    c_calibrate_joystick() - calibrates joystick.  determines extreme
    and center coordinates.  assumes that it can write to the interface
    screen.
   ------------------------------------------------------------------------- */
void c_calibrate_joystick () {
    char temp[512];
    int almost_done, done;
    unsigned char x_val, y_val;

    /* open joystick device if not open */
    if (js_fd < 0) {
	if (c_open_joystick()) {	/* problem opening device */
	    c_interface_print(
		    1, 21, "                                      " );
	    c_interface_print(
		    1, 22, "     cannot open joystick device.     " );
	    usleep(1500000);
	    c_interface_redo_bottom();
	    return;/* problem */
	}
    }

    c_interface_print(
	1, 21, "  Move joystick to all extremes then  " );
    c_interface_print(
	1, 22, "    center it and press a button.     " );
    usleep(1500000);
    memset(temp, '\0', 512);
    c_interface_print(
	1, 21, "                                      " );
    c_interface_print(
	1, 22, "                                      " );

    almost_done = done = 0;		/* not done calibrating */
    while ((read(js_fd, &js, JS_RETURN) > 0) && (!done))
    {
	sprintf (temp, "  x = %04x, y = %04x", js.x, js.y);
	c_interface_print(1, 22, temp);
	if (js_max_x < js.x)
	    js_max_x = js.x;
	if (js_max_y < js.y)
	    js_max_y = js.y;

	if (js_min_x > js.x)
	    js_min_x = js.x;
	if (js_min_y > js.y)
	    js_min_y = js.y;

	if (js.buttons != 0x00)				/* press */
	    almost_done = 1;
	if (almost_done && (js.buttons == 0x00))	/* release */
	    done = 1;
    }

    js_center_x = js.x;
    js_center_y = js.y;
    c_calculate_joystick_parms();	/* determine the parms */

    memset(temp, '\0', 512);
    c_interface_print(
	1, 21, "     Press a button to continue.      " );

    /* show the normalized values until user presses button */
    while ((read(js_fd, &js, JS_RETURN) > 0) && js.buttons == 0x00) {
	x_val = (js.x < js_center_x)
	    ? (js.x - js_offset_x) * js_adjustlow_x
	    : (js.x - js_center_x) * js_adjusthigh_x + half_joy_range;

	y_val = (js.y < js_center_y)
	    ? (js.y - js_offset_y) * js_adjustlow_y
	    : (js.y - js_center_y) * js_adjusthigh_y + half_joy_range;
	sprintf(temp, "    x = %02x,   y = %02x", x_val, y_val);
	c_interface_print(1, 22, temp);
    }
    c_interface_redo_bottom();
}
