/* code to send a request for GSC stars to an xephem or an ESO server.
 * ESO code based on sample from Miguel Albrecht <malbrech@eso.org>
 */

#include <stdio.h>
#include <math.h>

#ifdef __STDC__
#include <stdlib.h>
#include <string.h>
#else
extern void *realloc(), *malloc();
#endif

#if defined(_POSIX_SOURCE)
#include <unistd.h>
#else
extern int close();
#endif

#include "P_.h"
#include "astro.h"
#include "circum.h"
#include "gscd.h"
#include "net.h"

extern char *syserrstr P_((void));
extern void pm_set P_((int percent));
extern void zero_mem P_((void *loc, unsigned len));

typedef unsigned char UC;
typedef unsigned long UL;

/* xephem support functions */
static int sendrqst P_((int fd, double ra, double dec, double fov, double mag,
    char msg[]));
static void packbuf P_((double ra, double dec, double fov, double mag,
    UC buf[SPKTSZ]));
static void unpackbuf P_((Obj *op, UC buf[RPKTSZ]));
static int moreObjF P_((ObjF **opp, int nop, int nnew));

#define	VER	1	/* version we claim to be */
#define	NPKTS	50	/* how many star packets we slurp up at once */


/* fetch GSC stars around the given field of view from the named xephem host.
 * if find some add them to the list at *opp and return new total, else
 * return -1 with a diagnostic message in msg[].
 */
int
gscnetfetch (host, np, ra, dec, fov, mag, opp, nop, msg)
char *host;	/* name of server */
Now *np;	/* current circumstances */
double ra, dec;	/* RA/Dec of center of view, rads, J2000 */
double fov;	/* diamater of field of view, rads */
double mag;	/* limiting magnitude desired */
ObjF **opp;	/* *opp is malloced with stars found */
int nop;	/* number already at *opp */
char msg[];	/* return diagnostic message here, if returning -1 */
{
	int sockfd;
	char buf[1024];
	ObjF *op;
	int nnew;
	int rcv, rcvd;

	/* verify the request parameters.
	 * limits are essentially determined by the requests we can make to
	 * the GSC server.
	 */
	if (mag < BMAG)
	    mag = BMAG;
	if (mag > FMAG)
	    mag = FMAG;
	if (ra<0 || ra>2*PI || dec< -PI/2 || dec>PI/2 || fov<0 || fov>MAXFOV){
	    (void) sprintf (msg, "Bad net request parameters");
	    return (-1);
	}

	/* init the progress meter */
	pm_set (0);

	/* contact xephem server */
	sockfd = mkconnection (host, GSCPORT, msg);
	if (sockfd < 0)
	    return (-1);

	/* send the initial request -- get back count */
	nnew = sendrqst (sockfd, ra, dec, fov, mag, buf);
	if (nnew < 0) {
	    (void) sprintf (msg, "%s: %s", host, buf);
	    (void) close (sockfd);
	    return (-1);
	}
	if (nnew == 0) {
	    (void) close (sockfd);
	    return (0);
	}

	/* get room for nnew new ObjF's */
	if (moreObjF (opp, nop, nnew) < 0) {
	    (void) sprintf (msg, "No memory");
	    (void) close (sockfd);
	    return (-1);
	}
	op = *opp;

	/* read and digest the nnew stars */
	for (rcvd = 0; rcvd < nnew; rcvd += rcv) {
	    UC pkts[NPKTS][RPKTSZ];

	    pm_set (100*rcvd/nnew);

	    rcv = rcvd + NPKTS > nnew ? nnew - rcvd : NPKTS;
	    if (recvbytes (sockfd, (UC *)pkts, rcv*RPKTSZ) < 0) {
		(void) sprintf (msg, "%s: %s", host, syserrstr());
		nnew = -1;
		break;
	    } else {
		int i;
		for (i = 0; i < rcv; i++)
		    unpackbuf ((Obj *)&op[nop+rcvd+i], pkts[i]);
	    }
	}

	(void) close (sockfd);

	if (nnew > 0)
	    return (nop + nnew);

	return(nnew);
}

/* send the xephem gsc request to the server on the given socket fd.
 * if ok return count of new stars to be read
 * else fill msg[] with excuse and return -1.
 */
static int
sendrqst (fd, ra, dec, fov, mag, msg)
int fd;
double ra, dec;
double fov, mag;
char msg[];
{
	UC buf[80];

	packbuf (ra, dec, fov, mag, buf);

	/* send the message */
	if (sendbytes (fd, buf, SPKTSZ) < 0) {
	    strcpy (msg, syserrstr());
	    return (-1);
	}

	/* get ack -- first byte should be 0 */
	if (recvbytes (fd, buf, 80) < 0) {
	    strcpy (msg, syserrstr());
	    return (-1);
	}
	if (buf[0] != 0) {
	    strcpy (msg, (char *)buf+1); /* ASCIZ error message */
	    return (-1);
	}

	/* remaining bytes are ASCIZ count */
	return (atoi((char *)buf+1));
}

/* pack up a request buffer */
static void
packbuf (ra, dec, fov, mag, buf)
double ra, dec, fov, mag;
UC buf[SPKTSZ];
{
	UL t;

	/* version number */
	buf[0] = VER;

	/* 0 <= ra < 2*PI */
	t = ra/(2*PI)*(1L<<24) + 0.5;
	buf[1] = (t >> 16) & 0xff;
	buf[2] = (t >> 8)  & 0xff;
	buf[3] = (t)       & 0xff;

	/* -PI/2 <= dec <= PI/2 */
	t = (dec + PI/2)/PI*((1L<<24)-1) + 0.5;
	buf[4] = (t >> 16) & 0xff;
	buf[5] = (t >> 8)  & 0xff;
	buf[6] = (t)       & 0xff;

	/* 0 <= fov <= MAXFOV */
	t = fov/MAXFOV*((1L<<16)-1);
	buf[7] = (t >> 8)  & 0xff;
	buf[8] = (t)       & 0xff;

	/* BMAG <= mag <= FMAG */
	t = (mag - BMAG)/(FMAG-BMAG)*((1L<<16)-1);
	buf[9] = (t >> 8)  & 0xff;
	buf[10]= (t)       & 0xff;
}

/* break apart a reply buffer */
static void
unpackbuf (op, buf)
Obj *op;
UC buf[RPKTSZ];
{
	UL t0, t1;

	t0 = ((UL)buf[0] << 8) | (UL)buf[1];
	t1 = ((UL)buf[2] << 8) | (UL)buf[3];
	(void) sprintf (op->o_name, "GSC %04ld-%04ld", t0, t1);

	op->o_type = FIXED;

	op->f_class = buf[4] == 0 ? 'S' : 'T';

	t0 = ((UL)buf[5] << 16) | ((UL)buf[6] << 8) | (UL)buf[7];
	op->f_RA = t0*(2*PI)/(1L<<24);

	t0 = ((UL)buf[8] << 16) | ((UL)buf[9] << 8) | (UL)buf[10];
	op->f_dec = t0*PI/((1L<<24)-1) - PI/2;

	op->f_epoch = (float)J2000;

	t0 = ((UL)buf[11] << 8) | (UL)buf[12];
	set_fmag (op, t0*(FMAG-BMAG)/((1L<<16)-1)+BMAG);
}


/* ESO functions */

/* fetch GSC stars around the given field of view from the named ESO host.
 * if find some add them to the list at *opp and return new total, else
 * return -1 with a diagnostic message in msg[].
 */
int
gscesofetch (host, np, ra, dec, fov, mag, opp, nop, msg)
char *host;	/* name of server */
Now *np;	/* current circumstances */
double ra, dec;	/* RA/Dec of center of view, rads, J2000 */
double fov;	/* diamater of field of view, rads */
double mag;	/* limiting magnitude desired */
ObjF **opp;	/* *opp is malloced with stars found */
int nop;	/* number already at *opp */
char msg[];	/* return diagnostic message here, if returning -1 */
{
static char ifmt[] = "%s %lf %lf %lf %lf %lf %lf %*s %lf %*s %*s %d";
static char ofmt[] = "GET /skycat/servers/gsc-server?%.6f%c%.6f&r=0,%.1f&m=0,%g&s=R&f=1&* HTTP/1.0\n\n";
	char buf[1024];
	int sockfd;
	int n;

	/* contact the ESO server */
	sockfd = mkconnection (host, 80, msg);
	if (sockfd < 0)
	    return (-1);

	/* format the GET */
	(void) sprintf (buf, ofmt, radhr(ra), dec<0?'-':'+', fabs(raddeg(dec)),
							raddeg(fov/2)*60, mag);

	/* send the GET method */
	n = strlen(buf);
	if (sendbytes(sockfd, (UC *)buf, n) < 0) {
	    (void) sprintf (msg, "%s: send error: %s", host, syserrstr());
	    return (-1);
	}

	/* now read lines of stars from the socket and collect in opp array */
	while (sfgets (buf, sizeof(buf), sockfd)) {
	    double rh, rm, rs;
	    double dd, dm, ds;
	    char name[32];
	    double m;
	    Obj *op;
	    int c;

	    if (strncmp (buf, "[EOD]", 5) == 0)
		break;
	    if (sscanf (buf,ifmt,name,&rh,&rm,&rs,&dd,&dm,&ds,&m,&c) != 9)
		continue;
	    if (moreObjF (opp, nop, 1) < 0) {
		(void) sprintf (msg, "No memory");
		(void) close (sockfd);
		return (-1);
	    }
	    op = (Obj *)(&(*opp)[nop++]);

	    (void) sprintf (op->o_name, "GSC %.4s-%.4s", name+1, name+6);
	    op->o_type = FIXED;
	    op->f_class = c == 0 ? 'S' : 'T';
	    op->f_RA = hrrad (rs/3600.0 + rm/60.0 + rh);
	    if (dd < 0.0)
		op->f_dec = degrad (-ds/3600.0 - dm/60.0 + dd);
	    else
		op->f_dec = degrad (ds/3600.0 + dm/60.0 + dd);
	    op->f_epoch = (float)J2000;
	    set_fmag (op, m);
	}

	(void) close (sockfd);

	return (nop);
}

/* grow *opp, which already conatins nop, to hold nnew zeroed more.
 * if ok update *opp and return 0, else -1.
 */
static int
moreObjF (opp, nop, nnew)
ObjF **opp;
int nop;
int nnew;
{
	char *newmem;

	/* extend *opp for nnew more ObjF's */
	if (*opp)
	    newmem = realloc ((void *)*opp, (nop+nnew)*sizeof(ObjF));
	else
	    newmem = malloc (nnew * sizeof(ObjF));
	if (!newmem)
	    return (-1);
	*opp = (ObjF *)newmem;
	zero_mem ((void *)(*opp+nop), nnew*sizeof(ObjF));
	return (0);
}
