# =============================================================================
#
# File:		dsk_FileViewer.tcl
# Project:	TkDesk
#
# Started:	11.10.94
# Changed:	08.04.96
# Author:	cb
#
# Description:	Implements a class for the main file viewer window.
#
# Copyright (C) 1996  Christian Bolik
# 
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
# See the file "COPYING" in the base directory of this distribution
# for more.
#
# =============================================================================


#
# =============================================================================
#
# Class:	dsk_FileViewer
# Desc:		This creates a toplevel that is the (or a) main window
#		for the file viewer (menu, entries, listboxes etc.).
#
# Methods:	
# Procs:	
# Publics:
#

itcl_class dsk_FileViewer {
    constructor {config} {
	global [set this] tkdesk env cb_tools

	dsk_busy
	
	#
	# Create a toplevel with this object's name
	# (later accessible as $this-top):
	#
        set class [$this info class]
        ::rename $this $this-tmp-
        ::toplevel $this -class $class
        ::rename $this $this-top
        ::rename $this-tmp- $this

	wm withdraw $this

	#
	# Create menubar
	#
	frame $this.fMenu -bd 2 -relief raised
	pack $this.fMenu -fill x

	menubutton $this.fMenu.mbTkDesk -text "TkDesk" -underline 0 \
		-menu $this.fMenu.mbTkDesk.menu
	pack $this.fMenu.mbTkDesk -side left

	# ---- TkDesk Menu
	menu [set m $this.fMenu.mbTkDesk.menu]
	$m add command -label "New Browser..." -underline 0 \
		-command "dsk_ask_dir browser"
	$m add command -label "Clone Window" -underline 2 \
		-command "dsk_FileViewer .fv\[dsk_FileViewer :: id\] \
			-dir \[$this curdir\] -num_lbs \$tkdesk(num_lbs)"
	$m add command -label "Application Bar " -underline 0 \
		-command "dsk_appbar"
	$m add separator
	$m add cascade -label "Edit Config Files" -underline 0 \
		-menu $this.fMenu.mbTkDesk.menu.edmenu
	$m add cascade -label "Reread Config Files" -underline 0 \
		-menu $this.fMenu.mbTkDesk.menu.rdmenu		
	$m add cascade -label "Auto Save" -underline 1 \
		-menu $this.fMenu.mbTkDesk.menu.aumenu		
	$m add command -label "Save All Now" -underline 0 \
		-command {dsk_save_config 1}

	$m add separator
	if $tkdesk(in_development) {
	    $m add command -label "Restart" -underline 2 \
		-command "dsk_restart" -accelerator "Meta-r"
	    bind $this <Alt-r> "dsk_restart; break"
	    $m add command -label "Debug tkdesksh" -command "
	    exec rxvt -e gdb /home/root/tcl/TkDesk/tkdesksh [pid] &
	    "
	}
	$m add command -label "Close Window" -underline 0 \
		-command "$this close"
	if !$tkdesk(xmaster) {
	    $m add command -label "Quit" -underline 0 -command "dsk_exit"
	} else {
	    $m add command -label "Quit X Windows" -underline 0 \
			-command "dsk_exit"
	}

	menu [set m $this.fMenu.mbTkDesk.menu.edmenu]
	$m add command -label "All" -command "dsk_edit_configs"
	$m add separator
	foreach cf $tkdesk(configfiles) {
	    $m add command -label $cf -command "dsk_edit_configs $cf"
	}

	menu [set m $this.fMenu.mbTkDesk.menu.rdmenu]
	$m add command -label "All" -command "dsk_reread_config"
	$m add separator
	foreach cf $tkdesk(configfiles) {
	    $m add command -label $cf -command "dsk_reread_config $cf"
	}

	menu [set m $this.fMenu.mbTkDesk.menu.aumenu]
	$m add checkbutton -label "Annotations" \
		-variable tkdesk(autosave,annotations)
	$m add checkbutton -label "Bookmarks" \
		-variable tkdesk(autosave,bookmarks)
	$m add checkbutton -label "Histories" \
		-variable tkdesk(autosave,history)
	$m add checkbutton -label "Options" \
		-variable tkdesk(autosave,options)
	$m add checkbutton -label "Window Layout" \
		-variable tkdesk(autosave,layout)

	# ---- File Menu
	menubutton $this.fMenu.mbFile -text "File" -underline 0 \
		-menu $this.fMenu.mbFile.menu
	pack $this.fMenu.mbFile -side left

	menu [set m $this.fMenu.mbFile.menu]
	$m add command -label "Information" -underline 0 \
		-command "dsk_fileinfo" -accelerator "Ctrl-i"
	$m add command -label "New File..." -underline 0 \
		-command "dsk_create file"
	$m add command -label "New Directory..." -underline 4 \
		-command "dsk_create directory" -accelerator "Ctrl-d"
	$m add command -label "Copy, Move, Link... " -underline 0 \
		-command "dsk_copy" -accelerator "Ctrl-c"
	$m add command -label "Rename... " -underline 0 \
		-command "dsk_rename" -accelerator "Ctrl-r"
	$m add command -label "Delete..." -underline 1 \
		-command "dsk_delete" -accelerator "Del"
	$m add command -label "Print..." -underline 0 \
		-command "dsk_print" -accelerator "Ctrl-P"
	$m add separator
	$m add command -label "Find Files..." -underline 0 \
		-command "dsk_find_files" -accelerator "Ctrl-f"
	$m add command -label "Find Annotation..." -underline 5 \
		-command "dsk_find_annotation"
	$m add command -label "Copy To X Selection" -underline 8 \
		 -command "dsk_select X"
	$m add command -label "Open Selected Files" -underline 0 \
		-command "dsk_openall"
	#$m add command -label "Clear selection" -underline 6 \
		-command "dsk_select clear"

	# ---- Directories Menu
	menubutton $this.fMenu.mbDirs -text "Directories" -underline 0 \
		-menu $this.fMenu.mbDirs.menu
	pack $this.fMenu.mbDirs -side left

	menu [set m $this.fMenu.mbDirs.menu]
	$m add command -label "Open..." -underline 0 \
		-command "dsk_ask_dir" -accelerator "Ctrl-o"
	$m add command -label "New..." -underline 0 \
		-command "dsk_create directory" -accelerator "Ctrl-d"
	$m add command -label "Home Directory " -underline 0 \
		-command "$this config -dir \$env(HOME)"
	$m add cascade -label "Trees" -menu ${m}.fs
	menu ${m}.fs
	menu ${m}.fs.home -postcommand "dsk_casdirs $env(HOME) ${m}.fs.home 1"
	${m}.fs add cascade -label "Home " -menu ${m}.fs.home
	menu ${m}.fs.root -postcommand "dsk_casdirs / ${m}.fs.root 1"
	${m}.fs add cascade -label "Root " -menu ${m}.fs.root
	$m add separator
	
	$m add command -label "Trash" -underline 0 \
		-command "dsk_FileList .dfl\[dsk_FileList :: id\] \
			-directory $tkdesk(configdir)/.trash"
	$m add command -label "Empty Trash" -underline 0 \
		-command "dsk_empty_trash"
	$m add separator

	if [info exists tkdesk(directories)] {
	    foreach mdir $tkdesk(directories) {
		if {$mdir == "-"} {
		    $m add separator
		} else {
		    _add_dir_to_menu $this $m $mdir
		}
	    }
	}

	bind $m <ButtonRelease-1> "
		set tkdesk(menu,control) 0
		[bind Menu <ButtonRelease-1>]"
	bind $m <Control-ButtonRelease-1> "
		set tkdesk(menu,control) 1
		[bind Menu <ButtonRelease-1>]"

	# ---- Commands Menu
	menubutton $this.fMenu.mbCmds -text "Commands" -underline 0 \
		-menu $this.fMenu.mbCmds.menu
	pack $this.fMenu.mbCmds -side left

	menu [set m $this.fMenu.mbCmds.menu]
	$m add command -label "Execute..." -underline 0 \
		-command "dsk_ask_exec" -accelerator "Ctrl-x"
	$m add command -label "Periodic Execution..." -underline 0 \
		-command "dsk_periodic"
	$m add command -label "Job Control" -underline 0 -command "dsk_jobs"
	$m add separator

	if [info exists tkdesk(commands)] {
	    foreach cmd $tkdesk(commands) {
		if {[llength $cmd] > 1} {
		    _add_cmd_to_menu $m $cmd
		} else {
		    $m add separator
		}
	    }
	}

	# ---- Bookmarks Menu
	menubutton $this.fMenu.mbBook -text "Bookmarks" -underline 0 \
		-menu $this.fMenu.mbBook.menu
	pack $this.fMenu.mbBook -side left

	menu [set m $this.fMenu.mbBook.menu] \
		-postcommand "dsk_bookmark menu $m"
	# add dummy entry to work around bug in pre Tk 4.0p2:
	$m add command -label "dummy"
	bind $m <ButtonRelease-1> "
		set tkdesk(file_lb,control) 0
		[bind Menu <ButtonRelease-1>]"
	bind $m <Control-ButtonRelease-1> "
		set tkdesk(file_lb,control) 1
		[bind Menu <ButtonRelease-1>]"

	# ---- Options Menu
	menubutton $this.fMenu.mbOpts -text "Options" -underline 0 \
		-menu $this.fMenu.mbOpts.menu
	pack $this.fMenu.mbOpts -side left

	menu [set m $this.fMenu.mbOpts.menu]
	$m add checkbutton -label " Add Icons " -underline 1 \
	    -variable tkdesk(add_icons) \
	    -command "dsk_FileListbox :: addicons \$tkdesk(add_icons)
			$this refresh all"
	$m add checkbutton -label " Show All Files " -underline 1 \
	    -variable tkdesk(show_all_files) \
	    -command "dsk_FileListbox :: showall \$tkdesk(show_all_files)
			$this refresh all"
	$m add checkbutton -label " Folders On Top " -underline 1 \
	    -variable tkdesk(folders_on_top) \
	    -command "dsk_FileListbox :: topfolders \$tkdesk(folders_on_top)
			$this refresh all"
	$m add checkbutton -label " Append Type Char " -underline 2 \
	    -variable tkdesk(append_type_char) \
	    -command "dsk_FileListbox :: typechar \$tkdesk(append_type_char)
			$this refresh all"
	$m add checkbutton -label " Single Click (Dirs) " -underline 4 \
	    -variable tkdesk(single_click)
	$m add checkbutton -label " Always In Browser " -underline 11 \
	    -variable tkdesk(in_browser)
	$m add separator
	$m add checkbutton -label " Strip $env(HOME) " -underline 2 \
	    -variable tkdesk(strip_home) \
	    -command "$this refresh all"
	$m add checkbutton -label " Overwrite Always " -underline 1 \
	    -variable tkdesk(overwrite_always)
	$m add checkbutton -label " Really Delete " -underline 1 \
	    -variable tkdesk(really_delete)
	$m add checkbutton -label " Quick Drag'n'Drop " -underline 1 \
	    -variable tkdesk(quick_dragndrop)
	$m add checkbutton -label " Sort History " -underline 6 \
	    -variable tkdesk(sort_history)
	if [info exists tkdesk(soundcmd)] {
	    $m add checkbutton -label " Use Sound " -underline 8 \
		    -variable tkdesk(use_sound)
	}
	$m add checkbutton -label " Dialogs At Pointer " -underline 1 \
	    -variable tkdesk(at_pointer)
	$m add separator
	$m add cascade -label "Number Of Listboxes" -menu $m.numlbs

	menu [set m $m.numlbs]
	set [set this](num_lbs) $num_lbs
	$m add radiobutton -label " 1 Listbox" \
		-variable [set this](num_lbs) \
		-value 1 -command "$this _resize"
	$m add radiobutton -label " 2 Listboxes" \
		-variable [set this](num_lbs) \
		-value 2 -command "$this _resize"
	$m add radiobutton -label " 3 Listboxes" \
		-variable [set this](num_lbs) \
		-value 3 -command "$this _resize"
	$m add radiobutton -label " 4 Listboxes" \
		-variable [set this](num_lbs) \
		-value 4 -command "$this _resize"
	$m add radiobutton -label " 5 Listboxes" \
		-variable [set this](num_lbs) \
		-value 5 -command "$this _resize"
	$m add radiobutton -label " 6 Listboxes" \
		-variable [set this](num_lbs) \
		-value 6 -command "$this _resize"

	# ---- Help Menu
	menubutton $this.fMenu.mbHelp -text "Help" -underline 0 \
		-menu $this.fMenu.mbHelp.menu
	pack $this.fMenu.mbHelp -side right

	menu [set m $this.fMenu.mbHelp.menu]
	$m add command -label "User's Guide  " \
		-command "dsk_cbhelp $tkdesk(library)/doc/Guide howto" \
		-accelerator "F1"
	$m add command -label "FAQ" \
		-command "dsk_cbhelp $tkdesk(library)/doc/Guide#Frequently howto"
	$m add command -label "Changes" \
		-command "dsk_cbhelp $tkdesk(library)/doc/CHANGES"
	$m add command -label "License" \
		-command "dsk_cbhelp $tkdesk(library)/doc/License"
	$m add separator
	$m add checkbutton -label "Balloon Help" \
		-variable cb_tools(balloon_help)
	$m add command -label "About TkDesk..." \
		-command dsk_about

	# ---- create button bar
	_button_bar
	
	# ---- create path entry
	frame $this.fPE -bd 1 -relief raised
	pack $this.fPE -fill x

	entry $this.ePE -bd 2 -relief sunken
	bindtags $this.ePE "$this.ePE Entry All"
	pack $this.ePE -in $this.fPE -side left -fill x -expand yes \
		-padx $tkdesk(pad) -pady $tkdesk(pad) -ipady 2
	bind $this.ePE <Return> "$this config -dir \[$this.ePE get\]"
	bind $this.ePE <Control-Return> "set tkdesk(menu,control) 1
				$this config -dir \[$this.ePE get\]
				%W delete 0 end
				%W insert end \[$this curdir\]"
	bind $this.ePE <Tab> "focus $this; break"
	bind $this.ePE <3> "update idletasks; $this _path_popup %X %Y"
	cb_bindForCompletion $this.ePE <Control-Tab>
    	blt_drag&drop target $this.ePE \
				handler text "dd_handle_text $this.ePE 1"

	menubutton $this.mbHist -bd 2 -relief raised \
		-bitmap @$tkdesk(library)/cb_tools/bitmaps/combo.xbm \
		-menu $this.mbHist.menu
	pack $this.mbHist -in $this.fPE -side right \
		-padx $tkdesk(pad) -pady $tkdesk(pad) -ipadx 2 -ipady 2

	menu $this.mbHist.menu \
		-postcommand "history buildmenu $this.mbHist.menu ; update"
	# add dummy entry to work around bug in pre Tk 4.0p2:
	$this.mbHist.menu add command -label "dummy"
	bind $this.mbHist.menu <ButtonRelease-1> "
		set tkdesk(menu,control) 0
		[bind Menu <ButtonRelease-1>]"
	bind $this.mbHist.menu <Control-ButtonRelease-1> "
		set tkdesk(menu,control) 1
		[bind Menu <ButtonRelease-1>]"

	# ---- create horizontal scrollbar
	frame $this.fsb -bd 1 -relief raised
	pack $this.fsb -fill x
	
	scrollbar $this.hsb -orient horizontal -bd 2 -relief sunken \
		-command "$this _set_first_lb"
	$this.hsb set 1 3 0 2
	pack $this.hsb -in $this.fsb -fill x \
		-padx $tkdesk(pad) -pady $tkdesk(pad)
	
	#
	# ---- Create listboxes
	#
	frame $this.fLBs
	pack $this.fLBs -fill both -expand yes
	set j 0
	for {set i 0} {$i < $num_lbs} {incr i} {
	    set lbname $this.lb$j
	    incr j
	    dsk_FileListbox $lbname -viewer $this \
		-width $tkdesk(file_lb,minwidth) \
		-height $tkdesk(file_lb,minheight) -pad $tkdesk(pad)
	    pack $lbname -in $this.fLBs -side left -fill both -expand yes
	}
	set num_lbs_created $j

	# ---- create status bar
	frame $this.fStatus -bd 1 -relief raised 
	pack $this.fStatus -fill x 

	label $this.lStatus -text "Ready." -anchor w \
			-font $tkdesk(font,labels2)
	pack $this.lStatus -in $this.fStatus -fill x \
		-padx [expr $tkdesk(pad) / 2] -pady [expr $tkdesk(pad) / 2]

	#
	# Bindings
	#
	#bind $this <Any-Enter> \
		#"focus $this; set tkdesk(active_viewer) $this; break"
	bind $this <Any-Enter> \
		"set tkdesk(active_viewer) $this; break"
	bind $this <Tab> "focus $this.ePE; break"

	bind $this <Control-i> "dsk_fileinfo; break"
	bind $this <Control-f> "dsk_find_files; break"
	bind $this <Control-d> "dsk_create directory; break"	
	bind $this <Control-c> "dsk_copy; break"
	bind $this <Control-r> "dsk_rename; break"
	bind $this <Delete> "dsk_delete; break"
	bind $this <Control-x> "dsk_ask_exec; break"
	bind $this <Control-o> "dsk_ask_dir; break"
	bind $this <Control-b> "dsk_bookmark add; break"
	bind $this <Control-p> "dsk_print; break"
	bind $this <Return> "dsk_openall; break"
	bind $this <F1> "dsk_cbhelp $tkdesk(library)/doc/Guide howto"

	#
	# Window manager settings
	#
	update idletasks
	wm title $this "TkDesk Version $tkdesk(version)"
	wm minsize $this $tkdesk(file_lb,minwidth) $tkdesk(file_lb,minheight)
	#wm geometry $this $tkdesk(file_lb,width)x$tkdesk(file_lb,height)
	dsk_place_window $this fbrowser \
		$tkdesk(file_lb,width)x$tkdesk(file_lb,height) 1
	wm protocol $this WM_DELETE_WINDOW "$this close"

	if $tkdesk(fvwm) {
	    # create the icon window
	    # (this code is based upon the code posted by:
	    # kennykb@dssv01.crd.ge.com (Kevin B. Kenny))
	    toplevel $this-icon -bg $tkdesk(color,icon_background)
	    wm withdraw $this-icon
	    label $this-icon.label \
		    -image [cb_image $tkdesk(icon,filebrowser)] \
		    -width 32 -height 32 -bd 0 \
		    -bg $tkdesk(color,icon_background)
	    pack $this-icon.label -ipadx 2 -ipady 2
	    blt_drag&drop target $this-icon.label handler \
		    file "$this _dd_drophandler"
	    update idletasks
	    wm geometry $this-icon \
		    [winfo reqwidth $this-icon]x[winfo reqheight $this-icon]
	    wm protocol $this-icon WM_DELETE_WINDOW "$this delete"
	    wm iconwindow $this $this-icon
	} else {
	    wm iconbitmap $this @$tkdesk(library)/images/bigfiling.xbm
	}
	
	incr count		;# incr the counter of opened fv windows
	set tkdesk(menu,control) 0
	set tkdesk(file_lb,control) 0

        #                          
        #  Explicitly handle config's that may have been ignored earlier
        #                            
        foreach attr $config {
            config -$attr [set $attr]
        }

	dsk_sound dsk_new_filebrowser
	if !$dontmap {
	    wm deiconify $this
	    tkwait visibility $this
	    catch "lower $this .dsk_welcome"
	    update
	}

	# the status line may become quite large
	pack propagate $this.fStatus 0

	dsk_lazy
    }

    destructor {
	global tkdesk
	
	for {set i 0} {$i < $num_lbs_created} {incr i} {
	    catch "$this.lb$i delete"
	}
	incr count -1
        #after 10 rename $this-top {}	;# delete this name
        catch {destroy $this}	;# destroy associated windows
	catch {destroy $this-icon}
	set tkdesk(active_viewer) ""
	foreach fv "[itcl_info objects -class dsk_FileViewer] \
		[itcl_info objects -class dsk_FileList]" {
	    if {"$fv" != "$this"} {
		set tkdesk(active_viewer) $fv
	    }
	}
    }

    #
    # ----- Methods and Procs -------------------------------------------------
    #

    method config {config} {
    }

    method cget {var} {
	return [set [string trimleft $var -]]
    }

    method close {} {
	global tkdesk env

	# add current directory to history before closing window:
	if {[string first $env(HOME) $directory] == 0} {
	    history add [string_replace $directory $env(HOME) ~]
	} else {
	    history add $directory
	}
	
	if [winfo exists .dsk_appbar] {
	    $this delete
	} elseif {[dsk_active viewer] == 1} {
	    # about to close last window
	    dsk_exit 1
	} else {
	    $this delete
	}
    }

    method curdir {} {
	return $directory
    }

    method refresh {{mode ""}} {
	if {$mode == "all"} {
	    foreach fv [itcl_info objects -class dsk_FileViewer] {
	    	$fv refresh
	    }
	} else {
	    for {set i 0} {$i < $num_lbs_created} {incr i} {
		set lbdirs($i) ""
	    }
	    $this config -directory $directory
	}
    }

    method refreshdir {dir} {
	for {set i 0} {$i < $dir_depth} {incr i} {
	    if {$lbdirs($i) == $dir} {
		$this.lb$i refresh
		$this status "Ready."
		break
	    }
	}
    }

    method status {str} {
	$this.lStatus config -text $str
	update idletasks
    }

    method selstatus {} {
	global tkdesk_anno
	
	set sfl [$this select get]
	set l [llength $sfl]
	if {$l == 0} {
	    #$this status "Ready. [dsk_fs_status $directory]"
	    $this status "Ready."
	} elseif {$l == 1} {
	    set fi ""
	    catch "set fi \[dsk_ls -l -o $sfl\]"
	    if {$fi != ""} {
		#regsub -all {\\t} $fi " " fi
		regsub -all {\\t} $fi \t fi
		set fi [split [lindex $fi 0] \t]
		#ot_maplist [split $fi \t] p l o g s d1 d2 d3 n
		if [info exists tkdesk_anno($sfl)] {
		    set an "(A!) "
		} else {
		    set an ""
		}
		if {[lindex $fi 7] == ""} {
		    ot_maplist $fi n s d o g p l
		    set n [string trimright $n " "]
		    $this status "$an$n:  $s Bytes,  $d,  $o/$g  ($p)"
		} else {
		    # it's a symbolic link
		    set n [string trimright [lindex $fi 0] " "]
		    set lf [lindex [lindex $fi 7] 1]
		    $this status "$an$n: symbolic link to $lf"
		}
	    } else {
		$this status "[file tail $sfl]: broken symbolic link"
	    }
	} else {
	    set s 0

	    # sum up file sizes - another candidate for C coding
	    foreach f $sfl {
		catch "set s \[expr $s + \[file size $f\]\]"
	    }
	    $this status "$l Files selected ($s Bytes)."
	}
    }

    method select {cmd} {
	global tkdesk
	
	switch -glob -- $cmd {
	    get		{# return a list of all selected files
		set sfl ""
		for {set i 0} {$i < $dir_depth} {incr i} {
		    set sl [$this.lb$i.dlb select get]
		    if {$sl != ""} {
		        set fl [$this.lb$i.dlb get]
			foreach s $sl {
			    set file [lindex [split [lindex $fl $s] \t] 0]
			    set file [string trimright $file " "]
			    if $tkdesk(append_type_char) {
				set file [dsk_striptc $file]
			    }
			    lappend sfl "$lbdirs($i)$file"
			}
		    }
		}
		return $sfl
			}
	    clear	{# clear selection in all listboxes
		for {set i 0} {$i < $num_lbs_created} {incr i} {
		    $this.lb$i.dlb select clear
		}
		$this selstatus
			}
	    X		{# copy selected filenames to X selection
		set sfl [$this select get] 
		if {$sfl != ""} {
		    selection handle $this "$this _retrieve_X"
		    selection own $this
		} else {
		    cb_info "Please select one or more files first."
		}
			}
	    default	{
		error "$this select: unknown option $cmd"
			}
	}
    }

    method _retrieve_X {offset maxBytes} {
	# return "offset: $offset, maxBytes: $maxBytes"

	if {$offset != 0} {
	    # this should never happen, so return
	    return ""
	}

	set sfl [$this select get]
	set rl ""
	if {$sfl != ""} {
	    if {[llength $sfl] > 1} {
	    	foreach file $sfl {
		    append rl "$file\n"
	    	}
	    } else {
		set rl $sfl
	    }
	}
	if {[string length $rl] > $maxBytes} {
	    return [string range $rl 0 [expr $maxBytes - 1]]
	} else {
	    return $rl
	}
    }

    method _set_first_lb {num} {
	# lb$num is to become the leftmost listbox
	#puts "$num, $first_lb, $dir_depth; [$this.hsb get]"

	if {$first_lb == $num && \
		$dir_depth == [lindex [$this.hsb get] 0]} return

	if {$num < 0} {
	    set num 0
	}

	set max_flb [cb_max 0 [expr $dir_depth - $num_lbs]]
	if {$num > $max_flb} {
	    set num $max_flb
	}

	if {$first_lb == 0 && $num == 0} {
	    $this.hsb set $dir_depth $num_lbs $first_lb \
				[expr $first_lb + $num_lbs - 1]
	    return
	}
	if {$first_lb == $max_flb && $num == $max_flb} {
	    $this.hsb set $dir_depth $num_lbs $first_lb \
				[expr $first_lb + $num_lbs - 1]
	    return
	}

	set diff [expr $num - $first_lb]
	switch -- $diff {
	    1 {
		pack forget $this.lb$first_lb
		if {$num_lbs > 1} {
		    pack $this.lb[expr $num + $num_lbs - 1] \
			    -after $this.lb[expr $num + $num_lbs - 2] \
			    -in $this.fLBs -side left -fill both \
			    -expand yes
		} else {
		    pack $this.lb[expr $num + $num_lbs - 1] \
			    -in $this.fLBs -side left -fill both \
			    -expand yes
		}
	    }
	    -1 {
		pack forget $this.lb[expr $first_lb + $num_lbs - 1]
		if {$num_lbs > 1} {
		    pack $this.lb$num \
			    -before $this.lb$first_lb \
			    -side left -fill both \
			    -expand yes
		} else {
		    pack $this.lb$num \
			    -in $this.fLBs -side left -fill both \
			    -expand yes
		}
	    }
	    default	{
		for {set i $first_lb} \
			{$i < [expr $first_lb + $num_lbs]} {incr i} {
		    pack forget $this.lb$i
		}
		for {set i $num} \
			{$i < [expr $num + $num_lbs]} {incr i} {
		    pack $this.lb$i \
			    -in $this.fLBs -side left -fill both \
			    -expand yes
		}
	    }
	}

	set first_lb $num
	$this.hsb set $dir_depth $num_lbs $first_lb \
				[expr $first_lb + $num_lbs - 1]
    }

    method _selecting {lb} {
	global tkdesk

	if {$tkdesk(free_selection)} {
	    return
	}

	for {set i 0} {$i < $num_lbs_created} {incr i} {
	    if {"$this.lb$i" != "$lb"} {
		$this.lb$i.dlb select clear
	    }
	}
    }

    method _dblclick {lb sel} {
	global tkdesk
	
	if {$sel == "" || $lb == ""} {
	    return
	}
	if {$tkdesk(single_click) && [llength $sel] > 1} {
	    return
	}

	set dir [string trimright [$lb info public directory -value] "/"]
	#set file [lindex [lindex [$lb.dlb get] [lindex $sel 0]] 0]
	set file [string trimright [lindex [split [lindex [$lb.dlb get] \
		[lindex $sel 0]] \t] 0] " "]
	if $tkdesk(append_type_char) {
	    set file [dsk_striptc $file]
	}
	if {[string first "/" $file] == -1} {
	    set file "$dir/$file"
	}
	if {!$tkdesk(single_click) || \
		($tkdesk(single_click) && [file isdirectory $file])} {
	    ::dsk_open $this "$file"
	}
	if [file isdirectory $file] {
	    select clear
	}
    }

    method _popup {lb sel mx my} {
	if {$sel == "" || $lb == ""} {
	    return
	}
	set dir [string trimright [$lb info public directory -value] "/"]
	#set file [lindex [lindex [$lb.dlb get] [lindex $sel 0]] 0]
	set file [string trimright [lindex [split [lindex [$lb.dlb get] \
		[lindex $sel 0]] \t] 0] " "]
	::dsk_popup $lb "$dir/$file" $mx $my
	#$this selstatus
    }

    method _resize {} {
	global tkdesk [set this]

	if {$num_lbs != [set [set this](num_lbs)]} {
	    dsk_busy
	    if {$num_lbs_created < [set [set this](num_lbs)]} {
		for {set j $num_lbs_created} \
			{$j < [set [set this](num_lbs)]} {incr j} {
	    	    set lbname $this.lb$j
	    	    dsk_FileListbox $lbname -viewer $this \
			-width $tkdesk(file_lb,minwidth) -pad $tkdesk(pad)
	    	    pack $lbname -in $this.fLBs -side left \
			-fill both -expand yes
		}
		set num_lbs_created [set [set this](num_lbs)]
	    } else {
		$this _set_first_lb 0
		if {$num_lbs < [set [set this](num_lbs)]} {
		    for {set j $num_lbs} \
			    {$j < [set [set this](num_lbs)]} {incr j} {
		    	pack $this.lb$j -in $this.fLBs -side left \
				-fill both -expand yes
		    }
		} else {
		    for {set j [set [set this](num_lbs)]} \
			    {$j < $num_lbs} {incr j} {
		    	pack forget $this.lb$j
		    }
		}
	    }
	    set num_lbs [set [set this](num_lbs)]
	    $this config -directory $directory
	    dsk_lazy
	}
    }

    method _dd_drophandler {} {
	global DragDrop tkdesk

	catch "wm withdraw $tkdesk(dd_token_window)"
	update
	set dest $directory

	if ![file writable $dest] {
	    dsk_errbell
	    if {$dest != ""} {
	    	cb_error "You don't have write permission for this directory!"
	    } else {
		cb_error "This listbox is not a valid target (since it's empty)."
	    }
	    return
	}

	#dsk_debug "Rec.: $DragDrop(file)"
	#dsk_debug "dest: $dest"
	dsk_ddcopy $DragDrop(file) $dest
    }

    method _path_popup {x y} {
	global tkdesk
	
	set m $this-ppop
	catch {destroy $m}
	menu $m -tearoff 0
	bind $m <ButtonRelease-3> "
		set tkdesk(menu,control) 0
		[bind Menu <ButtonRelease-3>]"
	bind $m <Control-ButtonRelease-3> "
		set tkdesk(menu,control) 1
		[bind Menu <ButtonRelease-3>]"

	set p [cb_tilde [string trimright $directory "/"] collapse]
	if {$p != ""} {
	    while {[set p [file dirname $p]] != "/"} {
		$m add command -label $p -command "$this config -dir $p" \
			-font $tkdesk(font,entries)
	    }
	    $m add command -label "/" -command "$this config -dir /" \
			-font $tkdesk(font,entries)
	    tk_popup $m $x $y
	}
    }

    method _button_bar {} {
	global tkdesk
	
	if {[llength $tkdesk(button_bar)] > 0} {
	    catch {
		destroy $this.fBBo
		destroy $this.fBB
	    }
	    
	    frame $this.fBBo -bd 1 -relief raised
	    pack $this.fBBo -after $this.fMenu -fill x 

	    frame $this.fBB
	    pack $this.fBB -in $this.fBBo -fill x \
		    -padx $tkdesk(pad) -pady $tkdesk(pad)

	    set bcnt 0
	    foreach but $tkdesk(button_bar) {
		if {[llength $but] == 1} {
		    frame $this.fBB.f$bcnt -width $tkdesk(pad)
		    pack $this.fBB.f$bcnt -in $this.fBB -side left \
			    -padx $tkdesk(pad) -pady $tkdesk(pad)
		    incr bcnt
		    continue
		}

		set bitmap [lindex $but 0]
		set bgcolor [. cget -background]
		set fgcolor black
		if {[llength $bitmap] > 1} {
		    if {[lindex $bitmap 1] != ""} {
			set fgcolor [lindex $bitmap 1]
		    }
		    if {[llength $bitmap] > 2} {
		    	if {[lindex $bitmap 2] != ""} {
			    set bgcolor [lindex $bitmap 2]
		    	}
		    }
		    set bitmap [lindex $bitmap 0]
		}
		if {[string index $bitmap 0] == "/" || \
		    [string index $bitmap 0] == "~"} {
		    set bitmap $bitmap
		} else {
		    foreach p [split $tkdesk(path,images) ":"] {
			if [file exists $p/$bitmap] {
			    set bitmap $p/$bitmap
			    break
			}
		    }
		}
		if ![file exists $bitmap] {
		    set bitmap $tkdesk(library)/images/xlogo16.xpm
		}

	    	set action [string_replace [lindex $but 1] \" \\\"]

		set desc ""
		if {[llength $action] > 1} {
		    set desc [lindex $action 1]
		    set action [lindex $action 0]
		}
		dsk_debug "BB$bcnt action: $action"

		button $this.fBB.b$bcnt \
			-image [cb_image $bitmap -background $bgcolor \
			-foreground $fgcolor] \
			-takefocus 0 -highlightthickness 0 \
			-activebackground $bgcolor -activeforeground $fgcolor \
			-command "cd \[dsk_active dir\] ;\
			eval \[_expand_pc \"$action\"\]; cd ~"
		pack $this.fBB.b$bcnt -in $this.fBB -side left \
			-padx 0 -pady 0 \
			-ipadx 2 -ipady 2

		if {$desc != ""} {
		    cb_balloonHelp $this.fBB.b$bcnt "$desc"
		}

		incr bcnt
	    }
	}
    }

    proc id {{cmd ""}} {
	if {$cmd == ""} {
	    set i $id
	    incr id
	    return $i
	} elseif {$cmd == "reset"} {
	    set id 0
	    set count 0
	}
    }

    #
    # ----- Variables ---------------------------------------------------------
    #

    public num_lbs 3

    public dontmap 0

    public dir {} {
	global tkdesk

	set err [catch {$this-top config}]
	if !$err {
	    set err [catch {set isdir [file isdirectory $dir]}]
	    if !$err {
		if !$isdir {
		    catch {set dir [_make_path_valid $dir]}
		    catch {dsk_bell}
		    cb_alert "The path you specified is not completely valid."
		} elseif ![file readable $dir] {
		    dsk_errbell
		    cb_error "[file tail $dir]: Permission denied."
		    return
		}
	    } else {
		cb_error "Path (or user?) does not exist. Opening home directory."
		set dir ~
	    }

	    if $tkdesk(menu,control) {
		dsk_FileList .dfl[dsk_FileList :: id] -directory $dir
		set tkdesk(menu,control) 0
	    } else {
		$this config -directory $dir
	    }
	}
    }

    public directory "/" {
	global tkdesk env

	if ![winfo exists $this] return

	#set directory "[string trimright [glob $directory] "/"]/"
	set directory "[string trimright [cb_tilde $directory expand] "/"]/"
	dsk_debug "Directory $directory"

	set strip_i 0
	if $tkdesk(strip_home) {
	    if [string match "$env(HOME)/*" $directory] {
		set strip_i [string length "$env(HOME)"]
	    }
	}

	if [info exists tkdesk(strip_parents)] {
	    foreach d $tkdesk(strip_parents) {
		set d [string trimright $d /]
		if [string match "$d/*" $directory] {
		    set strip_i [string length $d]
		    break
		}
	    }
	}

	# determine depth of directory
	set dir_depth 0
	set first_i 0
	set cmask ""
	set l [string length $directory]
	for {set i $strip_i} {$i < $l} {incr i} {
	    if {[string index $directory $i] == "/"} {
		set ndir [string range $directory 0 $i]
		if ![info exists lbdirs($dir_depth)] {
		    set lbdirs($dir_depth) $ndir
		} elseif {$ndir != $lbdirs($dir_depth)} {
		    set lbdirs($dir_depth) $ndir
		} else {
		    catch {set cmask [$this.lb$first_i cget -mask]}
		    incr first_i
		}
		incr dir_depth
	    }
	}
	#puts $cmask
	if {$first_i == $dir_depth && $first_i} {
	    set first_i [expr $dir_depth - 1]
	}
	for {set i $dir_depth} {$i < $num_lbs_created} {incr i} {
	    set lbdirs($i) ""
	}

	#
	# fill list boxes
	#
	dsk_FileListbox :: print_ready 0
	for {set i $first_i} {$i < $dir_depth} {incr i} {
	    if {$i >= $num_lbs_created} {
	        dsk_FileListbox $this.lb$i -viewer $this \
			-width $tkdesk(file_lb,minwidth) \
			-height $tkdesk(file_lb,minheight) -pad $tkdesk(pad)
	    }

	    if {$cmask != ""} {
		$this.lb$i set_mask $cmask
	    }
	    $this.lb$i config -directory $lbdirs($i)
	    if {$i > 0} {
		$this.lb[expr $i - 1] tagpath $lbdirs($i)
	    }
	}
	$this.lb[expr $dir_depth - 1] tagpath

	if {$i > $num_lbs_created} {
	    set num_lbs_created $i
	} else {
	    while {$i < $num_lbs_created} {
		$this.lb$i clear
		incr i
	    }
	}

	set flb [cb_max 0 [expr $dir_depth - $num_lbs]]
	#puts "dd: $dir_depth"
	$this _set_first_lb $flb

	# add last directory to the path history:
	if {$last_directory != ""} {
	    if {[string first $env(HOME) $last_directory] == 0} {
		history add [string_replace $last_directory $env(HOME) ~]
	    } else {
		history add $last_directory
	    }
	}
	set last_directory $directory

	# update the path entry:
	$this.ePE delete 0 end
	$this.ePE insert end [cb_tilde $directory collapse]

	wm title $this [cb_tilde $directory collapse]
	wm iconname $this [file tail [string trimright $directory "/"]]/

	#$this status "Ready. [dsk_fs_status $directory]"
	$this status "Ready."
	dsk_FileListbox :: print_ready 1
    }

    protected dir_depth 0	;# depth of the current directory
    protected first_lb 0        ;# number of leftmost listbox
    protected num_lbs_created 0	;# number of created listboxes (>= num_lbs)
    protected lbdirs		;# array of each lb's directory
    protected sb_state "packed"
    protected last_directory ""

    common count 0
    common id 0
}


