# include <stdlib.h>
# include <string.h>

# include "hfs.h"
# include "glob.h"

typedef struct {
  char *mem;
  size_t memsz;
  char **eltend;
  char *strs;
} dlist;

typedef struct {
  char *str;
  int len;
  size_t space;
  char sbuf[50];
} dstring;

/* List Routines =========================================================== */

/*
 * NAME:	listinit()
 * DESCRIPTION:	initialize a new dynamic list
 */
static
int listinit(dlist *list)
{
  list->memsz = 100;
  list->mem   = malloc(list->memsz);
  if (list->mem == 0)
    return -1;

  list->eltend = (char **) list->mem;
  list->strs   = list->mem + list->memsz;

  return 0;
}

/*
 * NAME:	listfree()
 * DESCRIPTION:	free a dynamic list
 */
static
void listfree(dlist *list)
{
  free(list->mem);
}

/*
 * NAME:	listappend()
 * DESCRIPTION:	insert a string to the end of a list
 */
static
int listappend(dlist *list, char *str)
{
  int len;

  len = strlen(str) + 1;

  /* make sure there is room */

  if (sizeof(char *) + len > list->strs - (char *) list->eltend)
    {
      dlist newlist;
      size_t strsz;
      char **elt;

      strsz = (list->mem + list->memsz) - list->strs;

      newlist.memsz = list->memsz * 2 + sizeof(char *) + len;
      newlist.mem   = malloc(newlist.memsz);
      if (newlist.mem == 0)
	return -1;

      newlist.eltend = (char **) newlist.mem;
      newlist.strs   = newlist.mem + newlist.memsz - strsz;

      memcpy(newlist.strs, list->strs, strsz);

      for (elt = (char **) list->mem; elt < list->eltend; ++elt)
	*newlist.eltend++ = newlist.strs + (*elt - list->strs);

      free(list->mem);

      *list = newlist;
    }

  list->strs -= len;
  strcpy(list->strs, str);

  *list->eltend++ = list->strs;

  return 0;
}

/* String Routines ========================================================= */

/*
 * NAME:	strinit()
 * DESCRIPTION:	initialize a new dynamic string
 */
static
void strinit(dstring *string)
{
  string->str     = string->sbuf;
  string->len     = 0;
  string->space   = sizeof(string->sbuf);
  string->sbuf[0] = 0;
}

/*
 * NAME:	strappend()
 * DESCRIPTION:	append to a dynamic string
 */
static
int strappend(dstring *string, char *str, int len)
{
  int newlen;

  if (len < 0)
    len = strlen(str);

  newlen = string->len + len;

  /* make sure there is room */

  if (newlen >= string->space)
    {
      char *new;

      newlen *= 2;

      new = malloc(newlen);
      if (new == 0)
	return -1;

      string->space = newlen;

      memcpy(new, string->str, string->len);

      if (string->str != string->sbuf)
	free(string->str);

      string->str = new;
    }

  /* append the string */

  memcpy(string->str + string->len, str, len);

  string->len += len;
  string->str[string->len] = 0;

  return 0;
}

/*
 * NAME:	strfree()
 * DESCRIPTION:	free a dynamic string
 */
static
void strfree(dstring *string)
{
  if (string->str != string->sbuf)
    free(string->str);
}

/* Glob Routines =========================================================== */

/*
 * NAME:	strmatch()
 * DESCRIPTION:	return 1 iff a string matches a given (glob) pattern
 */
static
int strmatch(char *str, char *pat)
{
  while (1)
    {
      if (*str == 0 && *pat != 0 && *pat != '*')
	return 0;

      switch (*pat)
	{
	case 0:
	  return (*str == 0);

	case '*':
	  if (*++pat == 0)
	    return 1;

	  while (1)
	    {
	      if (strmatch(str, pat))
		return 1;

	      if (*str++ == 0)
		return 0;
	    }

	case '?':
	  break;

	case '[':
	  {
	    ++pat;

	    while (1)
	      {
		unsigned char p0, p1, s;

		p0 = *pat;

		if (p0 == ']' || p0 == 0)
		  return 0;

		s = *str;

		if (hfs_charorder[p0] == hfs_charorder[s])
		  break;

		if (pat[1] == '-')
		  {
		    p1 = pat[2];

		    if (p1 == 0)
		      return 0;

		    if ((hfs_charorder[p0] <= hfs_charorder[s] &&
			 hfs_charorder[p1] >= hfs_charorder[s]) ||
			(hfs_charorder[p0] >= hfs_charorder[s] &&
			 hfs_charorder[p1] <= hfs_charorder[s]))
		      break;

		    pat += 2;
		  }

		++pat;
	      }

	    while (*pat != ']')
	      {
		if (*pat == 0)
		  {
		    --pat;
		    break;
		  }

		++pat;
	      }
	  }
	  break;

	case '\\':
	  if (*++pat == 0)
	    return 0;

	  /* fall through */

	default:
	  if (hfs_charorder[(unsigned char) *pat] !=
	      hfs_charorder[(unsigned char) *str])
	    return 0;
	}

      ++pat, ++str;
    }
}

/*
 * NAME:	doglob()
 * DESCRIPTION:	perform recursive depth-first traversal of path to be globbed
 */
static
int doglob(hfsvol *vol, dlist *list, char *dir, char *rem)
{
  dstring new;
  int special, len, result = 0;
  char *obrace, *cbrace, *ptr;

  strinit(&new);

  special = 0;
  obrace = cbrace = 0;

  for (ptr = rem; *ptr && (obrace || *ptr != ':'); ++ptr)
    {
      switch (*ptr)
	{
	case '{':
	  if (obrace == 0)
	    obrace = ptr;
	  break;

	case '}':
	  if (obrace && cbrace == 0)
	    cbrace = ptr;
	  break;

	case '\\':
	  if (*++ptr == 0)
	    --ptr;

	case '*':
	case '[':
	case '?':
	  special = 1;
	  break;
	}
    }

  if (obrace)
    {
      char *elt;

      if (cbrace == 0 ||
	  strappend(&new, rem, obrace - rem) < 0)
	{
	  strfree(&new);
	  return -1;
	}
      len = new.len;

      for (ptr = obrace; *ptr != '}'; )
	{
	  elt = ptr + 1;

	  ptr = elt;
	  while (*ptr != '}' && *ptr != ',')
	    ++ptr;

	  if (strappend(&new, elt, ptr - elt) < 0 ||
	      strappend(&new, cbrace + 1, -1) < 0 ||
	      doglob(vol, list, dir, new.str) < 0)
	    {
	      strfree(&new);
	      return -1;
	    }

	  new.len = len;
	}

      strfree(&new);
      return 0;
    }

  if (strappend(&new, dir, -1) < 0)
    {
      strfree(&new);
      return -1;
    }
  len = new.len;

  if (special)
    {
      hfsdirent ent;
      hfsdir *d;
      dstring pat;
      int found = 0;

      strinit(&pat);
      if (strappend(&pat, rem, ptr - rem) < 0)
	{
	  strfree(&pat);
	  strfree(&new);
	  return -1;
	}

      if (*dir == 0 && strchr(rem, ':') == 0)
	d = hfs_opendir(vol, ":");
      else
	d = hfs_opendir(vol, dir);

      if (d == 0)
	{
	  strfree(&pat);
	  strfree(&new);
	  return -1;
	}

      while (hfs_readdir(d, &ent) >= 0)
	{
	  if (strmatch(ent.name, pat.str))
	    {
	      new.len = len;
	      if (strappend(&new, ent.name, -1) < 0)
		{
		  result = -1;
		  break;
		}

	      if (*ptr == 0)
		{
		  found = 1;
		  result = listappend(list, new.str);

		  if (result < 0)
		    break;
		}
	      else if (ent.flags & HFS_ISDIR)
		{
		  if (strappend(&new, ":", 1) < 0)
		    result = -1;
		  else
		    {
		      found = 1;
		      result = doglob(vol, list, new.str, ptr + 1);
		    }

		  if (result < 0)
		    break;
		}
	    }
	}

      hfs_closedir(d);

      if (result == 0 && ! found)
	{
	  new.len = len;
	  if (strappend(&new, rem, -1) < 0)
	    result = -1;
	  else
	    {
	      for (rem = new.str + len, ptr = rem; *rem; ++rem, ++ptr)
		{
		  if (*rem == '\\')
		    ++rem;

		  *ptr = *rem;
		}
	      *ptr = 0;

	      result = listappend(list, new.str);
	    }
	}

      strfree(&pat);
      strfree(&new);

      return result;
    }

  if (strappend(&new, rem, ptr - rem) < 0)
    result = -1;
  else
    {
      if (*ptr)
	{
	  if (strappend(&new, ":", 1) < 0)
	    result = -1;
	  else
	    result = doglob(vol, list, new.str, ptr + 1);

	  strfree(&new);
	  return result;
	}

      result = listappend(list, new.str);
    }

  strfree(&new);
  return result;
}

/* Interface Routine ======================================================= */

/*
 * NAME:	hfs->glob()
 * DESCRIPTION:	perform glob pattern matching
 */
char **hfs_glob(hfsvol *vol, int argc, char *argv[], int *nelts)
{
  dlist list;
  int i;

  if (listinit(&list) < 0)
    return 0;

  for (i = 0; i < argc; ++i)
    {
      if (doglob(vol, &list, "", argv[i]) < 0)
	{
	  listfree(&list);
	  return 0;
	}
    }

  *nelts = list.eltend - (char **) list.mem;

  return (char **) list.mem;
}
