%{
/*
 *                            COPYRIGHT
 *
 *  PCB, interactive printed circuit board design
 *  Copyright (C) 1994 Thomas Nau
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Contact addresses for paper mail and Email:
 *  Thomas Nau, Schlehenweg 15, 88471 Baustetten, Germany
 *  Thomas.Nau@medizin.uni-ulm.de
 *
 */

static	char	*rcsid = "$Header: parse_l.l,v 1.7 94/07/18 15:17:12 nau Exp $";

/* lexical definitions to parse ASCII input of PCB and Element description
 */

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <sys/stat.h>

#ifdef _POSIX_SOURCE
#include <unistd.h>
#endif

#include "global.h"
#include "data.h"
#include "error.h"
#include "memory.h"
#include "misc.h"
#include "parse_l.h"
#include "parse_y.h"

#ifdef	DBMALLOC_INCLUDE
#include DBMALLOC_INCLUDE
#endif

/* ---------------------------------------------------------------------------
 * some shared parser identifiers
 */
#ifdef FLEX_SCANNER
int				yylineno;		/* linenumber */
#endif

char			*yyfilename;	/* in this file */
PCBTypePtr		yyPCB;			/* used by parser */
ElementTypePtr	yyElement;
FontTypePtr		yyFont;
short			ParseType;

/* ---------------------------------------------------------------------------
 * some local prototypes
 */
static	int		Parse(char *, char *, char *);
static	char	*CreateParseCommandLine(char *, char *, char *);

%}

HEX					0x[0-9a-fA-F]+
DECIMAL				[0-9]+

%%

PCB			{ return(T_PCB); }
Grid		{ return(T_GRID); }
Cursor		{ return(T_CURSOR); }
Flags		{ return(T_FLAGS); }
Layer		{ return(T_LAYER); }
Pin			{ return(T_PIN); }
Via			{ return(T_VIA); }
Line		{ return(T_LINE); }
Rectangle	{ return(T_RECT); }
Text		{ return(T_TEXT); }
ElementLine	{ return(T_ELEMENTLINE); }
ElementArc	{ return(T_ELEMENTARC); }
Element		{ return(T_ELEMENT); }
SymbolLine	{ return(T_SYMBOLLINE); }
Symbol		{ return(T_SYMBOL); }
Mark		{ return(T_MARK); }
Groups		{ return(T_GROUPS); }

\'.\'				{ yylval.number = (unsigned) *(yytext+1); return(CHAR_CONST); }
{DECIMAL}|{HEX}		{
						sscanf((char *) yytext, "%i", &yylval.number);
						return(NUMBER);
					}
\"[^\"\n\r]*\"		{
							/* return NULL on empty string */
						if (yyleng == 2)
						{
							yylval.string = NULL;
							return(STRING);
						}

							/* remove ""  from string */
						*(yytext +yyleng-1) = '\0';
						yylval.string = MyStrdup((char *) (yytext+1), "LEX");
						return(STRING);
					}
#.*					{}
[ \t]+				{}
[\n]				{
#ifdef FLEX_SCANNER
						yylineno++;
#endif
					}
[\r]				{}
.					{ return(*yytext); }

%%

/* ---------------------------------------------------------------------------
 * creates the commandline for popen()
 * %f is replaced by the filename 
 * %p by the searchpath
 */
static char *CreateParseCommandLine(char *Template, char *Filename, char *Path)
{
	static	DynamicStringType	command;
			char				*p;

		/* clear old string */
	if (command.Data)
		command.Data[0] = '\0';

	for (p = Template; *p; p++)
	{
			/* copy character if not special or add string to command */
		if (!(*p == '%' && (*(p+1) == 'f' || *(p+1) == 'p')))
			DSAddCharacter(&command, *p);
		else
		{
			DSAddString(&command, *(p+1) == 'p' ? Path : Filename);

				/* skip the character */
			p++;
		}
	}
	DSAddCharacter(&command, '\0');
	return(command.Data);
}

/* ---------------------------------------------------------------------------
 * sets up the preprocessor command
 */
static int Parse(char *Filename, char *Executable, char *Path)
{
	struct	stat	buffer;
			char	*command;
			int		returncode,
					c;

#ifdef FLEX_SCANNER
	static	Boolean	firsttime = True;
#endif

		/* check if file exists */
	if (stat(Filename, &buffer))
	{
		MyWarningDialog("file '%s' doesn't exist\n", Filename);
		return(1);
	}

		/* create commandline from template */
	command = CreateParseCommandLine(Executable, Filename, Path);

		/* open pipe to stdout of command */
	if ((yyin = popen(command, "r")) == NULL)
	{
		MyWarningDialog("can't execute command '%s'\n", command);
		return(1);
	}

		/* check if input is empty */
	if ((c = fgetc(yyin)) == EOF)
	{
		MyWarningDialog("command '%s' produced no output\n"
			"check stderr of shell", command);
		pclose(yyin);
		return(1);
	}
	ungetc(c, yyin);

#ifdef FLEX_SCANNER
		/* reset parser if not called the first time */
	if (!firsttime)
		yyrestart(yyin);
	firsttime = False;
#endif

		/* init linenumber and filename for yyerror() */
	yylineno = 1;
	yyfilename = Filename;

		/* We need to save the data temorarily because lex-yacc are able
		 * to break the application if the input file has an illegal format.
		 * It's not necessary if the system supports the call of functions
		 * on termination.
		 * The backup alarm has to be deffered because LEX (FLEX) don't like
		 * signals too much
		 */
	StopBackupAlarm();

#if !defined(HAS_ATEXIT) && !defined(HAS_ON_EXIT)
	SaveTMPData();
	returncode = yyparse();
	RemoveTMPData();
#else
	returncode = yyparse();
#endif

	ContinueBackupAlarm();
	return(pclose(yyin) ? 1 : returncode);
}

/* ---------------------------------------------------------------------------
 * initializes LEX and calls parser for a single element
 */
int ParseElement(ElementTypePtr Ptr, char *Filename)
{
		/* set flag for parser to recognize a element only */
	ParseType = PARSE_ELEMENT;
	yyElement = Ptr;
	return(Parse(Filename, Settings.ElementCommand, Settings.ElementPath));
}

/* ---------------------------------------------------------------------------
 * initializes LEX and calls parser for a complete board
 */
int ParsePCB(PCBTypePtr Ptr, char *Filename)
{
		/* set flag for parser to recognize a PCB */
	ParseType = PARSE_PCB;
	yyPCB = Ptr;
	return(Parse(Filename, Settings.FileCommand, Settings.FilePath));
}

/* ---------------------------------------------------------------------------
 * initializes LEX and calls parser for a font
 */
int ParseFont(FontTypePtr Ptr, char *Filename)
{
		/* set flag for parser to recognize a PCB */
	ParseType = PARSE_FONT;
	yyFont = Ptr;
	return(Parse(Filename, Settings.FontCommand, Settings.FontPath));
}

