/*
 * ZyXEL_2864.c
 *
 * This file contains the ZyXEL 1496/2864 specific hardware stuff.
 *
 * This file is written and maintained by
 * Martin Seine <martin@erde.gun.de>
 *
 */

#include "../include/voice.h"

char *libvoice_ZyXEL_2864_c = "$Id: ZyXEL_2864.c,v 1.8 1996/08/06 19:50:26 marc Exp $";

/* Modem type temporary hardcoded */
#ifndef ZyXEL_2864
#define ZyXEL_2864
#endif

#define M_DISTRING_INIT  "ATS40.3=1 S40.4=1 S40.5=1 S40.6=1"
#define M_VOICE_INIT          "ATS39.7=0 S39.6=1 +VIT=60"

#define M_DTMF_INIT      "AT+VDD=%d,%d"

/*
 * Here we save the current mode of operation of the voice modem when
 * switching to voice mode, so that we can restore it afterwards.
 */

static char mode_save[VOICE_BUF_LEN] = "";

/*
 * Internal status variables for aborting some voice modem actions.
 */

static int stop_dialing;
static int stop_playing;
static int stop_recording;
static int stop_waiting;

/*
 * The ZyXEL samples with 9600 samples per second with a maximum of 4 bit
 * per sample. We want to buffer voice data for 0.1 second, so we need a
 * buffer less or equal to 9600 * 0.5 * 0.1 = 480 bytes.
 */

#define ZYXEL_BUFFER_SIZE 480
static int zyxel_buffer_size = ZYXEL_BUFFER_SIZE;

/*
 * This function handles the <DLE> shielded codes.
 */

#define ST_NO_INPUT (0x00)
#define ST_GOT_DLE  (0x01)

static int modem_answer_state = ST_NO_INPUT;

static void handle_modem_answer(char new_byte)
{

     switch (modem_answer_state) {
          case ST_NO_INPUT:

               switch (new_byte) {
                    case DLE:
                         modem_answer_state = ST_GOT_DLE;
                         break;
                    case XON:
                         lprintf(L_WARN, "%s: Received XON",
                          program_name, new_byte);
                         break;
                    case XOFF:
                         lprintf(L_WARN, "%s: Received XOFF",
                          program_name, new_byte);
                         break;
                    case NL:
                    case CR:
                         break;
                    default:
                         lprintf(L_ERROR, "%s: Illegal modem answer 0x%2x",
                          program_name, new_byte);
                    };

               break;
          case ST_GOT_DLE:

               switch (new_byte) {
                    case '0':
                    case '1':
                    case '2':
                    case '3':
                    case '4':
                    case '5':
                    case '6':
                    case '7':
                    case '8':
                    case '9':
                    case '*':
                    case '#':
#ifdef ZyXEL_2864
                    case 'A':
                    case 'B':
                    case 'C':
                    case 'D':
#endif
                         voice_handle_event(RECEIVED_DTMF,
                          (event_data) new_byte);
                         break;
                    case 'c':
                         voice_handle_event(FAX_CALLING_TONE,
                          (event_data) 0);
                         break;
                    case 'e':
                         voice_handle_event(DATA_CALLING_TONE,
                          (event_data) 0);
                         break;
                    case 's':
                         voice_handle_event(NO_VOICE_ENERGY,
                          (event_data) 0);
                         break;
                    case 'q':
                         voice_handle_event(SILENCE_DETECTED,
                          (event_data) 0);
                         break;
                    case 'b':
                         voice_handle_event(BUSY_TONE,
                          (event_data) 0);
                         break;
                    case 'd':
                         voice_handle_event(DIAL_TONE,
                          (event_data) 0);
                         break;
#ifdef ZyXEL_2864
                    case 'h':
                         voice_handle_event(HANDSET_ON_HOOK,
                          (event_data) 0);
                         break;
                    case 'H':
                         voice_handle_event(HANDSET_OFF_HOOK,
                          (event_data) 0);
                         break;
                    case 'l':
                    case 'L':
                         voice_handle_event(LOOP_BREAK,
                          (event_data) 0);
                         break;
#endif
                    case 'u':
                         lprintf(L_WARN, "%s: Modem reported data underrun",
                          program_name);
                         break;
                    default:
                         lprintf(L_ERROR,
                          "%s: Unknown <DLE> shielded code 0x%x",
                          program_name, new_byte);
                    };

               modem_answer_state = ST_NO_INPUT;
               break;
     };
};

/*
 * This is the main handle event routine for the ZyXEL 1496/2864.
 */
#ifdef ZyXEL_2864
int ZyXEL_2864_handle_event(int event, event_data data)
#else
int ZyXEL_1496_handle_event(int event, event_data data)
#endif
{
     char buffer[VOICE_BUF_LEN];

     switch (event) {
          case VOICE_ANSWER_PHONE:
               return(voice_command("ATA", "VCON"));
          case VOICE_BEEP:
               sprintf(buffer, "AT+VTS=[%d,0,%d]", data.beep.frequency,
                data.beep.length);

               if (voice_command(buffer, "OK") != VMA_USER_1)
                    return(ERROR);

               return(OK);
          case VOICE_DIAL:
               {
               int result = ERROR;

               voice_modem_state = DIALING;
               stop_dialing = FALSE;
               sprintf(buffer, "ATD%s", (char*) data.p);

               if (voice_command(buffer, "") != OK)
                    return(ERROR);

               while (!stop_dialing)
                    {
                    voice_read(buffer);
                    result = voice_analyze(buffer, NULL);

                    switch (result)
                         {
                         case VMA_BUSY:
                         case VMA_FAIL:
                         case VMA_ERROR:
                         case VMA_NO_ANSWER:
                         case VMA_NO_CARRIER:
                         case VMA_NO_DIAL_TONE:
                              stop_dialing = TRUE;
                              result = ERROR;
                              break;
                         case VMA_VCON:
                              stop_dialing = TRUE;
                              result = OK;
                              break;
                         };

                    };

               voice_modem_state = IDLE;
               return(result);
               };
          case VOICE_INIT:
#ifdef ZyXEL_2864
               lprintf(L_MESG, "initializing ZyXEL 2864 voice modem");
#else
               lprintf(L_MESG, "initializing ZyXEL 1496 voice modem");
#endif
               voice_modem_state = INITIALIZING;

#ifdef ZyXEL_2864
               if(cvd.dist_ring.d.i && (voice_command(M_DISTRING_INIT,"OK")
                 != VMA_USER_1))
#else
               if ((rom_release >= 611) && cvd.dist_ring.d.i &&
                (voice_command(M_DISTRING_INIT,
                "OK") != VMA_USER_1))
#endif
                    lprintf(L_WARN, "coudn't initialize distinctive RING");

#ifdef ZyXEL_2864
               if( voice_command(M_VOICE_INIT, "OK") != VMA_USER_1)
#else
               if ((rom_release >= 611) && voice_command(
                M_VOICE_INIT, "OK") != VMA_USER_1)
#endif
                    lprintf(L_WARN, "voice init failed, continuing");

#ifndef ZyXEL_2864
               if (rom_release >= 612)
#endif
                    {
                    sprintf(buffer, M_DTMF_INIT,
                     cvd.dtmf_threshold.d.i * 31 / 100, cvd.dtmf_len.d.i
                     / 5);

                    if (voice_command(buffer, "OK") != VMA_USER_1)
                         lprintf(L_WARN,
                          "setting DTMF threshold didn't work");

                    };

#ifndef ZyXEL_2864
               if (rom_release >= 613)
#endif
                    if (voice_command("AT+FLO=2", "OK") == VMA_USER_1)
                         {
                         TIO tio;

                         tio_get(voice_fd, &tio);
                         tio_set_flow_control(voice_fd, &tio, FLOW_BOTH);
                         tio_set(voice_fd, &tio);
                         }
                     else
                         lprintf(L_WARN,
                          "can't turn on hardware flow control");

               if (voice_command("ATS48.5=0", "OK") != VMA_USER_1)
                    lprintf(L_WARN, "couldn't set 1496 compatible mode");

               voice_modem_state = IDLE;
               return(OK);
          case VOICE_MODE_OFF:
               sprintf(buffer, "AT+FCLASS=%s", mode_save);
               voice_command(buffer, "OK");
               return(OK);
          case VOICE_MODE_ON:
               voice_command("AT+FCLASS?", "");
               voice_read(mode_save);
               voice_flush(1);
               voice_command("AT+FCLASS=8", "OK");
               return(OK);
          case VOICE_PLAY_FILE:
               {
               TIO tio_save;
               TIO tio;
               char input_buffer[ZYXEL_BUFFER_SIZE];
               char output_buffer[2 * ZYXEL_BUFFER_SIZE];
               int i;
               int bytes_in;
               int bytes_out;
               int bytes_written;

               stop_playing = FALSE;
               voice_modem_state = PLAYING;
               voice_command("AT+VTX", "CONNECT");
               tio_get(voice_fd, &tio);
               tio_save = tio;

#ifdef ZyXEL_2864
               if(1)
#else
               if (rom_release >= 613)
#endif
                    tio_set_flow_control(voice_fd, &tio, FLOW_HARD |
                     FLOW_XON_OUT);
               else
                    tio_set_flow_control(voice_fd, &tio, FLOW_XON_OUT);

               tio_set(voice_fd, &tio);

               while (!stop_playing)
                    {

                    if ((bytes_in = read(data.i, input_buffer,
                     zyxel_buffer_size)) <= 0)
                         break;

                    bytes_out = 0;

                    for(i = 0; i < bytes_in; i++)
                         {
                         output_buffer[bytes_out] = input_buffer[i];

                         if (output_buffer[bytes_out++] == DLE)
                              output_buffer[bytes_out++] = DLE;

                         };

                    lprintf(L_JUNK, "%s: <DATA %d bytes>", program_name,
                     bytes_out);

                    errno = 0;
                    bytes_written = 0;

                    while (((bytes_written += write(voice_fd,
                     &output_buffer[bytes_written], bytes_out -
                     bytes_written)) != bytes_out) && (errno == 0))
                         ;

                    if (bytes_written != bytes_out)
                         lprintf(L_ERROR,
                          "%s: could only write %d bytes of %d bytes (errno 0x%x)",
                          program_name, bytes_written, bytes_out, errno);

                    while (check_for_input(voice_fd))
                         {
                         char modem_byte;

                         if (read(voice_fd, &modem_byte, 1) != 1)
                              lprintf(L_ERROR,
                               "%s: could not read byte from voice modem",
                               program_name);
                         else
                              handle_modem_answer(modem_byte);

                         };

                    };

               if (stop_playing)
                    {
                    sprintf(output_buffer, "%c%c", DLE, DC4);
                    lprintf(L_JUNK, "%s: <DLE><DC4>", program_name);
                    }
               else
                    {
                    sprintf(output_buffer, "%c%c", DLE, ETX);
                    lprintf(L_JUNK, "%s: <DLE><ETX>", program_name);
                    };

               write(voice_fd, output_buffer, strlen(output_buffer));
               tio_set(voice_fd, &tio_save);
               voice_command("", "OK|VCON"); /* OK alone should do */
               voice_modem_state = IDLE;

               if (stop_playing)
                    return(INTERRUPTED);

               return(OK);
               };
          case VOICE_RECORD_FILE:
               {
               TIO tio_save;
               TIO tio;
               char input_buffer[ZYXEL_BUFFER_SIZE];
               char output_buffer[ZYXEL_BUFFER_SIZE];
               int i;
               int bytes_in;
               int bytes_out;
               int got_DLE_ETX = FALSE;
               int was_DLE = FALSE;

               stop_recording = FALSE;
               voice_modem_state = RECORDING;
               voice_command("AT+VRX", "CONNECT");
               tio_get(voice_fd, &tio);
               tio_save = tio;

#ifdef ZyXEL_2864
               if(1)
#else
               if (rom_release >= 613)
#endif
                    tio_set_flow_control(voice_fd, &tio, FLOW_HARD |
                     FLOW_XON_IN);
               else
                    tio_set_flow_control(voice_fd, &tio, FLOW_XON_IN);

               tio_set(voice_fd, &tio);

               while (!got_DLE_ETX)
                    {

                    if ((bytes_in = read(voice_fd, input_buffer,
                     zyxel_buffer_size)) <= 0)
                         {
                         lprintf(L_ERROR,
                          "%s: could not read byte from voice modem",
                          program_name);
                         return(FAIL);
                         };

                    bytes_out = 0;

                    for (i = 0; (i < bytes_in) && !got_DLE_ETX; i++)
                         {

                         if (was_DLE)
                              {
                              was_DLE = FALSE;

                              switch (input_buffer[i])
                                   {
                                   case DLE:
                                        output_buffer[bytes_out++] = DLE;
                                        break;
                                   case ETX:
                                        got_DLE_ETX = TRUE;
                                        break;
                                   default:
                                        handle_modem_answer(DLE);
                                        handle_modem_answer(input_buffer[i]);
                                   };

                              }
                         else
                              {

                              if (input_buffer[i] == DLE)
                                   was_DLE = TRUE;
                              else
                                   output_buffer[bytes_out++] =
                                    input_buffer[i];

                              };

                         };

                    write(data.i, output_buffer, bytes_out);
                    };

               tio_set(voice_fd, &tio_save);
               voice_command("", "OK|VCON");
               voice_modem_state = IDLE;
               return(OK);
               };
          case VOICE_SET_COMPRESSION:

               switch (data.i)
                    {
                    case 1:
                         zyxel_buffer_size = ZYXEL_BUFFER_SIZE * 1 / 4;
                         voice_command("AT+VSM=1", "OK");
                         return(OK);
                    case 0:
                    case 2:
                         zyxel_buffer_size = ZYXEL_BUFFER_SIZE * 2 / 4;
                         voice_command("AT+VSM=2", "OK");
                         return(OK);
                    case 3:
                         zyxel_buffer_size = ZYXEL_BUFFER_SIZE * 3 / 4;
                         voice_command("AT+VSM=3", "OK");
                         return(OK);
                    case 30:
                         zyxel_buffer_size = ZYXEL_BUFFER_SIZE * 3 / 4;
                         voice_command("AT+VSM=30", "OK");
                         return(OK);
                    case 4:
                         zyxel_buffer_size = ZYXEL_BUFFER_SIZE * 4 / 4;
                         voice_command("AT+VSM=4", "OK");
                         return(OK);
                    };

               lprintf(L_WARN,
                "ZyXEL 2864 handle event: Illegal voice compression method (%d)",
                data.i);
               return(FAIL);
          case VOICE_SET_DEVICE:

               switch (data.i)
                    {
                    case NO_DEVICE:
                         voice_command("AT+VLS=0", "OK");
                         return(OK);
#ifdef ZyXEL_2864
                    case LOCAL_HANDSET:
                         voice_command("AT+VLS=1", "VCON");
                         return(OK);
#endif
                    case DIALUP_LINE:
                         voice_command("AT+VLS=2", "VCON");
                         return(OK);
                    case EXTERNAL_MICROPHONE:
                         voice_command("AT+VLS=8", "VCON");
                         return(OK);
                    case INTERNAL_SPEAKER:
                         voice_command("AT+VLS=16", "VCON");
                         return(OK);
                    };

               lprintf(L_WARN,
                "ZyXEL 2864 handle event: Unknown output device (%d)",
                data.i);
               return(FAIL);
          case VOICE_STOP_DIALING:
               stop_dialing = TRUE;
               return(OK);
          case VOICE_STOP_PLAYING:
               stop_playing = TRUE;
               return(OK);
          case VOICE_STOP_RECORDING:
               stop_recording = TRUE;
               voice_write("AT");
               return(OK);
          case VOICE_STOP_WAITING:
               stop_waiting = TRUE;
               return(OK);
          case VOICE_SWITCH_TO_DATA_FAX:
               sprintf(buffer, "AT+FCLASS=%s", (char *) data.p);
               return(voice_command(buffer, "OK"));
          case VOICE_WAIT:
               {
               stop_waiting = FALSE;
               voice_modem_state = WAITING;
               alarm(data.i);

               while (!stop_waiting)
                    {

                    while (check_for_input(voice_fd))
                         {
                         char modem_byte;

                         if (read(voice_fd, &modem_byte, 1) != 1)
                              lprintf(L_ERROR,
                               "%s: could not read byte from voice modem",
                               program_name);
                         else
                              handle_modem_answer(modem_byte);

                         };

                    delay(100);
                    };

               voice_modem_state = IDLE;
               alarm(0);
               return(OK);
               };
          };

#ifdef ZyXEL_2864
     lprintf(L_WARN, "ZyXEL 2864 handle event: Unknown event %04x", event);
#else
     lprintf(L_WARN, "ZyXEL 1496 handle event: Unknown event %04x", event);
#endif
     return(FAIL);
     };
