/* buffer.cc
   Definitions of classes for holding files

    Author: John Collins, collins@phys.psu.edu.
    21 Jan 96

    (C) John Collins & Penn State University.
*/


#include "buffer.h"
#include <string.h>


// Null string.  Use pointer to it to avoid NULL pointers to strings.  
char Blank[] = "";
Line DummyLine;
Posn Nowhere(&DummyLine);


/*===========================================================
    class Line
    Designed to hold one of a linked set of lines of text.
*/


void Line::Init() {
   /* To be called only when setting up the buffer, to set the elements
      to sensible defaults.
   */
   prev = next = NULL;
   alloc = len = LineNum = 0;
   data = MYNULL;
}


Line::Line(){
   /* Constructs an empty unattached line.*/
   Init();
}

Line::Line(Line *&last, char *s){
   /* Constructs a line attached to the line last, if it exists.
      Adjusts all the pointers, and updates last to point to this.
      Initializes the data to be equal to the string.
   */
   Init();
   Attach(last);
   data = strdup (s);
   if (data){
      len = strlen (s);
      alloc = len + 1;
   } else {
      //?? This is probably an error to arrive here, e.g., lack of memory??
      lite_cerr << "Out of memory in making a Line.\n";
      alloc = len = 0;
      data = MYNULL;
   }
}


Line::Line(Line *&last, unsigned int size, char *s){
   /*  The same except the number of characters is defined, so that
       the string is not necessarily zero terminated.
       Remember to allow for 0 at end of data:
   */
   Init();
   Attach(last);
   data = new char [size+1];
   if (data){
      len = size;
      alloc = len + 1;
      strncpy (data, s, len);
      data[len] = 0;
   } else {
      //?? It is probably an error to arrive here, e.g., lack of memory??
      lite_cerr << "Out of memory in making a Line.\n";
      alloc = len = 0;
      data = MYNULL;
   }
}


Line::~Line(){
   Remove();
   Clear();
}


void Line::Remove() {
   //Remove from list, and adjust links of rest of list.
   if (prev) (prev -> next) = next;
   if (next) (next -> prev) = prev;
   prev = NULL;
   next = NULL;
}


void Line::Clear() {
   //Clear the data and delete its space
   if (alloc > 0) delete data;
   alloc = len = 0;
   data = MYNULL;
}


void Line::Attach(Line *&last) {
   /*  Attach to a list after the element last.
       Set last to point to this.
       Ignore values of prev and next.
   */
   prev = last;
   last = this;
   if (prev) {
      next = (prev->next);
      (prev->next) = this;
   } else {
      // There was no list to point to.  I am the sole element of the list.
      next = NULL;
   }
   if (next) {
      (next->prev) = this;
   }
}




/*======================================================
   class Buffer
*/


Buffer::Buffer(){
   /* Create an empty Buffer.
   */
   Init();
}


//Buffer::Buffer(istream & source){
Buffer::Buffer(FILE *source){
   /* Create the Buffer by reading from the file, which is assumed to be
      open for reading.  Leave the file open after reading it.
      See method Read for specifications of the reading.
   */
   Init();
   ErrorCode = Read (source);
}


Buffer::~Buffer() {
   // It's sufficient to delete all the data:
   Clear();
}


void Buffer::Clear() {
   /* Delete all the data in the Buffer.
   */
   while (first) {
      Line *next = first->next;
      delete first;
      first = next;
   }
}



//int Buffer::Read (istream &source) {
int Buffer::Read (FILE *source) {
   /* Fill the Buffer from the file source, which is assumed to be open
      for reading.  Append the results to any previous contents of the
      Buffer.  Start at the current position of the file, and leave the
      file open after the read, positioned at the end.

      Put one line of the file in each line of the Buffer.
      But if a line of the file is too long, then it is to span two or
      more lines of the Buffer.  New line characters are included at the
      end of each line of the Buffer when they are in the file.
      A future version will handle all of the UNIX, MSDOS and Macintosh
      conventions for the end of line character(s).  Note the MSDOS files
      will be OK on UNIX, but there will be an extra ^M character at the
      end of each line.

      Put a NULL character at the end of each Line.

      Return 0 for success or a non-zero error code for failure.
   */

   char *tmpbuf = new char[BUFSIZE];

   NumLastRead = 0;

   if (!tmpbuf) {
      lite_cerr << "I couldn't allocate a Buffer while reading a file.\n";
      return 1;
   }

   Line *last = NULL;
   int RetCode = 0; // My default error code: success.

   while (!feof(source)){
      tmpbuf[BUFSIZE-3] = 0;
      // This will be filled by a long read;
      fgets (tmpbuf, BUFSIZE-2, source);
      if (feof(source)) goto ExitPoint;
      /* Comments for streams case
      // That didn't get the delimiter, if any.
      if (!source.fail()) {
         int NumCh = source.gcount();
         char NextCh = source.get();
         if (NextCh == '\n') {
            tmpbuf[NumCh] = NextCh;
            NumCh++;
            tmpbuf[NumCh] = 0;
         } else {
            source.putback(NextCh);
            lite_cerr <<
               "In Buffer::Read:\n"
               "Line " << NumLastRead << " had no newline.  "
               "Perhaps this is not a text file." << endl;
         }
       } else if (source.bad()){
         lite_cerr  <<
            "Error while reading a file after line " << NumLastRead << endl;
         RetCode = 3;
         goto ExitPoint;
       } else {
         //Failure without error.  Is that only on end-of-file??
         //lite_cerr  <<
         //   "Failure without error while reading a file after line " << NumLastRead << endl;
         //RetCode = 4;
         //goto ExitPoint;
       }
      */
      if (ferror(source)){
          lite_cerr  <<
            "Error while reading a file after line " << (int)NumLastRead << endl;
          RetCode = 3;
          goto ExitPoint;
      }
      if ( (tmpbuf[BUFSIZE-3] != 0) && (tmpbuf[BUFSIZE-3] != '\n') ) {
         lite_cerr <<
            "In Buffer::Read:\n"
            "Line " << (int)NumLastRead << " had no newline.  "
            "Perhaps this is not a text file." << endl;
         tmpbuf[BUFSIZE-2] = '\n';
         tmpbuf[BUFSIZE-1] = 0;
      }
      new Line (last, tmpbuf);
      if (first == NULL) {
         // The list was empty, before the last line was read.
         first = last;
      }
      if (last->alloc == 0) {
         /* I must have run out of memory.  There was an error message,
            so there's no need to make a new one.  The calling routine
            will have to take action.
         */
         RetCode = 2;
         last->LineNum = NumLastRead + 1;
         goto ExitPoint;
      }
      /* Update NumLastRead only after we have succeeded in reading
         the line.
      */
      NumLastRead++;
      last->LineNum = NumLastRead;
   }

ExitPoint:
   delete tmpbuf;
   return RetCode;
}




int Buffer::Write (sos &dest) const {
   /* Write the Buffer into the file dest, which is assumed to be open
      for writing.  Put one line of the file in each line of the Buffer.
      Start at the current position of the file, and leave the
      file open after the write, positioned at the end.
      Just put the characters in the Buffer into the file.

      Assumes the lines are null-terminated.

      A future version will handle all of the UNIX, MSDOS and Macintosh
      conventions for the end of line character(s).

      Return 0 for success or a non-zero error code for failure.
   */
   for (Line *l = first; l; l = (l->next)){
      dest.write (l->data, l->len);
      if (dest.bad()) {
         lite_cerr << "Error while writing a file.\n";
         return 1;
       }
   }
   return 0;
}


unsigned int Buffer::NumLines() const {
   /* Number of lines in the Buffer.
      A future version will take into account the vagaries of incomplete
      lines.
   */
   unsigned int n = 0;
   for (Line *l = first; l; l = (l->next)) n++;
   return n;
}



/*======================================================
  class Posn
  Position of character in a line.
*/

void Posn::Show(sos &f) const {
   // Show line # and position
   f << "Char. " << (int)(index+1) << " in line " << LineNum();
}


void Posn::ShowContext(sos &f) const{
  if (l) {
     f << "In line " << LineNum() << "\n";
     f << l->data;
     for (int i = 0; i < index; i++)
       f << ' ';
     f << "^\n";
  }
}

int Posn::LineNum() const {
  if (l)
     return l->LineNum;
  else
     return 0;
}


char Posn::operator[](int offset) const {
   /* Return the character offset positions beyond the current position.
      Assume offset >=0, as for array.  (We can relax that later.)
   */
   if (offset < 0) {
      lite_cerr << "Error: Posn::[] with negative argument." << endl;
      return 0;
   }
   char c = 0;
   Line *curline = l;
   int i = index;
   while (curline && (offset + i >= curline->len )) {
      offset -= curline->len - i;
      curline = curline->next;
      i = 0;
   }
   if (curline && (l->len > 0) )
      c = curline->data[i+offset];
   return c;
}

void Posn::inc() {
   // Skip to next character
   if (index + 1 < l->len) {
      // There's still at least one character on this line:
      index ++;
   } else if (!l->next) {
      // I am at the end-of-file. Ensure index points at the ending null:
      index = l->len;
   } else {
      // I am off the end of a line.  Find the next non-empty line, if any.
      index = 0;
      do {
         l = l->next;
      } while ((l->next) && (l->len == 0));
   }
}


void Posn::dec() {
   /* Skip to previous character.
      If I am at the beginning of the file, stay put.
   */
   if (index > 0) {
      index --;
   } else if (!(l->prev)) {
      // Stay put, I am at the beginning of the file.
   } else {
      do {// Go back to the nearest non-empty line
         l = l->prev;
      } while ( (l->prev) && (l->len == 0));
      if (l->len>0)
         index = l->len -1;
      else
         index = 0;
   }
}

void Error (const char *s, const Posn & where) {
   // Show error message and context on stderr
   if (where.l)
      ShowWhere (where, stderr);
   fputs (s, stderr);
   fputc ('\n', stderr);
   if (where.l)
      ShowContext (where, stderr);
}


void Error (const char *s) {
   // Show error message on stderr
   fputs (s, stderr);
   fputc ('\n', stderr);
}


void ShowContext (const Posn &where, FILE *f) {
// Show line contents
  if (where.l) {
     fputs (where.l->data, f);
     for (int i = 0; i < where.index; i++)
       fputc (' ', f);
     fputs ("^\n", f);
  } else {
     fputs ("No specific line.\n", stderr);
  }
}


void ShowWhere (const Posn &where, FILE *f) {
// Show line number and 1-based index.
  if (where.l) {
     fprintf (f, "[%i:%il]", where.l->LineNum, where.index+1, f);
  } else {
     fprintf (f, "[0:0]");
  }
}


