package autoupdate::rpm;  # $Date: 2003/0/12 $

use strict;
# warnings causes problems under 5.005
#use warnings;
use POSIX;
use autoupdate::general;

BEGIN {
	use Exporter   ();
	use vars qw($VERSION $DEBUG @ISA @EXPORT @EXPORT_OK);

	# set the version for version checking
	$VERSION     = "1.5.4";
	@ISA         = qw(Exporter);
	@EXPORT      = qw(&CallRPM &CheckRPMVersion &SetArch &SetDistArch &GetArch &GetDistArch
			&GetAllowedArch &GetAllowedDistArch &TestArch
			&SetLocalRPMs &AddLocalRPMs &CheckLocal &CompareLocal &GetLocal &GetInstalled
			&SelectRPMs &LoadRpmBindings &TieRpmDatabase);
	@EXPORT_OK   = qw();
}
END {};

use vars qw($DEBUG $RpmModule $UseRpmVerCmp $RpmRoot $UsePipe $AddInstalled $RpmNoDigest $RpmArrayBug);

$DEBUG = 0;
$UsePipe = 0;
$AddInstalled = 0;
$RpmModule = 0;
$UseRpmVerCmp = 0;
$RpmRoot = "";
$RPM::err = "unknown rpm error";
#Don't compute digest when querying rpm's
$RpmNoDigest = 0;
#Some older versions of rpm segfault when the array of a tag is empty
$RpmArrayBug = 0;

my $RPM = "/bin/rpm";
my $RPMLIB = "/usr/lib/rpm/";
my $RPMDB = "/var/lib/rpm/";
my $TmpFile = "autoupdate_$$.tmp";
if ($ENV{'TMP'}) {
	$TmpFile = "$ENV{'TMP'}/$TmpFile";
} else {
	$TmpFile = "/tmp/$TmpFile";
}
# Tags to query an rpm for
my %TAGS = (
	'NAME' => 0,
	'VERSION' => 0,
	'RELEASE' => 0,
	'ARCH' => 0,
	'EPOCH' => 0,
	'VENDOR' => 0,
	'SOURCERPM' => 0,
	'PROVIDENAME' => 1,
	'PROVIDEFLAGS' => 1,
	'PROVIDEVERSION' => 1,
	'REQUIRENAME' => 1,
	'REQUIREFLAGS' => 1,
	'REQUIREVERSION' => 1,
	'OBSOLETENAME' => 1,
	'OBSOLETEFLAGS' => 1,
	'OBSOLETEVERSION' => 1 );
# Build up the query format from %TAGS
my $QueryFormat= "";

if (-x $RPM) {
	my @RPMOut;
	unless (CheckRPMVersion("3.0.3") ge 0) {
		$RpmArrayBug= 1;
	}
	if (CheckRPMVersion("4.1") ge 0) {
		$RpmNoDigest= 1;
	}
	if (CallRPM(\@RPMOut, "--querytags") == 0) {
		foreach my $tag (keys %TAGS) {
			unless (IsElement($tag, @RPMOut)) {
				print "AUPM: tag $tag not supported by rpm.\n" if $DEBUG;
				next;
			}
			if ($TAGS{$tag}) {
				$QueryFormat .= "[$tag:%{$tag}\\n]" unless ($RpmArrayBug);
			} else {
			$QueryFormat .= "$tag:%{$tag}\\n";
			}
		}
		if (!IsElement('EPOCH', @RPMOut) and IsElement('SERIAL', @RPMOut)) {
			$QueryFormat .= "EPOCH:%{SERIAL}\\n";
		}
		for my $tag ( qw(PROVIDE OBSOLETE) ) {
			if (!IsElement("${tag}NAME", @RPMOut) and IsElement("${tag}S", @RPMOut)) {
				$QueryFormat .= "[${tag}NAME:%{${tag}S}\\n${tag}FLAGS:\\n${tag}VERSION:\\n]" unless ($RpmArrayBug);
			}
		}
	} else {
		print STDERR "AUPM: Could not call rpm!\n" if $DEBUG;
	}
}

my $TieRpmDatabase = 0;
my $Arch;
my $DistArch;
my $RPMVersion;
my $RPM_DB;
my %LocalRPMs;
my %LocalRPMsArchs;
my @LocalTied;
my @LocalMissing;
my %RPM;

######################
# Calls RPM and saves all output
######################

sub CallRPM
{
	my $retval;
	my $rpmoutput= shift;
	# If a daemon gets restarted in the post script, the pipe
	# doesn't get closed until the daemon exists under some
	# older versions of rpm. Hence using a pipe is not safe. 

	my @progs= ($RPM);
	if ($_[0] eq "-U" and -x $RPMLIB . "rpmu") {
		push(@progs, $RPMLIB . "rpmu");
	} elsif ($_[0] eq "-i" and -x $RPMLIB . "rpmi") {
		push(@progs, $RPMLIB . "rpmi");
	} elsif ($_[0] =~ /^-q/ and -x $RPMLIB . "rpmq") {
		push(@progs, $RPMLIB . "rpmq");
	}
	
	foreach my $prog (@progs) {
		if ($UsePipe) {
			pipe(R, W);
		}

		my $pid = fork;
		die("Could not fork for RPM: $!") unless defined($pid);
		if(!$pid) {
			if ($UsePipe) {
				close(R);
			} else {
				# Just to be sure
				if (-e $TmpFile) {
					unlink($TmpFile) || die("Could not remove $TmpFile");
				}
				sysopen(W, $TmpFile, O_WRONLY|O_CREAT|O_EXCL, 00600) || die("Could not create: $TmpFile ($!)");
			}
			open(STDERR, ">&W") || die("Could not dup STDERR ($!)");
			open(STDOUT, ">&W") || die("Could not dup STDOUT ($!)");
			exec($prog, @_) || die("Could not exec $prog.");
			close(W);
			exit 1;
		} else {
			if ($UsePipe) {
				close(W);
			}	
			waitpid($pid,0);
			$retval= $?;
			unless ($UsePipe) {
				open(R,"< $TmpFile") || die("Could not open: $TmpFile ($!)");
			}
			@{$rpmoutput}= <R>;
			close(R);
			unless ($UsePipe) {
				unlink($TmpFile);
			}
			chomp(@{$rpmoutput});
			unless ($retval == 11) { #Segmentation fault
				return $retval;
			}
		}
	}
	return $retval;

}

######################
# Query an rpm using CallRPM
######################

sub QueryRPM {
	my @RPMOut;
	my $type = shift;
	my $item = shift;
	my @flags;
	if ($type eq "l") {
		@flags= ( "-q" );
		if ($RpmRoot) {
			push (@flags, "--root", $RpmRoot);
		}
	} else {
		@flags= ( "-qp" );
	}
	push (@flags, "--nodigest", "--nosignature") if ($RpmNoDigest);
	# Initialize arrays
	my %HDR = ();
	for my $tag (keys %TAGS) {
		if ($TAGS{$tag}) {
			$HDR{$tag} = [];
		} else {
			$HDR{$tag} = "";
		}
	}

	unless ( CallRPM(\@RPMOut, @flags, "--queryformat", $QueryFormat, $item) == 0 ) {
		$RPM::err= $RPMOut[0];
		return ();
	}
	for my $line (@RPMOut) {
		$line =~ s/^\(none\)//;
		next unless ($line =~ /^([A-Z]+):(.*)$/);
		my ($tag, $key) = ($1, $2);
		next unless ( defined $TAGS{$tag} );
		if ($TAGS{$tag}) {
			push(@{$HDR{$tag}}, $key);
		} else {
			if ($key eq '(none)') {
				$HDR{$tag}= "";
			} else {
				$HDR{$tag}= $key;
			}
		}	
	}
	return %HDR;
}

######################
# Check for a certain RPM version
######################

sub CheckRPMVersion {
	my $ReqVersion= shift;

	unless ($ReqVersion =~ /-/) {
		$ReqVersion .= "-0";
	}
	unless ($RPMVersion) {
		my @RPMOut;
		CallRPM(\@RPMOut, "-q", "rpm" );
		$RPMVersion="0.0-0";
		foreach my $item (@RPMOut) {
			if ($item =~ /^rpm-(\d+\..*)$/) {
				$RPMVersion= $1;
				last;
			}
		}
	}
	my $rpml= Header->new("rpm-$RPMVersion.noarch.rpm", "r");
	my $rpmr= Header->new("rpm-$ReqVersion.noarch.rpm", "r");
 
	return $rpml->compare($rpmr);
}

##################################
# Class for RPM header information
##################################

{ package Header;
use autoupdate::general;

sub new {
	my $proto = shift;
	my $class = ref($proto) || $proto;
	my $self  = {};
	my $file = "";
	my $path = "";
	my $name = "";
	my $version = "";
	my $release = "1";
	my $epoch = "";
	my $arch = "noarch";
	my $vendor = "";
	my $type = "r";
	my $hdr = "";
	return undef unless ($_[0]);
	$file = shift;
	chomp($file);
	if ($_[0] and $_[0] =~/^[flr]$/) {
		$type= $_[0];
	} elsif ( -f $file ) {
			$type= "f";
	}

	if ( $file =~ /^(.*\/)([^\/]+)$/ ) {
		$path = $1;
		$file = $2
	}
	if ($file =~ /^(.*)-([^-]+)-([^-]+)\.([^.]+)\.rpm$/) {
		$name= $1;
		$version= $2;
		$release= $3;
		$arch= $4;
	}
	if ($version =~ /^(\d+):(.+)$/) {
		$epoch= $1;
		$version=$2;
	} elsif ($version =~ /^\(none\):(.+)$/) {
		$version=$1;
	}

	$self->{NAME} = $name;
	$self->{EPOCH} = $epoch;
	$self->{VERSION} = $version;
	$self->{RELEASE} = $release;
	$self->{ARCH} = $arch;
	$self->{VENDOR} = $vendor;
	if ($type eq 'l') {
		$self->{FILENAME} = $name . "-" . $version . "-" . $release;
	} else {
		$self->{FILENAME} = "$path$file";
	}
	$self->{TYPE} = $type;
	$self->{HDR} = $hdr;
	bless ($self, $class);

	unless($name) {
		if ($type eq "f") {
			$hdr= $self->getheader();
			if ($hdr) {
				return $self;
			} else {
				return undef;
			}
		} elsif ($file =~ /^(.*)-(\d[^-]*)-(\d[^-]*)\.rpm$/) {
			$name= $1;
			$version= $2;
			$release= $3;
		} elsif ($file =~ /^(.*)-(\d[^-]*)\.(\D[^.]+)\.rpm$/) {
			$name= $1;
			$version= $2;
			$arch= $3;
		} elsif ($file =~ /^(.*)-(\d[^-]*)\.rpm$/) {
			$name= $1;
			$version= $2;
		} else {
			return undef;
		}
		$self->{NAME} = $name;
		$self->{VERSION} = $version;
		$self->{RELEASE} = $release;
		$self->{ARCH} = $arch;
	}
	if ($name =~ /^(.+)-[^-]+$/) {
		$self->{BASENAME}= $1;
	} else {
		$self->{BASENAME}= $name;
	}

	return $self;
}

sub name {
	my $self = shift;
	return $self->{NAME};
}

sub version {
	my $self = shift;
	return $self->{VERSION}."-".$self->{RELEASE};
}

sub arch {
	my $self = shift;
	return $self->{ARCH};
}

sub basename {
	my $self = shift;
	return $self->{BASENAME};
}

sub rpmname {
	my $self = shift;
	return $self->{NAME}."-".$self->{VERSION}."-".$self->{RELEASE}.".".$self->{ARCH}.".rpm";
}

sub filename {
	my $self = shift;
	if ($_[0]) {
		$self->{FILENAME} = shift;
	}
	return $self->{FILENAME};
}

sub getheader {
	my $self = shift;
	my $hdr;

	if ($self->{HDR}) {
		return $self->{HDR};
	}
	if ($self->{TYPE} eq "r") {
		return undef;
	}

	if ($self->{TYPE} eq "l") {
		if ($autoupdate::rpm::RpmModule == 1 and ! $autoupdate::rpm::RpmRoot) {
			&autoupdate::rpm::TieRpmDatabase(1);
			$hdr= $RPM{$self->name};
		} elsif ($autoupdate::rpm::RpmModule == 2) {
			&autoupdate::rpm::TieRpmDatabase(1);
			my $i = $RPM_DB->find_by_name_iter($self->name);
			unless ($i) {
				$RPM::err= "Could not create package iterator.";
				$hdr= undef;
			} else {
				while ($hdr= $i->next) {
					last if ( $hdr->version eq $self->{VERSION} and $hdr->release eq $self->{RELEASE});
				}
			}
		} else {
			my %HDR = &autoupdate::rpm::QueryRPM("l", $self->filename);
			$hdr= \%HDR if ( $HDR{'NAME'} );
		}
		if ($hdr) {
			push(@LocalTied, $self);
		}
	} elsif ($self->{TYPE} eq "f") {
		$RPM::err = "Failed to query";
		if ($autoupdate::rpm::RpmModule == 3) {
			tie my %HDR, "RPM::Perlonly", $self->filename;
		} elsif ($autoupdate::rpm::RpmModule == 2) {
			eval { $hdr = RPM2->open_package( $self->filename, $autoupdate::rpm::RpmNoDigest ); }
		} elsif ($autoupdate::rpm::RpmModule == 1) {
			$hdr = RPM::Header->new( $self->filename );
		} else {
			my %HDR = &autoupdate::rpm::QueryRPM("f", $self->filename);
			$hdr= \%HDR if ( $HDR{'NAME'} );
		}
	}
	unless ($hdr) {
		my $file= $self->filename;
		$RPM::err =~ s/error:/rpm error:/;
		if ( $RPM::err =~ /\Q$file/) {
       			print STDERR "Error: $RPM::err\n";
		} else {
       			print STDERR "Error: $file: $RPM::err\n";
		}
		$self->{TYPE} = "r"; # Don't try again
		return undef;
	}
	$self->{HDR} = $hdr;
	foreach my $tag ( qw(NAME EPOCH VERSION RELEASE ARCH VENDOR) ) {
		$self->{$tag} = $self->gettag($tag);
	}
	my $tmp= $self->gettag("SOURCERPM");
	if ($tmp and $tmp =~ /^(.*)-[^-]+-[^-]+\.src\.rpm$/) {
		$self->{BASENAME} = $1;
	}
	return $hdr;
}

sub gettag {
	my $self = shift;
	my $tag = shift;
	my $hdr= $self->{HDR};
	unless ($hdr) {
		$hdr= $self->getheader();
		return undef unless ($hdr);
	}
	
	if ($autoupdate::rpm::RpmModule == 2) {
		$tag= lc($tag);
		return $hdr->tag($tag);
	} else {
		$tag = uc($tag);
		my $tmp = $hdr->{$tag};
		return undef unless (defined $tmp);
		if (ref($tmp) eq "ARRAY") {
			return @{$tmp};
		} else {
			return $tmp;
		}
	}
}

sub files {
	my $self = shift;
	my ($flags, @files);

	if ($self->{TYPE} eq "r") {
		return ();
	}
	if ($autoupdate::rpm::RpmModule==0 or ($autoupdate::rpm::RpmModule==3 and $self->{TYPE} eq "l")) {
		if ($self->{TYPE} eq "l") {
			$flags= "-q";
		} else {
			$flags= "-qp";
		}
		unless ( &autoupdate::rpm::CallRPM(\@files, $flags, "--list", $self->filename) ==0 ) {
			return ();
		} else {
			return @files;
		}
	}

	my @base_names = $self->gettag('BASENAMES');
	return () unless ($base_names[0]);
	my @dir_names = $self->gettag('DIRNAMES');
	my @dir_indexes = $self->gettag('DIRINDEXES');

	foreach (0 .. $#base_names) {
		push (@files, $dir_names[$dir_indexes[$_]] . $base_names[$_]);
	}

	return @files;
}

sub provides {
	return decode_flags($_[0], "PROVIDE");
}

sub requires {
	return decode_flags($_[0], "REQUIRE");
}

sub obsoletes {
	return decode_flags($_[0], "OBSOLETE");
}

sub decode_flags {
	my $self = shift;
	my $item = shift;
	my @ret= ();
	my @name= $self->gettag("${item}NAME");
	return () unless ($name[0]);
	my @flag= $self->gettag("${item}FLAGS");
	my @vers= $self->gettag("${item}VERSION");
	for (my $i=0; $i< scalar(@name); $i++) {
		my $ret = $name[$i];
		next if (!$ret or $ret =~ /^rpmlib\(.*\)$/);
		my $flag = $flag[$i];
		if ($flag) {
			$flag = $flag & 14;
		}
		if ($flag) {
			$ret .= " ";
			$ret .= "<" if ($flag & 2);
			$ret .= ">" if ($flag & 4);
			$ret .= "=" if ($flag & 8);
			$ret .= " " . $vers[$i];
		}
		push(@ret, $ret) unless IsElement($ret, @ret);
	}

	return @ret;
}

#
# Version comparison code
#

sub compare {
	return 0 unless ($_[0]);
	my $rpma = shift;
	return 1 unless ($_[0]);
	my $rpmb = shift;

	my $a = $rpma->{NAME};
	my $b = $rpmb->{NAME};
	if ($a ne $b) { return ($a cmp $b); }
	if ($rpma->version eq $rpmb->version) { return 0; }
	if ($rpma and $rpmb and ($autoupdate::rpm::RpmModule==1 or $autoupdate::rpm::RpmModule==2)) {
		$a = $rpma->getheader();
		$b = $rpmb->getheader();
		if ($a and $b) {
			#Compare using rpm
			my $tmp;
			if ($autoupdate::rpm::RpmModule == 2) {
				$RPM::err = "Failed to compare ($a,$b)";
				$tmp= $a->compare($b);
			} else {
				$tmp= $a->cmpver($b);
			}
			if (defined($tmp)) {
				# rpm's internal version comparison != ordering!?
				# Try to compare the other way round
				unless ($tmp == 0) {
					my $tmpp;
					if ($autoupdate::rpm::RpmModule == 2) {
						$tmpp= $b->compare($a);
					} else {
						$tmpp= $b->cmpver($a);
					}
					if ($tmp == $tmpp) { # the result is useless
						$tmp= undef;
					}
				}
			} else {
				print STDERR "Error: $RPM::err\n";
			}
			if (defined($tmp)) {
				if ($tmp < 0) {
					$tmp = -1;
				} elsif ($tmp > 0) {
					$tmp = 1;
				}
				return $tmp;
			}
		}
	}
	if ($rpma->{TYPE} ne "r" and !$rpma->{HDR}) {
		$rpma->getheader();
	}
	if ($rpmb->{TYPE} ne "r" and !$rpmb->{HDR}) {
		$rpmb->getheader();
	}

	$a = $rpma->{EPOCH};
	$a = 0 unless ($a or $rpma->{TYPE} eq "r");
	$b = $rpmb->{EPOCH};
	$b = 0 unless ($b or $rpmb->{TYPE} eq "r");
	if ($a ne "" and $b ne "" and $a ne $b) { return ($a <=> $b)};
	$a = $rpma->{VERSION};
	$b = $rpmb->{VERSION};

	if ($a eq $b) {
		#Versions are equal, need to compare the release
		my $ra= $rpma->{RELEASE};
		my $rb= $rpmb->{RELEASE};
		# Save some time
		if ($ra eq $rb) { return 0 };

		unless ($autoupdate::rpm::UseRpmVerCmp and $autoupdate::rpm::RpmModule == 2) {
			# Rename 1fix is newer than 1
			if ( $ra =~ /^${rb}fix(ed)?$/i) { return 1 };
			if ( $rb =~ /^${ra}fix(ed)?$/i) { return -1 };
			#We only take the numeric part if present
			if ( $ra=~/^(.*\d)[^.\d]*$/ ) {
				$a=$1;
			} else {
				$a=$ra;
			}
			if ( $rb=~/^(.*\d)[^.\d]*$/ ) {
				$b= $1;
			} else {
				$b=$rb;
			}
		}
	}
	if ($autoupdate::rpm::UseRpmVerCmp and $autoupdate::rpm::RpmModule == 2) {
			#Compare using rpm
			$RPM::err = "Failed to compare ($a, $b)";
			my $tmp = RPM2::rpmvercmp($a, $b);
			if (defined($tmp)) {
				# rpm's internal version comparison != ordering!?
				# Try to compare the other way round
				unless ($tmp == 0) {
					my $tmpp = RPM2::rpmvercmp($b, $a);
					if ($tmp == $tmpp) { # the result is useless
						$tmp= undef;
					}
				}
			} else {
				print STDERR "Error: $RPM::err\n";
			}
			if (defined($tmp)) {
				if ($tmp < 0) {
					$tmp = -1;
				} elsif ($tmp > 0) {
					$tmp = 1;
				}
				return $tmp;
			}

	}
	return VersionCompare($a, $b);
}

} #END package Header


##############################
# Subs for arch
##############################

# Determine Arch
sub SetArch {
	if ($_[0]) {
		$Arch = shift;
		if ( $Arch eq "SRPMS" ) {
			$Arch= "src";
		}
	} else {
		$Arch=(uname())[4];
		if ($Arch eq "alpha") {
			if ( open(FILE,"</proc/cpuinfo") ) {
				my $item;
				foreach $item (<FILE>) {
					next unless ($item =~ /cpu model.*ev6/i);
					$Arch="alphaev6";
					last;
				}
			close(FILE);
			}
		} elsif ($Arch eq "i686") {
			if ( open(FILE,"</proc/cpuinfo") ) {
				my $item;
				foreach $item (<FILE>) {
					next unless ($item =~ /model name.*(athlon|duron)/i) ;
					$Arch="athlon";
					last;
				}
			close(FILE);
			}
		} elsif ($Arch =~ /00.+00/) { # AIX returns a long system id string like "000734CF4C00"
			$Arch= "ppc";
		}
	}
}

sub GetArch {
	unless ($Arch) { &SetArch };
	return $Arch;
}

# Determine DistArch
sub SetDistArch {
	if ($_[0]) {
		$DistArch = shift;
		if ( $DistArch eq "SRPMS" ) {
			$DistArch= "src";
		}
	} else {
		&SetArch unless ($Arch);
		if ( IsElement($Arch, qw(i486 i586 i686 athlon)) ) {
			$DistArch= "i386";
		} elsif ($Arch eq "sparc64") {
			$DistArch= "sparc";
		} elsif ($Arch eq "alphaev6") {
			$DistArch= "alpha";
		} else {
			$DistArch= $Arch;	
		}
	}
}

sub GetDistArch {
	unless ($DistArch) { &SetDistArch };
	return $DistArch;
}

# Determine allowedarch
sub GetAllowedArch {
	unless ($Arch) { &SetArch };
	# Safe default
	my @allowedarch= ($Arch, "noarch");

	if ( ($Arch eq "src") ) {
		@allowedarch= qw(src);
	} elsif ( ($Arch eq "i386") or ($Arch eq "i486") ) {
		@allowedarch= qw(i386 noarch);
	} elsif ( $Arch eq "i586" ) {
		@allowedarch= qw(i586 i386 noarch);
	} elsif ( $Arch eq "i686" ) {
		@allowedarch= qw(i686 i586 i386 noarch);
	} elsif ( $Arch eq "athlon" ) {
		@allowedarch= qw(athlon i686 i586 i386 noarch);
	} elsif ( $Arch eq "x86_64" ) {
		@allowedarch= qw(x86_64 athlon i686 i586 i386 noarch);
	} elsif ( $Arch eq "ia64" ) {
		@allowedarch= qw(ia64 i686 i586 i386 noarch);
	} elsif ( $Arch eq "sparc64" ) {
		@allowedarch= qw(sparc64 sparc noarch);
	} elsif ( $Arch eq "alphaev6" ) {
		@allowedarch= qw(alphaev6 alpha noarch);
	}
	return @allowedarch;
}

# Determine AllowedDistArch
sub GetAllowedDistArch {
	unless ($DistArch) { &SetDistArch };
	# Safe default
	my @allowedarch= ($DistArch, "noarch");

	if ($DistArch eq "src") {
		@allowedarch= qw(src);
	} elsif ( IsElement($DistArch, qw(i386 i486 i586 i686 athlon)) ) {
		@allowedarch= qw(athlon i686 i586 i386 noarch);
	} elsif ( IsElement($DistArch, qw(sparc sparc64)) ) {
		@allowedarch= qw(sparc64 sparc noarch);
	} elsif ( IsElement($DistArch, qw(alpha alphaev6)) ) {
		@allowedarch= qw(alphaev6 alpha noarch);
	}
	return @allowedarch;
}

# Test if arch is known
sub TestArch {
	&SetArch unless ($Arch);
	if ( IsElement($Arch, qw(src i386 i486 i586 i686 athlon ia64 x86_64 sparc sparc64 ppc alpha alphaev6 ia64)) ) {
		return 0;
	}
	print STDERR "Warning: Unknown architecture: $Arch.\n" if $_[0];
	print STDERR "Guessing valid architectures: $Arch noarch.\n" if $_[0];
	return 1;
}

######################
# Sets the hash of local RPMs
######################

sub SetLocalRPMs
{
	print "AUPM: Resetting local rpms.\n" if $DEBUG;
	%LocalRPMs= ();
	%LocalRPMsArchs= ();
	TieRpmDatabase(0);
	if ($_[0] and $_[0] eq "l") {
		print "AUPM: Adding installed rpms.\n" if $DEBUG;
		TieRpmDatabase(1);
		$AddInstalled= 1;
	} else {
		$AddInstalled= 0;
	}
}

######################
# Add a local rpm exists
######################

sub AddLocalRPMs
{
	my ($item, $tmp, $name, $res, $loc);
	my $type = "";
	if ($_[0] =~ /^[flr]$/) {
		$type= shift;
	}
	foreach $item ( @_ ) {
		$tmp = Header->new($item,$type);
		next unless $tmp;
		$name = $tmp->name;
		if ( CheckLocal($name) ) {
			$loc= $LocalRPMs{$name};
			$res= $tmp->compare($loc);
			if ($res ==0 and ($tmp->version eq $loc->version)) {
				push(@{$LocalRPMsArchs{$name}}, $tmp->arch); 
			}
			next if ($res < 1);
		}
		$LocalRPMs{$name}= $tmp;
		$LocalRPMsArchs{$name}= [ $tmp->arch ];
	}
}

######################
# Check if a local rpm exists
######################

sub CheckLocal
{
	my $name= shift;
	my $hdr= $LocalRPMs{$name};
	if ($AddInstalled and !defined $hdr) {
		$hdr= GetInstalled($name);
	}
	return 0 unless ($hdr);
	return $hdr->version;
}

######################
# Get the header of a local rpm
######################

sub GetLocal
{
	my $name= shift;
	my $hdr= $LocalRPMs{$name};
	if ($AddInstalled and !defined $hdr) {
		$hdr= GetInstalled($name);
	}
	return $hdr;
}

######################
# Compare an rpm against the local version
######################

sub CompareLocal
{
	my $hdr= GetLocal($_[0]->name);
	return 1 unless ($hdr);
	return $_[0]->compare($hdr);
}

######################
# Load the preferred perl module for rpm
######################

sub LoadRpmBindings {
	if ($RpmModule) {
		return $RpmModule;
	}
	$RpmModule = $_[0];

	my @modules;
	if ($RpmModule eq "0") {
		return $RpmModule;
	} elsif ($RpmModule eq "1") {
		@modules = ( "RPM::Header" );
	} elsif ($RpmModule eq "2") {
		@modules = ( "RPM2 0.62" );
	} elsif ($RpmModule eq "3") {
		@modules = ( "RPM::Perlonly" );
	} else {
		$RpmModule= 1;
		@modules = ( "RPM::Header 0.40", "RPM2 0.62", "RPM::Perlonly" );
	}
	foreach my $module (@modules) {
		print "AUPM: Loading $module module: " if $DEBUG;
		eval "use $module";
		unless ($@) {
			print "ok\n" if $DEBUG;
			last;
		}
		print "failed\n" if $DEBUG;
		$RpmModule++;
	}
	if ($@) { 
		warn('Failed to load RPM Module') if ($_[0]);
		$RpmModule= 0;
	}
	if ($RpmModule eq "2" and RPM2->rpm_api_version > 4.0 ) {
		$RpmNoDigest= RPM2->vsf_nosha1header | RPM2->vsf_nomd5header |
				RPM2->vsf_nosha1 | RPM2->vsf_nomd5 |
				RPM2->vsf_nodsa | RPM2->vsf_norsa;
	}
	return $RpmModule;
}

######################
# Tie/Untie the rpm database
######################

sub TieRpmDatabase {
	return 0 if ($_[0] eq $TieRpmDatabase);
	$TieRpmDatabase=0;
	unless ($_[0]) {
		print "AUPM: Closing rpm db.\n" if $DEBUG;
		foreach my $item (@LocalTied) {
			$item->{HDR}= "";
			if ($LocalRPMs{$item->name} and $LocalRPMs{$item->name} eq $item) {
				delete $LocalRPMs{$item->name};
			}
		}
		foreach my $item (@LocalMissing) {
			if ($LocalRPMs{$item} and $LocalRPMs{$item} ne "") {
				delete $LocalRPMs{$item};
			}
		}
		@LocalTied= ();
		@LocalMissing= ();
		if ($RpmModule == 1) {
			untie(%RPM);
		} elsif ($RpmModule == 2) {
			#$RPM_DB->close_rpm_db();
			$RPM_DB= undef;
		}
		return 0;
	}
	print "AUPM: Opening rpm db.\n" if $DEBUG;
	if ($RpmModule == 1) {
		unless ( tie(%RPM, "RPM::Database") ) {
			print STDERR "Fatal: Could not tie rpm database.\n";
			die ($RPM::err);
		}		
	} elsif ($RpmModule == 2) {
		my %db= ();
		if ($RpmRoot) {
			%db= ( -path => $RpmRoot . $RPMDB );
		}
		unless ( $RPM_DB = RPM2->open_rpm_db(%db) ) {
			die("Fatal: Could not tie rpm database.");
		}
	}
	$TieRpmDatabase=1;
	return 0;
}

######################
# Get a header from the rpm database
######################

sub GetInstalled {
	my $hdr;
	my $name = shift;

	&TieRpmDatabase(1);
	if ($RpmModule == 2) {
		my $i = $RPM_DB->find_by_name_iter($name);
		unless ($i) {
			$RPM::err= "Could not create package iterator.";
			return undef;
		}
		while (my $tmp = $i->next) {
			if (!$hdr or $tmp->compare($hdr) == 1) {
				$hdr= $tmp;
			}
		}
	} elsif ($RpmModule == 1) {
		$hdr= $RPM{$name};
	} else {
		my @RPMOut;
		if ( CallRPM(\@RPMOut, "-q", "--queryformat", "%{NAME}-%{EPOCH}:%{VERSION}-%{RELEASE}.%{ARCH}.rpm\n", $name) == 0 ) {
			foreach my $item ( @RPMOut ) {
				my $tmp = Header->new($item, "l");
				next unless $tmp;
				$name = $tmp->name;
				next if ($tmp->compare($LocalRPMs{$name}) < 1);
				$LocalRPMs{$name}= $tmp;
				$LocalRPMsArchs{$name}= [ $tmp->arch ];
			}
			return $LocalRPMs{$name};
		} else {
			# Not installed
			$hdr= undef;
		}
	}
	unless ($hdr) {
		$RPM::err= "Local header not found.";
		$LocalRPMs{$name}= ""; # Don't try again
		push(@LocalMissing, $name);
		return undef;
	}
	my $tmp;
	if ($RpmModule == 2) {
		$tmp= $hdr->name . "-" . $hdr->version . "-" . $hdr->release . "." . $hdr->arch . ".rpm";
	} else {
		$tmp= $hdr->{NAME} . "-" . $hdr->{VERSION} . "-" . $hdr->{RELEASE} . "." . $hdr->{ARCH} . ".rpm";
	}
	$tmp= Header->new($tmp, "l");
	$tmp->{HDR}= $hdr;
	$tmp->{EPOCH}= $tmp->gettag("EPOCH");
	$tmp->{VENDOR}= $tmp->gettag("VENDOR");
	$LocalRPMs{$name}= $tmp;
	$LocalRPMsArchs{$name}= [ $tmp->arch ];
	push(@LocalTied, $tmp);
	$hdr=$tmp;

	return $hdr;
}

######################
# Select rpms from a given array of rpms
# Returns array of selected rpms 
######################

sub SelectRPMs
{
	my $ref= shift;
	my @rpms =();
	my %Version=();
	my %Test=();
	my ($item, $rpm, $name, $arch, $rver, $dver, $ldver, $tmp, $res, $loc);
	my $BestMatch=1;
	my $CheckLocal=0;

	if (defined $_[0]) {
			$BestMatch= shift;
			if (defined $_[0]) {
				$CheckLocal= shift;
			}
	}

	# Get a list of allowed architectures
	my @AllowedArch;
	if ($BestMatch ==1 ) {
		@AllowedArch= GetAllowedArch();
	} else {
		@AllowedArch= GetAllowedDistArch();
	}
	print "AUPM: Selecting rpms (best=$BestMatch,local=$CheckLocal,arch=`@AllowedArch`):\n" if $DEBUG;
	# BestMatch=1  => Selects best arch if more than one is available
	# BestMatch=0  => Selects all arch if more than one is available
	# CheckLocal=0 => Do not check against local version
	# CheckLocal=1 => Exclude rpms with older or no local version
	# CheckLocal=2 => Exclude rpms with older local version

	# Loop through all rpms
	# Remember latest version in hash
	foreach $rpm ( @{$ref} ) {
		print "AUPM: " . $rpm->rpmname . " " if $DEBUG;
		if ( defined $Test{$rpm->rpmname} ) {
			print "(already present)\n" if $DEBUG;
			next;
		} else {
			$Test{$rpm->rpmname}= 1;
		}		
		unless ( IsElement($rpm->{ARCH}, @AllowedArch) ) {
			print "(wrong arch)\n" if $DEBUG;
			next;
		}
		$name=$rpm->name;
	
		if ($Version{$name}) {
			$tmp = $rpm->compare($Version{$name}->[0]);
		} elsif ( $CheckLocal ) { # Check against local version
			unless ( CheckLocal($name) or $CheckLocal == 2 ) {
				print "(no local version)\n" if $DEBUG;
				next;
			}
			$loc= $LocalRPMs{$name};
			$res= $rpm->compare($loc);
			if ( $res == -1 or ($res == 0 and ($BestMatch or ($rpm->version ne $loc->version) )) ) {
				print "(have local version)\n" if $DEBUG;
				next;
			} elsif ($res ==0 and ($rpm->arch eq $loc->arch or IsElement($rpm->arch, @{$LocalRPMsArchs{$name}}))) {
				print "(have local version)\n" if $DEBUG;
				next;
			}
			$tmp =1;
		} else {
			$tmp =1;
		}

		if ( $tmp == 1 ) { # Newer then what we currently have
			print "(new)\n" if $DEBUG;
			$Version{$name}= [ $rpm ];
		} elsif ( $tmp == 0) { # Same then what we currently have
			print "(same)\n" if $DEBUG;
			push(@{$Version{$name}}, $rpm);
		} else {
			print "(old)\n" if $DEBUG;
		}
	}

	if ($BestMatch) { #Select best match
		foreach $name (keys %Version) {
			my @tmp= @{$Version{$name}};
			next if (@tmp == 1);

			#Select best version
			# Try to match the nonnumeric part of the release
			# This usually stands for the distribution
			$tmp= $LocalRPMs{$name};
			if ( $tmp and $tmp->{RELEASE} =~ /^[^-]+-.*\d([^.\d]*)$/ ) {
				$ldver= $1;
			} else {
				$ldver= "";
			}
			$rpm= $tmp[0];
			($rver, $dver)= ($rpm->{RELEASE}, "");
			if ( $rver =~/^([^-]+-.*\d)([^.\d]*)$/ ) {
				($rver, $dver)= ($1, $2);
			}
			foreach $item ( @tmp ) {
				if ( $item->{RELEASE} eq "$rver$ldver" ) {
					$rpm= $item;
					last;
				} elsif ( $item->{RELEASE} eq $rver ) {
					$rpm= $item;
					last;
				}
			}

			@{$Version{$name}}= ();
			foreach $item ( @tmp ) {
				if ( $item->{RELEASE} eq $rpm->{RELEASE} ) {
					push(@{$Version{$name}}, $item);
				}
			}

			#Select best arch
			$rpm= undef;
			foreach $arch ( @AllowedArch ) { # Array ordered by preference!
				last if ($rpm);
				foreach $item ( @{$Version{$name}} ) {
					next unless ( $item->{ARCH} eq $arch );
					$rpm= $item;
					last;
				}
			}
			$Version{$name}= [ $rpm ];
		}
	}

	foreach $name (sort keys %Version) {
		push @rpms, @{$Version{$name}};
	}
	return @rpms;
}


1;

__END__

=head1 NAME

autoupdate::rpm

=head1 SYNOPSIS

 $rpm = Header->new("/path/foo-1.0-1.i386.rpm");
 $rpm->name
 $rpm->version
 $rpm->arch
 $rpm->rpmname
 $rpm->filename
 $rpm1->compare($rpm2)

 &SetArch &SetDistArch &TestArch
 &GetArch &GetDistArch &GetAllowedArch &GetAllowedDistArch
 &SetLocalRPMs &AddLocalRPMs &CheckLocal &CompareLocal &GetLocal
 &SelectRPMs &LoadRpmBindings &TieRpmDatabase

=head1 DESCRIPTION

This module implements an (stripped down) rpm header object plus some functions to
manipulate them.

=head2 Header objects

A header object can be crated with

 $rpm = Header->new("/path/foo-1.0-1.i386.rpm");

from a file name. An optional argument "f", "r", or "l" can
be used to specify the type.

 "f(ile)" the rpm corresponds to an existing file and can be
         queried using the rpm bindings.
 "l(ocal)" the rpm corresponds to a locally installed rpm
 "r(emote)" the rpm corresponds to a remote file which cannot
         be queried

The string shouold be of the form "[/path/]name-[epoch:]version-release.arch.rpm".

Methods implemented are

 $rpm->name     returns the rpm name (e.g.: foo)
 $rpm->version  returns the rpm version (e.g.: 1.0-1)
 $rpm->arch     returns the rpm architecutre (e.g.: i386)
 $rpm->rpmname  returns the full rpm name (e.g.: foo-1.0-1.i386.rpm)
 $rpm->filename returns the full file name (e.g.: /path/foo-1.0-1.i386.rpm)

 $rpm1->version($rpm2) compares the versions of $rpm1 and $rpm2.

 The remaining methods only work if rpm bindings are used

 $rpm->provides  like 'rpm -q --requires'
 $rpm->obsoletes like 'rpm -q --obsoletes'
 $rpm->requires  like 'rpm -q --requires'
 $rpm->files     like 'rpm -ql'
 $rpm->gettag('TAG') retrieve a tag from an rpm header

Each object contains the following date:

 $rpm->{NAME}     name tag of the rpm
 $rpm->{EPOCH}    epoch tag of the rpm
 $rpm->{VERSION}  version tag of the rpm
 $rpm->{RELEASE}  release tag of the rpm
 $rpm->{ARCH}     arch tag of the rpm
 $rpm->{FILENAME} the full file name
 $rpm->{TYPE}     the type (f|l|r)
 $rpm->{HDR}      pointer to the header object returned by the rpm bindings

=head2 Functions

First of all there the following functions related to architectures defined:

 &CallRPM(\@Out, @Args)    Calls rpm with the given arguments and saves the output
                           in @Out. Returns the exit code of rpm.
 &CheckRPMVersion($ver)    Check if rpm is newer than $ver.

 &SetArch([$string])       Sets the architecture to the given value or (if none is
                           given) tries to determine it from the current machine.
 &SetDistArch([$string])   Same for distribution architecture
 &GetArch                  Returns the architecture
 &GetDistArch              Returns the distribution architecture
 &GetAllowedArch           Returns an array of allowed architectures
 &GetAllowedDistArch       Returns an array of allowed architectures
 &TestArch                 Checks if the arhitecture is known

 &SetLocalRPMs([l])        Resets the hash of local rpms. Adds all installed rpms if
                           the first argument is "l".                           
 &AddLocalRPMs(@rpms)      Adds file names to the hash of local rpms
 &CheckLocal($name)        Checks if an rpm is in the hash of local rpms and returns
                           its version
 &CompareLocal($rpm)       Compares an rpm against the local version and returns the
                           result
 &GetLocal($name)          Returns the header of the newest local rpm with
                           the given name
 &GetInstalled($name)      Returns the header of the newest installed rpm with
                           the given name

 &SelectRPMs(\@rpms, 0|1, 0|1|2) Selects rpms from the given array by comparing them
                           to the local hash. Only the rpm with the highest version
                           are selected. If the second argument is 1, only the rpm is
                           selected for each version which matches best. If the third
                           argument is 1 only rpms which are newer than the local one
                           are selected, if it is 2 only rpms which are newer than the
                           local one or for which no local version exists are selected.

 &LoadRpmBindings(0|1|2|3) Load the preferred RPM bindings:
                           0 = automatic, 1 = RPM, 2 = RPM2, 3= RPM::Perlonly 
 &TieRpmDatabase(0|1)      Tie, untie the local rpm data base.
 
=head1 SEE ALSO

perl(1), rpm(8), RPM(3)

=head1 AUTHOR

Gerald Teschl

=head1 LICENSE

GPL

=cut
