package autoupdate::dld;  # $Date: 2003/06/12 $

use strict;
# warnings causes problems under 5.005
#use warnings;
use POSIX;
use File::Copy;
use autoupdate::general;

BEGIN {
	use Exporter   ();
	use vars qw($VERSION $DEBUG @ISA @EXPORT @EXPORT_OK);

	# set the version for version checking
	$VERSION     = "1.2";
	@ISA         = qw(Exporter);
	@EXPORT      = qw();
	@EXPORT_OK   = qw();
}
END {};

##################################
# Class for remote site
##################################

sub new {
	my $proto = shift;
	my $class = ref($proto) || $proto;
	my $self  = {
	Debug => 0,
	DebugFTP => 0,
	DebugSFTP => 0,
	DebugLWP => 0,
	Verbose => 1,
	Warnings => 0,
	Protocol => 'file',
	User => '',
	Pass => '',
	Host => 'localhost',
	Port => 0,
	Dirs => [],
	Passive => 0,
	FTPRetry => 0,
	FTPWait => 10,
	Firewall => '',
	FirewallType => 0,
	LWPKeepAlive => 1,
	LWPProtocols => [ "http", "https" ],
	Proxy => '',
	ProxyUser => '',
	ProxyPass => '',
	NoProxy => [],
	HTTPExt => [ "html", "htm", "php", "asp" ],
	SSHCipher => '',
	SSHArgs => [],
	Recursive => 0,
	MaxRecursive => 100,
	Timeout => '',
	NoWarnDirs => [],
	DirInclude => [],
	FileInclude => [ '^[^.]' ] };

	return undef unless ($_[0]);

	my %args= @_;
	if (exists $args{'Debug'}) {
		$self->{'Debug'}= $args{'Debug'};
		delete $args{'Debug'}
	}
	if (exists $args{'LWPProtocols'}) {
		my @tmp= @{$args{'LWPProtocols'}};
		$self->{'LWPProtocols'}= \@tmp;
		delete $args{'LWPProtocols'}
	}
	if (exists $args{'URL'}) {
		my ($host, $dir);
		my $URL= $args{'URL'};
		delete $args{'URL'};
		if ($URL=~ /^(\w+):\/\/(.*)$/) {
			$self->{'Protocol'}= $1;
			$URL=$2;
		} elsif ($URL=~ /^file:(.*)$/i) {
			$self->{'Protocol'}= "file";
			$URL=$1;
		}
		if ($self->{'Protocol'} eq "file") {
			$host= "localhost";
			$dir= $URL;
		} else {
			($host, $dir)= ($URL=~/([^\/]+)(.*)/);
			if ($host=~/^(.+)\@(.+)$/) {
				my $user= $1;
				$host= $2;
				if ($user=~/^(\w*):(.*)$/) {
					$self->{'User'}=  $1;
					$self->{'Pass'}=  $2;
				} else {
					$self->{'User'}=  $user;
				}
			}
		}
		$self->{'Host'}= $host;
		$self->{'Dirs'}= [ $dir ];
	}
	foreach my $item (keys %args) {
		if (exists $self->{$item}) {
			$self->{$item}= $args{$item};
		} else {
			print "DLD: Unknown option: $item\n" if $self->{'Debug'};
		}
	}
	
	$self->{'Protocol'} = lc($self->{'Protocol'});
	unless ($self->{'Protocol'} =~ /^[a-z0-9]+$/) {
		print STDERR "Error: Bad protocol: \'" . $self->{'Protocol'} . "\'\n";
		return undef;
	}
	$self->{'Connected'}= 0;
	$self->{'RetGen'} = 0,
	$self->{'DldResult'} = '',
	$self->{'Handle'}= '';

	if ( $self->{'Host'} =~ /^(.*):(\d+)$/ ) {
		$self->{'Host'}= $1;
		$self->{'Port'}= $2;
	}

	unless ($self->{'Port'}) {
		# Check for port
		my $Port= 0;
		if ($self->{'Protocol'} eq "sftp") {
			$Port= getservbyname("ssh", "tcp");
		} else {
			$Port= getservbyname($self->{'Protocol'}, "tcp");
		}

		unless ($Port) { #Just to be sure
			$Port= 0;
			$Port= 80  if ($self->{'Protocol'} eq "http");
			$Port= 443 if ($self->{'Protocol'} eq "https");
			$Port= 21  if ($self->{'Protocol'} eq "ftp");
			$Port= 22  if ($self->{'Protocol'} eq "sftp");
		}
		$self->{'Port'}= $Port;
	}
	if ( IsElement($self->{'Protocol'}, @{$self->{'LWPProtocols'}}) ) {
		$self->{'UseLWP'}= '1';
	} else {
		$self->{'UseLWP'}= '0';
	}
	for my $item ( qw(http https ftp file) ) { # Make sure we can follow e.g. ftp links
		push (@{$self->{'LWPProtocols'}}, $item) unless (IsElement($item, @{$self->{'LWPProtocols'}}));
	}

	bless ($self, $class);
	return $self;
}

sub open {
	my $self= shift;
	my $Host= $self->{'Host'};
	my $Protocol= $self->{'Protocol'};

	if ( $self->{'UseLWP'} ) {
		# Set up LWP

		##############################
		# Subpackage for LWP proxy support
		##############################

		{
			package AutoUpdateUserAgent;
			use vars qw(@ISA $VERSION);
			require LWP::UserAgent;

			@ISA = qw(LWP::UserAgent);
			$VERSION = "1.0";

			sub new
			{
				my $self = LWP::UserAgent::new(@_);
				$self->agent($VERSION);
				$self->{'ProxyUser'} = "";
				$self->{'ProxyPass'} = "";

				$self;
			}

			sub set_proxy_credentials
			{
				my $self = shift;
				$self->{'ProxyUser'} = shift;
				$self->{'ProxyPass'} = shift;
			}

			sub get_basic_credentials
			{
				my $self = shift;
				return ($self->{'ProxyUser'}, $self->{'ProxyPass'});
			}
		}

		my $LWPua = AutoUpdateUserAgent->new( keep_alive => $self->{'LWPKeepAlive'} );
		unless (defined $LWPua) {
			return 0;
		}

		$self->{'Handle'}= $LWPua;
		$self->{'Connected'}= 1;
		if ($self->{'Proxy'}) {
			$LWPua->proxy($self->{LWPProtocols}, $self->{'Proxy'});
			$LWPua->no_proxy(@{$self->{'NoProxy'}});
			if ($self->{'ProxyUser'} && $self->{'ProxyPass'}) {
				$LWPua->set_proxy_credentials($self->{'ProxyUser'}, $self->{'ProxyPass'});
			} else {
				$LWPua->env_proxy();
			}
		}
		if ($self->{'Timeout'}) {
			$LWPua->timeout($self->{'Timeout'});
		}
	} elsif ($Protocol eq "file") {
		$self->{'Connected'}= 1;
	} elsif ($Protocol eq "ftp") {
		require Net::FTP;
		my $FTPRetry= $self->{'FTPRetry'};
		my $ftp;

		unless ($FTPRetry >= 0) {
			$FTPRetry= 0;
		}
		$FTPRetry++;
		my %args = (Debug => $self->{'DebugFTP'},
			Passive => $self->{'Passive'},
			Firewall => $self->{'Firewall'},
			FirewallType => $self->{'FirewallType'},
			Port => $self->{'Port'});
		if ($self->{'Timeout'}) {
			$args{'Timeout'}= $self->{'Timeout'};
		}
		while ( ! ($ftp = Net::FTP->new($Host, %args)) ) {
			$FTPRetry--;
			if ($FTPRetry <= 0) {
				print STDERR "Error: Failed to connect to $Host.\n";
				$@ =~ s/Net::FTP/FTP errror/;
				print STDERR "$@\n";
				return 0;
			} else {
				print "Could not connect to $Host. Will try ". $FTPRetry ." more time(s).\n" if $self->{'Verbose'};
				$@ =~ s/Net::FTP/FTP errror/;
				print "$@\n" if $self->{'Debug'};
				sleep $self->{'FTPWait'};
			}
		}

		while ( ! $ftp->login($self->{'User'}, $self->{'Pass'}) ) {
			$FTPRetry--;
			if ($FTPRetry <= 0) {
				print STDERR "Error: Failed to login at $Host: ", $ftp->message;
				return 0;
			} else {
				print "Could not login to $Host. Will try ". $FTPRetry ." more time(s).\n" if $self->{'Verbose'};
				print "FTP message: ", $ftp->message if $self->{'$Debug'};
				sleep $self->{'FTPWait'};
			}
		}

		$ftp->binary();
		$self->{'Handle'}= $ftp;
		$self->{'Connected'}= 1;
	} elsif ( $Protocol eq "sftp" ) {
		require Net::SFTP;
		my @ssh_args= ();
		my %sftp_args= ();
		if ($self->{'User'}) {
			$sftp_args{'user'} = $self->{'User'};
		}
		if ($self->{'Pass'}) {
			$sftp_args{'password'} = $self->{'Pass'};
		}
		if ($self->{'DebugSFTP'}) {
			$sftp_args{'debug'} = $self->{'DebugSFTP'};
		}
		if ($self->{'Port'}) {
			push(@ssh_args, port => $self->{'Port'});
		}
		if ($self->{'SSHCipher'}) {
			push(@ssh_args, cipher => $self->{'SSHCipher'});
		}
		if ($self->{'SSHArgs'}) {
			$sftp_args{'ssh_args'} = $self->{'SSHArgs'};
		}
		my $sftp= Net::SFTP->new($Host, %sftp_args);
		unless ($sftp) {
			print STDERR "Error: Failed to connect to $Host\n";
			return 0;
		}
		$self->{'Handle'}= $sftp;
		$self->{'Connected'}= 1;
	} else {
		print STDERR "Error: Unsupported protocol: $Protocol://$Host\n";
	}

	return $self->{'Connected'};
}

sub close {
	my $self= shift;

	if ($self->{'UseLWP'}) {
		# do nothing
	} elsif ($self->{'Protocol'} eq "ftp") {
		$self->{'Handle'}->quit;
	}
	$self->{'Connected'}= 0;
	$self->{'Handle'}= '';
	return 0;
}

sub list {
	my $self= shift;
	$self->{'DldResult'}= '';
	my @files= @_;
	@files= @{$self->{'Dirs'}} unless (@files);
	return undef unless (@files);
	if ($self->{UseLWP}) {
		return $self->_List_Files_lwp(@files);
	} elsif ($self->{Protocol} eq 'file') {
		return $self->_List_Files_file(@files);
	} elsif ($self->{Protocol} eq 'ftp') {
		return $self->_List_Files_ftp(@files);
	} elsif ($self->{Protocol} eq 'sftp') {
		return $self->_List_Files_sftp(@files);
	}
	return ();
}

sub get {
	my $self= shift;
	my $file= shift; # remote path
	my $name= shift; # local file name
	my $time= shift; # time stamp
	my $rtime= 0;
	my $Protocol= $self->{'Protocol'};

	return undef unless $file;

	unless ($name) {
		$name= $file;
		if ($file =~ /\/([^\/]+)$/ ) {
			$name = $1;
		}
	}

	if ($self->{'UseLWP'}) {
		unless ($self->_Get_File_lwp($file, $name, $time)) {
			if ($self->{'DldResult'} =~ /^304/) {
				print "  $file: unmodified\n" if $self->{'Debug'};
				return 0;
			}
			print STDERR "Error: Failed to get: $file (". $self->{'DldResult'} . ")\n";
			return 1;
		}
	} elsif ($Protocol eq "file") {
		my $rtime= (stat($file))[9];
		if ($time and $rtime and $time >= $rtime) {
			print "  $file: unmodified\n" if $self->{'Debug'};
			return 0;
		}
		unless (copy($file, $name)) {
			print STDERR "Error: Failed to copy: $file ($?)\n";
			return 1;
		}
	} elsif ($Protocol eq "ftp") {
		my $ftp= $self->{'Handle'};
		$rtime= $ftp->mdtm($file);
		if ($time and $rtime and $time >= $rtime) {
			print "  $file: unmodified\n" if $self->{'Debug'};
			return 0;
		}
		my $lsize= 0;
		my $rsize= 0;
		if ( -f $name ) {
			$lsize= (stat($name))[7];
			$rsize= $ftp->size($file);
			if ( $rsize and $lsize < $rsize ) {
				print "  Resuming incomplete download.\n" if ($self->{'Verbose'});
			} else {
				unlink($file);
				$lsize= 0;
			}	
		}
		my $tmp;
		eval { $tmp= $ftp->get($file, $name, $lsize); };
		if ($@ or !$tmp) {
			my $msg;
			print STDERR "Error: Failed to ftp: ";
			$self->_get_ftp_message;
			if ($@ =~ /timeout/i ) {
				$msg= "Timeout";
			} else {
				$self->_get_ftp_message;
				$msg= $self->{'DldResult'};
			}
			if ($msg !~ /\Q$file/) {
				print STDERR "$file: ";
			}
			if ($msg) {
				print STDERR "$msg\n";
			} else {
				print STDERR "Timeout (?)\n";
			}
			if ($msg =~ /timeout/i) {
				$ftp->quit;
				$self->{'Connected'}= 0;
			}
			if ($msg =~ /connection closed/i) {
				$self->{'Connected'}= 0;
			}
			return 1;
		}
	} elsif ($Protocol eq "sftp") {
		my $sftp= $self->{'Handle'};
		$rtime= $sftp->do_stat($file)->atime;
		if ($time and $rtime and $time >= $rtime) {
			print "  $file: unmodified\n" if $self->{'Debug'};
			return 0;
		}
		unless (defined $sftp->get($file, $name)) {
			print STDERR "Error: Failed to get: $file\n";
			return 1;
		}
	} else {
		print STDERR "Error: Unsupported protocol: $Protocol\n";
		return 1;
	}

	# Preserve modification time
	utime (time, $rtime, $name) if $rtime;

	return 0;
}

# PRIVATE SUBS


######################
# Add file to filelist
######################

sub _my_push
{
	my $self= shift;
	my $file= shift;
	my $files= shift;
	my $maxrecursive= shift;
	my @patterns;
	if (defined $maxrecursive) {
		$maxrecursive += $self->{'MaxRecursive'};
		@patterns= @{$self->{'DirInclude'}};
	} else {
		@patterns= @{$self->{'FileInclude'}};
	}	
	
	# Count the number of subdirs
	if ($maxrecursive and $file =~ tr@/@@ > $maxrecursive ) {
		print STDERR "Warning: Maximum number of subdirs exceeded for: $file\n" if $self->{'Warnings'};
		return 0;
	}

	unless (@patterns) {
		push (@{$files}, $file) unless (IsElement($file, @{$files}));
		return 1;
	}
	foreach my $re (@patterns) {
		if ($file =~ /$re/) {
			push (@{$files}, $file) unless (IsElement($file, @{$files}));
			return 1;
		}
	}
	return 0;
}

######################
# Should we issue a warning
######################

sub _my_warn
{
	my $self= shift;
	my $dir= shift;
	my $warn= 1;

	foreach my $item (@{$self->{'NoWarnDirs'}}) {
		if ($dir eq $item) {
			$warn= 0;
			last;
		}
	}

	return 0 unless ($self->{'Debug'} or $warn);

	my $result= $self->{'DldResult'};
	print STDERR "Error: " . $self->{'Host'} . ": Failed to check: ";
	if ($result) {
		if ($result !~ /\Q$dir/) {
				print STDERR "$dir ($result)\n";
		} else {
			print STDERR "$result\n";
		}
	} else {
		print STDERR "$dir\n";
	}
	$self->{'RetDld'}= 1;

	return 1;
}

######################
# Get a list of remote files (lwp)
######################

sub _List_Files_lwp
{
	my $self= shift;
	my @rfiles = @_;
	my ($dir, $file, $rfile, @filelist);
	my $Host= $self->{'Host'};
	my $Protocol= $self->{'Protocol'};
	my @files= ();

	print "  Getting list of remote files (" . $self->{'Protocol'}. "):\n" if $self->{'Debug'};

	foreach $file (@rfiles) {
		$file="/$file" unless ($file =~ /^\//);
		print "    $file\n" if $self->{'Debug'};
		@filelist= $self->_Get_File_lwp($file);
		$rfile= shift(@filelist);
		unless ($rfile) {
			$self->_my_warn($file);
			next;	
		}
		$file= $rfile;
		if ($self->{'Protocol'} eq "ftp") {
			$dir= $file;
			$dir.=  "/" unless ($dir =~ /\/$/);
		} elsif ($file =~ /^(.*\/)[^\/]*$/) {
			$dir=$1;
		} else {
			$dir= "";
		}
		for my $item (@filelist) {
			next unless ($item);
			next if ($item =~ "^mailto:");
			$item =~ s/^$Protocol:\/\/$Host//;
			$item =~ s/#.*$//; # Remove labels
			$item =~ s/\/\?.*$/\//; #Remove queries
			if ($item =~ /^(\w+):(\/\/.*)$/) {
				my $prot= $1;
				my $path= $2;
				next unless IsElement($prot, @{$self->{'LWPProtocols'}}); # Ignore if we can't handle it
				$self->_my_push($item, \@files);
				next;
			}
			$item= "$dir$item" unless ($item =~ /^\//);
			Strip_Dir($item);
			next if $self->_my_push($item, \@files);
			next unless $self->{'Recursive'};
			next if ($item !~ /^$dir/); # Don't leave this directory
			my $isdir= 0; # Should this be considered a directory 
			if ($item =~ /\/$/) {
				$isdir= 1;
			} else {
				foreach my $ext (@{$self->{'HTTPExt'}}) {
					if ($item =~ /\.$ext$/i) {
						$isdir= 1;
						last;
					}
				}
			}
			next unless ($isdir);
			$self->_my_push($item, \@rfiles, ($dir =~ tr@/@@));
		}
	}

	return @files;
}

######################
# Get a file via LWP
# Saves a file to disk resp. parses html files for links
######################

sub _Get_File_lwp
{
	my $self= shift;
	my $location= shift;
	my $file= shift; # filename
	my $time= shift; # time stamp
	my ($res, $url);
	my $Host= $self->{'Host'};
	my $Port= $self->{'Port'};
	my $Protocol= $self->{'Protocol'};

	$self->{'DldResult'}= "";
	if ( $location =~ /^\w+:\/\// ) {
		$url= $location;
	} else {
		unless ($location =~ /^\//) {
			$location= "/" . $location;
		}
		if ($Protocol eq "file") {
			$url = "file:$location";
		} else {
			$url = "$Protocol://$Host:$Port$location";
		}
	}

	print "LWP: Getting url: $url\n" if $self->{'DebugLWP'};
	my $req = HTTP::Request->new( GET => $url );
	if ($self->{'User'}) {
		$req->authorization_basic($self->{'User'}, $self->{'Pass'});
	}

	if ($file) {  # Store file
		print "\nLWP: Saving data: $file\n" if $self->{'DebugLWP'};
		if ($time) {
			$req->if_modified_since($time);
		}
		if ( -f $file ) { #Resume download
			my $lsize= (stat($file))[7];
			print "  Resuming incomplete download.\n" if ($self->{'Verbose'});
			if ( CORE::open(LWPFILE, ">>$file") ) {
				$req->header(Range => "bytes=$lsize-");
				$res = $self->{'Handle'}->request( $req, sub { print LWPFILE $_[0] } , 4096);
				unless ($res->status_line =~ /^206/) { #Did the server understand our request?
					print "Warning: Failed to resume download (" . $res->status_line . ").\n" if $self->{'Warnings'};
					unlink $file;
					$req = HTTP::Request->new( GET => $url );
					if ($self->{'User'}) {
						$req->authorization_basic($self->{'User'}, $self->{'Pass'});
					}
					$res = $self->{'Handle'}->request( $req, $file );
				}
				CORE::close(LWPFILE);
			} else {
				print "Warning: Failed to append to: $file ($!).\n" if $self->{'Warnings'};
				unlink $file;
				$res = $self->{'Handle'}->request( $req, $file );
			}
		} else {
			$res = $self->{'Handle'}->request( $req, $file );
		}
	} else {
		$file= undef;
		$req->header(Accept => "text/html") unless ($self->{'Protocol'} eq "ftp");
		$res = $self->{'Handle'}->request( $req );
	}

	if( $res->is_success ) {
		print "LWP: Received url: $url\n" if $self->{'DebugLWP'};
		if (defined $file) {
			# Preserve modification time
			my $rtime= $res->last_modified;
			utime (time, $rtime, $file) if $rtime; 
			return $file
		}
	} else {
		print "LWP: $url: " . $res->status_line . "\n" if $self->{'DebugLWP'};
		$self->{'DldResult'}= $res->status_line;
		chomp($self->{'DldResult'});
		return undef;
	}

	# Parse data
	print "\nLWP: Parsing data (" . $res->content_type . "):\n" if $self->{'DebugLWP'};
   
	$file = $res->base->as_string;
	$file =~ s/^[a-z]+:\/\/[^\/]+(\/.*)$/$1/i;
	my @files = ($file);

	if ($res->content_type eq "text/ftp-dir-listing") {
		foreach (split(/\n/,$res->content)) {
			# |Attributes|HLs|Owner|Group|Size|Date|Name|
			my ($attr, $filename) = (split(/\s+/))[0,-1];
			next if ($filename eq "." or $filename eq "..");
			$filename .= "/" if ($attr =~ "^d" and $filename !~ /\/$/);
			print "LWP: HREF: $filename\n" if $self->{'DebugLWP'};
			push (@files, $filename);
		}
	} elsif ($res->content_type eq "text/html") {
		my $data = $res->content;
		foreach my $hrefMask ('<a [^>]*href="([^"]+)"', '<a [^>]*href=([^" >]+)' ) {
			while( $data =~ /$hrefMask/ig ) {
				next if ($1 eq ".." or $1 eq "../");
				push (@files, $1);
				print "LWP: HREF: $1\n" if $self->{'DebugLWP'};
			}
		}
	} else {
		print STDERR "Error: Cannot parse $url (unknown content-type).\n";
	}  
	print "LWP: Found " . (scalar(@files)-1) . " links.\n" if $self->{'DebugLWP'};

	return @files;
}



######################
# Get a list of all local files
######################

sub _List_Files_file
{
	my $self= shift;
	my @dirs = @_;
	my @files = ();

	print "DLD: Getting list of local files:\n" if ($self->{'Debug'});

	foreach my $dir (@dirs) {
		print "DLD:   $dir\n" if ($self->{'Debug'});
		unless ( -d $dir ) {
			$self->_my_warn($dir);
			next;
		}
		Strip_Dir($dir,"1");
		foreach my $item (glob("$dir/*")) {
			if ( -d $item ) {
				next unless ($self->{'Recursive'});
				# Recursively descent through all of the sub-directories
				if ( -l $item ) {
					$item= readlink($item);
					$item= "$dir/$item" unless ($item =~ /^\//);
					Strip_Dir($item,"1");
				}
				$self->_my_push($item, \@dirs, 0);
			} else {
				$self->_my_push($item, \@files);
			}			
		}
	}
	
	return @files;
}

######################
# Get a list of remote files (ftp)
######################

sub _List_Files_ftp
{
	my $self= shift;
	my @dirs = @_;
	my ($dir, $item);
	my @files = ();
	my $ftp= $self->{'Handle'};


	foreach $item (@dirs) {
		if ($item =~ /^\.(\/.*)$/) {
			$item= $ftp->pwd() . $1;
		}
		if ($item eq ".") {
			$item= $ftp->pwd();
		}
	}		

	print "  Getting list of remote files (ftp):\n" if $self->{'Debug'};

	foreach $dir (@dirs) {
		print "    $dir\n" if $self->{'Debug'};
		unless ( $ftp->cwd($dir) ) {
			$self->_get_ftp_message;
			$self->_my_warn($dir);
			next;
		}
		if ($dir =~ /^(.*)\/$/) {
			$dir=$1;
		}
		if ($self->{'Recursive'}) {
			# Recursively descent through all of the sub-directories
			my @tmp;
			eval { @tmp= $ftp->dir(); };
			foreach $item ( @tmp ) {
				# |Attributes|HLs|Owner|Group|Size|Date|Name|
				my ($attr, $filename) = (split(/\s+/,$item))[0,-1];
				next if ($filename eq "." or $filename eq "..");
				if ($attr !~ /^d/) {
					next if $self->_my_push("$dir/$filename", \@files);
				}
				if ( $attr =~ /^d/ ) {
					# Directory
				} elsif ( $attr =~ /^l/ ) {
					# Link
					next unless ($ftp->cwd($filename));
					$ftp->cwd($dir);
				} else {
					next;
				}
				$filename="$dir/$filename";
				$self->_my_push($filename, \@dirs, 0);
			}
		} else {
			my @tmp;
			eval { @tmp= $ftp->ls(); };
			foreach $item ( @tmp ) {
				$self->_my_push("$dir/$item", \@files);
			}
		}
	}

	return @files;
}

######################
# Get ftp message
######################

sub _get_ftp_message
{
	my $self= shift;
	my $ftp= $self->{'Handle'};

	if ($ftp->message) {
		$self->{'DldResult'}= (split("\n", $ftp->message))[-1];
	} else {
		$self->{'DldResult'}= "Timeout?";
	}
}

######################
# Get a list of remote files (sftp)
######################

sub _List_Files_sftp
{
	my $self= shift;
	my @dirs = @_;
	my ($dir, $item, $stat);
	my @files = ();
	my $sftp= $self->{'Handle'};

	print "  Getting list of remote files (sftp):\n" if $self->{'Debug'};

	foreach $dir (@dirs) {
		print "    $dir\n" if $self->{'Debug'};
		$stat= $sftp->do_stat($dir);
		unless ( $stat ) {
			$self->_my_warn($dir);
			next;
		}
		if ($dir =~ /^(.*)\/$/) {
			$dir=$1;
		}
		if ($self->{'Recursive'}) {
			# Recursively descent through all of the sub-directories
		
			foreach $item ( $sftp->ls($dir) ) {
				$item= $item->{'longname'};
				# |Attributes|HLs|Owner|Group|Size|Date|Name|
				my ($attr,$filename) = (split(/\s+/,$item))[0,-1];
				next if ($filename eq "." or $filename eq "..");
				if ($attr !~ /^d/) {
					next if $self->_my_push("$dir/$filename", \@files);
				}
				if ( $attr =~ /^d/ ) {
					# Directory
				} elsif ( $attr =~ /^l/ ) {
					next unless( $stat= $sftp->do_stat($item) );
					$stat= $stat->{'perm'} >> 12;
					next unless( $stat == "4");
				} else {
					next;
				}
				$filename="$dir/$filename";
				$self->_my_push($filename, \@dirs, 0);
			}
		} else {
			foreach $item ( $sftp->ls($dir) ) {
				$item= $item->{'filename'};
				$self->_my_push("$dir/$item", \@files);
			}
		}
	}

	return @files;
}

1;

__END__

=head1 NAME

autoupdate::dld - A class for parsing remote sites and download files

=head1 SYNOPSIS

 require autoupdate::dld;
 my $dld = autoupdate::dld->new( Protocol => "ftp", Host => "ftp.nowhere.org" [, ...]);
 $dld->open();
 $dld->list("/path");
 $dld->get("/path/file"[, file [, time]]);
 $dld->close();

=head1 DESCRIPTION

The C<autoupdate::dld> is a class providing an interface to get (recursive) directory
listings of remote sites and downloading files. Currently file, ftp, sftp, http, and
https are supported.

=head1 METHODS

=item $dld = autoupdate::dld->new( %options );

This class method constructs a new C<autoupdate::dld> object and
returns a reference to it.

The following options are available:

   KEY => DEFAULT
   -------------------------------
   Debug => 0
   DebugFTP => 0
   DebugSFTP => 0
   DebugLWP => 0
   Verbose => 1
   Warnings => 0
   Protocol => 'file'
   User => ''
   Pass => ''
   Host => 'localhost'
   Port => 0
   Dirs => []
   Passive => 0
   FTPRetry => 0
   FTPWait => 10
   Firewall => '',
   FirewallType => 0
   LWPKeepAlive => 1
   LWPProtocols => [ "http", "https" ]
   Proxy => ''
   ProxyUser => ''
   ProxyPass => ''
   NoProxy => []
   HTTPExt => [ "html", "htm", "php", "asp" ]
   SSHCipher => ''
   SSHArgs => []
   Recursive => 0
   MaxRecursive => 100
   Timeout => ''
   NoWarnDirs => []
   DirInclude => []
   FileInclude => [ '^[^.]' ] 

=item $dld->open();

   Opens the connection.

=item $dld->list([dir1 [, dir2, ...]]);

   Gets a directory listing for the given directories. Default is to list all directories in
   Dirs.

=item $dld->get(path/file [, file [,time]]);

   Gets a file. If the second argument is given, stores it under that name. If the third
   argument is given, only downloads the file if its mtime is newer than the given one.

=item $dld->close();

   Closes the connection.

=head1 SEE ALSO


autoupdate(8)
LWP(3)
Net::FTP(3)
Net::SFTP(3)

=head1 AUTHOR

Gerald Teschl

=head1 LICENSE

GPL

=cut
