<?php
/**
 * $Horde: passwd/lib/Driver/ldap.php,v 1.6.2.1 2002/10/09 17:51:01 ericr Exp $
 *
 * The ldap class attempts to change a user's password stored in an ldap
 * directory service.
 *
 * @author   Tjeerd van der Zee <admin@xar.nl>
 * @author   Mattias Webjrn Eriksson <mattias@webjorn.org>
 * @author   Eric Jon Rostetter <eric.rostetter@physics.utexas.edu>
 * @package  passwd
 */

    class Passwd_Driver_ldap extends Passwd_Driver {

    /** file pointer to the ldap connection. */
    var $ds;

    /** error string returned to user if an eror occurs. */
    var $err_str;

    var $params;

    /**
     * Constructs a new ldap Passwd_Driver object.
     *
     * @param array  $params    A hash containing connection parameters.
     */
    function Passwd_Driver_ldap($params = array())
    {
        $this->params = $params;
    }


    /**
     * Do an ldap connect and bind as the guest user or as the optional userdn.
     *
     * @param    $host     The ldap server hostname.
     * @param    $port     The ldap server port number to connect to.
     * @param    $userdn   The optional dn to use when binding non-anonymously.
     * @param    $oldpassword The optional password for $userdn.
     *
     * @return   boolean   True or False based on success of connect and bind.
     *
     */

    function connect($host, $port, $userdn, $oldpassword) {
        $this->ds = ldap_connect($host, $port);
        if (!$this->ds) {
           $this->err_str = _("Could not connect to ldap server");
           return false;
        }
        if (isset($userdn)) {
            $result = ldap_bind($this->ds, $userdn, $oldpassword);
        } else {
            $result = ldap_bind($this->ds);
        }
        if (!$result) {
          $this->err_str = _("Could not bind to ldap server");
          return false;
        }
        return true;
    }

    /**
     * Close the ldap connection.
     */

    function disconnect() {
        if (isset($this->ds)) {
           ldap_close($this->ds);
        }
    }

    /**
     * Lookup and return the user's dn.
     *
     * @param  $user     The username of the user.
     * @param  $realm    The realm (domain) name of the user.
     * @param  $basedn   The ldap basedn.
     * @param  $uid      The ldap uid.
     *
     * @return string    The ldap dn for the user.
     */

    function lookupdn($user, $realm, $basedn, $uid) {
      $sr = ldap_search($this->ds, $basedn, "$uid=$user@$realm");
      $entry = ldap_first_entry($this->ds, $sr);
      $dn = ldap_get_dn($this->ds, $entry);
      return $dn;
    }

    /**
     * Encrypt the new password string to use when updating the ldap server.
     *
     * @param   $encryption   The type of encryption (plain, crypt, SHA, MD5).
     * @param   $newPassword  The new password value to encrypt.
     *
     * @return  array         The array that contains the modified password.
     */

    function encryptPassword($encryption, $newPassword) {

      switch (strtolower($encryption)) {
      case "plain":
         $change_info["userPassword"] = $newPassword;
         break;
      case "sha":
         $change_info["userPassword"] = "{SHA}" .
         base64_encode(mHash(MHASH_SHA1, $newPassword));
         break;
      case "crypt":
         // The salt is left out, generated by php
         $change_info["userPassword"] = "{crypt}" . crypt($newPassword);
         break;
      case "md5":
         $change_info["userPassword"] = "{MD5}" .
         base64_encode(mHash(MHASH_MD5, $newPassword));
         break;
      default:
         $this->err_str = _("Password module is not properly configured");
         $change_info["userPassword"] = "";
         break;
      }
      return $change_info;
    }

    /**
     * Rebind to the ldap server if initially bound as guest
     *
     * @param   $user         The user whose record we will udpate.
     * @param   $realm        The realm (domain) name of the user.
     * @param   $oldPassword  The old (current) password value.
     * @param   $basedn       The ldap basedn.
     * @param   $uid          The ldap uid.
     *
     * @return  boolean       True or False based on success of the rebind.
     */

    function rebind($user, $realm, $oldPassword, $basedn, $uid) {

      $dn = $this->lookupdn($user, $realm, $basedn, $uid);

      // Now that I know who I am, then just rebind as myself,
      // instead of guest.

      if (!ldap_bind($this->ds, $dn, $oldPassword)) {
         $this->err_str = _("Your old password isn't correct");
         return false;
      }

      return $dn;
    }

    /**
     * Change the user's password.
     *
     * @param   $username     The user for which to change the password.
     * @param   $realm        The realm (domain) for the user.
     * @param   $oldpassword  The old (current) user password.
     * @param   $newpassword  The new user password to set.
     *
     * @return  boolean       True or false based on success of the change.
     */

    function change_password($username, $realm, $oldpassword, $newpassword) {

        $host       = @$this->params['default']['host'];
        $port       = @$this->params['default']['port'];
        $basedn     = @$this->params['default']['basedn'];
        $uid        = @$this->params['default']['uid'];
        $encryption = @$this->params['default']['encryption'];
        $userdn     = @$this->params['optional']['userdn'];

        if ( isset($realm) && $realm != "" ) {
           if ( isset( $this->params[$realm]['host'] ) )
              $host  = $this->params[$realm]['host'];
           if ( isset( $this->params[$realm]['port'] ) )
              $port  = $this->params[$realm]['port'];
           if ( isset( $this->params[$realm]['basedn'] ) )
              $basedn =$this->params[$realm]['basedn'];
           if ( isset( $this->params['optional']['userdn'] ) )
              $binddn = $this->params['optional']['userdn'];
        }

        if ( $host == "" || $port == "" || $basedn == "") {
           $this->err_str = _("Password module is not properly configured");
           return false;
        }

        $return_value = false;
        if ($this->connect($host, $port, $userdn, $oldpassword)) {
            $change_info = $this->encryptPassword($encryption,$newpassword);
            if ($change_info["userPassword"] != "") {
                if (!isset($userdn)) {
                    $userdn = $this->rebind($username, $realm, $oldpassword,
                                            $basedn, $uid);
                }
                if (ldap_mod_replace($this->ds,$userdn,$change_info)) {
                    $return_value = true;
                }
            }
            $this->disconnect();
        }
        return $return_value;
    }

}
?>
