<?php

require_once KRONOLITH_BASE . '/lib/Driver.php';
require_once KRONOLITH_BASE . '/lib/Event.php';

define('KRONOLITH_RECUR_NONE',          0);
define('KRONOLITH_RECUR_DAILY',         1);
define('KRONOLITH_RECUR_WEEKLY',        2);
define('KRONOLITH_RECUR_DAY_OF_MONTH',  3);
define('KRONOLITH_RECUR_WEEK_OF_MONTH', 4);
define('KRONOLITH_RECUR_YEARLY',        5);

define('KRONOLITH_MONDAY',    0);
define('KRONOLITH_TUESDAY',   1);
define('KRONOLITH_WEDNESDAY', 2);
define('KRONOLITH_THURSDAY',  3);
define('KRONOLITH_FRIDAY',    4);
define('KRONOLITH_SATURDAY',  5);
define('KRONOLITH_SUNDAY',    6);

define('KRONOLITH_MASK_SUNDAY',    1);
define('KRONOLITH_MASK_MONDAY',    2);
define('KRONOLITH_MASK_TUESDAY',   4);
define('KRONOLITH_MASK_WEDNESDAY', 8);
define('KRONOLITH_MASK_THURSDAY',  16);
define('KRONOLITH_MASK_FRIDAY',    32);
define('KRONOLITH_MASK_SATURDAY',  64);
define('KRONOLITH_MASK_WEEKDAYS',  62);
define('KRONOLITH_MASK_WEEKEND',   65);
define('KRONOLITH_MASK_ALLDAYS',   127);

define('KRONOLITH_JANUARY',    1);
define('KRONOLITH_FEBRUARY',   2);
define('KRONOLITH_MARCH',      3);
define('KRONOLITH_APRIL',      4);
define('KRONOLITH_MAY',        5);
define('KRONOLITH_JUNE',       6);
define('KRONOLITH_JULY',       7);
define('KRONOLITH_AUGUST',     8);
define('KRONOLITH_SEPTEMBER',  9);
define('KRONOLITH_OCTOBER',   10);
define('KRONOLITH_NOVEMBER',  11);
define('KRONOLITH_DECEMBER',  12);

define('KRONOLITH_DELETE_CATEGORY', 106);
define('KRONOLITH_RENAME_CATEGORY', 107);
define('KRONOLITH_ADD_CATEGORY', 108);

/**
 * The Kronolith:: class provides functionality common to all of
 * Kronolith.
 *
 * $Horde: kronolith/lib/Kronolith.php,v 1.49.2.10 2003/04/23 18:19:35 chuck Exp $
 *
 * @author  Chuck Hagenbuch <chuck@horde.org>
 * @version $Revision: 1.49.2.10 $
 * @since   Kronolith 0.1
 * @package kronolith
 */
class Kronolith {

    /**
     * Add a name=value pair to the end of an URL, taking care of whether
     * there are existing parameters and whether to use ? or & as the glue.
     * All data will be urlencoded.
     *
     * @access public
     *
     * @param string $url             The URL to modify
     * @param string $parameter       The name=value pair to add.
     * @param optional string $value  If specified, the value part ($parameter
     *                                is assumed to just be the parameter
     *                                name).
     *
     * @return string  The modified URL.
     *
     * @since Horde 2.1
     */
    function addParameter($url, $parameter, $value = null)
    {
        if (empty($parameter)) {
            return $url;
        }

        if (!is_array($parameter)) {
            if (is_null($value)) {
                @list($parameter, $value) = explode('=', $parameter, 2);
            }
            $add = array($parameter => $value);
        } else {
            $add = $parameter;
        }

        if (($pos = strpos($url, '?')) === false) {
            $url .= '?';
        } else {
            parse_str(substr($url, $pos + 1), $params);
            $url .= ini_get('arg_separator.output');
        }

        $url_params = array();
        foreach ($add as $parameter => $value) {
            if (!isset($params[$parameter])) {
                $url_params[] = urlencode($parameter) . '=' . urlencode($value);
            }
        }

        return $url . join(ini_get('arg_separator.output'), $url_params);
    }

    function dateObject($date = null)
    {
        $obj = new stdClass();
        $obj->year  = null;
        $obj->month = null;
        $obj->mday  = null;
        $obj->hour  = null;
        $obj->min   = null;
        $obj->sec   = null;

        if (is_array($date) || is_object($date)) {
            foreach ($date as $key => $val) {
                $obj->$key = $val;
            }
        }

        return $obj;
    }

    /**
     * Returns all the events that happen each day within a time period
     *
     * @param object $startDate    The start of the time range.
     * @param object $endDate      The end of the time range.
     * @param array  $calendars    The calendars to check for events.
     *
     * @return array  The events happening in this time period.
     */
    function listEventIds($startDate, $endDate, $calendars = null)
    {
        global $calendar;

        $eventIds = array();
        $calendars = array(Auth::getAuth());
        foreach ($calendars as $cal) {
            if ($calendar->getCalendar() != $cal) {
                $calendar->close();
                $calendar->open($cal);
            }
            $eventIds[$cal] = $calendar->listEvents($startDate, $endDate);
        }

        return $eventIds;
    }

    /**
     * Returns all the alarms active right on $date.
     *
     * @param object $date         The start of the time range.
     * @param array  $calendars    The calendars to check for events.
     *
     * @return array  The alarms active on $date.
     */
    function listAlarms($date, $calendars)
    {
        global $calendar;

        $alarms = array();
        foreach ($calendars as $cal) {
            if ($calendar->getCalendar() != $cal) {
                $calendar->close();
                $calendar->open($cal);
            }
            $alarms[$cal] = $calendar->listAlarms($date);
        }

        return $alarms;
    }

    /**
     * Returns all the events that happen each day within a time period
     *
     * @param object $startDate    The start of the time range.
     * @param object $endDate      The end of the time range.
     * @param array  $calendars    The calendars to check for events.
     *
     * @return array  The events happening in this time period.
     */
    function listEvents($startDate = null, $endDate = null, $calendars = null, $showRecurrence = true)
    {
        global $calendar, $prefs, $registry;

        if (!isset($startDate)) {
            $startDate = Kronolith::timestampToObject(time());
        }
        if (!isset($endDate)) {
            $endDate = Kronolith::timestampToObject(time());
        }

        $eventIds = Kronolith::listEventIds($startDate, $endDate, $calendars);

        $startOfPeriodTimestamp = mktime(0, 0, 0, $startDate->month, $startDate->mday, $startDate->year);
        $endOfPeriodTimestamp = mktime(23, 59, 59, $endDate->month, $endDate->mday, $endDate->year);
        $daysInPeriod = Date_Calc::dateDiff($startDate->mday, $startDate->month, $startDate->year, $endDate->mday, $endDate->month, $endDate->year);

        $results = array();
        foreach ($eventIds as $cal => $events) {
            if ($calendar->getCalendar() != $cal) {
                $calendar->close();
                $calendar->open($cal);
            }
            foreach ($events as $id) {
                // We MUST fetch each event right before getting its
                // recurrences; this is due to the way MCAL
                // works. MCAL's nextRecurrence() function gives you
                // the next recurrence for the event most recently
                // fetched. So if you fetch all events and then loop
                // through them, every recurrence you get will be for
                // the last event that you fetched.
                $event = $calendar->getEventObject($id);

                if (!$event->hasRecurType(KRONOLITH_RECUR_NONE) && $showRecurrence) {
                    /* Recurring Event */

                    if ($event->getStartTimestamp() < $startOfPeriodTimestamp) {
                        // The first time the event happens was before the
                        // period started. Start searching for recurrences
                        // from the start of the period.
                        $next = array('year' => $startDate->year, 'month' => $startDate->month, 'mday' => $startDate->mday);
                    } else {
                        // The first time the event happens is in the
                        // range; unless there is an exception for
                        // this ocurrence, add it.
                        if (!$event->hasException($event->getStartDate('Y'),
                                                  $event->getStartDate('n'),
                                                  $event->getStartDate('j'))) {
                            $results[$event->getStartDatestamp()][$id] = $event;
                        }

                        // Start searching for recurrences from the day
                        // after it starts.
                        $next = array('year' => $event->getStartDate('Y'), 'month' => $event->getStartDate('n'), 'mday' => $event->getStartDate('j') + 1);
                    }

                    // Add all recurences of the event.
                    $next = $event->nextRecurrence($next);
                    while ($next !== false && (Kronolith::compareDates($next, $endDate) <= 0)) {
                        if (!$event->hasException($next->year, $next->month, $next->mday)) {
                            $results[Kronolith::objectToDatestamp($next)][$id] = $event;
                        }
                        $next = $event->nextRecurrence(array('year' => $next->year,
                                                             'month' => $next->month,
                                                             'mday' => $next->mday + 1,
                                                             'hour' => $next->hour,
                                                             'min' => $next->min,
                                                             'sec' => $next->sec));
                    }
                } else {
                    /* Event only occurs once. */

                    // Work out what day it starts on.
                    if ($event->getStartTimestamp() < $startOfPeriodTimestamp) {
                        // It started before the beginning of the period.
                        $eventStartStamp = $startOfPeriodTimestamp;
                    } else {
                        $eventStartStamp = $event->getStartTimestamp();
                    }

                    // Work out what day it ends on.
                    if ($event->getEndTimestamp() >= $endOfPeriodTimestamp) {
                        // Ends after the end of the period.
                        $eventEndStamp = $endOfPeriodTimestamp;
                    } else {
                        // If the event doesn't end at 12am set the
                        // end date to the current end date. If it
                        // ends at 12am and does not end at the same
                        // time that it starts (0 duration), set the
                        // end date to the previous day's end date.
                        if ($event->getEndDate('G') != 0 ||
                            $event->getEndDate('i') != 0 ||
                            $event->getStartTimestamp() == $event->getEndTimestamp()) {
                            $eventEndStamp = $event->getEndTimestamp();
                        } else {
                            $eventEndStamp = mktime(23, 59, 59,
                                                    $event->getEndDate('n'),
                                                    $event->getEndDate('j') - 1,
                                                    $event->getEndDate('Y'));
                        }
                    }

                    // Add the event to all the days it covers.
                    $i = date('j', $eventStartStamp);
                    $loopStamp = mktime(0, 0, 0,
                                        date('n', $eventStartStamp),
                                        $i,
                                        date('Y', $eventStartStamp));
                    while ($loopStamp <= $eventEndStamp) {
                        if (!($event->isAllDay() && $loopStamp == $eventEndStamp)) {
                            $results[$loopStamp][$id] = $event;
                        }
                        $loopStamp = mktime(0, 0, 0,
                                            date('n', $eventStartStamp),
                                            ++$i,
                                            date('Y', $eventStartStamp));
                    }
                }
            }
        }

        /* Nag Tasks. */
        if ($prefs->getValue('show_tasks') &&
            (Auth::getAuth() || $registry->allowGuests($registry->hasMethod('tasks/list')))) {
            $taskList = $registry->call('tasks/list');
            $dueEndStamp = mktime(0, 0, 0, $endDate->month, $endDate->mday + 1, $endDate->year);
            if (!PEAR::isError($taskList)) {
                foreach ($taskList as $task) {
                    if ($task['due'] >= $startOfPeriodTimestamp && $task['due'] < $dueEndStamp) {
                        $event = $calendar->getEventObject();
                        $event->setTitle(sprintf(_("Due: %s"), $task['name']));
                        $event->taskID = $task['task_id'];
                        $event->setStartTimestamp($task['due']);
                        $event->setEndTimestamp($task['due'] + 1);
                        $dayStamp = mktime(0, 0, 0,
                                           date('n', $task['due']),
                                           date('j', $task['due']),
                                           date('Y', $task['due']));
                        $results[$dayStamp]['_task' . $task['task_id']] = $event;
                    }
                }
            }
        }

        foreach ($results as $day => $devents) {
            if (count($devents)) {
                uasort($devents, array('Kronolith', '_sortEventStartTime'));
                $results[$day] = $devents;
            }
        }

        return $results;
    }

    /**
     * Used with usort() to sort events based on their start times.
     * This function ignores the date component so recuring events can
     * be sorted correctly on a per day basis.
     */
    function _sortEventStartTime($a, $b)
    {
        return ((int)date('Gis', $a->startTimestamp) - (int)date('Gis', $b->startTimestamp));
    }

    function _sortEvents($a, $b)
    {
        return $a->startTimestamp - $b->startTimestamp;
    }

    function secondsToString($seconds)
    {
        $hours = floor($seconds / 3600);
        $minutes = ($seconds / 60) % 60;

        if ($hours > 1) {
            if ($minutes == 0) {
                return sprintf(_("%d hours"), $hours);
            } else if ($minutes == 1) {
                return sprintf(_("%d hours, %d minute"), $hours, $minutes);
            } else {
                return sprintf(_("%d hours, %d minutes"), $hours, $minutes);
            }
        } else if ($hours == 1) {
            if ($minutes == 0) {
                return sprintf(_("%d hour"), $hours);
            } else if ($minutes == 1) {
                return sprintf(_("%d hour, %d minute"), $hours, $minutes);
            } else {
                return sprintf(_("%d hour, %d minutes"), $hours, $minutes);
            }
        } else {
            if ($minutes == 0) {
                return _("no time");
            } else if ($minutes == 1) {
                return sprintf(_("%d minute"), $minutes);
            } else {
                return sprintf(_("%d minutes"), $minutes);
            }
        }
    }

    function recurToString($type)
    {
        switch ($type) {

        case KRONOLITH_RECUR_NONE:
            return _("Recurs not");
        case KRONOLITH_RECUR_DAILY:
            return _("Recurs daily");
        case KRONOLITH_RECUR_WEEKLY:
            return _("Recurs weekly");
        case KRONOLITH_RECUR_DAY_OF_MONTH:
        case KRONOLITH_RECUR_WEEK_OF_MONTH:
            return _("Recurs monthly");
        case KRONOLITH_RECUR_YEARLY:
            return _("Recurs yearly");

        }
    }

    /**
     * Returns week of the year, first Monday is first day of first week
     *
     * @param optional string $day    in format DD
     * @param optional string $month  in format MM
     * @param optional string $year   in format CCYY
     *
     * @return integer $week_number
     */
    function weekOfYear($day = null, $month = null, $year = null)
    {
        global $prefs;

        if (!isset($year)) $year = date('Y');
        if (!isset($month)) $month = date('n');
        if (!isset($day)) $day = date('j');
        if (!$prefs->getValue('week_start_monday') && Kronolith::dayOfWeek($year, $month, $day) == KRONOLITH_SUNDAY) {
            $day++;
        }

        $dayOfYear = Kronolith::dayOfYear($year, $month, $day);
        $dayOfWeek = Kronolith::dayOfWeek($year, $month, $day);
        $dayOfWeekJan1 = Kronolith::dayOfWeek($year, 1, 1);

        if ($dayOfYear <= 7 - $dayOfWeekJan1 && $dayOfWeekJan1 > 3 ) {
            if ($dayOfWeekJan1 == 4 || ($dayOfWeekJan1 == 5 && Kronolith::isLeapYear($year - 1))) {
                return '53';
            } else {
                return '52';
            }
        }

        if (Kronolith::isLeapYear($year)) {
            $daysInYear = 366;
        } else {
            $daysInYear = 365;
        }

        if ($daysInYear - $dayOfYear < 3 - $dayOfWeek) {
            return 1;
        }

        $WeekNumber = floor(($dayOfYear + (6 - $dayOfWeek) + $dayOfWeekJan1) / 7);
        if ($dayOfWeekJan1 > 3) { $WeekNumber -= 1; }

        return $WeekNumber;
    }

    /**
     * Return the number of weeks in the given year (52 or 53).
     *
     * @param optional integer $year  The year to count the number of weeks in.
     *
     * @return integer $numWeeks      The number of weeks in $year.
     */
    function weeksInYear($year = null)
    {
        if (!isset($year)) $year = date('Y');

        // Find the last Thursday of the year.
        $day = 31;
        while (date('w', mktime(0, 0, 0, 12, $day, $year)) != 4) {
            $day--;
        }
        return Kronolith::weekOfYear($day, 12, $year);
    }

    /**
     * Returns the day of the year (1-366) that corresponds to the
     * first day of the given week.
     *
     * @param int week        The week of the year to find the first day of.
     *
     * @return int dayOfYear  The day of the year of the first day of the given week.
     */
    function firstDayOfWeek($week = null, $year = null)
    {
        if (!isset($year)) $year = date('Y');
        if (!isset($week)) $week = Kronolith::weekOfYear(null, null, $year);

        $start = Kronolith::dayOfWeek($year, 1, 1);
        if ($start > 3) $start -= 7;
        return ((($week * 7) - (7 + $start)) + 1);
    }

    /**
     * Find the number of days in the given month.
     *
     * @param string month in format MM, default current local month
     *
     * @return int number of days
     */
    function daysInMonth($month = null, $year = null)
    {
        if (!isset($month)) {
            $month = date('n');
        }

        switch ($month) {
        case 2:
            if (!isset($year)) {
                $year = date('Y');
            }
            return Kronolith::isLeapYear($year) ? 29 : 28;
            break;

        case 4:
        case 6:
        case 9:
        case 11:
            return 30;
            break;

        default:
            return 31;
        }
    }

    function isLeapYear($year = null)
    {
        if (!isset($year)) {
            $year = date('Y');
        } else {
            if (strlen($year) != 4) {
                return false;
            }
            if (preg_match('/\D/', $year)) {
                return false;
            }
        }

        return (($year % 4 == 0 && $year % 100 != 0) || $year % 400 == 0);
    }

    /**
     * Return the day of the week (0 = Monday, 6 = Sunday) of the
     * given date.
     *
     * @param optional int $year
     * @param optional int $month
     * @param optional int $day
     *
     * @return int The day of the week.
     */
    function dayOfWeek($year = null, $month = null, $day = null)
    {
        if (!isset($year))  $year  = date('Y');
        if (!isset($month)) $month = date('n');
        if (!isset($day))   $day   = date('d');

        if ($month > 2) {
            $month -= 2;
        } else {
            $month += 10;
            $year--;
        }

        $day = (floor((13 * $month - 1) / 5) +
                $day + ($year % 100) +
                floor(($year % 100) / 4) +
                floor(($year / 100) / 4) - 2 *
                floor($year / 100) + 77);

        $weekday_number = (($day - 7 * floor($day / 7))) - 1;
        if ($weekday_number == -1) {
            // Wrap check.
            $weekday_number = 6;
        }

        return $weekday_number;
    }

    function dayOfYear($year = null, $month = null, $day = null)
    {
        if (!isset($year)) {
            $year = date('Y');
        }
        if (!isset($month)) {
            $month = date('n');
        }
        if (!isset($day)) {
            $day = date('d');
        }

        $days = 0;
        for ($i = 1; $i < $month; $i++) {
            $days += Kronolith::daysInMonth($i, $year);
        }
        $days += $day;

        return $days;
    }

    function compareDates($first, $second)
    {
        $first = Kronolith::dateObject($first);
        $second = Kronolith::dateObject($second);

        if ($first->year - $second->year != 0) {
            return $first->year - $second->year;
        } elseif ($first->month - $second->month != 0) {
            return $first->month - $second->month;
        } elseif ($first->mday - $second->mday != 0) {
            return $first->mday - $second->mday;
        } elseif ($first->hour - $second->hour != 0) {
            return $first->hour - $second->hour;
        } elseif ($first->min - $second->min != 0) {
            return $first->min - $second->min;
        } else {
            return $first->sec - $second->sec;
        }
    }

    function dateDiff($start, $end, $split = false)
    {
        $start = Kronolith::dateObject($start);
        $end = Kronolith::dateObject($end);

        $res = new stdClass();
        $res->year = $end->year - $start->year;
        $res->month = $end->month - $start->month;
        $res->mday = $end->mday - $start->mday;
        $res->hour = $end->hour - $start->hour;
        $res->min = $end->min - $start->min;
        $res->sec = $end->sec - $start->sec;
        if (!$split) {
            $res->month += $res->year * 12;
            $res->mday = Date_Calc::dateDiff($start->mday, $start->month, $start->year, $end->mday, $end->month, $end->year);
            $res->hour += $res->mday * 24;
            $res->min += $res->hour * 60;
            $res->sec += $res->min * 60;
        }

        return $res;
    }

    function correctDate($date)
    {
        if (isset($date->sec)) {
            while ($date->sec > 59) {
                $date->min++;
                $date->sec -= 60;
            }
        }
        if (isset($date->min)) {
            while ($date->min > 59) {
                $date->hour++;
                $date->min -= 60;
            }
        }
        if (isset($date->hour)) {
            while ($date->hour > 23) {
                $date->mday++;
                $date->hour -= 24;
            }
        }
        if (isset($date->mday)) {
            while ($date->mday > Date_Calc::daysInMonth($date->month, $date->year)) {
                $date->month++;
                $date->mday -= Date_Calc::daysInMonth($date->month - 1, $date->year);
            }
        }
        if (isset($date->month)) {
            while ($date->month > 12) {
                $date->year++;
                $date->month -= 12;
            }
        }

        return $date;
    }

    function timestampToObject($timestamp)
    {
        $res = new stdClass();
        list($res->hour, $res->min, $res->sec, $res->mday, $res->month, $res->year) = explode('/', date('H/i/s/j/n/Y', $timestamp));
        return $res;
    }

    function timestampToArray($timestamp)
    {
        $obj = Kronolith::timestampToObject($timestamp);

        return array('hour' => $obj->hour,
                     'min' => $obj->min,
                     'sec' => $obj->sec,
                     'month' => $obj->month,
                     'mday' => $obj->mday,
                     'year' => $obj->year);
    }

    function objectToTimestamp($obj)
    {
        return @mktime($obj->hour, $obj->min, $obj->sec, $obj->month, $obj->mday, $obj->year);
    }

    function objectToDatestamp($obj)
    {
        return @mktime(0, 0, 0, $obj->month, $obj->mday, $obj->year);
    }

    function arrayToTimestamp($arr)
    {
        return @mktime($arr['hour'], $arr['min'], $arr['sec'], $arr['month'], $arr['mday'], $arr['year']);
    }

    /**
     * Builds the HTML for a event category widget.
     *
     * @param string  $name       The name of the widget.
     * @param integer $selected   (optional) The default category.
     * @param boolean $newheader  (optional) Include a new category option.
     *
     * @return string       The HTML <select> widget.
     */
    function buildCategoryWidget($name, $selected = false, $newheader = false)
    {
        $html = "<select id=\"$name\" name=\"$name\">";

        if ($newheader) {
            $html .= '<option value="*new*">' . _("New Category") . "</option>\n";
            $html .= '<option value="">----</option>' . "\n";
        }

        foreach (Kronolith::listCategories() as $id => $name) {
            $html .= "<option value=\"$id\"";
            $html .= ($id == $selected && $selected !== false) ? ' selected="selected">' : '>';
            $html .= $name . "</option>\n";
        }
        $html .= '</select>';

        return $html;
    }

    /**
     * List a user's categories
     *
     * @return array A list of categories.
     */
    function listCategories()
    {
        global $prefs;

        static $catString, $categories;

        $cur = $prefs->getValue('event_categories');
        if (is_null($catString) || $catString != $cur) {
            $categories = array(0 => _("Unfiled"));

            $catString = $cur;
            if (empty($catString)) {
                return $categories;
            }

            $cats = explode('|', $catString);
            foreach ($cats as $cat) {
                list($key, $val) = explode(':', $cat);
                $categories[$key] = $val;
            }
        }

        asort($categories);
        return $categories;
    }

    /**
     * Add a new category
     *
     * @param string  $name     The name of the category to add.
     *
     * @return integer          A valid category id, 0 on failure or
     *                          the new category's id on success.
     */
    function addCategory($name)
    {
        global $prefs;

        if ($prefs->isLocked('event_categories') || empty($name)) {
            return 0;
        }

        $categories = Kronolith::listCategories();
        if (in_array($name, $categories)) {
            return 0;
        }

        $categories[] = $name;
        unset($categories[0]);

        $cats = array();
        $key = 0;
        foreach ($categories as $key => $cat) {
            $cat = array($key, $cat);
            $cats[] = implode(':', $cat);
        }

        $catString = implode('|', $cats);
        $prefs->setValue('event_categories', $catString);

        $prefs->store();

        return $key;
    }

    /**
     * Delete a category
     *
     * @param integer   $categoryID The id of the category to remove.
     *
     * @return boolean              True on success, false on failure.
     */
    function deleteCategory($categoryID)
    {
        global $prefs;

        $categories = Kronolith::listCategories();

        if ($prefs->isLocked('event_categories') ||
            !array_key_exists($categoryID, $categories)) {
            return false;
        }

        unset($categories[0]);
        unset($categories[$categoryID]);

        $cats = array();
        foreach ($categories as $key => $cat) {
            $cat = array($key, $cat);
            $cats[] = implode(':', $cat);
        }

        $catString = implode('|', $cats);
        $prefs->setValue('event_categories', $catString);

        $prefs->store();

        return true;
    }

    /**
     * Rename a category
     *
     * @param integer   $categoryID The id of the category to remove.
     * @param string    $name       The new name of the category.
     *
     * @return boolean              True on success, false on failure.
     */
    function renameCategory($categoryID, $name)
    {
        global $prefs;

        $categories = Kronolith::listCategories();

        if ($prefs->isLocked('event_categories') ||
            empty($name) ||
            !array_key_exists($categoryID, $categories)) {
            return false;
        }

        unset($categories[0]);
        $categories[$categoryID] = $name;

        $cats = array();
        foreach ($categories as $key => $cat) {
            $cat = array($key, $cat);
            $cats[] = implode(':', $cat);
        }

        $catString = implode('|', $cats);
        $prefs->setValue('event_categories', $catString);

        $prefs->store();

        return true;
    }

    /**
     * Returns the highlight colors for the categories
     *
     * @return array A list of colors, key matches listCategories keys.
     */
    function categoryColors()
    {
        global $prefs, $registry;

        static $colorString, $colors;
        if (!is_array($colors)) {
            $colors = array();
            $colorString = '';
        }

        $_colorString = $prefs->getValue('event_colors');
        if ($colorString != $_colorString) {
            $colors = array(0 => '#ccccff');
            $colorString = $_colorString;
            $cols = explode('|', $colorString);
            foreach ($cols as $col) {
                if (!empty($col)) {
                    list($key, $val) = explode(':', $col);
                    $colors[$key] = $val;
                }
            }
        }

        return $colors;
    }

    /**
     * Returns the string matching the given category ID.
     *
     * @param integer $categoryID     The category ID to look up.
     *
     * @return string       The formatted category string.
     */
    function formatCategory($categoryID = 0)
    {
        $categories = Kronolith::listCategories();
        return isset($categories[$categoryID]) ?
            $categories[$categoryID] :
            $categories[0];
    }

    /**
     * Calculate the border (darker) version of a color.
     *
     * @param string $color   An HTML color, e.g.: ffffcc.
     *
     * @return string  A darker html color.
     */
    function borderColor($color)
    {
        return Kronolith::modifyColor($color, -0x11);
    }

    /**
     * Calculate a lighter (or darker) version of a color.
     *
     * @param string $color    An HTML color, e.g.: #ffffcc.
     *
     * @return string  A modified HTML color.
     */
    function modifyColor($color, $factor = 0x11)
    {
        $r = hexdec(substr($color, 1, 2));
        $g = hexdec(substr($color, 3, 2));
        $b = hexdec(substr($color, 5, 2));

        if ($r >= $g && $r >= $b) {
            $g = $g / $r;
            $b = $b / $r;

            $r = $r + $factor;
            $g = floor($g * $r);
            $b = floor($b * $r);
        } elseif ($g >= $r && $g >= $b) {
            $r = $r / $g;
            $b = $b / $g;

            $g = $g + $factor;
            $r = floor($r * $g);
            $b = floor($b * $g);
        } else {
            $r = $r / $b;
            $g = $g / $b;

            $b = $b + $factor;
            $r = floor($r * $b);
            $g = floor($g * $b);
        }

        return '#' . str_pad(dechex(min($r, 255)), 2, '0', STR_PAD_LEFT) . str_pad(dechex(min($g, 255)), 2, '0', STR_PAD_LEFT) . str_pad(dechex(min($b, 255)), 2, '0', STR_PAD_LEFT);
    }

}
