<?php
// $Horde: kronolith/lib/Kronolith.php,v 1.45 2001/12/09 15:47:55 chuck Exp $

define('KRONOLITH_NAME', 'Kronolith');
require_once KRONOLITH_BASE . '/lib/Driver.php';
require_once KRONOLITH_BASE . '/lib/Event.php';

define('KRONOLITH_EVENT',  64);
define('KRONOLITH_ALARM', 128);

define('KRONOLITH_RECUR_NONE',          0);
define('KRONOLITH_RECUR_DAILY',         1);
define('KRONOLITH_RECUR_WEEKLY',        2);
define('KRONOLITH_RECUR_DAY_OF_MONTH',  3);
define('KRONOLITH_RECUR_WEEK_OF_MONTH', 4);
define('KRONOLITH_RECUR_YEARLY',        5);

define('KRONOLITH_MONDAY',    0);
define('KRONOLITH_TUESDAY',   1);
define('KRONOLITH_WEDNESDAY', 2);
define('KRONOLITH_THURSDAY',  3);
define('KRONOLITH_FRIDAY',    4);
define('KRONOLITH_SATURDAY',  5);
define('KRONOLITH_SUNDAY',    6);

define('KRONOLITH_MASK_SUNDAY',    1);
define('KRONOLITH_MASK_MONDAY',    2);
define('KRONOLITH_MASK_TUESDAY',   4);
define('KRONOLITH_MASK_WEDNESDAY', 8);
define('KRONOLITH_MASK_THURSDAY',  16);
define('KRONOLITH_MASK_FRIDAY',    32);
define('KRONOLITH_MASK_SATURDAY',  64);
define('KRONOLITH_MASK_WEEKDAYS',  62);
define('KRONOLITH_MASK_WEEKEND',   65);
define('KRONOLITH_MASK_ALLDAYS',   127);

define('KRONOLITH_JANUARY',    1);
define('KRONOLITH_FEBRUARY',   2);
define('KRONOLITH_MARCH',      3);
define('KRONOLITH_APRIL',      4);
define('KRONOLITH_MAY',        5);
define('KRONOLITH_JUNE',       6);
define('KRONOLITH_JULY',       7);
define('KRONOLITH_AUGUST',     8);
define('KRONOLITH_SEPTEMBER',  9);
define('KRONOLITH_OCTOBER',   10);
define('KRONOLITH_NOVEMBER',  11);
define('KRONOLITH_DECEMBER',  12);


/**
 * The Kronolith:: class provides the functionality common to all of
 * Kronolith.
 *
 * @author  Chuck Hagenbuch <chuck@horde.org>
 * @version $Revision: 1.45 $
 * @since   Kronolith 0.1
 * @package kronolith
 */
class Kronolith {
    
    function dateObject($date = null)
    {
        $obj->year  = null;
        $obj->month = null;
        $obj->mday  = null;
        $obj->hour  = null;
        $obj->min   = null;
        $obj->sec   = null;
        
        if (is_array($date) || is_object($date)) {
            foreach ($date as $key => $val) {
                $obj->$key = $val;
            }
        }
        
        return $obj;
    }
    
    function getEvents($date)
    {
        global $calendar, $registry, $prefs;
        
        $list = $calendar->listEvents($date, $date);
        
        $events = array();
        if (is_array($list)) {
            foreach ($list as $eventID) {
                $event = $calendar->getEventObject($eventID);
                $events[] = $event;
            }
        }
        
        if ($prefs->getValue('show_tasks')) {
            $taskList = $registry->call('tasks/list');
            if (!PEAR::isError($taskList)) {
                $start = mktime(0, 0, 0, $date->month, $date->mday, $date->year);
                $end = mktime(0, 0, 0, $date->month, $date->mday + 1, $date->year);
                foreach ($taskList as $task) {
                    if ($task['due'] >= $start && $task['due'] < $end) {
                        $event = $calendar->getEventObject();
                        $event->setTitle(_("Due:") . ' ' . $task['name']);
                        $event->taskID = $task['task_id'];
                        $event->setStartTimestamp($task['due']);
                        $event->setEndTimestamp($task['due'] + 1);
                        $events[] = $event;
                    }
                }
            }
        }
        
        usort($events, array('Kronolith', 'sortEvents'));
        return $events;
    }
    
    function sortEvents($a, $b)
    {
        return $a->startTimestamp - $b->startTimestamp;
    }
    
    function secondsToString($seconds)
    {
        $hours = floor($seconds / 3600);
        $minutes = ($seconds / 60) % 60;
        
        if ($hours > 1) {
            if ($minutes == 0) {
                return sprintf(_("%d hours"), $hours);
            } else if ($minutes == 1) {
                return sprintf(_("%d hours, %d minute"), $hours, $minutes);
            } else {
                return sprintf(_("%d hours, %d minutes"), $hours, $minutes);
            }
        } else if ($hours == 1) {
            if ($minutes == 0) {
                return sprintf(_("%d hour"), $hours);
            } else if ($minutes == 1) {
                return sprintf(_("%d hour, %d minute"), $hours, $minutes);
            } else {
                return sprintf(_("%d hour, %d minutes"), $hours, $minutes);
            }
        } else {
            if ($minutes == 0) {
                return _("no time");
            } else if ($minutes == 1) {
                return sprintf(_("%d minute"), $minutes);
            } else {
                return sprintf(_("%d minutes"), $minutes);
            }
        }
    }
    
    function recurToString($type)
    {
        switch ($type) {
        
        case KRONOLITH_RECUR_NONE:
            return _("Recurs not");
        case KRONOLITH_RECUR_DAILY:
            return _("Recurs daily");
        case KRONOLITH_RECUR_WEEKLY:
            return _("Recurs weekly");
        case KRONOLITH_RECUR_DAY_OF_MONTH:
        case KRONOLITH_RECUR_WEEK_OF_MONTH:
            return _("Recurs monthly");
        case KRONOLITH_RECUR_YEARLY:
            return _("Recurs yearly");
        
        }
    }
    
    
    /**
     * Returns week of the year, first Monday is first day of first week
     *
     * @param optional string $day    in format DD
     * @param optional string $month  in format MM
     * @param optional string $year   in format CCYY
     *
     * @return integer $week_number
     */
    function weekOfYear($day = null, $month = null, $year = null)
    {
        global $prefs;
        
        if (!isset($year)) $year = date('Y');
        if (!isset($month)) $month = date('m');
        if (!isset($day)) $day = date('d');
        if (!$prefs->getValue('week_start_monday') && Kronolith::dayOfWeek($year, $month, $day) == KRONOLITH_SUNDAY) {
            $day++;
        }

        $dayOfYear = Kronolith::dayOfYear($year,$month,$day);
        $dayOfWeek = Kronolith::dayOfWeek($year,$month,$day);
        $dayOfWeekJan1 = Kronolith::dayOfWeek($year,1,1);

        if ($dayOfYear <= 7-$dayOfWeekJan1 && $dayOfWeekJan1 > 3 ) {
             if ($dayOfWeekJan1 == 4 || ($dayOfWeekJan1 == 5 && Kronolith::isLeapYear($year-1)))
                  return "53";
             else
                  return "52";
        }

        if (Kronolith::isLeapYear($year))
            $daysInYear = 366;
        else
            $daysInYear = 365;

        if ($daysInYear - $dayOfYear < 3 - $dayOfWeek) return "1";

        $WeekNumber = floor(($dayOfYear + (6 - $dayOfWeek) + $dayOfWeekJan1) / 7);
        if ($dayOfWeekJan1 > 3) { $WeekNumber -= 1; }

        return $WeekNumber;
    }
    
    /**
     * Return the number of weeks in the given year (52 or 53).
     *
     * @param optional integer $year  The year to count the number of weeks in.
     *
     * @return integer $numWeeks      The number of weeks in $year.
     */
    function weeksInYear($year = null)
    {
        if (!isset($year)) $year = date('Y');
        
        // Find the last Thursday of the year.
        $day = 31;
        while (date('w', mktime(0, 0, 0, 12, $day, $year)) != 4) {
            $day--;
        }
        return Kronolith::weekOfYear($day, 12, $year);
    }
    
    /**
     * Returns the day of the year (1-366) that corresponds to the
     * first day of the given week.
     *
     * @param int week        The week of the year to find the first day of.
     *
     * @return int dayOfYear  The day of the year of the first day of the given week.
     */
    function firstDayOfWeek($week = null, $year = null)
    {
        if (!isset($year)) $year = date('Y');
        if (!isset($week)) $week = Kronolith::weekOfYear(null, null, $year);
        
        $start = Kronolith::dayOfWeek($year, 1, 1);
        if ($start > 3) $start -= 7;
        return ((($week * 7) - (7 + $start)) + 1);
    }
    
    /**
     * Find the number of days in the given month.
     *
     * @param string month in format MM, default current local month
     *
     * @return int number of days
     */
    function daysInMonth($month = null, $year = null)
    {
        if (!isset($month)) {
            $month = date('n');
        }
        if ($month == 2) {
            if (!isset($year))
                $year = date('Y');
            
            if (Kronolith::isLeapYear($year))
                return 29;
            else
                return 28;
        } elseif ($month == 4 || $month == 6 || $month == 9 || $month == 11) {
            return 30;
        } else {
            return 31;
        }
    }
    
    function isLeapYear($year = null)
    {
        if (!isset($year)) {
            $year = date('Y');
        } else {
            if (strlen($year) != 4)
                return false;
            
            if (preg_match('/\D/', $year))
                return false;
        }
        
        return (($year % 4 == 0 && $year % 100 != 0) || $year % 400 == 0);
    }
    
    /**
     * Return the day of the week (0 = Monday, 6 = Sunday) of the
     * given date.
     *
     * @param optional int $year
     * @param optional int $month
     * @param optional int $day
     *
     * @return int The day of the week.
     */
    function dayOfWeek($year = null, $month = null, $day = null)
    {
        if (!isset($year))  $year  = date('Y');
        if (!isset($month)) $month = date('m');
        if (!isset($day))   $day   = date('d');
        
        if ($month > 2)
            $month -= 2;
        else {
            $month += 10;
            $year--;
        }
        
        $day = (floor((13 * $month - 1) / 5) +
                $day + ($year % 100) +
                floor(($year % 100) / 4) +
                floor(($year / 100) / 4) - 2 *
                floor($year / 100) + 77);
        
        $weekday_number = (($day - 7 * floor($day / 7))) - 1;
        if ($weekday_number == -1) $weekday_number = 6; // wrap check
        
        return $weekday_number;
    }
    
    function dayOfYear($year = null, $month = null, $day = null)
    {
        if (!isset($year))  $year  = date('Y');
        if (!isset($month)) $month = date('m');
        if (!isset($day))   $day   = date('d');
        
        $days = 0;
        for ($i = 1; $i < $month; $i++)
            $days += Kronolith::daysInMonth($i, $year);
        $days += $day;
        
        return $days;
    }
    
    function compareDates($first, $second)
    {
        $first = Kronolith::dateObject($first);
        $second = Kronolith::dateObject($second);
        
        if ($first->year - $second->year != 0) {
            return $first->year - $second->year;
        } elseif ($first->month - $second->month != 0) {
            return $first->month - $second->month;
        } elseif ($first->mday - $second->mday != 0) {
            return $first->mday - $second->mday;
        } elseif ($first->hour - $second->hour != 0) {
            return $first->hour - $second->hour;
        } elseif ($first->min - $second->min != 0) {
            return $first->min - $second->min;
        } else {
            return $first->sec - $second->sec;
        }
    }
    
    function dateDiff($start, $end, $split = false)
    {
        $start = Kronolith::dateObject($start);
        $end = Kronolith::dateObject($end);
        
        $res->year = $end->year - $start->year;
        $res->month = $end->month - $start->month;
        $res->mday = $end->mday - $start->mday;
        $res->hour = $end->hour - $start->hour;
        $res->min = $end->min - $start->min;
        $res->sec = $end->sec - $start->sec;
        if (!$split) {
            $res->month += $res->year * 12;
            $res->mday = Date_Calc::dateDiff($start->mday, $start->month, $start->year, $end->mday, $end->month, $end->year);
            $res->hour += $res->mday * 24;
            $res->min += $res->hour * 60;
            $res->sec += $res->min * 60;
        }
        
        return $res;
    }
    
    function correctDate($date)
    {
        if (isset($date->sec)) {
            while ($date->sec > 59) {
                $date->min++;
                $date->sec -= 60;
            }
        }
        if (isset($date->min)) {
            while ($date->min > 59) {
                $date->hour++;
                $date->min -= 60;
            }
        }
        if (isset($date->hour)) {
            while ($date->hour > 23) {
                $date->mday++;
                $date->hour -= 24;
            }
        }
        if (isset($date->mday)) {
            while ($date->mday > Date_Calc::daysInMonth($date->month, $date->year)) {
                $date->month++;
                $date->mday -= Date_Calc::daysInMonth($date->month - 1, $date->year);
            }
        }
        if (isset($date->month)) {
            while ($date->month > 12) {
                $date->year++;
                $date->month -= 12;
            }
        }
        
        return $date;
    }
    
    function timestampToObject($timestamp)
    {
        list($res->hour, $res->min, $res->sec, $res->mday, $res->month, $res->year) = explode('/', date('H/i/s/d/m/Y', $timestamp));
        
        return $res;
    }
    
    function timestampToArray($timestamp)
    {
        $obj = Kronolith::timestampToObject($timestamp);
        
        return array('hour' => $obj->hour,
                     'min' => $obj->min,
                     'sec' => $obj->sec,
                     'month' => $obj->month,
                     'mday' => $obj->mday,
                     'year' => $obj->year);
    }
    
    function objectToTimestamp($obj)
    {
        return @mktime($obj->hour, $obj->min, $obj->sec, $obj->month, $obj->mday, $obj->year);
    }
    
    function arrayToTimestamp($arr)
    {
        return @mktime($arr['hour'], $arr['min'], $arr['sec'], $arr['month'], $arr['mday'], $arr['year']);
    }
    
}
?>
