<?php
// $Horde: kronolith/lib/Driver/sql.php,v 1.27 2001/12/10 17:51:19 chuck Exp $

/**
 * The Kronolith_Driver_sql:: class implements the Kronolith_Driver
 * API for a SQL backend.
 *
 * Table groups contains information for a group. Like the main owner
 * (which can change access to the group), and access mode (public,
 * private, share). For now, a group can't contains another group. If
 * we do this, we need another sub select request. Perhaps we can
 * propose this with a id like Huffman code. The can only have 26
 * subcategory.  A-->Z then AA-->AZ ... ZA-->ZZ then AAA--> .... -->
 * ZZZ.  So if we wen search main category , we search with A%
 *
 * @author  Luc Saillard <luc.saillard@fr.alcove.com>
 * @version $Revision: 1.27 $
 * @since   Kronolith 0.3
 * @package kronolith
 */
class Kronolith_Driver_sql {
    
    /** Hash containing connection parameters. */
    var $params = array();
    
    /** Handle for the current database connection. */
    var $db;
    
    /** Boolean indicating whether or not we're connected to the SQL server. */
    var $connected = false;
	
    /** The calendar to open. */
    var $calendar;
    
    function Kronolith_Driver_sql($params)
    {
        $this->params = $params;
    }
    
    /**
     * Attempts to open a persistent connection to the SQL server.
     *
     * @return boolean True.
     */
    function _connect()
    {
        if (!$this->connected) {
            include_once 'DB.php';

            if (!is_array($this->params)) {
                Horde::fatal(new PEAR_Error(_("No configuration information specified for SQL Calendar.")), __FILE__, __LINE__);
            }
            if (!isset($this->params['phptype'])) {
                Horde::fatal(new PEAR_Error(_("Required 'phptype' not specified in calendar configuration.")), __FILE__, __LINE__);
            }
            if (!isset($this->params['hostspec'])) {
                Horde::fatal(new PEAR_Error(_("Required 'hostspec' not specified in calendar configuration.")), __FILE__, __LINE__);
            }
            if (!isset($this->params['username'])) {
                Horde::fatal(new PEAR_Error(_("Required 'username' not specified in calendar configuration.")), __FILE__, __LINE__);
            }
            if (!isset($this->params['password'])) {
                Horde::fatal(new PEAR_Error(_("Required 'password' not specified in calendar configuration.")), __FILE__, __LINE__);
            }
            if (!isset($this->params['database'])) {
                Horde::fatal(new PEAR_Error(_("Required 'database' not specified in calendar configuration.")), __FILE__, __LINE__);
            }

            /* Connect to the SQL server using the supplied parameters. */
            $this->db = &DB::connect($this->params, true);
            if (DB::isError($this->db) || DB::isWarning($this->db)) {
                Horde::fatal($this->db, __FILE__, __LINE__);
            }

            $this->connected = true;
        }

        return true;
    }
	
    /**
     * Disconnect from the SQL server and clean up the connection.
     *
     * @return boolean true on success, false on failure.
     */
    function _disconnect()
    {
        if ($this->connected) {
            $this->connected = false;
            return $this->db->disconnect();
        }

        return true;
    }
	
    function open($calendar)
    {
        $this->calendar = $calendar;
        $this->_connect();
    }

    function listAlarms($date)
    {
        // TODO:
        $events = array();
        return is_array($events) ? $events : array();
    }

    function listEvents($startDate = null, $endDate = null)
    {
        $events = array();
        
        if (!isset($startDate)) {
            $startDate = Kronolith::dateObject();
        }
        if (!isset($endDate)) {
            $endDate = Kronolith::dateObject(array('mday' => 31, 'month' => 12, 'year' => 9999));
        } else {
            list($endDate->mday, $endDate->month, $endDate->year) = explode('/', Date_Calc::nextDay($endDate->mday, $endDate->month, $endDate->year, '%d/%m/%Y'));
        }

        $stime = sprintf("'%04d-%02d-%02d 00:00:00.000'", $startDate->year, $startDate->month, $startDate->mday);
        $etime = sprintf("'%04d-%02d-%02d 00:00:00.000'", $endDate->year, $endDate->month, $endDate->mday);

        $q = 'SELECT DISTINCT e.event_id, e.event_recurtype FROM ' . $this->params['table'] . ' e' .
             ' WHERE e.calendar_id = ' . $this->db->quote($this->calendar) .
             ' AND ((e.event_start < ' . $etime .
             ' AND e.event_end > ' . $stime . ')' .
             ' OR (e.event_recurenddate >= ' . $stime .
             ' AND e.event_start <= ' . $etime .
             ' AND e.event_recurtype != ' . KRONOLITH_RECUR_NONE . '))';
        
        $qr = $this->db->query($q);
        
        if (!DB::isError($qr)) {
            $row = $qr->fetchRow(DB_FETCHMODE_ASSOC);
            while ($row && !DB::isError($row)) {
                if ($row['event_recurtype'] == KRONOLITH_RECUR_NONE) {
                    $events[] = $row['event_id'];
                } else {
                    $next = $this->nextRecurrence($row['event_id'], $startDate);
                    if ($next && Kronolith::compareDates($next, $endDate) < 0) {
                        $events[] = $row['event_id'];
                    }
                }
                    
                $row = $qr->fetchRow(DB_FETCHMODE_ASSOC);
            }
        } else {
            var_dump($qr);
        }
        
        return $events;
    }
    
    function getEvent($eventID)
    {
        $qr = $this->db->query('SELECT event_id, event_description,event_location,event_keywords,event_title,event_category,' .
                               'event_recurtype, event_recurenddate,event_recurinterval,event_recurdays,event_start,event_end,event_alarm' .
                               ' FROM ' . $this->params['table'] . ' WHERE event_id = ' . $eventID);
        
        if (!DB::isError($qr)) {
            $row = $qr->fetchRow(DB_FETCHMODE_ASSOC);
            if ($row && !DB::isError($row)) {
                return $row;
            }
        }
        
        var_dump($qr);
        return false;
    }
    
    function getEventObject($eventID = null)
    {
        if (isset($eventID)) {
            return new Kronolith_Event_sql($this, $this->getEvent($eventID, true));
        } else {
            return new Kronolith_Event_sql($this);
        }
    }
    
    function saveEvent($eventID, $properties)
    {
        if ($eventID !== null) {
            $query = 'UPDATE ' . $this->params['table'] . ' SET ';
            
            foreach ($properties as $key => $val) {
                $query .= " $key = " . $this->db->quote($val) . ',';
            }
            $query = substr($query, 0, -1);
            $query .= " WHERE event_id = $eventID ";
            
            if (DB::isError($res = $this->db->query($query))) {
                var_dump($res);
                return false;
            }
            
            return true;
        } else {
            $id = $this->db->nextId($this->params['table']);
            if (DB::isError($id)) {
                var_dump($id);
                return $id;
            }
            
            $query = 'INSERT INTO '.$this->params['table'].' ';
            $cols_name = '(event_id,';
            $cols_values = 'values (' . $id . ',';
            
            foreach ($properties as $key => $val) {
                $cols_name .= " $key,";
                $cols_values .= $this->db->quote($val) . ',';
            }
            
            $cols_name .= ' calendar_id)';
            $cols_values .= $this->db->quote($this->calendar) . ')';
            
            if (DB::isError($res = $this->db->query($query . $cols_name . $cols_values))) {
                var_dump($res);
                return false;
            }
            
            return true;
        }
    }
    
    function nextRecurrence($eventID, $afterDate, $weekstart = KRONOLITH_SUNDAY)
    {
        $event = $this->getEventObject($eventID);
        $afterDate = Kronolith::dateObject($afterDate);
        
        if (Kronolith::compareDates($event->start, $afterDate) > 0) {
            return $event->start;
        }
        
        $event->recurEnd->hour = 23;
        $event->recurEnd->min  = 59;
        $event->recurEnd->sec  = 59;
        
        switch ($event->recurType) {
        
        case KRONOLITH_RECUR_DAILY:
            $diff = Kronolith::dateDiff($event->start, $afterDate);
            $recur = (ceil($diff->mday / $event->recurInterval)) * $event->recurInterval;
            $next = $event->start;
            list($next->mday, $next->month, $next->year) = explode('/', Date_Calc::daysToDate(Date_Calc::dateToDays($next->mday, $next->month, $next->year) + $recur, '%e/%m/%Y'));
            if (Kronolith::compareDates($next, $event->recurEnd) <= 0 &&
                Kronolith::compareDates($next, $afterDate) >= 0) {
                return $next;
            }
            break;
        
        case KRONOLITH_RECUR_WEEKLY:
            list($start_week->mday, $start_week->month, $start_week->year) = explode('/', Date_Calc::beginOfWeek($event->start->mday, $event->start->month, $event->start->year, '%e/%m/%Y'));
            $start_week->hour = $event->start->hour;
            $start_week->min = $event->start->min;
            $start_week->sec = $event->start->sec;
            list($after_week->mday, $after_week->month, $after_week->year) = explode('/', Date_Calc::beginOfWeek($afterDate->mday, $afterDate->month, $afterDate->year, '%e/%m/%Y'));
            $after_week_end = $after_week;
            $after_week_end->mday += 7;
            $after_week_end = Kronolith::correctDate($after_week_end);
            $diff = Kronolith::dateDiff($start_week, $after_week);
            $recur = $diff->mday + $diff->mday % ($event->recurInterval * 7);
            $next = $start_week;
            list($next->mday, $next->month, $next->year) = explode('/', Date_Calc::daysToDate(Date_Calc::dateToDays($next->mday, $next->month, $next->year) + $recur, '%e/%m/%Y'));
            while (Kronolith::compareDates($next, $afterDate) < 0 && Kronolith::compareDates($next, $after_week_end) < 0) {
                $next->mday++;
                $next = Kronolith::correctDate($next);
            }
            if (Kronolith::compareDates($next, $event->recurEnd) <= 0) {
                if (Kronolith::compareDates($next, $after_week_end) >= 0) {
                    return $GLOBALS['calendar']->nextRecurrence($eventID, $after_week_end);
                }
                while (!$event->recurOnDay((int)pow(2, (int)Date_Calc::dayOfWeek($next->mday, $next->month, $next->year))) && Kronolith::compareDates($next, $after_week_end) < 0) {
                    $next->mday++;
                    $next = Kronolith::correctDate($next);
                }
                if (Kronolith::compareDates($next, $event->recurEnd) <= 0) {
                    if (Kronolith::compareDates($next, $after_week_end) >= 0) {
                        return $GLOBALS['calendar']->nextRecurrence($eventID, $after_week_end);
                    } else {
                        return Kronolith::dateObject($next);
                    }
                }
            }
            break;
            
        case KRONOLITH_RECUR_DAY_OF_MONTH:
            $diff = Kronolith::dateDiff($event->start, $afterDate);
            $recur = $diff->month + $diff->month % $event->recurInterval;
            $next = $event->start;
            $next->month += $recur;
            $next = Kronolith::correctDate($next);
            if (Kronolith::compareDates($next, $event->recurEnd) <= 0 &&
                Kronolith::compareDates($next, $afterDate) >= 0) {
                return $next;
            }
            break;
        
        case KRONOLITH_RECUR_WEEK_OF_MONTH:
            break;
        
        case KRONOLITH_RECUR_YEARLY:
            $diff = Kronolith::dateDiff($event->start, $afterDate);
            $recur = $diff->year + $diff->year % $event->recurInterval;
            $next = $event->start;
            $next->year += $recur;
            if (Kronolith::compareDates($next, $event->recurEnd) <= 0 &&
                Kronolith::compareDates($next, $afterDate) >= 0) {
                return $next;
            }
            break;
            
        }
        
        return false;
    }
    
    function deleteEvent($eventID)
    {
        $query = 'DELETE from ' . $this->params['table'] . " WHERE event_id = $eventID";
        if (DB::isError($res = $this->db->query($query))) {
            var_dump($res);
            return false;
        }
        
        return true;
    }
    
}

class Kronolith_Event_sql extends Kronolith_Event {
    
    var $initialized = false;
    
    function Kronolith_Event_sql(&$driver, $eventObject = null)
    {
        parent::Kronolith_Event($driver, $eventObject);
        
        $this->properties = array();
        $this->eventID = null;
        
        if (isset($eventObject)) {
            $this->parseSQLEvent($eventObject);
        }
    }
    
    function save()
    {
        if (!$this->initialized) {
            return false;
        }
        
        return $this->driver->saveEvent($this->eventID, $this->properties);
    }
    
    function parseSQLEvent($SQLEvent)
    {
        list($this->start->year, $this->start->month, $this->start->mday, $this->start->hour, $this->start->min, $this->start->sec) = sscanf($SQLEvent['event_start'], '%04d-%02d-%02d %02d:%02d:%02d');
        list($this->end->year, $this->end->month, $this->end->mday, $this->end->hour, $this->end->min, $this->end->sec) = sscanf($SQLEvent['event_end'], '%04d-%02d-%02d %02d:%02d:%02d');
        
        $this->startTimestamp = mktime($this->start->hour, $this->start->min, $this->start->sec, $this->start->month, $this->start->mday, $this->start->year);
        $this->endTimestamp = mktime($this->end->hour, $this->end->min, $this->end->sec, $this->end->month, $this->end->mday, $this->end->year);
        
        $this->durMin = ($this->endTimestamp - $this->startTimestamp) / 60;
        
        if (isset($SQLEvent['event_recurenddate'])) {
            list($this->recurEnd->year, $this->recurEnd->month, $this->recurEnd->mday, $this->recurEnd->hour, $this->recurEnd->min, $this->recurEnd->sec) = sscanf($SQLEvent['event_recurenddate'], '%04d-%02d-%02d %02d:%02d:%02d');
            $this->recurEndTimestamp = mktime($this->recurEnd->hour, $this->recurEnd->min, $this->recurEnd->sec, $this->recurEnd->month, $this->recurEnd->mday, $this->recurEnd->year);
        }
        
        $this->title = $SQLEvent['event_title'];
        if (isset($SQLEvent['event_category'])) {
            $this->category = $SQLEvent['event_category'];
        }
        if (isset($SQLEvent['event_location'])) {
            $this->location = $SQLEvent['event_location'];
        }
        if (isset($SQLEvent['event_keywords'])) {
            $this->keywords = explode(',', $SQLEvent['event_keywords']);
        }
        if (isset($SQLEvent['event_description'])) {
            $this->description = $SQLEvent['event_description'];
        }
        $this->eventID = $SQLEvent['event_id'];
        
        if (isset($SQLEvent['event_alarm'])) {
            $this->alarm = (int)$SQLEvent['event_alarm'];
        }
        
        $this->recurType = (int)$SQLEvent['event_recurtype'];
        $this->recurInterval = (int)$SQLEvent['event_recurinterval'];
        
        if (isset($SQLEvent['event_recurdays'])) {
            $this->recurData = (int)$SQLEvent['event_recurdays'];
        }
        
        $this->initialized = true;
    }
    
    function createEvent()
    {
        if (isset($this->title))
            $this->properties['event_title'] = $this->title;
        if (isset($this->description))
            $this->properties['event_description'] = $this->description;
        if (isset($this->category))
            $this->properties['event_category'] = $this->category;
        if (isset($this->location))
            $this->properties['event_location'] = $this->location;
        if (isset($this->keywords))
            $this->properties['event_keywords'] = implode(',', $this->keywords);
        if (isset($this->startTimestamp)) {
            $this->properties['event_start'] = date('Y-m-d H:i:s', $this->startTimestamp);
        }
        if (isset($this->endTimestamp)) {
            $this->properties['event_end'] = date('Y-m-d H:i:s', $this->endTimestamp);
        }
        if (isset($this->alarm)) {
            $this->properties['event_alarm'] = $this->alarm;
        }
        if (isset($this->recurType)) {
            $this->properties['event_recurtype'] = $this->recurType;
            if (isset($this->recurEndTimestamp)) {
                $this->properties['event_recurenddate'] = date('Y-m-d H:i:s', $this->recurEndTimestamp);
            }
            if (!isset($this->recurInterval)) $this->recurInterval = 1;
            $this->properties['event_recurinterval'] = $this->recurInterval;
            $this->properties['event_recurdays'] = $this->recurData;
        }
        $this->initialized = true;
    }
    
    function readFormData($eventData)
    {
        global $prefs;
        
        // basic fields
        if (isset($eventData['title'])) {
            $this->properties['event_title'] = Horde::dispelMagicQuotes($eventData['title']);
        }
        if (isset($eventData['description'])) {
            $this->properties['event_description'] = Horde::dispelMagicQuotes($eventData['description']);
        }
        if (isset($eventData['category'])) {
            $this->properties['event_category'] = Horde::dispelMagicQuotes($eventData['category']);
        }
        if (isset($eventData['location'])) {
            $this->properties['event_location'] = Horde::dispelMagicQuotes($eventData['location']);
        }
        if (isset($eventData['keywords'])) {
            $this->properties['event_keywords'] = implode(',', $eventData['keywords']);
        }
        
        // event start
        if (isset($eventData['start_year'])
            && isset($eventData['start_month'])
            && isset($eventData['start_day'])
            && isset($eventData['start_hour'])
            && isset($eventData['start_min'])
            && (isset($eventData['am_pm'])
                || ($prefs->getValue('24hr')))) {
            if (!$prefs->getValue('24hr')) {
                if ($eventData['am_pm'] == 'PM') {
                    if ($eventData['start_hour'] != 12)
                        $eventData['start_hour'] += 12;
                } elseif ($eventData['start_hour'] == 12) {
                    $eventData['start_hour'] = 0;
                }
            }
            if (isset($eventData['end_or_dur'])
                && $eventData['end_or_dur'] == 1
                && isset($eventData['whole_day'])
                && $eventData['whole_day'] == 1) {
                $eventData['start_hour'] = 0;
                $eventData['start_min'] = 0;
                $eventDate['dur_day'] = 0;
                $eventData['dur_hour'] = 24;
                $eventData['dur_min'] = 0;
            }
            $this->properties['event_start'] = sprintf('%04d-%02d-%02d %02d:%02d:00',
                                                       $eventData['start_year'],$eventData['start_month'],$eventData['start_day'],
                                                       $eventData['start_hour'],$eventData['start_min']);
            
            if (isset($eventData['end_or_dur'])
                && $eventData['end_or_dur'] == 1) {
                // event duration
                if (isset($eventData['dur_day']) 
                    && isset($eventData['dur_hour']) 
                    && isset($eventData['dur_min'])) {
                    $endstamp = mktime($eventData['start_hour'] + $eventData['dur_hour'],
                                       $eventData['start_min'] + $eventData['dur_min'],
                                       0,
                                       $eventData['start_month'],
                                       $eventData['start_day'] + $eventData['dur_day'],
                                       $eventData['start_year']);
                    
                    $this->properties['event_end'] = date('Y-m-d H:i:s', $endstamp);
                }
            } else {
                // event end
                if (isset($eventData['end_year'])
                    && isset($eventData['end_month'])
                    && isset($eventData['end_day'])
                    && isset($eventData['end_hour'])
                    && isset($eventData['end_min'])
                    && (isset($eventData['end_am_pm'])
                        || ($prefs->getValue('24hr')))) {
                    if (!$prefs->getValue('24hr')) {
                        if ($eventData['end_am_pm'] == 'PM') {
                            if ($eventData['end_hour'] != 12)
                                $eventData['end_hour'] += 12;
                        } elseif ($eventData['end_hour'] == 12) {
                            $eventData['end_hour'] = 0;
                        }
                    }
                    $endstamp = mktime($eventData['end_hour'],
                                       $eventData['end_min'],
                                       0,
                                       $eventData['end_month'],
                                       $eventData['end_day'],
                                       $eventData['end_year']);
                    //                    if ($endstamp < $startstamp)
                    //                        $endstamp = $startstamp;
                    $this->properties['event_end'] = date('Y-m-d H:i:s', $endstamp);
                }
            }
        }
        
        // alarm
        
        if (isset($eventData['alarm'])) {
            if ($eventData['alarm'] == 1 &&
                isset($eventData['alarm_value']) &&
                isset($eventData['alarm_unit'])) {
                $this->properties['event_alarm'] = ($eventData['alarm_value'] * $eventData['alarm_unit']);
            } else {
                $this->properties['event_alarm'] = 0;
            }
        }
        
        // recurrence
        if (isset($eventData['recur'])) {
            if (isset($eventData['recur_enddate_type']) &&
                $eventData['recur_enddate_type'] == 'none') {
                $eventData['recur_enddate_year'] = 9999;
                $eventData['recur_enddate_month'] = 12;
                $eventData['recur_enddate_day'] = 31;
            }
            
            switch ($eventData['recur']) {
            case KRONOLITH_RECUR_NONE:
                $this->properties['event_recurtype']=KRONOLITH_RECUR_NONE;
                break;
                
            case KRONOLITH_RECUR_DAILY:
                if (isset($eventData['recur_enddate_year'])
                    && isset($eventData['recur_enddate_month'])
                    && isset($eventData['recur_enddate_day'])) {
                    if (empty($eventData['recur_daily_interval'])) {
                        $eventData['recur_daily_interval'] = 1;
                    }
                    
                    $this->properties['event_recurtype'] = KRONOLITH_RECUR_DAILY;
                    $this->properties['event_recurinterval'] = $eventData['recur_daily_interval'];
                    $this->properties['event_recurenddate'] = sprintf("%04d%02d%02d",
                                                                      $eventData['recur_enddate_year'],
                                                                      $eventData['recur_enddate_month'],
                                                                      $eventData['recur_enddate_day']);
                }
                break;
                
            case KRONOLITH_RECUR_WEEKLY:
                if (isset($eventData['recur_enddate_year'])
                    && isset($eventData['recur_enddate_month'])
                    && isset($eventData['recur_enddate_day'])
                    && isset($eventData['weekly'])
                    && is_array($eventData['weekly'])) {
                    $weekdays = 0;
                    foreach ($eventData['weekly'] as $day) {
                        $weekdays |= $day;
                    }
                    if (empty($eventData['recur_weekly_interval'])) {
                        $eventData['recur_weekly_interval'] = 1;
                    }
                    
                    $this->properties['event_recurtype'] = KRONOLITH_RECUR_WEEKLY;
                    $this->properties['event_recurinterval'] = $eventData['recur_weekly_interval'];
                    $this->properties['event_recurdays'] = $weekdays;
                    $this->properties['event_recurenddate'] = sprintf("%04d%02d%02d",
                                                                      $eventData['recur_enddate_year'],
                                                                      $eventData['recur_enddate_month'],
                                                                      $eventData['recur_enddate_day']);
                }
                break;
                
            case KRONOLITH_RECUR_DAY_OF_MONTH:
                if (isset($eventData['recur_enddate_year'])
                    && isset($eventData['recur_enddate_month'])
                    && isset($eventData['recur_enddate_day'])) {
                    if (empty($eventData['recur_day_of_month_interval']))
                        $eventData['recur_day_of_month_interval'] = 1;
                    $this->properties['event_recurtype'] = KRONOLITH_RECUR_DAY_OF_MONTH;
                    $this->properties['event_recurinterval'] = $eventData['recur_day_of_month_interval'];
                    $this->properties['event_recurenddate'] = sprintf("%04d%02d%02d",
                                                                      $eventData['recur_enddate_year'],
                                                                      $eventData['recur_enddate_month'],
                                                                      $eventData['recur_enddate_day']);
                }
                break;
                
            case KRONOLITH_RECUR_WEEK_OF_MONTH:
                if (isset($eventData['recur_enddate_year'])
                    && isset($eventData['recur_enddate_month'])
                    && isset($eventData['recur_enddate_day'])) {
                    if (empty($eventData['recur_week_of_month_interval'])) $eventData['recur_week_of_month_interval'] = 1;
                    $this->properties['event_recurtype'] = KRONOLITH_RECUR_WEEK_OF_MONTH;
                    $this->properties['event_recurinterval'] = $eventData['recur_week_of_month_interval'];
                    $this->properties['event_recurenddate'] = sprintf("%04d%02d%02d",
                                                                    $eventData['recur_enddate_year'],
                                                                    $eventData['recur_enddate_month'],
                                                                    $eventData['recur_enddate_day']);
                }
                break;
                
            case KRONOLITH_RECUR_YEARLY:
                if (isset($eventData['recur_enddate_year'])
                    && isset($eventData['recur_enddate_month'])
                    && isset($eventData['recur_enddate_day'])) {
                    if (empty($eventData['recur_yearly_interval'])) $eventData['recur_yearly_interval'] = 1;
                    
                    $this->properties['event_recurtype'] = KRONOLITH_RECUR_YEARLY;
                    $this->properties['event_recurinterval'] = $eventData['recur_yearly_interval'];
                    $this->properties['event_recurenddate'] = sprintf("%04d%02d%02d",
                                                                      $eventData['recur_enddate_year'],
                                                                      $eventData['recur_enddate_month'],
                                                                      $eventData['recur_enddate_day']);
                }
                break;
                
            }
        }
        
        $this->initialized = true;
    }
    
}
?>
