<?php
// $Horde: kronolith/lib/Driver/mcal.php,v 1.23 2001/10/28 13:17:35 jan Exp $

/**
 * The Kronolith_Driver_mcal:: class implements the Kronolith_Driver
 * API for an MCAL backend.
 *
 * @author  Chuck Hagenbuch <chuck@horde.org>
 * @version $Revision: 1.23 $
 * @since   Kronolith 0.1
 * @package kronolith
 */
class Kronolith_Driver_mcal {
    
    var $mcalUsername;
    var $mcalPassword;
    var $stream;
    
    function Kronolith_Driver_mcal($params)
    {
        if (empty($params['username']) || empty($params['password'])) {
            return false;
        }
        
        $this->mcalUsername = $params['username'];
        $this->mcalPassword = $params['password'];
    }
    
    function open($calendar)
    {
        $this->stream = @mcal_popen("{/mstore}<$calendar>", $this->mcalUsername, $this->mcalPassword);
    }
    
    function listEvents($startDate = null, $endDate = null)
    {
        $events = @mcal_list_events($this->stream, $startDate->year, $startDate->month, $startDate->mday, $endDate->year, $endDate->month, $endDate->mday);
        return is_array($events) ? $events : array();
    }
    
    function listAlarms($date)
    {
        $events = @mcal_list_alarms($this->stream, $date->year, $date->month, $date->mday, $date->hour, $date->min, $date->sec);
        return is_array($events) ? $events : array();
    }
    
    function getEvent($eventID)
    {
        return mcal_fetch_event($this->stream, $eventID);
    }
    
    function getEventObject($eventID = null)
    {
        if (isset($eventID))
            return new Kronolith_Event_mcal($this, $this->getEvent($eventID));
        else
            return new Kronolith_Event_mcal($this);
    }
    
    function saveEvent($eventExists)
    {
        if ($eventExists) {
            return mcal_store_event($this->stream);
        } else {
            return mcal_append_event($this->stream);
        }
    }
    
    function nextRecurrence($eventID, $afterDate, $weekstart = KRONOLITH_SUNDAY)
    {
        return Kronolith::dateObject(mcal_next_recurrence($this->stream, $weekstart, $afterDate));
    }
    
    function deleteEvent($eventID)
    {
        return mcal_delete_event($this->stream, $eventID);
    }
    
    function parseMcalDate($dateObject)
    {
        if (count($dateObject) === 0) {
            return 0;
        }
        
        $year = isset($dateObject->year) ? $dateObject->year : 0;
        $month = isset($dateObject->month) ? $dateObject->month : 0;
        $day = isset($dateObject->mday) ? $dateObject->mday : 0;
        
        // Check for events with no recur_enddate
        if ($year == 9999 && $month == 12 && $day == 31) {
            return 0;
        }
        
        $hour = isset($dateObject->hour) ? $dateObject->hour : 0;
        $minute = isset($dateObject->min) ? $dateObject->min : 0;
        $second = isset($dateObject->sec) ? $dateObject->sec : 0;
        
        return mktime($hour, $minute, $second, $month, $day, $year);
    }
    
}

class Kronolith_Event_mcal extends Kronolith_Event {
    
    var $initialized = false;
    
    function Kronolith_Event_mcal(&$driver, $eventObject = null)
    {
        parent::Kronolith_Event($driver, $eventObject);
        
        if (isset($eventObject)) {
            $this->parseMcalEvent($eventObject);
        }
    }
    
    function save()
    {
        if (!$this->initialized) {
            return false;
        }
        
        return $this->driver->saveEvent(isset($this->eventID));
    }
    
    function parseMcalEvent($mcalEvent)
    {
        $this->title = $mcalEvent->title;
        if (isset($mcalEvent->category)) {
            $this->category = $mcalEvent->category;
        }
        if (isset($mcalEvent->attrlist['location'])) {
            $this->location = $mcalEvent->attrlist['location'];
        }
        if (isset($mcalEvent->attrlist['keywords'])) {
            $this->keywords = explode(',', $mcalEvent->attrlist['keywords']);
        }
        if (isset($mcalEvent->description)) {
            $this->description = $mcalEvent->description;
        }
        $this->eventID = $mcalEvent->id;
        
        $this->startTimestamp = Kronolith_Driver_mcal::parseMcalDate($mcalEvent->start);
        $this->start = Kronolith::timestampToObject($this->startTimestamp);
        
        $this->endTimestamp = Kronolith_Driver_mcal::parseMcalDate($mcalEvent->end);
        $this->end = Kronolith::timestampToObject($this->endTimestamp);
        
        $this->durMin = ($this->endTimestamp - $this->startTimestamp) / 60;
        
        if (isset($mcalEvent->recur_enddate)) {
            $this->recurEndTimestamp = Kronolith_Driver_mcal::parseMcalDate($mcalEvent->recur_enddate);
            $this->recurEnd = $mcalEvent->recur_enddate;
        }
        
        $this->alarm = $mcalEvent->alarm;
        
        $this->recurType = $mcalEvent->recur_type;
        $this->recurInterval = $mcalEvent->recur_interval;
        if (isset($mcalEvent->recur_data)) {
            $this->recurData = $mcalEvent->recur_data;
        }
        
        $this->initialized = true;
    }
    
    function createEvent()
    {
        if (isset($this->title))
            mcal_event_set_title($this->driver->stream, $this->title);
        if (isset($this->description))
            mcal_event_set_description($this->driver->stream, $this->description);
        if (isset($this->category))
            mcal_event_set_category($this->driver->stream, $this->category);
        if (isset($this->location))
            mcal_event_add_attribute($this->driver->stream, 'location', $this->location);
        if (isset($this->keywords))
            mcal_event_add_attribute($this->driver->stream, 'keywords', implode(',', $this->keywords));
        if (isset($this->startTimestamp)) {
            list($year, $month, $day) = explode(':', date('Y:n:j', $this->startTimestamp));
            list($hour, $min, $sec) = explode(':', date('G:i:s', $this->startTimestamp));
            mcal_event_set_start($this->driver->stream, $year, $month, $day, $hour, $min, $sec);
        }
        if (isset($this->endTimestamp)) {
            list($year, $month, $day) = explode(':', date('Y:n:j', $this->endTimestamp));
            list($hour, $min, $sec) = explode(':', date('G:i:s', $this->endTimestamp));
            mcal_event_set_end($this->driver->stream, $year, $month, $day, $hour, $min, $sec);
        }
        if (isset($this->recurType)) {
            switch ($this->recurType) {
            case KRONOLITH_RECUR_NONE:
                mcal_event_set_recur_none($this->driver->stream);
                break;

            case KRONOLITH_RECUR_DAILY:
                if (isset($this->recurEndTimestamp)) {
                    if (!isset($this->recurInterval)) $this->recurInterval = 1;
                    list($year, $month, $day) = explode(':', date('Y:n:j', $this->recurEndTimestamp));
                    mcal_event_set_recur_daily($this->driver->stream, $year, $month, $day, $this->recurInterval);
                }
                break;

            case KRONOLITH_RECUR_WEEKLY:
                if (isset($this->recurEndTimestamp) && isset($this->recurData)) {
                    if (!isset($this->recurInterval)) $this->recurInterval = 1;
                    list($year, $month, $day) = explode(':', date('Y:n:j', $this->recurEndTimestamp));
                    mcal_event_set_recur_weekly($this->driver->stream, $year, $month, $day, $this->recurInterval, $this->recurData);
                }
                break;

            case KRONOLITH_RECUR_DAY_OF_MONTH:
                if (isset($this->recurEndTimestamp)) {
                    if (!isset($this->recurInterval)) $this->recurInterval = 1;
                    list($year, $month, $day) = explode(':', date('Y:n:j', $this->recurEndTimestamp));
                    mcal_event_set_recur_monthly_mday($this->driver->stream, $year, $month, $day, $this->recurInterval);
                }
                break;

            case KRONOLITH_RECUR_WEEK_OF_MONTH:
                if (isset($this->recurEndTimestamp)) {
                    if (!isset($this->recurInterval)) $this->recurInterval = 1;
                    list($year, $month, $day) = explode(':', date('Y:n:j', $this->recurEndTimestamp));
                    mcal_event_set_recur_monthly_wday($this->driver->stream, $year, $month, $day, $this->recurInterval);
                }
                break;

            case KRONOLITH_RECUR_YEARLY:
                if (isset($this->recurEndTimestamp)) {
                    if (!isset($this->recurInterval)) $this->recurInterval = 1;
                    list($year, $month, $day) = explode(':', date('Y:n:j', $this->recurEndTimestamp));
                    mcal_event_set_recur_yearly($this->driver->stream, $year, $month, $day, $this->recurInterval);
                }
                break;

            }
        }
        $this->initialized = true;
    }

    function readFormData($eventData)
    {
        global $prefs;
        
        // basic fields
        if (isset($eventData['title'])) {
            mcal_event_set_title($this->driver->stream, Horde::dispelMagicQuotes($eventData['title']));
        }
        if (isset($eventData['description'])) {
            mcal_event_set_description($this->driver->stream, Horde::dispelMagicQuotes($eventData['description']));
        }
        if (isset($eventData['category'])) {
            mcal_event_set_category($this->driver->stream, Horde::dispelMagicQuotes($eventData['category']));
        }
        if (isset($eventData['location'])) {
            mcal_event_add_attribute($this->driver->stream, 'location', Horde::dispelMagicQuotes($eventData['location']));
        }
        if (isset($eventData['keywords'])) {
            mcal_event_add_attribute($this->driver->stream, 'keywords', implode(',', $eventData['keywords']));
        }
        
        // event start
        if (isset($eventData['start_year'])
            && isset($eventData['start_month'])
            && isset($eventData['start_day'])
            && isset($eventData['start_hour'])
            && isset($eventData['start_min'])
            && (isset($eventData['am_pm'])
            || ($prefs->getValue('24hr')))) {
            if (!$prefs->getValue('24hr')) {
                if ($eventData['am_pm'] == 'PM') {
                    if ($eventData['start_hour'] != 12)
                        $eventData['start_hour'] += 12;
                } elseif ($eventData['start_hour'] == 12) {
                    $eventData['start_hour'] = 0;
                }
            }
            if (isset($eventData['end_or_dur'])
                && $eventData['end_or_dur'] == 1
                && isset($eventData['whole_day'])
                && $eventData['whole_day'] == 1) {
                $eventData['start_hour'] = 0;
                $eventData['start_min'] = 0;
                $eventDate['dur_day'] = 0;
                $eventData['dur_hour'] = 24;
                $eventData['dur_min'] = 0;
            }
            $startstamp = mktime($eventData['start_hour'],
                                 $eventData['start_min'],
                                 0,
                                 $eventData['start_month'],
                                 $eventData['start_day'],
                                 $eventData['start_year']);
            $start = explode(':', date('Y:n:j:G:i', $startstamp));
            mcal_event_set_start($this->driver->stream, $start[0], $start[1], $start[2], $start[3], $start[4]);
            
            if (isset($eventData['end_or_dur'])
                && $eventData['end_or_dur'] == 1) {
                // event duration
                if (isset($eventData['dur_day']) 
                    && isset($eventData['dur_hour']) 
                    && isset($eventData['dur_min'])) {
                    $endstamp = mktime($eventData['start_hour'] + $eventData['dur_hour'],
                                       $eventData['start_min'] + $eventData['dur_min'],
                                       0,
                                       $eventData['start_month'],
                                       $eventData['start_day'] + $eventData['dur_day'],
                                       $eventData['start_year']);
                    $end = explode(':', date('Y:n:j:G:i', $endstamp));
                    mcal_event_set_end($this->driver->stream, $end[0], $end[1], $end[2], $end[3], $end[4]);
                }
            } else {
                // event end
                if (isset($eventData['end_year'])
                    && isset($eventData['end_month'])
                    && isset($eventData['end_day'])
                    && isset($eventData['end_hour'])
                    && isset($eventData['end_min'])
                    && (isset($eventData['end_am_pm'])
                    || ($prefs->getValue('24hr')))) {
                    if (!$prefs->getValue('24hr')) {
                        if ($eventData['end_am_pm'] == 'PM') {
                            if ($eventData['end_hour'] != 12)
                                $eventData['end_hour'] += 12;
                        } elseif ($eventData['end_hour'] == 12) {
                            $eventData['end_hour'] = 0;
                        }
                    }
                    $endstamp = mktime($eventData['end_hour'],
                                       $eventData['end_min'],
                                       0,
                                       $eventData['end_month'],
                                       $eventData['end_day'],
                                       $eventData['end_year']);
                    if ($endstamp < $startstamp)
                        $endstamp = $startstamp;
                    $end = explode(':', date('Y:n:j:G:i', $endstamp));
                    mcal_event_set_end($this->driver->stream, $end[0], $end[1], $end[2], $end[3], $end[4]);
                }
            }
        }
        
        // alarm
        
        if (isset($eventData['alarm'])) {
            if ($eventData['alarm'] == 1 &&
                isset($eventData['alarm_value']) &&
                isset($eventData['alarm_unit'])) {
                mcal_event_set_alarm($this->driver->stream, ($eventData['alarm_value'] * $eventData['alarm_unit']));
            } else {
                mcal_event_set_alarm($this->driver->stream, 0);
            }
        }
        
        // recurrence
        if (isset($eventData['recur'])) {
            if (isset($eventData['recur_enddate_type']) &&
                $eventData['recur_enddate_type'] == 'none') {
                $eventData['recur_enddate_year'] = 9999;
                $eventData['recur_enddate_month'] = 12;
                $eventData['recur_enddate_day'] = 31;
            }
            
            switch ($eventData['recur']) {
            case KRONOLITH_RECUR_NONE:
                mcal_event_set_recur_none($this->driver->stream);
                break;
                
            case KRONOLITH_RECUR_DAILY:
                if (isset($eventData['recur_enddate_year'])
                    && isset($eventData['recur_enddate_month'])
                    && isset($eventData['recur_enddate_day'])) {
                    if (empty($eventData['recur_daily_interval'])) $eventData['recur_daily_interval'] = 1;
                    mcal_event_set_recur_daily($this->driver->stream,
                                               $eventData['recur_enddate_year'],
                                               $eventData['recur_enddate_month'],
                                               $eventData['recur_enddate_day'],
                                               $eventData['recur_daily_interval']);
                }
                break;
                
            case KRONOLITH_RECUR_WEEKLY:
                if (isset($eventData['recur_enddate_year'])
                    && isset($eventData['recur_enddate_month'])
                    && isset($eventData['recur_enddate_day'])
                    && isset($eventData['weekly'])
                    && is_array($eventData['weekly'])) {
                    $weekdays = 0;
                    foreach ($eventData['weekly'] as $day) {
                        $weekdays |= $day;
                    }
                    if (empty($eventData['recur_weekly_interval'])) $eventData['recur_weekly_interval'] = 1;
                    mcal_event_set_recur_weekly($this->driver->stream,
                                                $eventData['recur_enddate_year'],
                                                $eventData['recur_enddate_month'],
                                                $eventData['recur_enddate_day'],
                                                $eventData['recur_weekly_interval'],
                                                $weekdays);
                }
                break;
                
            case KRONOLITH_RECUR_DAY_OF_MONTH:
                if (isset($eventData['recur_enddate_year'])
                    && isset($eventData['recur_enddate_month'])
                    && isset($eventData['recur_enddate_day'])) {
                    if (empty($eventData['recur_day_of_month_interval'])) $eventData['recur_day_of_month_interval'] = 1;
                    mcal_event_set_recur_monthly_mday($this->driver->stream,
                                                      $eventData['recur_enddate_year'],
                                                      $eventData['recur_enddate_month'],
                                                      $eventData['recur_enddate_day'],
                                                      $eventData['recur_day_of_month_interval']);
                }
                break;

            case KRONOLITH_RECUR_WEEK_OF_MONTH:
                if (isset($eventData['recur_enddate_year'])
                    && isset($eventData['recur_enddate_month'])
                    && isset($eventData['recur_enddate_day'])) {
                    if (empty($eventData['recur_week_of_month_interval'])) $eventData['recur_week_of_month_interval'] = 1;
                    mcal_event_set_recur_monthly_wday($this->driver->stream,
                                                      $eventData['recur_enddate_year'],
                                                      $eventData['recur_enddate_month'],
                                                      $eventData['recur_enddate_day'],
                                                      $eventData['recur_week_of_month_interval']);
                }
                break;
                
            case KRONOLITH_RECUR_YEARLY:
                if (isset($eventData['recur_enddate_year'])
                    && isset($eventData['recur_enddate_month'])
                    && isset($eventData['recur_enddate_day'])) {
                    if (empty($eventData['recur_yearly_interval'])) $eventData['recur_yearly_interval'] = 1;
                    mcal_event_set_recur_yearly($this->driver->stream,
                                                $eventData['recur_enddate_year'],
                                                $eventData['recur_enddate_month'],
                                                $eventData['recur_enddate_day'],
                                                $eventData['recur_yearly_interval']);
                }
                break;
                
            }
        }
        
        $this->initialized = true;
    }
    
}
?>
