<?php

require_once 'Horde/History.php';

/**
 * The IMP_Maillog:: class contains all functions related to handling
 * logging of responses to individual e-mail messages.
 *
 * $Horde: imp/lib/Maillog.php,v 1.15 2004/09/28 15:58:35 chuck Exp $
 *
 * Copyright 2003-2004 Michael Slusarz <slusarz@bigworm.colorado.edu>
 *
 * See the enclosed file COPYING for license information (GPL). If you
 * did not receive this file, see http://www.fsf.org/copyleft/gpl.html.
 *
 * @author  Michael Slusarz <slusarz@bigworm.colorado.edu>
 * @version $Revision: 1.15 $
 * @since   IMP 4.0
 * @package IMP
 */
class IMP_Maillog {

    /**
     * Create a log entry.
     *
     * @access public
     *
     * @param string $type           Either 'forward', 'redirect', 'reply',
     *                               'reply', or 'mdn'.
     * @param mixed $msg_ids         Either a single Message-ID or an array
     *                               of Message-IDs to log.
     * @param optional string $data  Any additional data to store. For
     *                               'forward' and 'redirect' this is the list
     *                               of recipients the message was sent to.
     *                               For 'mdn' this is the MDN-type of the
     *                               message that was sent.
     */
    function log($type, $msg_ids, $data = null)
    {
        $history = &Horde_History::singleton();

        if (!is_array($msg_ids)) {
            $msg_ids = array($msg_ids);
        }

        foreach ($msg_ids as $val) {
            $id = IMP_Maillog::_getUniqueHistoryId($val);
            switch ($type) {
            case 'forward':
                $history->log($id, array('recipients' => $data, 'action' => 'forward'));
                break;

            case 'mdn':
                $history->log($id, array('type' => $data, 'action' => 'mdn'));
                break;

            case 'redirect':
                $history->log($id, array('recipients' => $data, 'action' => 'redirect'));
                break;

            case 'reply':
                $history->log($id, array('action' => 'reply'));
                break;
            }
        }
    }

    /**
     * Retrieve any history for the given Message-ID.
     *
     * @access public
     *
     * @param string $msg_id  The Message-ID of the message.
     *
     * @return object DataTreeObject  The DataTreeObject object containing the
     *                                log information.
     *                                Returns PEAR_Error object on error.
     */
    function &getLog($msg_id)
    {
        $history = &Horde_History::singleton();
        return $log = &$history->getHistory(IMP_Maillog::_getUniqueHistoryId($msg_id));
    }

    /**
     * Determines if an MDN notification of a certain type has been sent
     * previously for this message-ID.
     *
     * @access public
     *
     * @param string $msg_id  The Message-ID of the message.
     * @param string $type    The type of MDN.
     *
     * @return boolean  True if a MDN has been sent for this message with
     *                  the given type.
     */
    function sentMDN($msg_id, $type)
    {
        $msg_history = IMP_Maillog::getLog($msg_id);
        if (!is_a($msg_history, 'PEAR_Error')) {
            foreach ($msg_history->getData() as $entry) {
                if (($entry['action'] == 'mdn') && ($entry['type'] == $type)) {
                    return true;
                }
            }
        }

        return false;
    }

    /**
     * Retrieve any history for the given Message-ID and display via the
     * Horde notification system.
     *
     * @access public
     *
     * @param string $msg_id  The Message-ID of the message.
     */
    function displayLog($msg_id)
    {
        global $notification, $prefs;

        $msg_history = IMP_Maillog::getLog($msg_id);
        if (!is_a($msg_history, 'PEAR_Error')) {
            foreach ($msg_history->getData() as $entry) {
                $msg = null;
                if (isset($entry['desc'])) {
                    $msg = $entry['desc'];
                } else {
                    switch ($entry['action']) {
                    case 'forward':
                        $msg = sprintf(_("You forwarded this message on %%s to the following recipients: %s."), $entry['recipients']);
                        break;

                    case 'mdn':
                        /* We don't display 'mdn' log entries. */
                        break;

                    case 'redirect':
                        $msg = sprintf(_("You redirected this message to %s on %%s."), $entry['recipients']);
                        break;

                    case 'reply':
                        $msg = _("You replied to this message on %s.");
                        break;
                    }
                }
                if (!is_null($msg)) {
                    $notification->push(htmlspecialchars(@sprintf($msg, strftime($prefs->getValue('date_format') . ' ' . $prefs->getValue('time_format'), $entry['ts']))), 'imp.' . $entry['action']);
                }
            }
        }
    }

    /**
     * Delete the log entries for a given list of Message-IDs.
     *
     * @access public
     *
     * @param mixed $msg_ids  Either a single Message-ID or an array
     *                        of Message-IDs to delete.
     */
    function deleteLog($msg_ids)
    {
        if (!is_array($msg_ids)) {
            $msg_ids = array(IMP_Maillog::_getUniqueHistoryId($msg_ids));
        } else {
            foreach ($msg_ids as $key => $msg_id) {
                $msg_ids[$key] = IMP_Maillog::_getUniqueHistoryId($msg_id);
            }
        }

        $history = &Horde_History::singleton();
        return $history->removeByNames($msg_ids);
    }

    /**
     * Generate the unique log ID for a forward/reply/redirect event.
     *
     * @access private
     *
     * @param string $msgid  The Message-ID of the original message.
     *
     * @return string  The unique log ID to use with Horde_History::.
     */
    function _getUniqueHistoryId($msgid)
    {
        if (is_array($msgid)) {
            return '';
        }

        return implode('.', array('imp', str_replace('.', '*', Auth::getAuth()), $msgid));
    }

}
