<?php
/**
 * The IMP_Filter:: class contains all functions related to handling
 * filtering messages in IMP.
 *
 * For full use, the following Horde API calls should be defined
 * (These API methods are not defined in IMP):
 *   mail/applyFilters
 *   mail/canApplyFilters
 *   mail/showFilters
 *   mail/blacklistFrom
 *   mail/showBlacklist
 *   mail/whitelistFrom
 *   mail/showWhitelist
 *
 * $Horde: imp/lib/Filter.php,v 1.56 2004/09/29 05:53:15 slusarz Exp $
 *
 * Copyright 2002-2004 Michael Slusarz <slusarz@bigworm.colorado.edu>
 *
 * See the enclosed file COPYING for license information (GPL). If you
 * did not receive this file, see http://www.fsf.org/copyleft/gpl.html.
 *
 * @author  Michael Slusarz <slusarz@bigworm.colorado.edu>
 * @version $Revision: 1.56 $
 * @since   IMP 4.0
 * @package IMP
 */
class IMP_Filter {

    /**
     * Runs the filters if they are able to be applied manually.
     *
     * @access public
     *
     * @param string $mbox  The mailbox to apply the filters to.
     */
    function filter($mbox)
    {
        global $imp, $registry;

        if ($imp['filteravail']) {
            $params = array('imap' => $imp['stream'], 'mailbox' => IMP::serverString($mbox));
            $registry->call('mail/applyFilters', array($params));
        }
    }

    /**
     * Adds the From address from the message(s) to the blacklist.
     *
     * @access public
     *
     * @param array $indices  An array of message indices to blacklist.
     */
    function blacklistMessage($indices)
    {
        IMP_Filter::_processBWlist($indices, _("your blacklist"), 'blacklistFrom', 'showBlacklist');
    }

    /**
     * Adds the From address from the message(s) to the whitelist.
     *
     * @access public
     *
     * @param array $indices  An array of message indices to whitelist.
     */
    function whitelistMessage($indices)
    {
        IMP_Filter::_processBWlist($indices, _("your whitelist"), 'whitelistFrom', 'showWhitelist');
    }

    /**
     * Internal function to handle adding addresses to [black|white]list.
     *
     * @access private
     *
     * @param array $indices   An array of message indices to whitelist.
     * @param string $descrip  The textual description to use.
     * @param string $reg1     The name of the mail/ registry call to use for
     *                         adding the addresses.
     * @param string $reg2     The name of the mail/ registry call to use for
     *                         linking to the filter management page.
     */
    function _processBWlist($indices, $descrip, $reg1, $reg2)
    {
        global $notification, $registry;

        require_once IMP_BASE . '/lib/MIME/Headers.php';

        /* Get the list of from addresses. */
        $addr = array();
        foreach ($indices as $msg) {
            $imp_headers = &new IMP_Headers($msg);
            $from = $imp_headers->getFromAddress();
            $notification->push(sprintf(_("The address \"%s\" has been added to %s."), $from, $descrip));
            $addr[] = $from;
        }

        $registry->call('mail/' . $reg1, array($addr));

        /* Add link to filter management page. */
        if ($registry->hasMethod('mail/' . $reg2)) {
            $manage_link = Horde::link($registry->link('mail/' . $reg2), sprintf(_("Filters: %s management page"), $descrip)) . _("HERE") . '</a>';
            $notification->push(sprintf(_("Click %s to go to %s management page."), $manage_link, $descrip), 'horde.message', array('content.raw'));
        }
    }

}
