#!/usr/bin/perl -w

# Simplified module build script

sub print_usage {
   print <<"USAGE"

build.pl : Library build program.

   This program builds the libraries for IMP.

   It can take as arguments the location of a file with more directives:
      ./build.pl --with-config=./build.opts

   Another example would be :
      ./build.pl --with-config=/etc/horde/imp/build.opts

   Or the directives themselves:
      ./build.pl --database=postgres

   The config file currently can only contain one thing, which database you want.

   For example, It should be in the form:
       database=postgres

USAGE
}

# File containing all of the options
$Options_File = 'build.opts';

# Final resting place of the module's libraries
$IMP_Library_File = "../imp.lib";
$MIME_Types_Library_File = "../mimetypes.lib";
$Database_Library_File = "../db.lib";

    
# Parse command-line arguments
foreach $Argument (@ARGV) {

    # Check to see if they gave us a database
    if($Argument =~ /--database=(.*)/) {
	$Options{"database"} = $1;

    # Check to see if they tell us a new config file location
    } elsif($Argument =~ /--with-config=(.*)/) {
	$Options_File = $1;

    # We have no idea what this is
    } else {
	&print_usage;
	exit;
    }
}

# If it exists, Open the options file and set all of the options into a hash
if(-e $Options_File) {
    open(OPTIONS, $Options_File) ||
	die "Error couldn't open specified options file $Options_File: $!\n";
    
    while(<OPTIONS>) {
	chomp();
	($Option_Name, $Option_Value) = split('=', $_);
	$Options{$Option_Name} = $Option_Value;
	print "Set $Option_Name equal to $Option_Value\n";  # debugging
    }
    close(OPTIONS);
} else {
   open(OPTIONS, $Options_File) ||
       warn "[Note] Ignoring optional configuration file $Options_File.\n";
}


# Construct the basic list of files to include for the main IMP library
@IMP_Library_Files = (
		      # Imp includes
		      'includes/HordeCopyright.inc',

		      # Imp definitions
		      'defines/ImapServers.def',
		      'defines/ImpActions.def',
		      'defines/ImpLibVersion.def',
		      
		      # Imp specific objects
		      'objects/IMAPServer.object',
		      'objects/ImpSession.object',
		      'objects/ImpDefaults.object',
		      'objects/LDAPServer.object',
		      
		      # Imp specific functions
		      'functions/decode_mime_string.func',
		      'functions/full_mailbox_list.func',
		      'functions/get_barefrom.func',
		      'functions/mailbox_list.func',
		      'functions/unsub_mailbox_list.func',
		      'functions/reload_frame.func',

		      # Imp initialization files
		      'initialization/IncludeDbLib.init',
		      'initialization/IncludeHordeLib.init',
		      'initialization/IncludeImpLib.init',
		      'initialization/InitImpDefaults.init'
		      );


# Now the MIME-types
@MIME_Types_Library_Files = (
			     'objects/mime_part_data.object',
			     'functions/mime_action_images.func',
			     'functions/mime_action_ximpdata.func',
			     'functions/mime_view_html.func',
			     'functions/mime_view_msword.func',
			     'functions/mime_view_rpm.func',
			     'functions/mime_view_text.func',
			     'functions/mime_view_tgz.func',
			     'functions/mime_view_zip.func',
			     'initialization/InitMimeTypes.init'
			     );


# Construct the list of database files (taking care to use the right ones for each database)
if(exists($Options{"database"})) { $Database_Driver = $Options{"database"}; }
else { $Database_Driver = "dummy"; }

$Function_Root = "db/v1/functions/" . $Database_Driver;
$Include_Root = "db/v1/includes/" . $Database_Driver;

@Database_Library_Files = (
			   'includes/HordeCopyright.inc',
			   $Include_Root . '/table_format.inc',
			   $Function_Root . '/imp_add_address.func',
			   $Function_Root . '/imp_check_prefs.func',
			   $Function_Root . '/imp_delete_address.func',
			   $Function_Root . '/imp_get_addresses.func',
			   $Function_Root . '/imp_get_from.func',
			   $Function_Root . '/imp_get_fullname.func',
			   $Function_Root . '/imp_get_lang.func',
			   $Function_Root . '/imp_get_signature.func',
			   $Function_Root . '/imp_set_from.func',
			   $Function_Root . '/imp_set_fullname.func',
			   $Function_Root . '/imp_set_lang.func',
			   $Function_Root . '/imp_set_signature.func',
			   $Function_Root . '/imp_update_address.func'
			   );

# Only mysql & postgres support for sessions at the moment 
if($Database_Driver eq "mysql") {
    push(@Database_Library_Files,
        $Function_Root . '/imp_get_session.func',
        $Function_Root . '/imp_set_session.func',
        $Function_Root . '/imp_purge_sessions.func'
        );
}

if($Database_Driver eq "postgres") {
    push(@Database_Library_Files,
         $Function_Root . '/imp_open_pg_db.func',
	 $Function_Root . '/imp_get_session.func',
	 $Function_Root . '/imp_set_session.func',
	 $Function_Root . '/imp_purge_sessions.func'
	 );
}

if($Database_Driver eq "oracle") {
    push(@Database_Library_Files,
	 $Function_Root . '/enterOracle.func',
	 $Function_Root . '/leaveOracle.func'
	 );
}


# Build all of the libraries
print "Building IMP libraries...";
build_lib($IMP_Library_File, @IMP_Library_Files);
build_lib($MIME_Types_Library_File, @MIME_Types_Library_Files);
build_lib($Database_Library_File, @Database_Library_Files);
print "...done.\n";


# Helper functions
sub build_lib {

    # Get the library name and the array of files to build from the argument list
    local($Library_Name, @Library_Files) = @_;

    # Open the library file up or die trying
    open(LIBRARY, "> $Library_Name")
   	|| die "Can\'t open $Library_Name for writing: $!\n";

    # Start the file with the requisite PHP magic
    print LIBRARY "<?php\n\n";

    # Put all of the files into the library
    foreach $Library_File (@Library_Files) {
	#print "I should be adding: $Library_File\n"; # debugging
	add_file(LIBRARY, $Library_File);
    }

    # Close the file with the requisite PHP magic
    print LIBRARY "?>\n";
    close(LIBRARY);
}

sub add_file {
   local(*OUT_FH, $Filename) = @_;
   #print "\tAdding file : $Filename\n"; #debugging

   # Open the library section with a comment on the upcoming source file
   print OUT_FH "// Begin $Filename\n";

   # Open the source file and dump the contents into the library
   open(IN_FH, $Filename) || die "Unable to open $Filename: $!\n";
   while(<IN_FH>) { print OUT_FH $_; }
   close(IN_FH);   
   
   # Close the library section with a comment
   print OUT_FH "// End $Filename\n\n";
}
