<?php
/**
 * IMAP_Thread provides functions for working with imap_thread() output.
 *
 * $Horde: framework/IMAP/IMAP/Thread.php,v 1.4 2004/10/16 09:00:58 jan Exp $
 *
 * Copyright 2004 Michael Slusarz <slusarz@curecanti.org>
 *
 * See the enclosed file COPYING for license information (GPL). If you
 * did not receive this file, see http://www.fsf.org/copyleft/gpl.html.
 *
 * @author  Michael Slusarz <slusarz@curecanti.org>
 * @version $Revision: 1.4 $
 * @since   Horde 3.0
 * @package Horde_IMAP
 */
class IMAP_Thread {

    /**
     * imap_thread() output.
     *
     * @var array $_thread
     */
    var $_threadarray;

    /**
     * The list of message IDs to ignore.
     *
     * @var array $_ignore
     */
    var $_ignore = array();

    /**
     * Internal thread data structure.
     *
     * @var array $_thread
     */
    var $_thread = array();

    /**
     * Array index to Message index lookup array.
     *
     * @var array $_lookup
     */
    var $_lookup = array();

    /**
     * Constructor.
     *
     * @access public
     *
     * @param array $ob               Output from imap_thread().
     * @param optional array $ignore  Ignore list.
     */
    function IMAP_Thread($ob, $ignore = array())
    {
        $this->_threadarray = $ob;
        $this->_ignore = $ignore;
        $this->_processStructure();
    }

   /**
     * Gets the indention level for an IMAP message index.
     *
     * @access public
     *
     * @param integer $index  The IMAP message index.
     *
     * @return mixed  Returns the thread indent level if $index found.
     *                Returns false on failure.
     */
    function getThreadIndent($index)
    {
        if ($key = $this->_getKey($index)) {
            if (isset($this->_thread[$key]['level'])) {
                return $this->_thread[$key]['level'];
            }
        }

         return false;
    }

    /**
     * Gets the base thread index for an IMAP message index.
     *
     * @access public
     *
     * @param integer $index  The IMAP message index.
     *
     * @return mixed  Returns the base IMAP index if $index is part of a
     *                thread.
     *                Returns false on failure.
     */
    function getThreadBase($index)
    {
        if ($key = $this->_getKey($index)) {
            if (!empty($this->_thread[$key]['base'])) {
                return $this->_thread[$key]['base'];
            }
        }

        return false;
    }

    /**
     * Is this index the last in the current level?
     *
     * @access public
     *
     * @param integer $index  The IMAP message index.
     *
     * @return boolean  Returns true if $index is the last element in the
     *                  current thread level.
     *                  Returns false if not, or on failure.
     */
    function lastInLevel($index)
    {
        if ($key = $this->_getKey($index)) {
            if (!empty($this->_thread[$key]['last'])) {
                return $this->_thread[$key]['last'];
            }
        }

        return false;
    }

    /**
     * Do the message index -> array index lookup.
     *
     * @access private
     *
     * @param integer $index  The IMAP message index.
     *
     * @return mixed  The array index value.
     */
    function _getKey($index)
    {
        return (isset($this->_lookup[$index])) ? $this->_lookup[$index] : false;
    }

    /**
     * Return the sorted list of messages indices.
     *
     * @access public
     *
     * @param boolean $new  True for newest first, false for oldest first.
     *
     * @return array  The sorted list of messages.
     */
    function messageList($new)
    {
        return ($new) ? array_reverse(array_keys($this->_lookup)) : array_keys($this->_lookup);
    }

    /**
     * Returns the list of messages in the current thread.
     *
     * @access public
     *
     * @param integer $index  The IMAP index of the current message.
     *
     * @return array  A list of IMAP message indices.
     */
    function getThread($index)
    {
        /* Find the beginning of the thread. */
        $begin = $this->getThreadBase($index);
        if (empty($begin) ||
            !($key = $this->_getKey($begin))) {
            return array($index);
        }

        /* Work forward from the first thread element to find the end of the
         * thread. */
        $thread_list = array($this->_thread[$key]['index']);
        while ($key++) {
            if (!isset($this->_thread[$key])) {
                break;
            }
            $curr = $this->_thread[$key];
            if ($curr['base'] == $begin) {
                $thread_list[] = $this->_thread[$key]['index'];
            } else {
                break;
            }
        }

        return $thread_list;
    }

    /**
     * Process the output from imap_thread() into an internal data structure.
     *
     * @access private
     */
    function _processStructure()
    {
        $last_index = $thread_base = null;
        $ignore_list = $indices = array();
        $i = $last_i = $level = 0;

        foreach ($this->_threadarray as $key => $val) {
            $pos = strpos($key, '.');
            $index = substr($key, 0, $pos);
            $type = substr($key, $pos + 1);

            if (isset($ignore_list[$index])) {
                continue;
            }

            switch ($type) {
            case 'num':
                if (($val === 0) ||
                    (!empty($this->_ignore) &&
                     in_array($val, $this->_ignore))) {
                    $ignore_list[$index] = 1;
                } else {
                    $i++;
                    if (empty($level)) {
                        $thread_base = $val;
                    }
                    $this->_lookup[$val] = $index;
                    $this->_thread[$index] = array();
                    $this->_thread[$index]['index'] = $val;
                }
                break;

            case 'next':
                $i++;
                $this->_thread[$index]['base'] = (!empty($val) || !empty($level)) ? $thread_base : null;
                $level++;
                break;

            case 'branch':
                $this->_thread[$index]['level'] = $level--;
                $this->_thread[$index]['last'] = true;
                if (!empty($level)) {
                    $this->_thread[$last_index]['last'] = ($last_i == ($i - 1));
                }
                $last_index = $index;
                $last_i = $i++;
                break;
            }
        }
    }

}
