<?php
/**
 * The Horde_Block_Collection:: class provides an API to the blocks
 * (applets) framework.
 *
 * $Horde: framework/Block/Block/Collection.php,v 1.36 2004/11/25 17:19:51 jan Exp $
 *
 * Copyright 2003-2004 Mike Cochrane <mike@graftonhall.co.nz>
 * Copyright 2003-2004 Jan Schneider <jan@horde.org>
 *
 * See the enclosed file COPYING for license information (LGPL). If you
 * did not receive this file, see http://www.fsf.org/copyleft/lgpl.html.
 *
 * @author  Mike Cochrane <mike@graftonhall.co.nz>
 * @author  Jan Schneider <jan@horde.org>
 * @version $Revision: 1.36 $
 * @since   Horde 3.0
 * @package Horde_Block
 */
class Horde_Block_Collection {

    /**
     * What kind of blocks are we collecting? Defaults to any.
     *
     * @var string $_type
     */
    var $_type = 'portal';

    /**
     * A hash storing the information about all available blocks from
     * all applications.
     *
     * @var array $_blocks
     */
    var $_blocks = array();

    /**
     * Constructor.
     *
     * @param string $type  (optional) The kind of blocks to list.
     */
    function Horde_Block_Collection($type = null)
    {
        if (!is_null($type)) {
            $this->_type = $type;
        }

        if (isset($_SESSION['horde']['blocks'])) {
            $this->_blocks = &$_SESSION['horde']['blocks'];
            return;
        }

        global $registry;
        require_once 'Horde/Block.php';

        foreach ($registry->listApps() as $app) {
            if (is_a($result = $registry->pushApp($app), 'PEAR_Error')) {
                continue;
            }
            $blockdir = $registry->get('fileroot', $app) . '/lib/Block';
            $dh = @opendir($blockdir);
            if (is_resource($dh)) {
                while (($file = readdir($dh)) !== false) {
                    if (substr($file, -4) == '.php') {
                        $block_name = null;
                        $block_type = null;
                        if (is_readable($blockdir . '/' . $file)) {
                            include_once $blockdir . '/' . $file;
                        }
                        if (!is_null($block_type) && !is_null($this->_type) && $block_type != $this->_type) {
                            continue;
                        }
                        if (!empty($block_name)) {
                            $this->_blocks[$app][substr($file, 0, -4)]['name'] = $block_name;
                        }
                    }
                }
                closedir($dh);
            }
            $registry->popApp($app);
        }

        uksort($this->_blocks,
               create_function('$a, $b',
                               'global $registry;
                                return strcasecmp($registry->get("name", $a), $registry->get("name", $b));')
               );

        $_SESSION['horde']['blocks'] = &$this->_blocks;
    }

    /**
     * Returns a single instance of the Horde_Blocks class.
     *
     * @static
     *
     * @param string $type  (optional) The kind of blocks to list.
     *
     * @return object Horde_Blocks  The Horde_Blocks intance.
     */
    function &singleton($type = null)
    {
        static $instances = array();

        // Use array_key_exists since the default key is NULL.
        if (!array_key_exists($type, $instances)) {
            $instances[$type] = new Horde_Block_Collection($type);
        }

        return $instances[$type];
    }

    function &getBlock($app, $name, $params = null)
    {
        global $registry;

        $class = 'Horde_Block_' . $app . '_' . $name;
        include_once $registry->get('fileroot', $app) . '/lib/Block/' . $name . '.php';
        if (!class_exists($class)) {
            return PEAR::raiseError(sprintf(_("%s not found."), $class));
        }

        return $block = &new $class($params);
    }

    /**
     * Returns a pretty printed list of all available blocks.
     *
     * @return array  A hash with block IDs as keys and application plus block
     *                block names as values.
     */
    function getBlocksList()
    {
        static $blocks = array();
        if (!empty($blocks)) {
            return $blocks;
        }

        global $registry;

        /* Get available blocks from all apps. */
        foreach ($this->_blocks as $app => $app_blocks) {
            foreach ($app_blocks as $block_id => $block) {
                $blocks[$app . ':' . $block_id] = $registry->get('name', $app) . ': ' . $block['name'];
            }
        }

        return $blocks;
    }

    /**
     * Returns a select widget with all available blocks.
     *
     * @param string $cur_app    The block from this application gets selected.
     * @param string $cur_block  The block with this name gets selected.
     *
     * @return string  The select tag with all available blocks.
     */
    function getBlocksWidget($cur_app = null, $cur_block = null, $onchange = false)
    {
        global $registry;

        $widget = '<select name="app"';
        if ($onchange) {
            $widget .= ' onchange="document.blockform.action.value=\'save-resume\';document.blockform.submit()"';
        }
        $widget .= ">\n";
        $blocks_list = $this->getBlocksList();
        foreach ($blocks_list as $id => $name) {
            $widget .= sprintf("<option value=\"%s\"%s>%s</option>\n",
                                   $id,
                                   ($id == $cur_app . ':' . $cur_block) ? ' selected="selected"' : '',
                                   $name);
        }
        $widget .= "</select>\n";

        return $widget;
    }

    /**
     * Returns the option type.
     */
    function getOptionType($app, $block, $param_id)
    {
        $this->getParams($app, $block);
        return $this->_blocks[$app][$block]['params'][$param_id]['type'];
    }

    /**
     * Returns whether the option is required or not. Defaults to true.
     */
    function getOptionRequired($app, $block, $param_id)
    {
        $this->getParams($app, $block);
        if (!isset($this->_blocks[$app][$block]['params'][$param_id]['required'])) {
            return true;
        } else {
            return $this->_blocks[$app][$block]['params'][$param_id]['required'];
        }
    }

    /**
     * Returns the values for an option.
     */
    function &getOptionValues($app, $block, $param_id)
    {
        $this->getParams($app, $block);
        return $this->_blocks[$app][$block]['params'][$param_id]['values'];
    }

    /**
     * Returns the widget necessary to configure this block.
     */
    function getOptionsWidget($app, $block, $param_id, $val = null)
    {
        $widget = '';

        $this->getParams($app, $block);
        $param = $this->_blocks[$app][$block]['params'][$param_id];
        switch ($param['type']) {
        case 'boolean':
        case 'checkbox':
            $checked = !empty($val[$param_id]) ? ' checked="checked"' : '';
            $widget = sprintf('<input type="checkbox" name="params[%s]"%s>', $param_id, $checked);
            break;

        case 'enum':
            $widget = sprintf('<select name="params[%s]">', $param_id);
            foreach ($param['values'] as $key => $name) {
                if (String::length($name) > 30) {
                    $name = substr($name, 0, 27) . '...';
                }
                $widget .= sprintf("<option value=\"%s\"%s>%s</option>\n",
                                   htmlspecialchars($key),
                                   (isset($val[$param_id]) && $val[$param_id] == $key) ? ' selected="selected"' : '',
                                   htmlspecialchars($name));
            }

            $widget .= '</select>';
            break;

        case 'multienum':
            $widget = sprintf('<select multiple="multiple" name="params[%s][]">', $param_id);
            foreach ($param['values'] as $key => $name) {
                if (String::length($name) > 30) {
                    $name = substr($name, 0, 27) . '...';
                }
                $widget .= sprintf("<option value=\"%s\"%s>%s</option>\n",
                                   htmlspecialchars($key),
                                   (isset($val[$param_id]) && in_array($key, $val[$param_id])) ? ' selected="selected"' : '',
                                   htmlspecialchars($name));
            }

            $widget .= '</select>';
            break;

        case 'mlenum':
            // Multi-level enum.
            if (is_array($val) && isset($val['__' . $param_id])) {
                $firstval = $val['__' . $param_id];
            } else {
                $firstval = current(array_keys($param['values']));
            }
            $blockvalues = $param['values'][$firstval];
            asort($blockvalues);

            $widget = sprintf('<select name="params[__%s]" onchange="document.blockform.action.value=\'save-resume\';document.blockform.submit()">', $param_id) . "\n";
            foreach ($param['values'] as $key => $values) {
                $name = String::length($key) > 30 ? String::substr($key, 0, 27) . '...' : $key;
                $widget .= sprintf("<option value=\"%s\"%s>%s</option>\n",
                                   htmlspecialchars($key),
                                   $key == $firstval ? ' selected="selected"' : '',
                                   htmlspecialchars($name));
            }
            $widget .= "</select><br/>\n";

            $widget .= sprintf("<select name=\"params[%s]\">\n", $param_id);
            foreach ($blockvalues as $key => $name) {
                $name = (String::length($name) > 30) ? String::substr($name, 0, 27) . '...' : $name;
                $widget .= sprintf("<option value=\"%s\"%s>%s</option>\n",
                                   htmlspecialchars($key),
                                   $val[$param_id] == $key ? ' selected="selected"' : '',
                                   htmlspecialchars($name));
            }
            $widget .= "</select><br/>\n";
            break;

        case 'color':
            $val = isset($block->_params[$param_id]) ? $block->_params[$param_id] : '';
            $widget = sprintf('<input style="background-color:%s" type="text" name="params[%s]" value="%s" />', $val, $param_id, $val);
            $url  = $registry->get('webroot', 'horde');
            $url .= Util::addParameter('/services/images/colorpicker.php?target=params[' . $param_id . ']', 'form', 'blockform');
            $widget .= sprintf('<a href="%s" onclick="window.open(\'%s\', \'colorpicker\', \'toolbar=no,location=no,status=no,scrollbars=no,resizable=no,width=120,height=250,left=100,top=100\'); return false;" onmouseout="window.status=\'\';" onmouseover="window.status=\'%s\'; return true;" class="widget" target="colorpicker">',
                               $url, $url, _("Color Picker"));
            $widget .= Horde::img('colorpicker.png', _("Color Picker"), 'height="16"', $registry->getImageDir('horde'));
            $widget .= '</a>';
            break;

        case 'int':
        case 'text':
            $widget = sprintf('<input type="text" name="params[%s]" value="%s" />', $param_id, empty($val) ? $param['default'] : $val[$param_id]);
            break;

        case 'error':
            $widget = '<span class="form-error">' . $val[$param_id] . '</span>';
            break;
        }

        return $widget;
    }

    /**
     * Returns the name of the specified block.
     *
     * @param string $app    An application name.
     * @param string $block  A block name.
     *
     * @return string  The name of the specified block.
     */
    function getName($app, $block)
    {
        if (!isset($this->_blocks[$app][$block])) {
            return sprintf(_("Block \"%s\" of application \"%s\" not found."), $block, $app);
        }
        return $this->_blocks[$app][$block]['name'];
    }

    /**
     * Returns the parameter list of the specified block.
     *
     * @param string $app    An application name.
     * @param string $block  A block name.
     *
     * @return array  An array with all paramter names.
     */
    function getParams($app, $block)
    {
        if (!isset($this->_blocks[$app][$block]['params'])) {
            $blockOb = &$this->getBlock($app, $block);
            if (is_a($blockOb, 'PEAR_Error')) {
                return $blockOb;
            }
            $this->_blocks[$app][$block]['params'] = $blockOb->getParams();
        }

        if (isset($this->_blocks[$app][$block]['params']) &&
            is_array($this->_blocks[$app][$block]['params'])) {
            return array_keys($this->_blocks[$app][$block]['params']);
        } else {
            return array();
        }
    }

    /**
     * Returns the (clear text) name of the specified parameter.
     *
     * @param string $app    An application name.
     * @param string $block  A block name.
     * @param string $param  A parameter name.
     *
     * @return string  The name of the specified parameter.
     */
    function getParamName($app, $block, $param)
    {
        $this->getParams($app, $block);
        return $this->_blocks[$app][$block]['params'][$param]['name'];
    }

    /**
     * Returns the default value of the specified parameter.
     *
     * @param string $app    An application name.
     * @param string $block  A block name.
     * @param string $param  A parameter name.
     *
     * @return string  The default value of the specified parameter or null.
     */
    function getDefaultValue($app, $block, $param)
    {
        $this->getParams($app, $block);
        if (isset($this->_blocks[$app][$block]['params'][$param]['default'])) {
            return $this->_blocks[$app][$block]['params'][$param]['default'];
        }
        return null;
    }

    /**
     * Returns if the specified block is customizeable by the user.
     *
     * @param string $app    An application name.
     * @param string $block  A block name.
     *
     * @return boolean  True is the block is customizeable.
     */
    function isEditable($app, $block)
    {
        $this->getParams($app, $block);
        return isset($this->_blocks[$app][$block]['params']) &&
            count($this->_blocks[$app][$block]['params']);
    }

}
