<?php
// $Horde: horde/lib/Perms/sql.php,v 1.10 2001/09/07 15:21:56 chuck Exp $

/**
 * The Perms_sql class provides a sql implementation of the Horde
 * permissions system.
 *
 * Required values for $params:
 *      'phptype'       The database type (ie. 'pgsql', 'mysql, etc.).
 *      'hostspec'      The hostname of the database server.
 *      'protocol'      The communication protocol ('tcp', 'unix', etc.).
 *      'username'      The username with which to connect to the database.
 *      'password'      The password associated with 'username'.
 *      'database'      The name of the database.
 *      'table'         The name of the preferences table in 'database'.
 * 
 * Required by some database implementations:
 *      'options'       Additional options to pass to the database.
 *      'tty'           The TTY on which to connect to the database.
 *      'port'          The port on which to connect to the database.
 *
 * The table structure for the preferences is as follows:
 *
 *  create table horde_perms (
 *      perm_uid        varchar(255) not null,
 *      perm_name       varchar(255) not null,
 *      perm_value      int not null default 0,
 *      primary key (perm_uid, perm_name)
 *  );
 *
 * @author  Chuck Hagenbuch <chuck@horde.org>
 * @version $Revision: 1.10 $
 * @since   Horde 1.3
 * @package horde.perms
 */
class Perms_sql extends Perms {
    
    /** Hash containing connection parameters. */
    var $params = array();
    
    /** Handle for the current database connection. */
    var $db;
    
    /** Boolean indicating whether or not we're connected to the SQL server. */
    var $connected = false;
	
    
    /**
     * Constructs a new SQL permissions object.
     *
     * @param optional array $params   A hash containing connection parameters.
     */
    function Perms_sql($params = array())
    {
        $this->params = $params;
    }
	
	
    /**
     * Attempts to open a persistent connection to the SQL server.
     *
     * @return int PERMS_OK on success, PERMS_ERROR_* on failure.
     */
    function _connect()
    {
        if (!$this->connected) {
            include_once 'DB.php';
            if (!is_array($this->params)) return PERMS_ERROR_PARAMS;
            if (!isset($this->params['phptype'])) return PERMS_ERROR_PARAMS;
            if (!isset($this->params['hostspec'])) return PERMS_ERROR_PARAMS;
            if (!isset($this->params['username'])) return PERMS_ERROR_PARAMS;
            if (!isset($this->params['password'])) return PERMS_ERROR_PARAMS;
            if (!isset($this->params['database'])) return PERMS_ERROR_PARAMS;
            if (!isset($this->params['table'])) return PERMS_ERROR_PARAMS;

            /* Connect to the SQL server using the supplied parameters. */
            $this->db = &DB::connect($this->params, true);
            if (DB::isError($this->db) || DB::isWarning($this->db)) {
                return PERMS_ERROR_CONNECT;
            }

            $this->connected = true;
        }

        return PERMS_OK;
    }
	
    /**
     * Disconnect from the SQL server and clean up the connection.
     *
     * @return boolean true on success, false on failure.
     */
    function _disconnect()
    {
        if ($this->connected) {
            $this->connected = false;
            return $this->db->disconnect();
        }

        return true;
    }
	
    /**
	 * Find out what rights the given user has to the specified object.
	 * 
     * @param string $objectpath The full path to the piece of
     *                           content/whatever to check the permissions of.
	 * @param string $user The user to check for.
	 * @param int    $default What to return if nothing is found in the db.
	 * 
	 * @return int Any permissions the user has, or $default if there
	 *             are none.
	 */
    function check($objectpath, $user, $default = PERMS_NONE)
    {
		if ($this->_connect() != PERMS_OK) return $default;
		
		if (empty($objectpath)) {
			return $default;
        }
		
		if (!is_array($objectpath)) {
			$objectpath = explode($this->path_seperator, $objectpath);
        }
		
        /* Build the SQL query. */
        $query = 'SELECT perm_value FROM ' . $this->params['table'];
        $query .= ' WHERE perm_name = ' . $this->db->quote(array_pop($objectpath)) . ' AND perm_uid = ' . $this->db->quote($user);
        
        /* Execute the query. */
        $result = $this->db->query($query);
		
        if (!DB::isError($result)) {
            $row = $result->fetchRow(DB_GETMODE_ASSOC);
            if ($row && !DB::isError($row) && (isset($row['perm_value']))) {
                $result->free();
                return $row['perm_value'];
            } else {
                $result->free();
                return $this->check($objectpath, $user, $default);
            }
        } else {
            return PERMS_ERROR;
        }
    }
	
    /**
     * Add the permission(s) specified to the given user and object.
     *
     * @param string $objectpath The full path to the piece of
     *                           content/whatever to add permissions to.
     * @param string $user The user to add permissions for.
     * @param int    $perm The permission to be added.
     */
    function addPerms($object, $user, $perm)
    {
        if ($this->_connect() != PERMS_OK) return false;
        
        /* Build the SQL query. */
        $query = 'UPDATE ' . $this->params['table'] . ' SET perm_value = perm_value | ' . $perm;
        $query .= 'where perm_uid = ' . $this->db->quote($user);
        
        /* Execute the query. */
        $result = $this->db->query($query);
        
        if ($result !== DB_OK) {
            return PERMS_ERROR;
        }
        return PERMS_OK;
    }
    
	/**
     * Remove the permission(s) specified from the given user and object.
     *
     * @param string $objectpath The full path to the piece of
     *                           content/whatever to remove permissions from.
     * @param string $user The user to remove permissions from.
     * @param int    $perm The permission to be removed.
     */
    function removePerms($object, $user, $perm)
    {
        if ($this->_connect() != PERMS_OK) return false;
        
        /* Build the SQL query. */
        $query = 'UPDATE ' . $this->params['table'] . ' SET perm_value = perm_value & ~' . $perm;
        $query .= 'where perm_uid = ' . $this->db->quote($user);
        
        /* Execute the query. */
        $result = $this->db->query($query);
        
        if ($result !== DB_OK) {
            return PERMS_ERROR;
        }
        return PERMS_OK;
    }
    
}
?>
