<?php
/**
 * $Horde: horde/lib/Category/sql.php,v 1.8.2.3 2002/01/02 17:05:43 jan Exp $
 * Copyright 1999-2002 Original Author <shuther@bigfoot.com>
 * See the enclosed file COPYING for license information (LGPL).  If you 
 * did not receive this file, see http://www.fsf.org/copyleft/lgpl.html.
 */

/**
 * The Category_sql:: class provides an SQL implementation of the Horde
 * caqtegory system.
 * 
 * Required values for $params:
 *      'phptype'        The database type (ie. 'pgsql', 'mysql, etc.).
 *      'hostspec'       The hostname of the database server.
 *      'protocol'       The communication protocol ('tcp', 'unix', etc.).
 *      'username'       The username with which to connect to the database.
 *      'password'       The password associated with 'username'.
 *      'database'       The name of the database.
 *      'table'          The name of the data table in 'database'.
 *      'tableRelations' The name of the relation parents/children table in 
 *                      'database'.
 *
 * The table structure for the category system is as follows:
 *
 * Note: A group may be part of several groups.
 * 
 * @todo Use executeMultiple when required, try to setup transactions,
 * undo/redo if possible
 *
 * @author  Stephane Huther <shuther@bigfoot.com>
 * @version $Revision: 1.8.2.3 $
 * @since   Horde 2.1
 * @package horde.category
 *
 * 
 */
class Category_sql extends Category {

    /**
     * Handle for the current database connection.
     * @var resource $db
     */
    var $db;
    
    /**
     * Boolean indicating whether or not we're connected to the SQL server.
     * @var boolean $connected
     */
    var $_connected = false;


    /**
     * Constructs a new SQL authentication object.
     *
     * @param array $params   A hash containing connection parameters.
     */
    function Category_sql($params)
    {
        parent::Category($params);
        $this->_connect();
    }
    
    /**
     * Attempts to open a persistent connection to the SQL server.
     *
     * @return boolean true.
     */
    function _connect()
    {
        if (!$this->_connected) {
            if (!is_array($this->params)) {
                Horde::fatal(new PEAR_Error(_("No configuration information specified for SQL Categories.")), __FILE__, __LINE__);
            }
            if (!isset($this->params['phptype'])) {
                Horde::fatal(new PEAR_Error(_("Required 'phptype' not specified in categories configuration.")), __FILE__, __LINE__);
            }
            if (!isset($this->params['hostspec'])) {
                Horde::fatal(new PEAR_Error(_("Required 'hostspec' not specified in categories configuration.")), __FILE__, __LINE__);
            }
            if (!isset($this->params['username'])) {
                Horde::fatal(new PEAR_Error(_("Required 'username' not specified in categories configuration.")), __FILE__, __LINE__);
            }
            if (!isset($this->params['password'])) {
                Horde::fatal(new PEAR_Error(_("Required 'password' not specified in categories configuration.")), __FILE__, __LINE__);
            }
            if (!isset($this->params['database'])) {
                Horde::fatal(new PEAR_Error(_("Required 'database' not specified in categories configuration.")), __FILE__, __LINE__);
            }
            if (!isset($this->params['table'])) {
                Horde::fatal(new PEAR_Error(_("Required 'table' not specified in categories configuration.")), __FILE__, __LINE__);
            }
            if (!isset($this->params['tableRelations'])) {
                Horde::fatal(new PEAR_Error(_("Required 'tableRelations' not specified in categories configuration.")), __FILE__, __LINE__);
            }
            
            /* Connect to the SQL server using the supplied parameters. */
            include_once 'DB.php';
            $this->db = &DB::connect($this->params, true);
            if (DB::isError($this->db) || DB::isWarning($this->db)) {
                Horde::fatal($this->db, __FILE__, __LINE__);
            }
            
            $this->_connected = true;

            $r = $this->fullyload(); // Load everything in memory.
            if (PEAR::isError($r)) {
                Horde::fatal($r, __FILE__, __LINE__);
            }
        }
        
        return true;
    }
    
    /**
     * Disconnect from the SQL server and clean up the connection.
     *
     * @return boolean true on success, false on failure.
     */
    function _disconnect()
    {

        if ($this->_connected) {
            $this->_connected = false;
            return $this->db->disconnect();
        }

        return true;
    }

    /**
     * Dump the database to the array of this class
     * it is a way to do a one way synchronize
     * 
     * @return mixed may return a DB error if not an array
     *
     * note: there is no check against circular reference!!!
     */
    function fullyload()
    {
        $query = sprintf('SELECT t.category_name as name, t2.category_name as category_id_parent FROM %s t, %s t2, %s cp' .
                         ' WHERE t.group_uid=%s AND cp.category_id_parent=t2.category_id AND cp.category_id_child=t.category_id' .
                         ' AND t2.group_uid=%s ',
                         $this->params['table'], $this->params['table'], $this->params['tableRelations'],
                         $this->db->quote($this->params['group']), $this->db->quote($this->params['group']));
        
        /* Execute the query. */
        $result = $this->db->getAssoc($query, true);
        
        if (DB::isError($result)) {
            return $result;
        }
        
        $query = sprintf('SELECT t.category_name as name, \'-1\' as parent FROM %s t, %s cp' .
                         ' WHERE t.group_uid=%s AND t.category_id=cp.category_id_child AND cp.category_id_parent=\'-1\'',
                         $this->params['table'], $this->params['tableRelations'], $this->db->quote($this->params['group']));
        
        /* Execute the query. */
        $result2 = $this->db->getAssoc($query, true);
        
        if (DB::isError($result2)) {
            return $result2;
        }
        
        $this->import(CATEGORY_FORMAT_FETCH, $result);
        $this->import(CATEGORY_FORMAT_FETCH, $result2);
        
        return true;
    }

    /**
     * Add a category
     * 
     * note: there is no check against circular reference!!!
     * @param string $name       The name of the category.
     * @param optional string $parent   the name of the parent category
     */
    function addCategory($category, $parent = '-1')
    {
        $this->_connect();

        $retval = parent::addCategory($category, $parent);
        if (PEAR::isError($retval)) {
            return $retval;
        }

        if (is_subclass_of($category, 'CategoryObject')) {
            $name = $category->getName();
            $data = SerializeUnit::serializeUnit($category->getData(), CATEGORY_SERIALIZE_FORMAT);
            $ser = CATEGORY_SERIALIZE_FORMAT;
        } else {
            $name = $category;
            $data = '';
            $ser = SERIALIZEUNIT_NONE;
        }

        $id = $this->db->nextId($this->params['table']);
        if (DB::isError($id)) {
            return $id;
        }

        if ('-1' == $parent) {
            $parentid = '-1';
        } else {
            $query = sprintf('SELECT t.category_id as id FROM %s t WHERE t.category_name=%s',
                             $this->params['table'], $this->db->quote($parent));
            $parentid = $this->db->getCol($query);
            if (DB::isError($parentid)) {
                return $parentid;
            }
            
            if (count($parentid) == 0) {
                $parentid = '-1';
            } else {
                $parentid = $parentid[0];
            }
        }

        $query = sprintf('INSERT INTO %s (category_id, group_uid, category_name, category_data, user_uid, category_serialized)' .
                         ' VALUES (%s, %s, %s, %s, %s, %s)',
                         $this->params['table'], $id, $this->db->quote($this->params['group']),
                         $this->db->quote($name), $this->db->quote($data),
                         $this->db->quote(Auth::getAuth()), $ser);

        /* Execute the query. */
        $result = $this->db->query($query);
        if (DB::isError($result)) {
            return $result;
        }

        $query = sprintf('INSERT INTO %s (category_id_child, category_id_parent) VALUES (%s, %s)',
                         $this->params['tableRelations'], $id, $parentid);

        /* Execute the query. */
        $result = $this->db->query($query);
        if (DB::isError($result)) {
            return $result;
        }
        
        return true;
    }

    /**
     * Remove a category from another one
     * 
     * @param string $name       The name of the category.
     * @param optional string $parent       The name of the parent from where
     *                           we remove it. If default, we removed it from
     *                           every category.
     *                           0 means every branch
     *                           -1 mean the root
     *                           other means just one, not deeper
     */
    function removeCategory($category, $parent)
    {
        $this->_connect();

        $retval = parent::removeCategory($category, $parent);
        if (PEAR::isError($retval)) {
            return $retval;
        }

        if (is_subclass_of($category, 'CategoryObject')) {
            $name = $category->getName();
        } else {
            $name = $category;
        }

        if (!isset($this->categories[$name])) {
            return new PEAR_Error('The category does not exist.');
        }
        if (in_array($name, array_values($this->categories))) {
            return new PEAR_Error('Removal failed.');
        }

        $query = sprintf('DELETE FROM %s WHERE group_uid=%s AND category_name=%s',
                         $this->params['table'],
                         $this->db->quote($this->params['group']),
                         $this->db->quote($name));

        /* Execute the query. */
        $result = $this->db->query($query);
        if (DB::isError($result)) {
            return $result;
        }

        return true;
    }
     
    /**
     * Move a category from one parent to a new one.
     * 
     * note: there is no check against circular reference!!!
     * @param string $name       The name of the category.
     * @param string $new_parent The name of the new parent.
     */
    function moveCategory($name, $old_parent, $new_parent = '-1')
    {
        $this->_connect();
        
        settype($old_parent, 'string');
        settype($new_parent, 'string');

        $ret = parent::moveCategory($name, $old_parent, $new_parent);
        if (PEAR::isError($ret)) {
            return $ret;
        }
        
        if (is_subclass_of($name, 'CategoryObject')) {
            $name = SerializeUnit::serializeUnit($name->getData(), CATEGORY_SERIALIZE_FORMAT);
            $ser = CATEGORY_SERIALIZE_FORMAT;
        } else {
            $ser = SERIALIZEUNIT_NONE;
        }
        
        if ('-1' == $new_parent) {
            $parentid = '-1';
        } else {
            $query = sprintf('SELECT t.category_id as id FROM %s t WHERE t.category_name=%s AND t.group_uid=%s',
                             $this->params['table'], $this->db->quote($new_parent),
                             $this->db->quote($this->params['group']));
            $parentid = $this->db->getCol($query);
            if (DB::isError($parentid)) {
                return $parentid;
            }
            if ('0' == $parentid) {
                $parentid = '-1';
            } else {
                $parentid = $parentid[0];
            }
        }

        $query = sprintf('SELECT cp.category_id_child as child, cp.category_id_parent as parent FROM %s cp, %s t, %s told' .
                         ' WHERE t.group_uid=%s AND t.category_name=%s AND t.category_serialized=%s' . 
                         ' AND t.category_id=cp.category_id_child AND cp.category_id_parent=told.category_id' . 
                         ' AND t.group_uid=told.group_uid AND told.category_name=%s',
                         $this->params['tableRelations'], $this->params['table'],
                         $this->params['table'], $this->db->quote($this->params['group']),
                         $this->db->quote($name), $ser, $this->db->quote($old_parent));
        
        $row = $this->db->getRow($query, array(), DB_FETCHMODE_ASSOC);
        if (DB::isError($row)) {
            return $row;
        }

        $old_parentid = $row['parent'];
        $childid = $row['child'];
        
        $query = sprintf('UPDATE %s SET category_id_parent=%s WHERE category_id_child=%s AND category_id_parent=%s',
                         $this->params['tableRelations'], $parentid, $childid, $old_parentid);
        
        /* Execute the query. */
        $result = $this->db->query($query);
        if (DB::isError($result)) {
            return $result;
        }

        return true;
    }
    
    /**
     * Update the data in a category. Does not change the category's
     * parent, just name and/or serialized data.
     *
     * @param string $old_category  The old category data.
     * @param string $new_category  The new category data.
     */
    function updateCategory($old_category, $new_category)
    {
        $this->_connect();

        $retval = parent::updateCategory($old_category, $new_category);
        if (PEAR::isError($retval)) {
            return $retval;
        }
        
        if (is_subclass_of($old_category, 'CategoryObject')) {
            $old_name = $old_category->getName();
        } else {
            $old_name = $old_category;
        }

        if (is_subclass_of($new_category, 'CategoryObject')) {
            $new_name = $new_category->getName();
            $new_data = SerializeUnit::serializeUnit($new_category->getData(), CATEGORY_SERIALIZE_FORMAT);
            $new_ser = CATEGORY_SERIALIZE_FORMAT;
        } else {
            $new_name = $new_category;
            $new_data = null;
            $new_ser = SERIALIZEUNIT_NONE;
        }

        $query = sprintf('UPDATE %s SET category_name=%s, category_data=%s, category_serialized=%s' .
                         ' WHERE category_name=%s',
                         $this->params['table'], $this->db->quote($new_name), $this->db->quote($new_data),
                         $this->db->quote($new_ser), $this->db->quote($old_name));
        
        /* Execute the query. */
        $result = $this->db->query($query);
        if (DB::isError($result)) {
            return $result;
        }

        return true;
    }

}
?>
