<?php
// $Horde: horde/lib/Auth.php,v 1.22.2.2 2001/12/03 16:44:35 chuck Exp $

require_once HORDE_BASE . '/lib/Secret.php';

/**
 * The Auth:: class provides a common abstracted interface into the
 * various backends for the Horde authentication system.
 *
 * @author  Chuck Hagenbuch <chuck@horde.org>
 * @version $Revision: 1.22.2.2 $
 * @since   Horde 1.3
 * @package horde.auth
 */
class Auth {
    
    /** An array of capabilities, so that the driver can report which
        operations it supports and which it doesn't.
        @var array $capabilities */
    var $capabilities = array('add'    => false,
                              'update' => false,
                              'remove' => false,
                              'list'   => false);
     
    
    /**
     * Attempts to return a concrete Auth instance based on $driver.
     * 
     * @param string $driver The type of concrete Auth subclass to return.
     *                       This is based on the storage driver ($driver). The
     *                       code is dynamically included.
     * @param array  $params (optional) A hash containing any additional
     *                       configuration or connection parameters a subclass
     *                       might need.
     *
     * @return object Auth   The newly created concrete Auth instance, or false
     *                       on an error.
     */
    function &factory($driver, $params = array())
    {
        $driver = strtolower(basename($driver));

        if (empty($driver) || (strcmp($driver, 'none') == 0)) {
            return new Auth;
        }

        @include_once dirname(__FILE__) . '/Auth/' . $driver . '.php';
        $class = 'Auth_' . $driver;
        if (class_exists($class)) {
            return new $class($params);
        } else {
            return false;
        }
    }
    
    /**
     * Attempts to return a reference to a concrete Auth instance based on
     * $driver. It will only create a new instance if no Auth instance
     * with the same parameters currently exists.
     *
     * This should be used if multiple permissions sources (and, thus,
     * multiple Auth instances) are required.
     *
     * This method must be invoked as: $var = &Auth::singleton()
     *
     * @param string $driver The type of concrete Auth subclass to return.
     *                       This is based on the storage driver ($driver). The
     *                       code is dynamically included.
     * @param array  $params (optional) A hash containing any additional
     *                       configuration or connection parameters a subclass
     *                       might need.
     *
     * @return object Auth  The concrete Auth reference, or false on an error.
     */
    function &singleton($driver, $params = array())
    {
        static $instances;

        if (!isset($instances)) $instances = array();

        $signature = md5(strtolower($driver) . '][' . implode('][', $params));
        if (!isset($instances[$signature])) {
            $instances[$signature] = &Auth::factory($driver, $params);
        }

        return $instances[$signature];
    }

    /**
     * Find out if a set of login credentials are valid.
     * 
     * @param string $userID       The userID to check
     * @param array  $credentials  The credentials to check
     * 
     * @return boolean Whether or not the credentials are valid.
     */
    function authenticate($userID, $credentials)
    {
        return Auth::checkAuth($userID);
    }

    /**
     * Add a set of authentication credentials.
     * 
     * @param string $userID       The userID to add.
     * @param array  $credentials  The credentials to use.
     *
     * @return mixed        True on success or a PEAR_Error object on failure.
     */
    function addUser($userID, $credentials)
    {
        return (new PEAR_Error('unsupported'));
    }

    /**
     * Update a set of authentication credentials.
     * 
     * @param string $oldID        The old userID.
     * @param string $newID        The new userID.
     * @param array  $credentials  The new credentials
     *
     * @return mixed        True on success or a PEAR_Error object on failure.
     */
    function updateUser($oldID, $newID, $credentials)
    {
        return (new PEAR_Error('unsupported'));
    }

    /**
     * Delete a set of authentication credentials.
     * 
     * @param string $userID  The userID to delete.
     *
     * @return mixed        True on success or a PEAR_Error object on failure.
     */
    function removeUser($userID)
    {
        return (new PEAR_Error('unsupported'));
    }
    
    /**
     * List all users in the system.
     * 
     * @return mixed   The array of userIDs, or a PEAR_Error object on failure.
     */
    function listUsers()
    {
        return (new PEAR_Error('unsupported'));
    }
    
    /**
     * Checks if there is a session with valid auth information.
     *
     * @param string $userID          The userID we are expecting in the session.
     * @param optional string $realm  The authentication realm to check for the user.
     *
     * @return boolean                Whether or not the user is authenticated.
     */
    function checkAuth($userID, $realm = null)
    {
        if (!empty($realm)) {
            if (isset($GLOBALS['HTTP_SESSION_VARS']['__auth'][$realm])) {
                if (!empty($GLOBALS['HTTP_SESSION_VARS']['__auth'][$realm]['authenticated']) &&
                    !empty($GLOBALS['HTTP_SESSION_VARS']['__auth'][$realm]['userID']) &&
                    $GLOBALS['HTTP_SESSION_VARS']['__auth'][$realm]['userID'] == $userID) {
                    return true;
                }
            }
        } else {
            if (isset($GLOBALS['HTTP_SESSION_VARS']['__auth'])) {
                if (!empty($GLOBALS['HTTP_SESSION_VARS']['__auth']['authenticated']) &&
                    !empty($GLOBALS['HTTP_SESSION_VARS']['__auth']['userID']) &&
                    $GLOBALS['HTTP_SESSION_VARS']['__auth']['userID'] == $userID) {
                    return true;
                }
            }
        }
        return false;
    }
	
    /**
     * Return the currently logged in user, if there is one.
     *
     * @param optional string $realm  The authentication realm to check for a user.
     * 
     * @return mixed $userID The userID of the current user, or false
     *                       if no user is logged in.
     */
    function getAuth($realm = null)
    {
        if (!empty($realm)) {
            if (isset($GLOBALS['HTTP_SESSION_VARS']['__auth'][$realm])) {
                if (!empty($GLOBALS['HTTP_SESSION_VARS']['__auth'][$realm]['authenticated']) &&
                    !empty($GLOBALS['HTTP_SESSION_VARS']['__auth'][$realm]['userID'])) {
                    return $GLOBALS['HTTP_SESSION_VARS']['__auth'][$realm]['userID'];
                }
            }
        } else {
            if (isset($GLOBALS['HTTP_SESSION_VARS']['__auth'])) {
                if (!empty($GLOBALS['HTTP_SESSION_VARS']['__auth']['authenticated']) &&
                    !empty($GLOBALS['HTTP_SESSION_VARS']['__auth']['userID'])) {
                    return $GLOBALS['HTTP_SESSION_VARS']['__auth']['userID'];
                }
            }
        }
        return false;
    }

    /**
     * Return a credential of the currently logged in user, if there is one.
     *
     * @param          string $credential  The credential to retrieve.
     * @param optional string $realm       The authentication realm to check
     *                                     for a user.
     * 
     * @return mixed   The requested credential, or false
     *                 if no user is logged in.
     */
    function getCredential($credential, $realm = null)
    {
        if (!empty($realm) &&
            !empty($GLOBALS['HTTP_SESSION_VARS']['__auth'][$realm]) &&
            !empty($GLOBALS['HTTP_SESSION_VARS']['__auth'][$realm]['authenticated'])) {
            $credentials = unserialize(Secret::read(Secret::getKey('auth'), $GLOBALS['HTTP_SESSION_VARS']['__auth'][$realm]['credentials']));
        } else if (!empty($GLOBALS['HTTP_SESSION_VARS']['__auth']) &&
                   !empty($GLOBALS['HTTP_SESSION_VARS']['__auth']['authenticated'])) {
            $credentials = unserialize(Secret::read(Secret::getKey('auth'), $GLOBALS['HTTP_SESSION_VARS']['__auth']['credentials']));
        } else {
            return false;
        }

        if (isset($credentials[$credential])) {
            return $credentials[$credential];
        } else {
            return false;
        }
    }

    /**
     * Set a variable in the session saying that authorization has
     * succeeded, note which userID was authorized, and note when the
     * login took place.
     *
     * @param          string $userID       The userID who has been authorized.
     * @param          array  $credentials  The credentials of the user.
     * @param optional string $realm        The realm the user is authorized in.
     */
    function setAuth($userID, $credentials, $realm = null)
    {
        if (!isset($GLOBALS['HTTP_SESSION_VARS']['__auth'])) {
            session_register('__auth');
        }
        $GLOBALS['__auth'] = &$GLOBALS['HTTP_SESSION_VARS']['__auth'];
        
        $credentials = Secret::write(Secret::getKey('auth'), serialize($credentials));
        $auth = array('authenticated' => true,
                      'userID' => $userID,
                      'credentials' => $credentials,
                      'timestamp' => time());
        
        if (!empty($realm)) {
            $GLOBALS['__auth'][$realm] = $auth;
        } else {
            $GLOBALS['__auth'] = $auth;
        }
    }
    
    /**
     * Clear any authentication tokens in the current session.
     */
    function clearAuth()
    {
        if (isset($GLOBALS['HTTP_SESSION_VARS']['__auth'])) {
            $GLOBALS['__auth'] = &$GLOBALS['HTTP_SESSION_VARS']['__auth'];
            $GLOBALS['__auth'] = array();
            $GLOBALS['__auth']['authenticated'] = null;
        }
    }
    
    /**
     * Is the current user a guest user?
     *
     * @return boolean   Whether or not this is a guest user.
     */
    function isGuest()
    {
        return false;
    }
    
    /**
     * Query the current Auth object to find out if it supports the
     * given capability.
     *
     * @param string $capability  The capability to test for.
     * @return boolean            Whether or not the capability is supported.
     */
    function hasCapability($capability)
    {
        return !empty($this->capabilities[$capability]);
    }
    
}
?>
