<?php
/**
 * PostgreSQL Session Handler for PHP
 *
 * Copyright 2000 Jon Parise <jon@csh.rit.edu>.  All rights reserved.
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *  1. Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 *  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 *  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 *  ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 *  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 *  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 *  OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 *  HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 *  LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 *  OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 *  SUCH DAMAGE. 
 *
 * Usage Notes
 * ~~~~~~~~~~~
 * - Set session.save_handler to 'user'.
 * - Set session.save_path to the name of the database table.
 * - Modify the $params string in pgsql_session_open() to match your setup.
 * - Create the table structure using the follow schema:
 *
 *      CREATE TABLE php_sessions (
 *          session_id  CHAR(64)    NOT NULL PRIMARY KEY,
 *          last_active INTEGER     NOT NULL,
 *          data        TEXT
 *      );
 * 
 * @author  Jon Parise <jon@csh.rit.edu>
 * @version $Revision: 1.3 $
 */

/* Make sure PHP is configured for our custom session handler. */
assert(get_cfg_var('session.save_handler') == 'user');

/* Get the name of the session table.  Default to 'php_sessions'. */
$pgsql_session_table = get_cfg_var('session.save_path');
if (empty($pgsql_session_table)) $pgsql_session_table = 'php_sessions';

/* Global PostgreSQL database connection handle. */
$pgsql_session_handle = 0;


/**
 * Opens a new session.
 *
 * @param $save_path    The value of session.save_path.
 * @param $session_name The name of the session ('PHPSESSID').
 *
 * @return  boolean     True on success, false on failure.
 */
function pgsql_session_open($save_path, $session_name)
{
    global $pgsql_session_handle;

    $params = 'host=localhost dbname=gaia user=jon password=foobie';

    $pgsql_session_handle = pg_pconnect($params);
    return ($pgsql_session_handle != 0);
}

/**
 * Closes the current session.
 *
 * @return  boolean     True on success, false on failure.
 */
function pgsql_session_close()
{
    global $pgsql_session_handle;

    if (isset($pgsql_session_handle)) {
        return pg_close($pgsql_session_handle);
    }
    return true;
}

/**
 * Reads the requested session data from the database.
 *
 * @param $key          Unique session ID of the requested entry.
 *
 * @return  string      The requested session data, or false on failure.
 */
function pgsql_session_read($key)
{
    global $pgsql_session_handle, $pgsql_session_table;

    $key = addslashes($key);

    /* Build an execute the database query. */
    $query = "select data from $pgsql_session_table where session_id = '$key'";
    $result = @pg_exec($pgsql_session_handle, $query);

    if (isset($result) && $result && (pg_numrows($result) == 1)) {
        $data = pg_result($result, 0, 'data');
        pg_freeresult($result);
        return $data;
    } else {
        return false;
    }
}

/**
 * Writes the provided session data with the requested key to the database.
 *
 * @param $key          Unique session ID of the current entry.
 * @param $val          String containing the session data.
 *
 * @return  boolean     True on success, false on failure.
 */
function pgsql_session_write($key, $val)
{
    global $pgsql_session_handle, $pgsql_session_table;

    $key = addslashes($key);
    $val = addslashes($val);

    /* Check if there's existing session data for this $key. */
    $query = "select count(*) from $pgsql_session_table where session_id = '$key'";
    $exists = (pg_result(@pg_exec($pgsql_session_handle, $query), 0, 0) == 1);

    /* Build the appropriate query. */
    $now = time();
    if ($exists) {
        $query = "update $pgsql_session_table set data = '$val', " .
                 "last_active = '$now' where session_id = '$key'";
    } else {
        $query = "insert into $pgsql_session_table " .
                 "values('$key', '$now', '$val')";
    }

    $result = @pg_exec($pgsql_session_handle, $query);

    $ret = ($result != 0);
    pg_freeresult($result);

    return $ret;
}

/**
 * Destroys the requested session.
 *
 * @param $key          Unique session ID of the requested entry.
 *
 * @return  boolean     True on success, false on failure.
 */
function pgsql_session_destroy($key)
{
    global $pgsql_session_handle, $pgsql_session_table;

    $key = addslashes($key);

    $query = "delete from $pgsql_session_table where session_id = '$key'";
    $result = @pg_exec($pgsql_session_handle, $query);

    $ret = ($result != 0);
    pg_freeresult($result);

    return $ret;
}

/**
 * Performs session garbage collection based on the provided lifetime.
 *
 * @param $maxlifetime  Maximum lifetime of a session.
 *
 * @return  boolean     True on success, false on failure.
 */
function pgsql_session_gc($maxlifetime)
{
    global $pgsql_session_handle, $pgsql_session_table;

    $expiry = time() - $maxlifetime;

    $query = "delete from $pgsql_session_table where last_active < $expiry";
    $result = @pg_exec($pgsql_session_handle, $query);

    $ret = ($result != 0);
    pg_freeresult($result);

    return $ret;
}

/* Register the session handling functions with PHP. */
session_set_save_handler(
    'pgsql_session_open',
    'pgsql_session_close',
    'pgsql_session_read',
    'pgsql_session_write',
    'pgsql_session_destroy',
    'pgsql_session_gc'
);
?>
