<?php
// $Horde: horde/lib/Registry.php,v 1.13 2000/12/01 03:42:25 chuck Exp $

/**
 * The Registry:: class provides a set of methods for communication
 * between Horde applications.
 *
 * @author   Chuck Hagenbuch <chuck@horde.org>
 * @author   Jon Parise <jon@horde.org>
 * @version  $Revision: 1.13 $
 * @since Horde 1.3
 */
class Registry {
    
    var $registry = array();
    var $services = array();
    var $applications = array();
    var $used;
    
    /**
     * Create a registry instance.
     */
    function Registry()
    {
        // read in and evaluate the registry file
        include_once dirname(__FILE__) . '/../config/registry.php';
        
        // assign the variables to class variables for local storage/reference
        $this->registry = $__horde_registry;
        $this->applications = $__horde_applications;
        $this->services = $__horde_registered_services; 
    }
    
    /**
     * Return the hook corresponding to the default package that
     * provides the functionality requested by the $method
     * parameter. $method is a string consisting of
     * "packagetype/methodname".
     *
     * @param $method the method to call
     * @param $args (optional) arguments to the method
     * @param $extra (optional) extra, non-standard arguments to the method
     */
    function call($method, $args = array(), $extra = array())
    {
        list($type, $subtype) = explode('/', $method);
        return $this->callByPackage($this->registry[$type][$subtype], $method, $args, $extra);
    }
    
    /**
     * Echo the output of call()
     *
     * @param $method the method to call
     * @param $args (optional) arguments to the method
     * @param $extra (optional) extra, non-standard arguments to the method
     */
    function pcall($method, $args = array(), $extra = array())
    {
        echo $this->call($method, $args, $extra);
    }
    
    /**
     * Output the hook corresponding to the specific package named.
     *
     * @param $package the desired package.
     * @param $method the method to call.
     * @param $args (optional) arguments to the method.
     * @param $extra (optional) extra, non-standard arguments to the method.
     */
    function callByPackage($package, $method, $args = array(), $extra = array())
    {
        // mark this service as used so that we can output support
        // functions at the end of the page
        $this->used[$method] = true;
        list($type, $subtype) = explode('/', $method);
        
        if (empty($this->services[$this->registry[$type][$subtype]][$type][$subtype]['callback'])) {
            // no callback defined
            return null;
        }
        
        $callback = $this->services[$this->registry[$type][$subtype]][$type][$subtype]['callback'];
        
        // fill in urlencoded arguments
        foreach ($args as $key => $val) {
            $callback = str_replace('%' . $key . '%', htmlentities($val), $callback);
        }
        if (isset($this->applications[$package]['webroot'])) {
            $callback = str_replace('%application%', $this->applications[$package]['webroot'], $callback);
        }
        
        // repace urlencoded arguments that haven't been specified with an
        // empty string (this is where the default would be substituted in
        // a stricter registry).
        $callback = preg_replace("|%.+%|U", '', $callback);
    
        // fill in html-encoded arguments
        foreach ($args as $key => $val) {
            $callback = str_replace('|' . $key . '|', urlencode($val), $callback);
        }
        
        // append any extra, non-standard arguments
        $extra_args = '';
        foreach ($extra as $key => $val) {
            $key = urlencode($key);
            $val = urlencode($val);
            $extra_args .= "&$key=$val";
        }
        $callback = str_replace('|extra|', $extra_args, $callback);
        
        // repace html-encoded arguments that haven't been specified with
        // an empty string (this is where the default would be substituted
        // in a stricter registry).
        $callback = preg_replace('|\|.+\||U', '', $callback);

        return htmlspecialchars($callback);
    }
    
    /**
     * Echo the output of callByPackage()
     *
     * @param $package the desired package
     * @param $method the method to call
     * @param $args (optional) arguments to the method
     */
    function pcallByPackage($package, $method, $args = array())
    {
        echo $this->callByPackage($package, $method, $args);
    }
    
    /**
     * Return the code corresponding to the default package necessary
     * to directly invoke the method. If you want to print results,
     * this will print the results, not a link to get the results. If
     * you want to invoke a javascript function, this will invoke it, not
     * print a link that when clicked will invoke it.
     *
     * @param $method the method to invoke.
     * @param $args (optional) arguments to the method.
     * @param $extra (optional) extra, non-standard arguments to the method.
     */
    function invoke($method, $args = array(), $extra = array())
    {
        list($type, $subtype) = explode('/', $method);
        return $this->invokeByPackage($this->registry[$type][$subtype], $method, $args, $extra);
    }
    
    /**
     * Echo the output of invoke()
     *
     * @param $method the method to invoke
     * @param $args (optional) arguments to the method
     * @param $extra (optional) extra, non-standard arguments to the method
     */
    function pinvoke($method, $args = array(), $extra = array())
    {
        echo $this->invoke($method, $args, $extra);
    }
    
    /**
     * Return the code corresponding to the default package necessary
     * to directly invoke the method. If you want to print results,
     * this will print the results, not a link to get the results. If
     * you want to invoke a javascript function, this will invoke it, not
     * print a link that when clicked will invoke it.
     *
     * @param $package the desired package.
     * @param $method the method to invoke.
     * @param $args (optional) arguments to the method.
     * @param $extra (optional) extra, non-standard arguments to the method.
     */
    function invokeByPackage($package, $method, $args = array(), $extra = array())
    {
        // mark this service as used so that we can output support
        // functions at the end of the page
        $this->used[$method] = true;
        list($type, $subtype) = explode('/', $method);
        
        if (empty($this->services[$this->registry[$type][$subtype]][$type][$subtype]['invocation'])) {
            // no invocation defined
            return null;
        }
        
        $invocation = $this->services[$this->registry[$type][$subtype]][$type][$subtype]['invocation'];
        
        // fill in urlencoded arguments
        foreach ($args as $key => $val) {
            $invocation = str_replace('%' . $key . '%', htmlentities($val), $invocation);
        }
        $invocation = str_replace('%application%', $this->applications[$package]['webroot'], $invocation);
        
        // repace urlencoded arguments that haven't been specified with an
        // empty string (this is where the default would be substituted in
        // a stricter registry).
        $invocation = preg_replace("|%.+%|U", '', $invocation);
        
        // fill in html-encoded arguments
        foreach ($args as $key => $val) {
            $invocation = str_replace('|' . $key . '|', urlencode($val), $invocation);
        }
        
        // append any extra, non-standard arguments
        $extra_args = '';
        foreach ($extra as $key => $val) {
            $key = urlencode($key);
            $val = urlencode($val);
            $extra_args .= "&$key=$val";
        }
        $invocation = str_replace('|extra|', $extra_args, $invocation);
        
        // repace html-encoded arguments that haven't been specified with
        // an empty string (this is where the default would be substituted
        // in a stricter registry).
        $invocation = preg_replace('|\|.+\||U', '', $invocation);
        
        return htmlspecialchars($invocation);
    }
    
    /**
     * Echo the output of invokeByPackage()
     *
     * @param $package the desired package.
     * @param $method the method to invoke.
     * @param $args (optional) arguments to the method.
     */
    function pinvokeByPackage($package, $method, $args = array())
    {
        echo $this->invokeByPackage($package, $method, $args);
    }
    
    /**
     * Go through the list of methods that have been used, and
     * evaluate/include any support files needed.
     */
    function shutdown()
    {
        if (!empty($this->used) && is_array($this->used)) {
            foreach ($this->used as $key => $val) {
                list($type, $subtype) = explode('/', $key);
                $application = $this->registry[$type][$subtype];
                if (!empty($this->services[$application][$type][$subtype]['includeFile'])) {
                    include_once $this->applicationFilePath($this->services[$application][$type][$subtype]['includeFile'], $application);
                }
            }
        }
    }
    
    /**
     * Replace any %application% strings with the filesystem path to
     * the application.
     */
    function applicationFilePath($path, $app)
    {
        return str_replace('%application%', $this->applications[$app]['fileroot'], $path);
    }
    
    /**
     * Replace any %application% strings with the web path to the
     * application.
     */
    function applicationWebPath($path, $app)
    {
        return str_replace('%application%', $this->applications[$app]['webroot'], $path);
    }
    
}
?>
