<?php
// $Horde: horde/lib/Prefs/sql.php,v 1.12 2000/12/11 02:43:12 chuck Exp $

/**
 * Preferences storage implementation for PHP's PEAR database abstraction
 * layer.
 *
 * Required values for $params:
 *      'phptype'       The database type (ie. 'pgsql', 'mysql, etc.).
 *      'hostspec'      The hostname of the database server.
 *      'username'      The username with which to connect to the database.
 *      'password'      The password associated with 'username'.
 *      'database'      The name of the database.
 *      'table'         The name of the preferences table in 'database'.
 * 
 * Required by some database implementations:
 *      'options'       Additional options to pass to the database.
 *      'tty'           The TTY on which to connect to the database.
 *      'port'          The port on which to connect to the database.
 *
 * The table structure for the preferences is as follows:
 *
 *  create table user_webmail_prefs (
 *      uid             char(32) not null,
 *      pref_name       char(32) not null,
 *      pref_value      text null,
 *      primary key (uid, pref_name)
 *  );
 *
 * @author  Jon Parise <jon@csh.rit.edu>
 * @version $Revision: 1.12 $
 * @since   Horde 1.3
 */
class Prefs_sql extends Prefs {

    /** Hash containing connection parameters. */
    var $params = array();

    /** Handle for the current database connection. */
    var $db;

    /** Boolean indicating whether or not we're connected to the SQL server. */
    var $connected = false;

    /**
     * Constructs a new SQL preferences object.
     *
     * @param $user     The user who owns these preferences.
     * @param $password The password associated with $user. (unused)
     * @param $params   A hash containing connection parameters.
     */
    function Prefs_sql($user, $password = '', $params = array())
    {
        $this->user = $user;
        $this->params = $params;
    }
    
    /**
     * Attempts to open a persistent connection to the SQL server.
     *
     * @return          PREFS_OK on success, PREFS_ERROR_* on failure.
     */
    function connect()
    {
        if (!$this->connected) {
            include_once 'DB.php';
            
            if (!is_array($this->params)) return PREFS_ERROR_PARAMS;
            if (!isset($this->params['phptype'])) return PREFS_ERROR_PARAMS;
            if (!isset($this->params['hostspec'])) return PREFS_ERROR_PARAMS;
            if (!isset($this->params['username'])) return PREFS_ERROR_PARAMS;
            if (!isset($this->params['password'])) return PREFS_ERROR_PARAMS;
            if (!isset($this->params['database'])) return PREFS_ERROR_PARAMS;
            if (!isset($this->params['table'])) return PREFS_ERROR_PARAMS;
            
            /* Connect to the SQL server using the supplied parameters. */
            $this->db = &DB::connect($this->params, true);
            if (DB::isError($this->db) || DB::isWarning($this->db)) {
                return PREFS_ERROR_CONNECT;
            }
            
            $this->connected = true;
        }
        
        return PREFS_OK;
    }

    /**
     * Disconnect from the SQL server and clean up the connection.
     *
     * @return      true on success, false on failure.
     */
    function disconnect()
    {
        if ($this->connected) {
            $this->connected = false;
            return $this->db->disconnect();
        }

        return true;
    }

    /**
     * Retrieves the requested set of preferences from the user's database
     * entry.
     *
     * @param $prefs    (optional) An array listing the preferences to
     *                  retrieve.  If not specified, retrieve all of the
     *                  preferences listed in the $prefs hash.
     *
     * @return          PREFS_OK on success, PREFS_ERROR_* on failure.
     */
    function retrieve($prefs = array())
    {
        /* If we're not already connected, invoke the connect() method. */
        if (!$this->connected) {
            if ($this->connect() != PREFS_OK) return PREFS_ERROR_CONNECT;
        }

        /*
         * If a list of preferences to retrieve hasn't been provided in
         * $prefs, assume all preferences are desired.
         */
        if (count($prefs) == 0) {
            $prefs = $this->listAll();
        }
        if (!is_array($prefs) || (count($prefs) == 0)) return PREFS_ERROR;

        /* Build the SQL query. */
        $query = 'select pref_name, pref_value from ';
        $query .= $this->params['table'] . ' '; 
        $query .= "where uid = '" . $this->user . "'";
        
        /* Execute the query. */
        $result = $this->db->query($query);
        
        if (isset($result) && is_object($result)) {
            $row = $result->fetchRow(DB_FETCHMODE_ASSOC);
            if (DB::isError($row)) return PREFS_ERROR_EMPTY;
            
            /*
             * Set the requested values in the $this->prefs hash based on
             * the contents of the SQL result.
             *
             * Note that Prefs::setValue() can't be used here because of the
             * check for the "changeable" bit.  We want to override that
             * check when populating the $this->prefs hash from the SQL
             * server.
             */
            while ($row && !DB::isError($row)) {
                $name = trim($row['pref_name']);
                if (in_array($name, $prefs)) {
                    $value = trim($row['pref_value']);
                    $this->prefs[$name]['val'] = $value;
                    $this->setDirty($name, false);
                }
                $row = $result->fetchRow(DB_FETCHMODE_ASSOC);
            }
            $result->free();
            
        } else return PREFS_ERROR;
        
        return PREFS_OK;
    }

    /**
     * Stores preferences to SQL server.
     *
     * @param $prefs    (optional) An array listing the preferences to be
     *                  stored.  If not specified, store all of the
     *                  preferences listed in the $prefs hash.
     *
     * @return bool     PREFS_OK on success, PREFS_ERROR_* on failure.
     */
    function store($prefs = array())
    {
        /* If we're not already connected, invoke the connect() method. */
        if (!$this->connected) {
            if ($this->connect() != PREFS_OK) return PREFS_ERROR_CONNECT;
        }

        /*
         * If a list of preferences to store hasn't been provided in
         * $prefs, assume all preferences are desired.
         */
        if (count($prefs) == 0) {
            $prefs = $this->listAll();
        }
        if (!is_array($prefs)) return PREFS_ERROR;
        
        /* Check for any "dirty" preferences. */
        $dirty_prefs = array();
        foreach ($prefs as $pref) {
            if ($this->isDirty($pref)) {
                $dirty_prefs[] = $pref;
            }
        }
        
        /*
         * If no "dirty" preferences were found, there's no need to update
         * the SQL server.  Exit successfully.
         */
        if (count($dirty_prefs) == 0) {
            return PREFS_OK;
        }
        
        /*
         * Loop through the "dirty" preferences.  If our attempt to insert
         * a new row fails, try to update an existing one.
         */
        foreach ($dirty_prefs as $name) {
            $value = addslashes($this->getValue($name));
            
            /* Attempt an insert. */
            $query = 'insert into ' . $this->params['table'] . ' ';
            $query .= '(uid, pref_name, pref_value) ';
            $query .= "values('$this->user', '$name', '$value')";
            $result = $this->db->query($query);
            
            /* If the insert failed, attempt an update. */
            if (($result != DB_OK) || DB::isError($result)) {
                $query = 'update ' . $this->params['table'] . ' ';
                $query .= "set pref_value = '$value' where uid = ";
                $query .= "'$this->user' and pref_name = '$name'";
                $result = $this->db->query($query);
                
                /* Return an error if the update fails, too. */
                if (($result != DB_OK) || DB::isError($result)) return PREFS_ERROR;
            }
            
            /* Mark this preference as "clean" now. */
            $this->setDirty($name, false);
        }
        return PREFS_OK;
    }
    
}
?>
