<?php
// $Horde: horde/lib/Prefs/ldap.php,v 1.9 2000/10/18 17:35:58 jon Exp $

/**
 * Preferences storage implementation for PHP's LDAP extention.
 *
 * Required values for $params:
 *      'hostspec'        The hostname of the LDAP server.
 *      'basedn'        The base DN for the LDAP server.
 *      'uid'           The username search key.
 *      'username'      The user as which to bind for write operations.
 *      'password'      'user's password for bind authentication.
 *
 * @author  Jon Parise <jon@csh.rit.edu>
 * @version $Revision: 1.9 $
 * @since   Horde 1.3
 */
class Prefs_ldap extends Prefs {

    /**
     * Hash containing connection parameters.
     */
    var $params = array();

    /**
     * Handle for the current LDAP connection.
     */
    var $connection = '';

    /**
     * Boolean indicating whether or not we're connected to the LDAP server.
     */
    var $connected = false;

    /**
     * String holding the user's DN.
     */
    var $dn = '';

    /**
     * String holding the authenticated user's DN (for binding).
     */
    var $auth_dn = '';

    /**
     * Boolean indicating whether $connection has bound to the LDAP server
     * as an authenticated user (non-anonymous).
     */
    var $anonymous = true;

    /**
     * Constructs a new LDAP preferences object.
     *
     * @param $user     The user who owns these preferences.
     * @param $password The password associated with $user.
     * @param $params   A hash containing connection parameters.
     */
    function Prefs_ldap($user, $password, $params = array())
    {
        $this->user = $user;
        $this->params = $params;

        /*
         * If $params['username'] is empty, set it to the name of the
         * current user.  Also, the $params['password'] to the current
         * user's password.
         *
         * Note: This assumes the user is allowed to modify their own LDAP
         *       entry.
         */
        if (empty($this->params['username'])) {
            $this->params['username'] = $user;
            $this->params['password'] = $password;
        }
    }

    /**
     * Opens a connection to the LDAP server.
     *
     * @param $bind     Boolean indicating whether or not to bind to the
     *                  LDAP server using an authenticated connected.  
     *
     * @return          PREFS_OK on success, PREFS_ERROR_* on failure.
     */
    function connect($bind = false)
    {
        if (!is_array($this->params)) return PREFS_ERROR_PARAMS;

        /* Connect to the LDAP server anonymously. */
        if (!isset($this->params['hostspec'])) return PREFS_ERROR_PARAMS;
        $conn = ldap_connect($this->params['hostspec']);
        if (!$conn) return PREFS_ERROR_CONNECT;

        /* Search for the user's full DN. */
        $search = ldap_search($conn, $this->params['basedn'],
            $this->params['uid'] . '=' . $this->user,
            array($this->params['uid']));
        $result = ldap_get_entries($conn, $search);
        if (is_array($result)) {
            $uid = $this->params['uid'];
            $this->dn = $uid . '=' . $result[0][$uid][0] . ',' .
                $this->params['basedn'];
        } else return PREFS_ERROR_CONNECT;

        /* Store the connection handle at the instance level. */
        $this->connection = $conn;
        $this->connected = true;
        $this->anonymous = true;

        /* Bind to the LDAP server, if requested. */
        if ($bind) {
            if ($this->bind() != PREFS_OK) {
                return PREFS_ERROR_CONNECT;
            }
        }

        return PREFS_OK;
    }

    /**
     * Bind to the LDAP server using authentication.
     *
     * @return          PREFS_OK on success, PREFS_ERROR_* on failure.
     */
    function bind()
    {
        /*
         * If we're not already connected, return an error.  Invoking
         * connect() again could lead to circular function calls.
         */
        if (!$this->connected) return PREFS_ERROR_CONNECT;

        /* Make sure the connection values for ldap_bind() were provided. */
        if (!isset($this->params['basedn'])) return PREFS_ERROR_PARAMS;
        if (!isset($this->params['uid'])) return PREFS_ERROR_PARAMS;
        if (!isset($this->params['username'])) return PREFS_ERROR_PARAMS;
        if (!isset($this->params['password'])) return PREFS_ERROR_PARAMS;

        /* Search for the full DN and bind to the LDAP server as that user. */
        $search = ldap_search($this->connection, $this->params['basedn'],
            $this->params['uid'] . '=' . $this->params['username'],
            array($this->params['uid']));
        $result = ldap_get_entries($this->connection, $search);
        if (is_array($result)) {
            $uid = $this->params['uid'];
            $this->auth_dn = $uid . '=' . $result[0][$uid][0] . ',' .
                $this->params['basedn'];
            $bind = @ldap_bind($this->connection, $this->auth_dn,
                $this->params['password']);
            if (!$bind) return PREFS_ERROR_AUTH;
        } else return PREFS_ERROR_CONNECT;

        $this->anonymous = false;

        return PREFS_OK;
    }

    /**
     * Disconnect from the LDAP server and clean up the connection.
     *
     * @return      true on success, false on failure.
     */
    function disconnect()
    {
        $this->dn = '';
        $this->auth_dn = '';
        $this->connected = false;

        return ldap_close($this->connection);
    }

    /**
     * Retrieves the requested set of preferences from the user's LDAP
     * entry.
     *
     * @param $prefs    (optional) An array listing the preferences to
     *                  retrieve.  If not specified, retrieve all of the
     *                  preferences listed in the $prefs hash.
     *
     * @return          PREFS_OK on success, PREFS_ERROR_* on failure.
     */
    function retrieve($prefs = array())
    {
        /* If we're not already connected, invoke the connect() method. */
        if (!$this->connected) {
            if ($this->connect(false) != PREFS_OK) return PREFS_ERROR_CONNECT;
        }

        /*
         * If a list of preferences to retrieve hasn't been provided in
         * $prefs, assume all preferences are desired.
         */
        if (count($prefs) == 0) {
            $prefs = Prefs::listAll();
        }
        if (!is_array($prefs)) return PREFS_ERROR;

        $search = ldap_search($this->connection, $this->params['basedn'],
            $this->params['uid'] . '=' . $this->user, $prefs);
        if ($search) {
            $result = ldap_get_entries($this->connection, $search);
        } else return PREFS_ERROR_CONNECT;

        if (isset($result) && is_array($result)) {
            /*
             * Set the requested values in the $this->prefs hash based on
             * the contents of the LDAP result.
             *
             * Note that Prefs::setValue() can't be used here because of the
             * check for the "changeable" bit.  We want to override that
             * check when populating the $this->prefs hash from the LDAP
             * server.
             */
            foreach ($prefs as $pref) {
                if (isset($result[0][$pref][0])) {
                    $this->prefs[$pref]['val'] = $result[0][$pref][0];
                }
            }
        } else return PREFS_ERROR_EMPTY;

        return PREFS_OK;
    }

    /**
     * Stores preferences to LDAP server.
     *
     * @param $prefs    (optional) An array listing the preferences to be
     *                  stored.  If not specified, store all of the
     *                  preferences listed in the $prefs hash.
     *
     * @return bool     PREFS_OK on success, PREFS_ERROR_* on failure.
     */
    function store($prefs = array())
    {
        /* If we're not already connected, invoke the connect(). */
        if (!$this->connected) {
            if ($this->connect(true) != PREFS_OK) return PREFS_ERROR_CONNECT;
        } else {
            /* Bind if the LDAP server if we haven't already. */
            if ($this->anonymous) {
                if ($this->bind() != PREFS_OK) return PREFS_ERROR_CONNECT;
            }
        }

        /*
         * If a list of preferences to store hasn't been provided in
         * $prefs, assume all preferences are desired.
         */
        if (count($prefs) == 0) {
            $prefs = Prefs::listAll();
        }
        if (!is_array($prefs)) return PREFS_ERROR;

        /* Check for any "dirty" preferences. */
        $dirty_prefs = array();
        foreach ($prefs as $pref) {
            if (Prefs::isDirty($pref)) {
                $dirty_prefs[] = $pref;
            }
        }

        /*
         * If no "dirty" preferences were found, there's no need to update
         * the LDAP server.  Exit successfully.
         */
        if (count($dirty_prefs) == 0) {
            return PREFS_OK;
        }

        /*
         * Build a hash of the preferences and their values that need to be
         * stored in the LDAP server.
         */
        $new_values = array();
        foreach($dirty_prefs as $pref) {
            $new_values[$pref] = Prefs::getValue($pref);
        }

        /* Sent the hash of new values off to the LDAP server. */
        if (!ldap_modify($this->connection, $this->dn, $new_values)) {
            return PREFS_ERROR;
        }

        return PREFS_OK;
    }
}

?>
