<?php
// $Horde: horde/lib/Prefs.php,v 1.19 2001/01/07 23:45:55 chuck Exp $

/* Return codes */
define('PREFS_OK',               0);    // Operation succeeded
define('PREFS_ERROR',           -1);    // operation failed
define('PREFS_ERROR_PARAMS',    -2);    // Bad or missing parameters ($params)
define('PREFS_ERROR_CONNECT',   -3);    // Connection failure
define('PREFS_ERROR_AUTH',      -4);    // Authentication failure
define('PREFS_ERROR_EMPTY',     -5);    // Empty retrieval result

/* Bitmask constants */
define('_PREFERENCE_CHANGEABLE', 1);    // Value is changeable by the user
define('_PREFERENCE_DIRTY',      2);    // Value has been changed by the user

/**
 * The Prefs:: class provides a common abstracted interface into the
 * various preferences storage mediums.  It also includes all of the
 * functions for retrieving, storing, and checking preference values.
 *
 * The preferences configuration file (used in setDefaults) takes this form:
 *
 *      preference_key, default_value, user_changeable
 *
 * The 'user_changeable' field is either a 0 (locked) or a 1 (user-editable).
 *
 * Lines starting with a '#' are considered comments.
 *
 * @author  Jon Parise <jon@csh.rit.edu>
 * @version $Revision: 1.19 $
 * @since   Horde 1.3
 */
class Prefs {
    
    /**
     * Hash holding all of the user's preferences. Each preference is
     * itself a hash, so this will ultimately be multi-dimensional.
     * @var array $prefs
     */
    var $prefs = array();
	
	/**
     * Boolean saying whether or not we've read a set of defaults
     * yet, so that we can make sure not to do it twice.
     * @var bool $defaultsRead
     */
	var $defaultsRead = false;
	
    /**
     * String containing the current username. This indicates the
     * owner of the preferences.
     * @var string $user
     */
    var $user = '';
    
    /**
     * Attempts to return a concrete Prefs instance based on $driver.
     * 
     * @param $driver   The type of concrete Prefs subclass to return.
     *                  This is based on the storage driver ($driver).  The
     *                  code is dynamically included.
     *
     * @param $user     (optional) The name of the user who owns this set of
     *                  preferences.
     *
     * @param $password (optional) The password associated with $user.
     *
     * @param $params   (optional) A hash containing any additional
     *                  configuration or connection parameters a subclass
     *                  might need.
     *
     * @return          The newly created concrete Prefs instance, or false
     *                  on error.
     */
    function factory($driver, $user = '', $password = '', $params = array())
    {
        if (empty($driver) || (strcasecmp($driver, 'none') == 0)) {
            return new Prefs;
        }

        /*
         * If $params['user_hook'] is defined, use it to retrieve the value
         * to use for the username ($this->user).  Otherwise, just use the
         * value passed in the $user parameter.
         */
        if (!empty($params['user_hook'])) {
            $user = call_user_func($params['user_hook'], $user);
        }
        
        $classfile = dirname(__FILE__) . '/Prefs/' . $driver . '.php';
        
        if (@is_readable($classfile)) {
            include_once $classfile;
            $class = 'Prefs_' . $driver;
            return new $class($user, $password, $params);
        } else {
            return false;
        }
    }

    /**
     * Attempts to return a reference to a concrete Prefs instance based on
     * $driver. It will only create a new instance if no Prefs instance
     * with the same parameters currently exists.
     *
     * This should be used if multiple preference sources (and, thus,
     * multiple Prefs instances) are required.
     *
     * This method must be invoked as: $var = &Prefs::singleton()
     *
     * @param $driver   The type of concrete Prefs subclass to return.
     *                  This is based on the storage driver ($driver). The
     *                  code is dynamically included.
     *
     * @param $user     (optional) The name of the user who owns this set of
     *                  preferences.
     *
     * @param $password (optional) The password associated with $user.
     *
     * @param $params   (optional) A hash containing any additional
     *                  configuration or connection parameters a subclass
     *                  might need.
     *
     * @return          The concrete Prefs reference, or false on an error.
     */
    function &singleton($driver, $user = '', $password = '', $params = array())
    {
        static $instances;

        if (!isset($instances)) $instances = array();

        $signature = md5($driver . '][' . $user . '][' . implode('][', $params));
        if (!isset($instances[$signature])) {
            $instances[$signature] = Prefs::factory($driver, $user, $password, $params);
        }

        return $instances[$signature];
    }

    /**
     * Populates the $prefs hash with new entries and externally defined
     * default values.
     *
     * @param $filename The filename of the file from which to read the list
     *                  of preferences and their default values.
     */
    function setDefaults($filename)
    {
        if ($this->defaultsRead) return;
        $this->defaultsRead = true;

        if (!@is_readable($filename)) return;

        /* Read the entire file ($filename) into the $lines array. */
        $lines = file($filename);

        foreach ($lines as $line) {

            /* Trim whitespace and ignore comments (lines starting with a #). */
            $line = trim($line);
            $line = preg_replace("|#.*$|", '', $line);

            if (!empty($line)) {

                $values = explode(',', $line);
                for ($i = 0; $i < count($values); $i++) {
                    $values[$i] = trim($values[$i]);
                }

                $pref = str_replace('.', '_', $values[0]);
                $val = $values[1];
                $acl = ($values[2] == 1) ? _PREFERENCE_CHANGEABLE : 0;
                $acl &= ~_PREFERENCE_DIRTY;

                $this->add($pref, $val, $acl);
            }
        }

    }

    /**
     * Adds a new preference entry to the $prefs hash.
     *
     * @param $pref     The name of the preference to add.
     * @param $val      (optional) The initial value of the preference.
     * @param $acl      (optional) The initial ACL of the preference.
     */
    function add($pref, $val = '', $acl = 0)
    {
        if (is_array($this->prefs)) {
            $this->prefs[$pref] = array('val' => $val, 'acl' => $acl);
        }
    }

    /**
     * Removes a preference entry from the $prefs hash.
     *
     * @param $pref     The name of the preference to remove.
     */
    function remove($pref)
    {
        if (is_array($this->prefs)) {
            unset($this->prefs[$pref]);
        }
    }

    /**
     * Returns an array containing the names of all of the preferences
     * stored in the $prefs hash.
     *
     * @return          An array containiing the names of the preferences
     *                  in the $prefs hash.
     */
    function listAll()
    {
        return array_keys($this->prefs);
    }

    /**
     * Sets the given preferences to the specific value, if the preference
     * is modifiable.
     *
     * @param $pref     The name of the preference to modify.
     * @param $val      The new value for this preference.
     *
     * @return          True if the value was successfully set, false on a
     *                  failure.
     */
    function setValue($pref, $val)
    {
        /* Exit early if this preference is not user-changeable. */
        if (!$this->isChangeable($pref)) {
            return false;
        }

        /*
         * If the preference's value is already equal to $val, don't bother
         * changing it.  Changing it would set the "dirty" bit, causing an
         * unnecessary update later on in the storage routine.
         */
        if ($this->prefs[$pref]['val'] == $val) return true;

        /* Assign the new value and set the "dirty" bit. */
        $this->prefs[$pref]['val'] = $val;
        $this->setDirty($pref, true);

        return true;
    }

    /**
     * Returns the value of the requested preference.
     *
     * @param $pref     The name of the preference to retrieve.
     *
     * @return          The value of the preference.
     */
    function getValue($pref)
    {
        if (isset($this->prefs[$pref]['val'])) {
            return $this->prefs[$pref]['val'];
        }
    }

    /**
     * Modifies the "changeable" bit for the given preference.
     *
     * @param $pref     The name of the preference to modify.
     * @param $bool     The new boolean value for the "changeable" bit.
     */
    function setChangeable($pref, $bool)
    {
        if ($bool != $this->isChangeable($pref)) {
            if ($bool) {
                $this->prefs[$pref]['acl'] |=  _PREFERENCE_CHANGEABLE;
            } else {
                $this->prefs[$pref]['acl'] &= ~_PREFERENCE_CHANGEABLE;
            }
        }
    }

    /**
     * Returns the state of the "changeable" bit for the given preference.
     *
     * @param $pref     The name of the preference the check.
     *
     * @return          The boolean state of $pref's "changeable" bit.
     */
    function isChangeable($pref)
    {
        if (isset($this->prefs[$pref]['acl'])) {
            return ($this->prefs[$pref]['acl'] & _PREFERENCE_CHANGEABLE);
        }
        return false;
    }

    /**
     * Modifies the "dirty" bit for the given preference.
     *
     * @param $pref     The name of the preference to modify.
     * @param $bool     The new boolean value for the "dirty" bit.
     */
    function setDirty($pref, $bool)
    {
        if ($bool != $this->isDirty($pref)) {
            if ($bool) {
                $this->prefs[$pref]['acl'] |=  _PREFERENCE_DIRTY;
            } else {
                $this->prefs[$pref]['acl'] &= ~_PREFERENCE_DIRTY;
            }
        }
    }

    /**
     * Returns the state of the "dirty" bit for the given preference.
     *
     * @param $pref     The name of the preference the check.
     *
     * @return          The boolean state of $pref's "dirty" bit.
     */
    function isDirty($pref)
    {
        return ($this->prefs[$pref]['acl'] & _PREFERENCE_DIRTY);
    }

    /**
     * This is basically an abstract method that should be overridden by a
     * subclass implementation. It's here to retain code integrity in the
     * case that no subclass is loaded ($driver == 'none').
     */
    function retrieve()
    {
        return true;
    }

    /**
     * This is basically an abstract method that should be overridden by a
     * subclass implementation. It's here to retain code integrity in the
     * case that no subclass is loaded ($driver == 'none').
     */
    function store()
    {
        return true;
    }

}
?>
