<?php
// $Horde: horde/lib/Perms.php,v 1.4 2000/08/28 03:45:53 chuck Exp $

/* Return codes */
define('PERMS_OK',               0);    // Operation succeeded
define('PERMS_ERROR',           -1);    // operation failed
define('PERMS_ERROR_PARAMS',    -2);    // Bad or missing parameters ($params)
define('PERMS_ERROR_CONNECT',   -3);    // Connection failure
define('PERMS_ERROR_AUTH',      -4);    // Authentication failure
define('PERMS_ERROR_EMPTY',     -5);    // Empty retrieval result

/* Bitmask constants */
define('PERMS_NONE', 0);    // No rights.
define('PERMS_SHOW', 1);    // Existence of object is known - object is shown to user.
define('PERMS_READ', 2);    // Contents of the object can be read.
define('PERMS_EDIT', 4);    // Contents of the object can be edited.
define('PERMS_DELE', 8);    // The object can be deleted.

/**
 * The Perms:: class provides a common abstracted interface into the
 * various backends for the Horde permissions system. It also includes
 * all of the functions for retrieving, storing, and checking
 * permissions.
 *
 * Permissions are based on a generic, granular model. The Horde
 * permission system doesn't need to be associated with, or control,
 * the objects being restricted - as long as you control all access to
 * them through Horde interfaces, or take other steps.
 *
 * We just expect some kind of unique identifier and a userid, and
 * check that against the storage being used. So if we used the ids
 * assigned to MCAL calendar events, and made sure they were unique
 * across servers, we could store permissions for those events in a
 * completely seperate SQL database.
 *
 * @author  Chuck Hagenbuch <chuck@horde.org>
 * @version $Revision: 1.4 $
 * @since   Horde 1.3
 */
class Perms {
    
    /** The character to seperate levels of permissions hierarchy with. */
    var $path_seperator = '|';
    
	
    /**
     * Attempts to return a concrete Perms instance based on $driver.
     * 
     * @param $driver The type of concrete Perms subclass to return.
     *                This is based on the storage driver ($driver).  The
     *                code is dynamically included.
     * @param $params (optional) A hash containing any additional
     *                configuration or connection parameters a subclass
     *                might need.
     *
     * @return The newly created concrete Perms instance, or false
     *         on an error.
     */
    function factory($driver, $params = array())
    {
        if (empty($driver) || (strcasecmp($driver, 'none') == 0)) {
            return new Perms;
        }
        
        $classfile = dirname(__FILE__) . '/Perms/' . $driver . '.php';
        
        if (@is_readable($classfile)) {
            include_once $classfile;
            $class = 'Perms_' . $driver;
            return new $class($params);
        } else {
            return false;
        }
    }
	
    /**
     * Attempts to return a reference to a concrete Perms instance based on
     * $driver. It will only create a new instance if no Perms instance
     * with the same parameters currently exists.
     *
     * This should be used if multiple permissions sources (and, thus,
     * multiple Perms instances) are required.
     *
     * This method must be invoked as: $var = &Perms::singleton()
     *
     * @param $driver The type of concrete Perms subclass to return.
     *                This is based on the storage driver ($driver). The
     *                code is dynamically included.
     * @param $params (optional) A hash containing any additional
     *                configuration or connection parameters a subclass
     *                might need.
     *
     * @return The concrete Perms reference, or false on an error.
     */
    function &singleton($driver, $user = '', $params = array())
    {
        static $instances;

        if (!isset($instances)) $instances = array();

        $signature = md5($driver . '][' . $user . '][' . implode('][', $params));
        if (!isset($instances[$signature])) {
            $instances[$signature] = Perms::factory($driver, $user, $params);
        }

        return $instances[$signature];
    }
	
    /**
	 * Find out what rights the given user has to the specified object.
	 * 
     * @param $objectpath The full path to the piece of
     *                    content/whatever to check the permissions
     *                    of.
	 * @param $user The user to check for.
	 * @param $default What to return if nothing is found in the db.
	 * 
	 * @return Any permissions the user has, or $default if there
	 *         are none.
	 */
    function check ($objectpath, $user, $default = PERMS_NONE)
    {
        return $default;
    }
	
    /**
	 * Find out what rights the given user has to the specified object.
	 * 
     * @param $object The piece of content/whatever to check the
     *                permissions of.
	 * @param $user The user to check for.
     * @param $perm The permission level that needs to be checked.
	 * 
	 * @return True if the user has the specified permissions, and
	 *         false otherwise.
     */
    function hasPerm ($object, $user, $perm)
    {
        return ($this->check($object, $user) | $perm);
    }
	
    /**
     * Add the permission(s) specified to the given user and object.
     *
     * @param $object The piece of content/whatever to add
     *                permissions to.
	 * @param $user The user to add permissions for.
     * @param $perm The permission to be added.
     */
    function addPerms ($object, $user, $perm)
    {
        return PERMS_OK;
    }
	
    /**
     * Remove the permission(s) specified from the given user and object.
     *
     * @param $object The piece of content/whatever to remove
     *                permissions from.
	 * @param $user The user to remove permissions from.
     * @param $perm The permission to be removed.
     */
    function removePerms ($object, $user, $perm)
		{
        return PERMS_OK;
    }
	
    /**
     * Set the object to be showable to the given user.
     *
     * @param $object The piece of content to set permissions for.
     * @param $user The user to assign the permissions to.
     */
    function setShowable($object, $user)
    {
        $this->addPerms($object, $user, PREFS_SHOW);
    }
	
    /**
     * Set the object to not be showable to the given user.
     *
     * @param $object The piece of content to set permissions for.
     * @param $user The user to assign the permissions to.
     */
    function setUnshowable($object, $user)
    {
        $this->removePerms($object, $user, PREFS_SHOW);
    }
	
    /**
     * Set the object to be readable by the given user.
     *
     * @param $object The piece of content to set permissions for.
     * @param $user The user to assign the permissions to.
     */
    function setReadable($object, $user)
    {
        $this->addPerms($object, $user, PREFS_READ);
    }
	
    /**
     * Set the object to be unreadable by the given user.
     *
     * @param $object The piece of content to set permissions for.
     * @param $user The user to assign the permissions to.
     */
    function setUnreadable($object, $user)
    {
        $this->removePerms($object, $user, PREFS_READ);
    }
	
    /**
     * Set the object to be editable by the given user.
     *
     * @param $object The piece of content to set permissions for.
     * @param $user The user to assign the permissions to.
     */
    function setEditable($object, $user)
    {
        $this->addPerms($object, $user, PREFS_EDIT);
    }
	
    /**
     * Set the object to be uneditable by the given user.
     *
     * @param $object The piece of content to set permissions for.
     * @param $user The user to assign the permissions to.
     */
    function setUneditable($object, $user)
    {
        $this->removePerms($object, $user, PREFS_EDIT);
    }
	
    /**
     * Set the object to be deleteable by the given user.
     *
     * @param $object The piece of content to set permissions for.
     * @param $user The user to assign the permissions to.
     */
    function setDeleteable($object, $user)
    {
        $this->addPerms($object, $user, PREFS_DELE);
    }
	
    /**
     * Set the object to be undeleteable by the given user.
     *
     * @param $object The piece of content to set permissions for.
     * @param $user The user to assign the permissions to.
     */
    function setUndeleteable($object, $user)
    {
        $this->removePerms($object, $user, PREFS_DELE);
    }
    
}
?>
