<?php
/**
 * The MIME_Structure:: class provides methods for dealing with MIME mail.
 *
 * @author  Chuck Hagenbuch <chuck@horde.org>
 * @version $Revision: 1.1 $
 * @since   Horde 1.3
 */
class MIME_Structure {
    
    /* Mime parsing function based on code by justin <justinc@andrew.cmu.edu> */
    function parse($body, $index, &$MimeID)
    {
        if (!is_array($MimeID)) $MimeID = array();
        $attachments = array();
        return MIME_Structure::_doParse($body, $index, $MimeID, $attachments);
    }
    
    function _doParse($body, $index, &$MimeID, &$attachments, $ref = '')
    {
        if (!strlen($ref)) // top multiparts don't get their own line
            $ref = (isset($body->type) && $body->type == TYPEMULTIPART) ? '' : 1;
        
        if (isset($body->subtype) && $body->subtype == 'RFC822') {
            $href = "$ref.0";
            $attachments[$href] = new stdClass;
            $attachments[$href]->header = true;
            $attachments[$href]->header_imap_id = $href;
            $attachments[$href]->index = $index;
        }
        if (!empty($body->bytes)) {
            $attachments[$ref] = MIME_Structure::setInfo($body, $MimeID, $ref);
            $attachments[$ref]->header = false;
            $attachments[$ref]->imap_id = $ref;
            $attachments[$ref]->index = $index;
        }
        if (isset($body->parts)) {
            $parts = $body->parts;
            $i = 0;
            while (list(,$sub_part) = each($parts)) {
                if (isset($body->type) && ($body->type == TYPEMESSAGE) && isset($sub_part->type) && ($sub_part->type == TYPEMULTIPART))
                    $sub_ref = $ref;
                else
                    $sub_ref = (!strlen($ref)) ? '' . ($i+1) : $ref . '.' . ($i+1);
                MIME_Structure::_doParse($sub_part, $index, $MimeID, $attachments, $sub_ref);
                $i++;
            }
        }
        
        return $attachments;
    }
    
    /**
     * Given a mime part from imap_fetchstructure(), munge it into a
     * useful form and make sure that any parameters which are missing
     * are given default values.
     *
     * @param object $part     The original part info.
     * @param array &$MimeID   An array of Mime IDs to write this part's ID to.
     * @param string $ref      The number of this part.
     */
    function setInfo($part, &$MimeID, $ref)
    {
        $object = new stdClass;
        $object->type = (isset($part->type)) ? $part->type : TYPETEXT;
        $object->subtype = ($part->ifsubtype) ? strtolower($part->subtype) : 'x-unknown';
        $object->encoding = empty($part->encoding) ? MIME_DEFAULT_ENCODING : $part->encoding;
        $object->charset = empty($part->charset) ? MIME_DEFAULT_CHARSET : $part->charset;
        $object->bytes = empty($part->bytes) ? '?' : $part->bytes;
        $object->size = ($object->bytes != '?') ? sprintf('%0.2f', $object->bytes/1024) : '?';
        $object->disposition = ($part->ifdisposition) ? strtolower($part->disposition) : 'inline';
        
        if ($part->ifid) {
            $object->id = $part->id;
            $MimeID[$ref] = $object->id;
        } else
            $object->id = false;
        
        // go through the parameters, if any
        if ($part->ifparameters) {
            while (list(,$param) = each($part->parameters)) {
                $field = strtolower($param->attribute);
                $object->$field = $param->value;
            }
        }
        
        // go through the disposition parameters, if any
        if ($part->ifdparameters) {
            while (list(,$param) = each($part->dparameters)) {
                $field = strtolower($param->attribute);
                $object->$field = $param->value;
            }
        }
        
        // make sure a name is set
        if (empty($object->name) && !empty($object->filename))
            $object->name = $object->filename;
        
        // make sure there's a description
        if (!empty($part->description))
            $object->description = $part->description;
        else {
            if (!empty($object->name))
                $object->description = $object->name;
            else
                $object->description = MIME_DEFAULT_DESCRIPTION;
        }
        
        if (empty($object->name)) $object->name = preg_replace('|\W|', '_', $object->description);
        
        global $mime_types, $mime_actions;
        $object->TYPE = isset($mime_types[$object->type]) ? $mime_types[$object->type] : $mime_types[TYPETEXT];
        // set conf information
        if (isset($mime_actions["$object->TYPE/$object->subtype"]))
            $object->conf = $mime_actions["$object->TYPE/$object->subtype"];
        elseif (isset($mime_actions["$object->TYPE/"]))
            $object->conf = $mime_actions["$object->TYPE/"];
        elseif (isset($mime_actions['/']))
            $object->conf = $mime_actions['/'];
        else
           $object->conf = array('action' => 'unknown', 'icon' => 'mime_unknown.gif');
        
        return $object;
    }
    
}
?>
