<?php
// $Horde: horde/lib/Horde.php,v 1.78 2001/01/10 19:24:31 chuck Exp $

define('HORDE_NO_ACTION',     0);
define('HORDE_LOGIN',         1);
define('HORDE_MAILTO',        2);
define('HORDE_SIGNUP',        3);
define('HORDE_SEND_MESSAGE',  4);

// define Horde error types
define('HORDE_MESSAGE', 0);
define('HORDE_SUCCESS', 1);
define('HORDE_WARNING', 2);
define('HORDE_ERROR',   4);


/**
 * If called with a filename, register that filename to be deleted at
 * request shutdown. If called with no arguments, unlink all files
 * registered. The first time it is called, it initializes the array
 * of files, and registers itself as a shutdown function - no need to
 * do so manually.
 *
 * @param string (optional) The filename to be deleted at the end of
 *                          the request.
 */
function _fileCleanup($filename = false)
{
    static $files;
    
    if (!isset($files) || !is_array($files)) {
        $files = array();
        register_shutdown_function('_fileCleanup');
    }
    
    if (!$filename) {
        foreach ($files as $file) {
            if (@file_exists($file))
                @unlink($file);
        }
    } else {
        $files[] = $filename;
    }
}


/**
 * The Lang:: class provides common methods for handling language detection
 * and selection.
 *
 * @author  Jon Parise <jon@horde.org>
 * @author  Chuck Hagenbuch <chuck@horde.org>
 * @version $Revision: 1.78 $
 * @since   Horde 1.3
 */
class Lang {

    /**
     * Selects the most preferred language for the current client session.
     *
     * @param $set_cookie       (optional) Boolean determining whether a
     *                          cookie containing the select language should
     *                          be sent.
     *
     * @return  string          The selected language abbreviation.
     */
    function select($set_cookie = false)
    {
        global $HTTP_ACCEPT_LANGUAGE, $chuckmIMPlang, $language, $conf, $horde;

        /* Use a directly supplied parameter, if given. */
        if (isset($language) && Lang::valid($language)) {
            $language = strtolower($language);

        /* If we have already set a cookie, simply use that. */
        } elseif (isset($chuckmIMPlang) && Lang::valid($chuckmIMPlang)) {
            $language = $chuckmIMPlang;

        /* No browser cookie, try browser-accepted languages, then default. */
        } elseif (isset($HTTP_ACCEPT_LANGUAGE)) {

            /* Start with a known default value. */
            $language = $horde['lang']['default'];

            /* The browser supplies a list, so return the first valid one. */
            $browser_langs = explode(',', $HTTP_ACCEPT_LANGUAGE);
            for ($i = 0; $i < count($browser_langs); $i++) {
                $browser_langs[$i] = trim($browser_langs[$i]);
                if (Lang::valid($browser_langs[$i])) {
                    $language = $browser_langs[$i];
                    break;
                } elseif (Lang::valid(substr($browser_langs[$i], 0, 2))) {
                    $language = substr($browser_langs[$i], 0, 2);
                    break;
                }
            }

        /* No dice auto-detecting, so give them the server default. */
        } else {
            if (isset($conf['lang']['default'])) {
                $language = $conf['lang']['default'];
            } else {
                $language = $horde['lang']['default'];
            }
        }

        /* Set the language cookie to expire in a year. */
        if ($set_cookie) {
            setcookie('chuckmIMPlang', $language, time()+31536000, '/', '', 0);
        }

        return $language;
    }

    /**
     * Determines whether the supplied language is valid.
     *
     * @param $language         The abbreviated name of the language.
     *
     * @return  boolean         True if the language is valid, false if it's
     *                          not valid or unknown.
     */
    function valid($language)
    {
        global $nls;

        return isset($nls['languages'][strtolower($language)]);
    }
}

/**
 * The Text:: class provides common methods for manipulating text.
 *
 * @author  Jon Parise <jon@horde.org>
 * @version $Revision: 1.78 $
 * @since   Horde 1.3
 */
class Text {

    /**
     * Filter the given text based on the words found in $words.
     *
     * @param $text         The text to filter.
     * @param $words_file   Filename containing the words to replace.
     * @param $replacement  The replacement string.
     *
     * @return  string      The filtered version of $text.
     */
    function filter($text, $words_file, $replacement)
    {
        if (@is_readable($words_file)) {
            
            /* Read the file and iterate through the lines. */
            $lines = file($words_file);
            foreach ($lines as $line) {
                
                /* Strip whitespace and comments. */
                $line = trim($line);
                $line = preg_replace('|#.*$|', '', $line);
                
                /* Filter the text. */
                if (!empty($line)) {
                    $text = preg_replace("/(\b(\w*)$line\b|\b$line(\w*)\b)/i",
                                         $replacement, $text);
                }
            }
        }
        
        return $text;
    }
    
    /**
     * Wraps the text of a message.
     *
     * @param $text         String containing the text to wrap.
     * @param $length       Wrap $text at this number of characters.
     * @param $break_char   Character to use when breaking lines.
     *
     * @return  string      String containing the wrapped text.
     */
    function wrap($text, $length = 80, $break_char = "\n")
    {
        $paragraphs = explode("\n", $text);
        for ($i = 0; $i < count($paragraphs); $i++) {
            $paragraphs[$i] = wordwrap($paragraphs[$i], $length, $break_char);
        }
        return implode($break_char, $paragraphs);
    }
    
    /**
     * Replace occurences of %VAR% with VAR, if VAR exists in the
     * webserver's environment. Ignores all text after a # character
     * (shell-style comments).
     *
     * @param string $text	The text to expand.
     * @return string		The expanded text.
     */
    function expandEnvironment($text)
    {
        if (preg_match("|([^#]*)#.*|", $text, $regs)) {
            $text = $regs[1];
            
            if (strlen($text) > 0) {
                $text = $text . "\n";
            }
        }
        
        while (preg_match("|%([A-Za-z_]+)%|", $text, $regs)) {
            $text = preg_replace("|%([A-Za-z_]+)%|", getenv($regs[1]), $text);
        }
        return $text;
    }
    
}


class Secret {
    
    function write($key, $message)
    {
        if (function_exists('mcrypt_module_open')) {
            Secret::srand();
            $td = mcrypt_module_open(MCRYPT_BLOWFISH, '', MCRYPT_MODE_ECB, '');
            $iv = mcrypt_create_iv(mcrypt_enc_get_iv_size($td), MCRYPT_RAND);
            @mcrypt_generic_init($td, $key, $iv);
            $encrypted_data = mcrypt_generic($td, $message);
            mcrypt_generic_end($td);
            
            return $encrypted_data;
        } else {
            include_once 'Crypt/HCEMD5.php';
            
            $hcemd5 = new Crypt_HCEMD5($key);
            return $hcemd5->encodeMimeSelfRand($message);
        }
    }
    
    function read($key, $ciphertext)
    {
        if (function_exists('mcrypt_module_open')) {
            Secret::srand();
            $td = mcrypt_module_open(MCRYPT_BLOWFISH, '', MCRYPT_MODE_ECB, '');
            $iv = mcrypt_create_iv(mcrypt_enc_get_iv_size($td), MCRYPT_RAND);
            @mcrypt_generic_init($td, $key, $iv);
            $decrypted_data = mdecrypt_generic($td, $ciphertext);
            mcrypt_generic_end($td);
            
            return $decrypted_data;
        } else {
            include_once 'Crypt/HCEMD5.php';
            
            $hcemd5 = new Crypt_HCEMD5($key);
            return $hcemd5->decodeMimeSelfRand($ciphertext);
        }
    }
    
    function srand()
    {
        static $initialized;
        
        if (empty($initialized)) {
            srand((double)microtime() * 1000000);
            $initialized = true;
        }
    }
    
}


class Horde {
    
    /**
     * Add a message to the Horde message stack.
     *
     * @access public
     *
     * @param string $message The text description of the message.
     * @param constant $type  The type of message: HORDE_ERROR,
     *                        HORDE_WARNING, HORDE_SUCCESS, or HORDE_MESSAGE.
     */
    function raiseMessage($message, $type = HORDE_MESSAGE)
    {
        global $hordeMessageStack;
        $hordeMessageStack[] = array('type' => $type, 'message' => $message);
    }
    
    
    //
    // Session utility functions
    //
    
    /**
     * Return a session-id-ified version of $uri.
     *
     * @param $uri The URI to be modified.
     * @param $full Generate a full (http://server/path/) URL.
     * @param $always_append_session Tack on the session ID even if
     *                               cookies are present.
     *
     * @return the url with the session id appended
     */
    function url($uri, $full = false, $always_append_session = false)
    {
        global $conf, $HTTP_SERVER_VARS, $HTTP_COOKIE_VARS;
        
        $protocol = 'http';
        
        if ($full) {
            if (!isset($conf['use_ssl'])) {
                $conf['use_ssl'] = 2;
            }
            
            if ($conf['use_ssl'] == 1) {
                $protocol = 'https';
            } else if ($conf['use_ssl'] == 2) {
                if ((isset($HTTP_SERVER_VARS['HTTPS']) && $HTTP_SERVER_VARS['HTTPS'] == 'on') || getenv('SSL_PROTOCOL_VERSION')) {
                    $protocol = 'https';
                }
            }
            
            if (($HTTP_SERVER_VARS['SERVER_PORT'] != 80) && ($HTTP_SERVER_VARS['SERVER_PORT'] != 443)) {
                $server = $HTTP_SERVER_VARS['SERVER_NAME'] . ':' . $HTTP_SERVER_VARS['SERVER_PORT'];
            } else {
                $server = $HTTP_SERVER_VARS['SERVER_NAME'];
            }
            
            $url = $protocol . '://' . $server . $conf['paths']['root'];
            if (substr($conf['paths']['root'], -1) == '/') {
                $url .= $uri;
            } else {
                $url .= '/' . $uri;
            }
        } else {
            $url = $uri;
        }
        
        if ($always_append_session ||
            !isset($HTTP_COOKIE_VARS[session_name()])) {
            $url .= (strpos($url, '?') != false ?  '&' : '?') .
                urlencode(session_name()) . '=' . session_id();
        }
        
        return ($full ? $url : htmlentities($url));
    }

    /**
     * Return an anchor tag with the relevant parameters 
     *
     * @param $url The full URL to be linked to
     * @param $status An optional JavaScript mouse-over string
     * @param $class The CSS class of the link
     * @param $target The window target to point this link too
     *
     * @return the full <a href> tag
     */
    function link($url, $status = false, $class = false, $target = false)
    {
        $ret =  '<a href="'.$url.'"';
        if ($status) {
            $ret .= ' onmouseout="status=\'\';" onmouseover="status=\''.
                    $status.'\'; return true;" ';
        }
        if ($class) {
            $ret .= ' class="'.$class.'"';
        }
        if ($target) {
            $ret .= ' target="'.$target.'"';
        }

        return "$ret>";
    }

    /**
     * Print an anchor tag with the relevant parameters
     *
     * @param $url The full URL to be linked to
     * @param $status An optional JavaScript mouse-over string
     * @param $class The CSS class of the link
     * @param $target The window target to point this link too
     */
    function plink($url, $status = false, $class = false, $target = false)
    {
       echo Horde::link($url, $status, $class, $target);
    }

    /**
     * Print a session-id-ified version of the URI.
     *
     * @param $uri the URI to be modified
     * @param $full Generate a full (http://server/path/) URL.
     * @param $always_append_session Tack on the session ID even if
     *                               cookies are present.
     */
    function purl($uri, $full = false, $always_append_session = false)
    {
        echo Horde::url($uri, $full, $always_append_session);
    }

    /**
     * Return a session-id-ified version of $PHP_SELF.
     *
     * @param $query_string (optional) include the query string?
     */
    function selfURL($query_string = false)
    {
        global $HTTP_SERVER_VARS;
        
        $url = $HTTP_SERVER_VARS['SCRIPT_NAME'];
        
        if ($query_string && !empty($HTTP_SERVER_VARS['QUERY_STRING'])) {
            $url .= '?' . $HTTP_SERVER_VARS['QUERY_STRING'];
        }
        
        return Horde::url($url);
    }

    /**
     * Print a session-id-ified version of $PHP_SELF.
     *
     * @param $query_string (optional) include the query string?
     */
    function pselfURL($query_string = false)
    {
        echo Horde::selfURL($query_string);
    }

    /**
     * Return a hidden form input containing the session name and id.
     */
    function formInput()
    {
        return '<input type="hidden" name="' . session_name() . '" value="' . session_id() . '" />';
    }

    /**
     * Print a hidden form input containing the session name and id.
     */
    function pformInput()
    {
        echo Horde::formInput();
    }
    
    
    //
    // Image utility functions
    //
    
    /**
     * Construct a correctly-pathed link to an image
     *
     * @param $src The image file.
     * @param $attr Any extra parameters for the image tag.
     * @param $dir The root graphics directory.
     *
     * @return string The full image tag.
     */
    function img($src, $attr='', $dir='')
    {
        if (empty($dir)) $dir = 'graphics';
        
        $img = '<img';
        $img .= ' src="' . $dir . '/' . $src . '"';
        $img .= ' border="0"';
        if (!empty($attr)) $img .= ' ' . $attr;
        $img .= ' />';
        
        return $img;
    }
    
    /**
     * Construct a correctly-pathed link to an image
     *
     * @param $src The image file.
     * @param $attr Any extra parameters for the image tag.
     * @param $dir The root graphics directory.
     */
    function pimg ($src, $attr='', $dir='') { echo Horde::img($src, $attr, $dir); }
    
    /**
     * If magic_quotes_gpc is in use, run stripslashes() on $var.
     *
     * @param  string $var The string to un-quote, if necessary.
     * @return string      $var, minus any magic quotes.
     */
    function dispelMagicQuotes($var)
    {
        static $magic_quotes;
        
        if (!isset($magic_quotes))
            $magic_quotes = get_magic_quotes_gpc();
        
        if ($magic_quotes)
            $var = stripslashes($var);
        
        return $var;
    }
    
    /**
     * Get a form variable from GET or POST data, stripped of magic
     * quotes if necessary.
     *
     * @param string $var The name of the form variable to look for.
     * @param string $default (optional) The value to return if the
     *                                   variable is not there.
     *
     * @return string     The cleaned form variable, or $default.
     */
    function getFormData($var, $default = null)
    {
        if (isset($GLOBALS['HTTP_POST_VARS'][$var])) {
            return Horde::dispelMagicQuotes($GLOBALS['HTTP_POST_VARS'][$var]);
        } elseif (isset($GLOBALS['HTTP_GET_VARS'][$var])) {
            return Horde::dispelMagicQuotes($GLOBALS['HTTP_GET_VARS'][$var]);
        } else {
            return $default;
        }
    }
    
}


class SessionCache {
    
    function putObject($object)
    {
        $store = serialize($object);
        $objectID = md5($store);
        if (!isset($GLOBALS['HTTP_SESSION_VARS']['hordeObjectCache'])) {
            global $hordeObjectCache;
            $hordeObjectCache = array();
            $hordeObjectCache[$objectID] = $store;
            $GLOBALS['HTTP_SESSION_VARS']['hordeObjectCache'] = &$hordeObjectCache;
            session_register('hordeObjectCache');
        } else {
            /* Prune the cache if there are more than 20 items in it. */
            if (count($GLOBALS['HTTP_SESSION_VARS']['hordeObjectCache']) > 20) {
                array_shift($GLOBALS['HTTP_SESSION_VARS']['hordeObjectCache']);
            }
            $GLOBALS['HTTP_SESSION_VARS']['hordeObjectCache'][$objectID] = $store;
        }
        
        return $objectID;
    }
    
    function getObject($objectID)
    {
        if (!isset($GLOBALS['HTTP_SESSION_VARS']['hordeObjectCache'][$objectID])) {
            return false;
        }
        
        return unserialize($GLOBALS['HTTP_SESSION_VARS']['hordeObjectCache'][$objectID]);
    }
    
}


/**
 * HTTP_Cache:: provides a wrapper around php's output buffering
 * mechanisms and also does compression, generates headers - ETag,
 * Content-Length, etc. - which may be beneficial to bandwidth
 * usage and performance.
 *
 * @author Mark Nottingham <mnot@pobox.com>
 * @author Chuck Hagenbuch <chuck@horde.org>
 * @version $Revision: 1.78 $
 * @since Horde 1.3
 */
class HTTP_Cache {
    
    /**
     * Start the output buffer, and make sure that implicit flush is
     * off so that data is always buffered.
     */
    function start()
    {
        ob_start();
        ob_implicit_flush(0);
    }
    
    /**
     * Output the contents of the output buffer, compressed if
     * desired, along with any relevant headers.
     *
     * @param $compress (optional) Use gzip compression, if the browser supports it.
     * @param $use_etag Generate an ETag, and don't send the body if the browser has the same object cached.
     * @param $send_body Send the body of the request? Might be false for HEAD requests.
     */
    function output($compress = true, $use_etag = true, $send_body = true)
    {
        $min_gz_size = 1024;
        $page = ob_get_contents();
        $length = strlen($page);
        ob_end_clean();
        
        if ($compress && extension_loaded('zlib') && (strlen($page) > $min_gz_size) && isset($GLOBALS['HTTP_SERVER_VARS']['HTTP_ACCEPT_ENCODING'])) {
            $ae = explode(',', str_replace(' ', '', $GLOBALS['HTTP_SERVER_VARS']['HTTP_ACCEPT_ENCODING']));
            $enc = false;
            if (in_array('gzip', $ae)) {
                $enc = 'gzip';
            } else if (in_array('x-gzip', $ae)) {
                $enc = 'x-gzip';
            }
            
            if ($enc) {
                $page = gzencode($page);
                $length = strlen($page);
                header('Content-Encoding: ' . $enc);
                header('Vary: Accept-Encoding');
            } else {
                $compress = false;
            }
        } else {
            $compress = false;
        }
        
        if ($use_etag) {
            $etag = '"' . md5($page) . '"';
            header('ETag: ' . $etag);
            if (isset($GLOBALS['HTTP_SERVER_VARS']['HTTP_IF_NONE_MATCH'])) {
                $inm = explode(',', $GLOBALS['HTTP_SERVER_VARS']['HTTP_IF_NONE_MATCH']);
                foreach ($inm as $i) {
                    if (trim($i) == $etag) {
                        header('HTTP/1.0 304 Not Modified');
                        $send_body = false;
                        break;
                    }
                }
            }
        }
        
        if ($send_body) {
            header('Content-Length: ' . $length);
            echo $page;
        }
    }
    
}
?>
