<?php
// $Horde: horde/lib/Connection/file.php,v 1.2 2000/12/21 06:39:17 max Exp $

/**
 * Connection tracking implementation for stub files.
 *
 * Optional values for $params:
 *      'stub_dir'      The directory where to keep connection stub files.
 *      'timeout'       The period (in seconds) after which an id is purged.
 *
 * @author  Max Kalika <max@horde.org>
 * @version $Revision: 1.2 $
 * @since   Horde 1.3
 */
class Connection_file extends Connection {
    
    /** Handle for the open file descriptor. */
    var $fd = '';
    
    /** Boolean indicating whether or not we have an open file descriptor. */
    var $connected = false;
    
    /**
     * Create a new file based tracking storage container.
     *
     * @param $params   A hash containing storage parameters.
     */
    function Connection_file($params = array())
    {
        $this->params = $params;
        
        /* Choose the directory to save the stub files. */
        if (!isset($this->params['stub_dir'])) {
            if (ini_get('upload_tmp_dir')) {
                $this->params['stub_dir'] = ini_get('upload_tmp_dir');
            }
            elseif (getenv('TMPDIR')) {
                $this->params['stub_dir'] = getenv('TMPDIR');
            }
            else {
                $this->params['stub_dir'] = '/tmp';
            }
        }
        
        /* Set timeout to 24 hours if not specified. */
        if (!isset($this->params['timeout'])) {
            $this->params['timeout'] = 86400;
        }
    }
    
    /**
     * Opens a file descriptor to a new or existing file.
     *
     * @return bool         CONN_OK on success, CONN_ERROR_* on failure.
     */
    function connect()
    {
        if (!$this->connected) {
            
            /*
             * Open a file descriptor to the connection stub file.
             */
            $this->fd = fopen($this->params['stub_dir'] . "/conn_" . $this->hexRemoteAddr(), "a");
            if (!$this->fd) {
                return CONN_ERROR_CONNECT;
            }
            
            $this->connected = true;
        }
        
        return CONN_OK;
    }
    
    /**
     * Closes the file descriptor.
     *
     * @return bool         true on success, false on failure.
     */
    function disconnect()
    {
        if ($this->connected) {
            $this->connected = false;
            return fclose($this->fd);
        }
        
        return true;
    }
    
    /**
     * Deletes all expired connection id's from the SQL server.
     *
     * @return bool         CONN_OK on success, CONN_ERROR_* on failure.
     */
    function purge()
    {
        /* If we're already connected, disconnect as we will be unlinking files. */
        if ($this->connected) {
            if ($this->disconnect()) return CONN_ERROR_CONNECT;
        }
        
        /* Build stub file list. */
        if (!$dirFD = opendir($this->params['stub_dir'])) {
            return CONN_ERROR_CONNECT;
        }
        
        /* Find expired stub files */
        while (($dirEntry = readdir($dirFD)) != '') {
            if (preg_match('|^conn_\w{8}$|', $dirEntry) && (time() - filemtime($this->params['stub_dir'] . "/" . $dirEntry) >= $this->params['timeout'])) {
                if (!unlink($this->params['stub_dir'] . "/" . $dirEntry)) {
                    return CONN_ERROR;
                }
            }
        }
        
        closedir($dirFD);
        
        return CONN_OK;
    }
    
    function exists($connID)
    {
        /* If we're already connected, disconnect as we will be reading one file. */
        if ($this->connected) {
            if ($this->disconnect()) return CONN_ERROR_CONNECT;
        }
        
        /* Find already used IDs */
        $fileContents = @file($this->params['stub_dir'] . '/conn_' . $this->hexRemoteAddr());
        if ($fileContents) {
            for ($i = 0; $i < count($fileContents); $i++) {
                if (chop($fileContents[$i]) == $connID) {
                    return true;
                }
            }
        }
        
        return false;
    }
    
    function add($connID)
    {
        /* If we're not already connected, invoke the connect(). */
        if (!$this->connected) {
            if ($this->connect() != CONN_OK) return CONN_ERROR_CONNECT;
        }
        
        /* Write the entry. */
        fwrite($this->fd, "$connID\n");
        
        /* Return an error if the update fails, too. */
        if (!$this->disconnect()) return CONN_ERROR;
        
        return CONN_OK;
    }
    
}
?>
