<?php
// $Horde: horde/lib/Connection.php,v 1.3 2001/01/07 23:45:56 chuck Exp $

/* Return codes */
define('CONN_OK',               0);    // Operation succeeded
define('CONN_ERROR',           -1);    // operation failed
define('CONN_ERROR_PARAMS',    -2);    // Bad or missing parameters ($params)
define('CONN_ERROR_CONNECT',   -3);    // Connection failure
define('CONN_ERROR_AUTH',      -4);    // Authentication failure
define('CONN_ERROR_EMPTY',     -5);    // Empty retrieval result

/**
 * The Connection:: class provides a common abstracted interface into the
 * various connection tracking mediums. It also includes all of the
 * functions for retrieving, storing, and checking connection identities.
 *
 * @author  Max Kalika <max@horde.org>
 * @version $Revision: 1.3 $
 * @since   Horde 1.3
 */
class Connection {
    
    /**
     * Attempts to return a concrete Connection instance based on $driver.
     * 
     * @param $driver   The type of concrete Connection subclass to return.
     *                  This is based on the storage driver ($driver).  The
     *                  code is dynamically included.
     *
     * @param $params   (optional) A hash containing any additional
     *                  configuration or connection parameters a subclass
     *                  might need.
     *
     * @return          The newly created concrete Connection instance, or
     *                  false an error.
     */
    function factory($driver, $params = array())
    {
        if (empty($driver) || (strcasecmp($driver, 'none') == 0)) {
            return new Connection;
        }
        
        $classfile = dirname(__FILE__) . '/Connection/' . $driver . '.php';
        
        if (@is_readable($classfile)) {
            include_once $classfile;
            $class = 'Connection_' . $driver;
            return new $class($params);
        } else {
            return false;
        }
    }

    function hexRemoteAddr()
    {
        global $HTTP_SERVER_VARS;
        
        $addr = explode('.', $HTTP_SERVER_VARS['REMOTE_ADDR']);
        return sprintf("%02x%02x%02x%02x", $addr[0], $addr[1], $addr[2], $addr[3]);
    }
    
    /**
     * Generates a connection id and returns it.
     *
     * @return string   The generated id string.
     */
    function generateID()
    {
        return md5(time() . '][' . Connection::hexRemoteAddr());
    }
    
    /**
     * Checks if the given connection id has been previously used.
     * First purges all expired connection id's. Then retreives
     * current connections for the given ip address. If the specified
     * connection id was not found, adds it.
     *
     * @param $connID     The value of the connection id to check.
     *
     * @return bool       True if the connection id has not been used,
     *                    false otherwise.
     */
    function verify($connID)
    {
        $this->purge();
        if ($this->exists($connID)) {
            return false;
        }
        else {
            $this->add($connID);
            return true;
        }
    }

    /**
     * This is basically an abstract method that should be overridden by a
     * subclass implementation. It's here to retain code integrity in the
     * case that no subclass is loaded ($driver == 'none').
     */
    function exists()
    {
        return false;
    }

    /**
     * This is basically an abstract method that should be overridden by a
     * subclass implementation. It's here to retain code integrity in the
     * case that no subclass is loaded ($driver == 'none').
     */
    function add()
    {
        return true;
    }

    /**
     * This is basically an abstract method that should be overridden by a
     * subclass implementation. It's here to retain code integrity in the
     * case that no subclass is loaded ($driver == 'none').
     */
    function purge()
    {
        return true;
    }
}
?>
