<?php
// $Horde: chain/lib/Driver/sql.php,v 1.29 2002/06/14 17:57:52 chuck Exp $

/**
 * Chain storage implementation for PHP's PEAR database abstraction layer.
 *
 * Required values for $params:<pre>
 *      'phptype'                The database type (e.g. 'pgsql', 'mysql, etc.).
 *      'hostspec'               The hostname of the database server.
 *      'protocol'               The communication protocol ('tcp', 'unix', etc.).
 *      'username'               The username with which to connect to the database.
 *      'password'               The password associated with 'username'.
 *      'database'               The name of the database.
 *      'table'                  The name of the links table in 'database'.
 *      'table_categories'       The name of the categories table in 'database'.
 *      'table_link_listing'     The name of the link listing table in 'database'.
 *      'table_category_listing' The name of the category listing table in 'database'.</pre>
 *
 * Required by some database implementations:
 *      'options'       Additional options to pass to the database.
 *      'tty'           The TTY on which to connect to the database.
 *      'port'          The port on which to connect to the database.
 *
 * The table structure can be created by the scripts/drivers/chain_links.sql
 * script.
 *
 * @author  Jon Parise <jon@horde.org>
 * @author  Eric Rechlin <eric@hpcalc.org>
 * @version $Revision: 1.00 $
 * @since   Chain 0.1
 * @package chain
 */
class Chain_Driver_sql extends Chain_Driver {

    /** Hash containing connection parameters. */
    var $params = array();

    /** Handle for the current database connection.
        @var object DB $db */
    var $db;

    /** Boolean indicating whether or not we're connected to the SQL server. */
    var $_connected = false;


    /**
     * Constructs a new SQL storage object.
     *
     * @param string $user      The user who owns these links.
     * @param array  $params    A hash containing connection parameters.
     */
    function Chain_Driver_sql($user, $params = array())
    {
        $this->user = $user;
        $this->params = $params;
    }

    /**
     * Attempts to open a persistent connection to the SQL server.
     *
     * @return boolean    True on success; exits (Horde::fatal()) on error.
     */
    function _connect()
    {
        if (!$this->_connected) {
            if (!is_array($this->params)) {
                Horde::fatal(new PEAR_Error(_("No configuration information specified for SQL Links.")), __FILE__, __LINE__);
            }
            if (!isset($this->params['phptype'])) {
                Horde::fatal(new PEAR_Error(_("Required 'phptype' not specified in configuration.")), __FILE__, __LINE__);
            }
            if (!isset($this->params['hostspec'])) {
                Horde::fatal(new PEAR_Error(_("Required 'hostspec' not specified in configuration.")), __FILE__, __LINE__);
            }
            if (!isset($this->params['username'])) {
                Horde::fatal(new PEAR_Error(_("Required 'username' not specified in configuration.")), __FILE__, __LINE__);
            }
            if (!isset($this->params['password'])) {
                Horde::fatal(new PEAR_Error(_("Required 'password' not specified in configuration.")), __FILE__, __LINE__);
            }
            if (!isset($this->params['database'])) {
                Horde::fatal(new PEAR_Error(_("Required 'database' not specified in configuration.")), __FILE__, __LINE__);
            }
            if (!isset($this->params['table'])) {
                Horde::fatal(new PEAR_Error(_("Required 'table' not specified in configuration.")), __FILE__, __LINE__);
            }
            if (!isset($this->params['table_categories'])) {
                Horde::fatal(new PEAR_Error(_("Required 'table_categories' not specified in configuration.")), __FILE__, __LINE__);
            }

            /* Connect to the SQL server using the supplied parameters. */
            include_once 'DB.php';
            $this->db = &DB::connect($this->params, true);
            if (DB::isError($this->db) || DB::isWarning($this->db)) {
                Horde::fatal($this->db, __FILE__, __LINE__);
            }

            /* Enable the "portability" option. */
            $this->db->setOption('optimize', 'portability');

            $this->_connected = true;
        }

        return true;
    }

    /**
     * Disconnect from the SQL server and clean up the connection.
     *
     * @return boolean     true on success, false on failure.
     */
    function _disconnect()
    {
        if ($this->_connected) {
            $this->_connected = false;
            return $this->db->disconnect();
        }

        return true;
    }

    /**
     * Adds the given link to the database
     *
     * @param location Filename to local file or address to external file
     * @param title Name to use for link
     * @param category Category ID in which to list the link
     * @param subcategory Set to 1 if it should be treated as subcategory (optional)
     * @param desc Detailed description of link (optional)
     *
     * @return mixed  True on success, PEAR_Error on failure.
     */
    function addLink($location, $title, $category, $subcategory, $desc = "") {
        /* Make sure we have a valid database connection. */
        $this->_connect();

        $currtime = time();
        $query = sprintf(
            'insert into %s (link_location, link_title, ' .
            'link_desc, link_subcategory, link_date, link_category) ' .
            'values(%s, %s, %s, %d, %d, %d)',
            $this->params['table'],
            $this->db->quote($location),
            $this->db->quote($title),
            $this->db->quote($desc),
            $subcategory,
            $category,
            $currtime);
        
        /* Attempt the insertion query. */
        $result = $this->db->query($query);

        /* Return an error immediately if the query failed. */
        if (DB::isError($result)) {
            return $result;
        }

    }

     /**
     * Deletes the specified link from the database
     *
     * @param link_id ID number of link to delete
     *
     * @return mixed True on success, PEAR_Error on failure.
     */
    function removeLink($link_id) {
        /* Make sure we have a valid database connection. */
        $this->_connect();

        /* Delete from links table. */
        $query = sprintf(
            'delete from %s where link_id=%d',
            $this->params['table'],
            $link_id);

        /* Attempt the deletion query. */
        $result = $this->db->query($query);

        /* Return an error immediately if the query failed. */
        if (DB::isError($result)) {
            return $result;
        }

    }

    /**
     * Modifies the given link in the database
     *
     * @param link_id ID number of link to modify
     * @param location Filename to local file or address to external file
     * @param title Name to use for link
     * @param category Category ID in which to list the link
     * @param subcategory Set to 1 if it should be treated as subcategory (optional)
     * @param desc Detailed description of link (optional)
     *
     * @return mixed  True on success, PEAR_Error on failure.
     */
    function changeLink($link_id, $location, $title, $category, $subcategory = 0, $desc = "") {
        $currtime = time();
        $query = sprintf(
            'update %s set link_location=%s, link_title=%s, ' .
            'link_desc=%s, link_subcategory=%s, link_date=%d, link_category=%d where link_id=%d',
            $this->params['table'],
            $this->db->quote($location),
            $this->db->quote($title),
            $this->db->quote($desc),
            $subcategory,
            $currtime,
            $category,
            $link_id);
        /* Attempt the insertion query. */
        $result = $this->db->query($query);

        /* Return an error immediately if the query failed. */
        if (DB::isError($result)) {
            return $result;
        }

    }

    /**
     * Adds the given link to the today's links database
     *
     * @param location Filename to local file or address to external file
     * @param title Name to use for link
     * @param desc Detailed description of link (optional)
     *
     * @return mixed  True on success, PEAR_Error on failure.
     */
    function addToday($location, $title, $desc = "") {
        /* Make sure we have a valid database connection. */
        $this->_connect();

        $currtime = time();
        $query = sprintf(
            'insert into %s (today_location, today_title, ' .
            'today_desc, today_date) ' .
            'values(%s, %s, %s, %d)',
            $this->params['table_today'],
            $this->db->quote($location),
            $this->db->quote($title),
            $this->db->quote($desc),
            $currtime);
        
        /* Attempt the insertion query. */
        $result = $this->db->query($query);

        /* Return an error immediately if the query failed. */
        if (DB::isError($result)) {
            return $result;
        }

    }

    /**
     * Modifies the given link in the today's links database
     *
     * @param link_id ID number of link to modify
     * @param location Filename to local file or address to external file
     * @param title Name to use for link
     * @param desc Detailed description of link (optional)
     *
     * @return mixed  True on success, PEAR_Error on failure.
     */
    function changeToday($today_id, $location, $title, $desc = "") {
        /* Make sure we have a valid database connection. */
        $this->_connect();
        
        $currtime = time();
        $query = sprintf(
            'update %s set today_location=%s, today_title=%s, ' .
            'today_desc=%s, today_date=%s where today_id=%d',
            $this->params['table_today'],
            $this->db->quote($location),
            $this->db->quote($title),
            $this->db->quote($desc),
            $currtime,
            $today_id);

        /* Attempt the insertion query. */
        $result = $this->db->query($query);

        /* Return an error immediately if the query failed. */
        if (DB::isError($result)) {
            return $result;
        }

    }

     /**
     * Deletes the specified link from the today's links database
     *
     * @param link_id ID number of link to delete
     *
     * @return mixed True on success, PEAR_Error on failure.
     */
    function removeToday($today_id) {
        /* Make sure we have a valid database connection. */
        $this->_connect();

        /* Delete from links table. */
        $query = sprintf(
            'delete from %s where today_id=%d',
            $this->params['table_today'],
            $today_id);

        /* Attempt the deletion query. */
        $result = $this->db->query($query);

        /* Return an error immediately if the query failed. */
        if (DB::isError($result)) {
            return $result;
        }

    }

    /**
     * Adds the given category to the database
     *
     * @param title Name to use for category
     * @param category Category ID in which to list as subcategory
     * @param desc Detailed description of category (optional)
     * @param content Extended content, formatted as HTML or plain text (optional)
     *
     * @return mixed  True on success, PEAR_Error on failure.
     */
    function addCategory($title, $category, $desc = "", $content = "") {
        /* Make sure we have a valid database connection. */
        $this->_connect();

        $query = sprintf(
            'insert into %s (category_title, ' .
            'category_desc, category_content, category_category) ' .
            'values(%s, %s, %s, %d)',
            $this->params['table_categories'],
            $this->db->quote($title),
            $this->db->quote($desc),
            $this->db->quote($content),
            $category);

        /* Attempt the insertion query. */
        $result = $this->db->query($query);

        /* Return an error immediately if the query failed. */
        if (DB::isError($result)) {
            return $result;
        }

    }

     /**
     * Deletes the specified category from the database
     *
     * @param category_id ID number of category to delete
     *
     * @return mixed True on success, PEAR_Error on failure.
     */
    function removeCategory($category_id) {
        /* Make sure we have a valid database connection. */
        $this->_connect();

        /* Delete from categories table. */
        $query = sprintf(
            'delete from %s where category_id=%d',
            $this->params['table_categories'],
            $category_id);

        /* Attempt the deletion query. */
        $result = $this->db->query($query);

        /* Delete all contained links from links table. */
        $query = sprintf(
            'delete from %s where link_category=%d',
            $this->params['table'],
            $category_id);

        /* Attempt the deletion query. */
        $result = $this->db->query($query);

        /* Return an error immediately if the query failed. */
        if (DB::isError($result)) {
            return $result;
        }

        $query = sprintf('select category_id from %s where category_category=%d',
            $this->params['table_categories'],
            $category_id);

        /* Delete all contained categories from categories table. */
        $result = $this->db->query($query);

        if (isset($result) && !DB::isError($result)) {
            $row = $result->fetchRow(DB_FETCHMODE_ASSOC);
            if (DB::isError($row)) {
                return $row;
            }

            while ($row && !DB::isError($row)) {
                /* Create a new category based on this row's values. */
                $this->removeCategory($row['category_id']);

                /* Advance to the new row in the result set. */
                $row = $result->fetchRow(DB_FETCHMODE_ASSOC);
            }
            $result->free();
        } else {
            return $result;
        }
    }

    /**
     * Modifies the given category in the database.
     *
     * @param category_id ID number of category to modify
     * @param title Name to use for link
     * @param category Category ID in which to list as subcategory
     * @param desc Detailed description of category (optional)
     * @param content Extended content, formatted as HTML (optional)
     *
     * @return mixed  True on success, PEAR_Error on failure.
     */
    function changeCategory($category_id, $title, $category, $desc = "", $content = "", $content_html = 0) {
        /* Make sure we have a valid database connection. */
        $this->_connect();

        $query = sprintf(
            'update %s set category_title=%s, category_desc=%s, ' .
            'category_content=%s, category_category=%d where category_id=%d',
            $this->params['table_categories'],
            $this->db->quote($title),
            $this->db->quote($desc),
            $this->db->quote($content),
            $category,
            $category_id);

        /* Attempt the insertion query. */
        $result = $this->db->query($query);

        /* Return an error immediately if the query failed. */
        if (DB::isError($result)) {
            return $result;
        }

    }

   /**
    * Gets the links from the specified category.
    * Stores listing in superclass $links.
    *
    * @param category Category from which to retrieve links.
    *
    * @return True on success, PEAR_Error on failure.
    */
    function getLinks($category_id) {
        /* Make sure we have a valid database connection. */
        $this->_connect();

        $query = sprintf('select link_location, link_title, link_desc, ' .
            'link_date, link_id, link_subcategory from %s where link_category=%d',
            $this->params['table'],
            $category_id);

        /* Attempt the selection query. */
        $result = $this->db->query($query);

        if (isset($result) && !DB::isError($result)) {
            $row = $result->fetchRow(DB_FETCHMODE_ASSOC);
            if (DB::isError($row)) {
                return $row;
            }

            /* Store the retrieved values in a fresh $links list. */
            $this->links = array();
            while ($row && !DB::isError($row)) {
                /* Create a new link based on this row's values. */
                $link = array();
                $link['id'] = $row['link_id'];
                $link['title'] = $row['link_title'];
                $link['desc'] = $row['link_desc'];
                $link['location'] = $row['link_location'];
                $link['date'] = $row['link_date'];
                $link['subcategory'] = $row['link_subcategory'];

                /* Add this new link to the $links list. */
                $this->links[$row['link_id']] = $link;

                /* Advance to the new row in the result set. */
                $row = $result->fetchRow(DB_FETCHMODE_ASSOC);
            }
            $result->free();
        } else {
            return $result;
        }
    }

   /**
    * Gets today's links from the today table.
    * Stores listing in superclass $links.
    *
    * @return True on success, PEAR_Error on failure.
    */
    function getToday() {
        /* Make sure we have a valid database connection. */
        $this->_connect();

        $query = sprintf('select * from %s', $this->params['table_today']);

        /* Attempt the selection query. */
        $result = $this->db->query($query);

        if (isset($result) && !DB::isError($result)) {
            $row = $result->fetchRow(DB_FETCHMODE_ASSOC);
            if (DB::isError($row)) {
                return $row;
            }

            /* Store the retrieved values in a fresh $links list. */
            $this->today = array();
            while ($row && !DB::isError($row)) {
                /* Create a new link based on this row's values. */
                $item = array();
                $item['id'] = $row['today_id'];
                $item['title'] = $row['today_title'];
                $item['desc'] = $row['today_desc'];
                $item['location'] = $row['today_location'];
                $item['date'] = $row['today_date'];

                /* Add this new item to the $today list. */
                $this->today[$row['today_id']] = $item;

                /* Advance to the new row in the result set. */
                $row = $result->fetchRow(DB_FETCHMODE_ASSOC);
            }
            $result->free();
        } else {
            return $result;
        }
    }

   /**
    * Gets all information about the specified link.
    * Also includes an array of container categories.
    *
    * @param link_id ID number of link which to retrieve.
    *
    * @return True on success, PEAR_Error on failure.
    */
    function getLink($link_id) {
        /* Make sure we have a valid database connection. */
        $this->_connect();

        $query = sprintf('select link_location, link_title, link_desc, ' .
            'link_date, link_subcategory, link_category from %s where link_id=%d',
            $this->params['table'],
            $link_id);

        /* Attempt the selection query. */
        $result = $this->db->query($query);

        if (isset($result) && !DB::isError($result)) {
            $row = $result->fetchRow(DB_FETCHMODE_ASSOC);
            if (DB::isError($row)) {
                return $row;
            }

            /* Store the retrieved values in a fresh $links list. */
            $this->currlink = array();

            /* Create a new link based on this row's values. */
            $this->currlink['id'] = $link_id;
            $this->currlink['title'] = $row['link_title'];
            $this->currlink['desc'] = $row['link_desc'];
            $this->currlink['location'] = $row['link_location'];
            $this->currlink['date'] = $row['link_date'];
            $this->currlink['subcategory'] = $row['link_subcategory'];
            $this->currlink['category'] = $row['link_category'];

            $result->free();
            
            return true;
        } else {
            return $result;
        }
    }

   /**
    * Gets an array of all categories.
    *
    * @param link_id ID number of link which to retrieve.
    *
    * @return True on success, PEAR_Error on failure.
    */
    function getAllCategories() {
        /* Make sure we have a valid database connection. */
        $this->_connect();

        $query = sprintf('select * from %s', $this->params['table_categories']);

        /* Attempt the selection query. */
        $result = $this->db->query($query);

        if (isset($result) && !DB::isError($result)) {
            $row = $result->fetchRow(DB_FETCHMODE_ASSOC);
            if (DB::isError($row)) {
                return $row;
            }

            /* Store the retrieved values in a fresh $allCategories list. */
            $this->allCategories = array();
            while ($row && !DB::isError($row)) {
                /* Create a new category array based on this row's values. */
                $category = array();
                $category['id'] = $row['category_id'];
                $category['title'] = $row['category_title'];
                $category['desc'] = $row['category_desc'];

                /* Add this new link to the $links list. */
                $this->allCategories[$row['category_id']] = $category;

                /* Advance to the new row in the result set. */
                $row = $result->fetchRow(DB_FETCHMODE_ASSOC);
            }
            $result->free();
        } else {
            return $result;
        }
    }

   /**
    * Gets the subcategories from the specified category.
    * Stores listing in superclass $categories.
    *
    * @param category Category from which to retrieve subcategories.
    *
    * @return True on success, PEAR_Error on failure.
    */
    function getCategories($category_id) {
        /* Make sure we have a valid database connection. */
        $this->_connect();

        $query = sprintf('select category_id, category_title, category_desc ' .
            'from %s where category_category=%d',
            $this->params['table_categories'],
            $category_id);

        /* Attempt the selection query. */
        $result = $this->db->query($query);

        if (isset($result) && !DB::isError($result)) {
            $row = $result->fetchRow(DB_FETCHMODE_ASSOC);
            if (DB::isError($row)) {
                return $row;
            }

            /* Store the retrieved values in a fresh $categories list. */
            $this->categories = array();
            while ($row && !DB::isError($row)) {
                /* Create a new category based on this row's values. */
                $category = array();
                $category['id'] = $row['category_id'];
                $category['title'] = $row['category_title'];
                $category['desc'] = $row['category_desc'];

                /* Add this new category to the $categories list. */
                $this->categories[$row['category_id']] = $category;

                /* Advance to the new row in the result set. */
                $row = $result->fetchRow(DB_FETCHMODE_ASSOC);
            }
            $result->free();
        } else {
            return $result;
        }
    }

   /**
    * Gets all information about the specified category.
    * Stores listing in superclass $category.
    *
    * @param category Category about which to retrieve information.
    *
    * @return True on success, PEAR_Error on failure.
    */
    function getCategory($category_id) {
        /* Make sure we have a valid database connection. */
        $this->_connect();

        $query = sprintf('select * from %s where category_id=%d',
            $this->params['table_categories'],
            $category_id);
        /* Attempt the selection query. */
        $result = $this->db->query($query);

        if (isset($result) && !DB::isError($result)) {
            $row = $result->fetchRow(DB_FETCHMODE_ASSOC);
            if (DB::isError($row)) {
                return $row;
            }

            /* Store the retrieved values in a fresh $categories list. */
            $this->category = array();
            $this->category['id'] = $row['category_id'];
            $this->category['title'] = $row['category_title'];
            $this->category['desc'] = $row['category_desc'];
            $this->category['content'] = $row['category_content'];
            $this->category['category'] = $row['category_category'];

            $result->free();
            
            return true;
        } else {
            return $result;
        }
    }

}
?>
