<?php
// $Horde: chain/lib/Chain.php,v 1.43 2002/06/06 12:48:17 bjn Exp $

require_once CHAIN_BASE . '/lib/constants.php';

/**
 * Chain Base Class.
 *
 * @author Jon Parise <jon@horde.org>
 * @version $Revision: 1.43 $
 * @package chain
 */
class Chain {

    /**
     * Retrieves the current user's link list from storage.
     * This function will also sort the resulting list, if requested.
     *
     * @param object Chain_Driver  $storage   The current storage object.
     * @param constant $sortby   (optional) The field by which to sort.
     *                           (CHAIN_SORT_PRIORITY, CHAIN_SORT_NAME
     *                            CHAIN_SORT_DUE, CHAIN_SORT_COMPLETION)
     * @param constant $sortdir  (optional) The direction by which to sort.
     *                           (CHAIN_SORT_ASC, CHAIN_SORT_DESC)
     *
     * @return array        Returns a list of the requested links.
     *
     * @see Chain_Driver::listLinks()
     */
    function listLinks($storage, $sortby = CHAIN_SORT_TITLE,
                       $sortdir = CHAIN_SORT_ASCEND) {
        global $prefs;

        /* Retrieve the link list for storage. */
        $links = $storage->listLinks();

        /* Sort the link list. */
        $sort_functions = array(
            CHAIN_SORT_TITLE => 'ByTitle',
            CHAIN_SORT_DATE => 'ByDate',
            CHAIN_SORT_LOCATION => 'ByLocation',
            CHAIN_SORT_DESCRIPTION => 'ByDescription'
        );

        /* Sort the array if we have a sort function defined for this field. */
        if (isset($sort_functions[$sortby])) {
            $prefix = ($sortdir == CHAIN_SORT_DESCEND) ? '_rsort' : '_sort';
            uasort($links, $prefix . $sort_functions[$sortby]);
        }

        return $links;
    }

    /**
     * Retrieves today's featured link list from storage.
     * This function will also sort the resulting list, if requested.
     *
     * @param object Chain_Driver  $storage   The current storage object.
     * @param constant $sortby   (optional) The field by which to sort.
     *                           (CHAIN_SORT_PRIORITY, CHAIN_SORT_NAME
     *                            CHAIN_SORT_DUE, CHAIN_SORT_COMPLETION)
     * @param constant $sortdir  (optional) The direction by which to sort.
     *                           (CHAIN_SORT_ASC, CHAIN_SORT_DESC)
     *
     * @return array        Returns a list of the requested links.
     *
     * @see Chain_Driver::listToday()
     */
    function listToday($storage, $sortby = CHAIN_SORT_TITLE,
                       $sortdir = CHAIN_SORT_ASCEND) {
        /* Retrieve the link list for storage. */
        $links = $storage->listToday();

        /* Sort the link list. */
        $sort_functions = array(
            CHAIN_SORT_TITLE => 'ByTitle',
            CHAIN_SORT_DATE => 'ByDate',
            CHAIN_SORT_LOCATION => 'ByLocation',
            CHAIN_SORT_DESCRIPTION => 'ByDescription'
        );

        /* Sort the array if we have a sort function defined for this field. */
        if (isset($sort_functions[$sortby])) {
            $prefix = ($sortdir == CHAIN_SORT_DESCEND) ? '_rsort' : '_sort';
            uasort($links, $prefix . $sort_functions[$sortby]);
        }

        return $links;
    }

    /**
     * Retrieves the current category list from storage.
     *
     * @param object Chain_Driver  $storage   The current storage object.
     *
     * @return array        Returns a list of the requested categories.
     *
     * @see Chain_Driver::listCategories()
     */
    function listCategories($storage, $sortby = CHAIN_SORT_TITLE,
                       $sortdir = CHAIN_SORT_ASCEND) {

        global $prefs;

        /* Retrieve the category list from storage. */
        $categories = $storage->listCategories();
        
        /* Sort the category list. */
        $sort_functions = array(
            CHAIN_SORT_TITLE => 'ByTitle',
            CHAIN_SORT_DESCRIPTION => 'ByDescription'
        );

        /* Sort the array if we have a sort function defined for this field. */
        if (isset($sort_functions[$sortby])) {
            $prefix = ($sortdir == CHAIN_SORT_DESCEND) ? '_rsort' : '_sort';
            uasort($categories, $prefix . $sort_functions[$sortby]);
        }

        return $categories;
    }

    /**
     * Retrieves the current category info from storage.
     *
     * @param object Chain_Driver  $storage   The current storage object.
     *
     * @return array        Returns a list of the category information.
     *
     * @see Chain_Driver::listCategory()
     */
    function listCategory($storage) {

        /* Retrieve the category information from storage. */
        return $storage->listCategory();
    }

    /**
     * Checks whether a link points to a local file instead of an Internet link.
     *
     * @param link The target address or file of the link.
     *
     * @return integer        Returns 1 if local file, 0 if Internet link.
     *
     * @see Chain_Driver::listCategory()
     */
    function isLocalFile($link) {

        if (strstr($link, "://")) {
            return 0;
        } else {
            return 1;
        }
    }

    /**
     * Builds the HTML for a link category widget.
     *
     * @param string  $name         The name of the widget.
     * @param integer $selected     The default category.
     *
     * @return string       The HTML <select> widget.
     */
/*    function buildCategoryWidget($name, $allCategories, $selectedCategories)
    {
        $html = "<select multiple=\"multiple\" size=6 name=\"" . $name . "[]\">";
        foreach ($allCategories as $cat) {
            $html .= "<option value=\"" . $cat['id'] . "\"";
            $html .= (in_array($cat['id'], $selectedCategories)) ? ' selected="selected">' : '>';
            $html .= $cat['title'] . '</option>';
        }
        $html .= '</select>';

        return $html;
    }*/

    /**
     * Formats the given Unix-style date string.
     *
     * @param string $unixdate     The Unix-style date value to format.
     *
     * @return string       The formatted due date string.
     */
    function formatDate($unixdate = '')
    {
        global $conf;

        if (empty($unixdate)) {
            return '';
        }

        $s  = strftime($conf['view']['date_format'], $unixdate);
        $s .= ' ' . _("at") . ' ';
        $s .= strftime($conf['view']['time_format'], $unixdate);

        return $s;
    }

    /**
     * Returns the string matching the given category ID.
     *
     * @param integer $categoryID     The category ID to look up.
     *
     * @return string       The formatted category string.
     */
    function formatCategory($categoryID = 0)
    {
        $categories = Chain::listCategories();
        return isset($categories[$categoryID]) ?
            $categories[$categoryID] :
            $categories[0];
    }

}

/**
 * Comparison function for sorting links by name.
 *
 * @param array $a  Link one.
 * @param array $b  Link two.
 *
 * @return integer  1 if link one is greater, -1 if link two is greater; 0 if they are equal.
 */
function _sortByTitle($a, $b)
{
    return strcmp($a['title'], $b['title']);
}

/**
 * Comparison function for reverse sorting links by name.
 *
 * @param array $a  Link one.
 * @param array $b  Link two.
 *
 * @return integer  -1 if link one is greater, 1 if link two is greater; 0 if they are equal.
 */
function _rsortByTitle($a, $b)
{
    return strcmp($b['title'], $a['title']);
}

/**
 * Comparison function for sorting links by location.
 *
 * @param array $a  Link one.
 * @param array $b  Link two.
 *
 * @return integer  1 if link one is greater, -1 if link two is greater; 0 if they are equal.
 */
function _sortByLocation($a, $b)
{
    return strcmp($a['location'], $b['location']);
}

/**
 * Comparison function for reverse sorting links by location.
 *
 * @param array $a  Link one.
 * @param array $b  Link two.
 *
 * @return integer  -1 if link one is greater, 1 if link two is greater; 0 if they are equal.
 */
function _rsortByLocation($a, $b)
{
    return strcmp($b['location'], $a['location']);
}

/**
 * Comparison function for sorting links by description.
 *
 * @param array $a  Link one.
 * @param array $b  Link two.
 *
 * @return integer  1 if link one is greater, -1 if link two is greater; 0 if they are equal.
 */
function _sortByDescription($a, $b)
{
    return strcmp($a['description'], $b['description']);
}

/**
 * Comparison function for reverse sorting links by description.
 *
 * @param array $a  Link one.
 * @param array $b  Link two.
 *
 * @return integer  -1 if link one is greater, 1 if link two is greater; 0 if they are equal.
 */
function _rsortByDescription($a, $b)
{
    return strcmp($b['description'], $a['description']);
}

/**
 * Comparison function for sorting links by modification date.
 *
 * @param array $a  Link one.
 * @param array $b  Link two.
 *
 * @return integer  1 if link one is greater, -1 if link two is greater; 0 if they are equal.
 */
function _sortByDate($a, $b)
{
    if ($a['date'] == $b['date']) return 0;
    return ($a['date'] > $b['date']) ? -1 : 1;
}

/**
 * Comparison function for reverse sorting links by modification date.
 *
 * @param array $a  Link one.
 * @param array $b  Link two.
 *
 * @return integer  -1 if link one is greater, 1 if link two is greater; 0 if they are equal.
 */
function _rsortByDate($a, $b)
{
    if ($a['date'] == $b['date']) return 0;
    return ($a['date'] < $b['date']) ? -1 : 1;
}


?>
